/* irqhigh.c -- set high priority IRQ's */

/*
// Copyright 1996, 1997 by Craig Estey -- See the file COPYING for details
*/

#include <irqhigh.h>
#include <asm/io.h>

#if defined(__GNUC__) && (! defined(__OPTIMIZE__))
#error irqhigh: yes, we really need the optimizer enabled
#endif

/* current value */
u_char irqhigh_list[2] = { IRQHIGH_MST_DFT, IRQHIGH_SLV_DFT };

/* list of resultant priorites */
u_char irq_prior_list[IRQHIGH_MAX];
u_char irq_prior_bias[IRQHIGH_MAX];

/* irqhigh_offset -- compute offset into irqhigh_list */
extern inline u_char
irqhigh_offset(u_char irq_no)
{

	irq_no >>= 3;

	return irq_no;
}

/* irqhigh_calc -- calculate priority table */
static void
irqhigh_calc(void)
{
	u_char mst_no;
	u_char mst_idx;
	u_char slv_no;
	u_char slv_idx;
	u_char prior;
	u_char bias;

	prior = 0;
	bias = 0;

	/* start at master highest */
	mst_no = irqhigh_list[0];
	for (mst_idx = 0;  mst_idx <= 7;  ++mst_idx) {
		irq_prior_list[mst_no] = prior++;
		irq_prior_bias[mst_no] = bias;

		/* when we hit the slave cascade, process slave, then resume master */
		if (mst_no == 2) {
			bias = 1;
			slv_no = irqhigh_list[1];
			for (slv_idx = 0;  slv_idx <= 7;  ++slv_idx) {
				irq_prior_list[slv_no] = prior++;
				irq_prior_bias[slv_no] = bias;
				slv_no = (slv_no + 1) & 0x0F;
				if (slv_no == 0)
					slv_no = IRQHIGH_SLV_LO;
			}
		}

		mst_no = (mst_no + 1) & 0x07;
	}
}

/* irqhigh_insane -- check request sanity */
/* RETURNS: error string (or NULL) */
static char *
irqhigh_insane(u_char mst_no,u_char slv_no)
{
	char *err;
	u_char off;

	err = NULL;

	do {
		/* check for master within range */
		if (mst_no >= IRQHIGH_SLV_HI) {
			err = "master value out of range";
			break;
		}

		/* don't check slave value if master is actually a slave */
		if (RNGE(slv_no,IRQHIGH_SLV_LO,IRQHIGH_SLV_HI))
			break;

		/* check slave value */
		if (! RNGE(slv_no,IRQHIGH_SLV_LO,IRQHIGH_SLV_HI)) {
			err = "slave value out of range";
			break;
		}

		/* insure that we have a genuine master/slave combination */
		off = irqhigh_offset(mst_no);
		if (off == irqhigh_offset(slv_no)) {
			if (off > 0)
				err = "two slaves specified";
			else
				err = "two masters specified";
			break;
		}
	} while (0);

	return err;
}

/* irqhigh_set1 -- set highest priority IRQ device on given 8259 */
static void
irqhigh_set1(int goflg,u_char irq_no)
{
	u_char lo;
	u_short port;
	u_char idx;

	/* decide on master vs. slave 8259 */
	idx = irqhigh_offset(irq_no);
	if (idx > 0) {
		port = 0xA0;
		irqhigh_set1(goflg,2);
	}

	/* master */
	else
		port = 0x20;

	irqhigh_list[idx] = irq_no;
	irq_no &= 0x07;

	/* set specific priority (of lowest priority device) */
	if (goflg) {
		/*
		// on the odd chance that the PIC state is indeterminate.  that is:
		// (1)  somebody did an outb to select an alternate PIC register
		// (2A) forgot to do the followup inb/outb.
		// (2B) this was done from task level _without_ interrupts locked
		// in that case, we'd be writing to the wrong register.  this is
		// unlikely because (2A) and (2B) are bugs, but this is CYA
		*/
		inb(port);

		lo = (irq_no - 1) & 0x07;
		lo |= 0xC0;
		outb(lo,port);
	}
}

/* irqhigh_set -- set highest priority IRQ device for both 8259's */
/* RETURNS: error code if things don't match */
static int
irqhigh_set(int goflg,u_char mst_no,u_char slv_no)
{
	int err;
	int slvflg;

	/* order is important -- set slave then master */
	slvflg = (irqhigh_offset(mst_no) == 0);
	if (slvflg)
		irqhigh_set1(goflg,slv_no);
	irqhigh_set1(goflg,mst_no);

	/* check for errors */
	err = -EINVAL;
	do {
		/* check slave number */
		if (slvflg) {
			if (irqhigh_list[1] != slv_no)
				break;
		}

		/* check master number */
		if (irqhigh_list[0] != mst_no)
			break;

		/* no errors */
		err = 0;
	} while (0);

	return err;
}

/* irqhigh_init -- initialize to default */
extern inline void
irqhigh_init(int goflg)
{

	/* set standard COM ports highest on master and disk controller on slave */
	irqhigh_set(goflg,IRQHIGH_MST_DFT,IRQHIGH_SLV_DFT);
}
