/* lib/file -- irqtune support library (file routines) */

#pragma member irqtune.h

/* filecpy -- copy file */
/* RETURNS: error code */
int
filecpy(int goflg,const char *dirdst,const char *dirsrc,const char *tail)
{
	int perm;
	int srcunit;
	int dstunit;
	int err;
	int rlen;
	struct stat stdst;
	struct stat stsrc;
	struct stat stself;
	char dstfile[1024];
	char srcfile[1024];

	do {
		/* insure self exists (and it's a directory) */
		err = stat(dirtop,&stself);
		if (err < 0) {
			zprtx("filecpy: unable to stat self '%s' -- %s\n",
				dirtop,strerror(errno));
			break;
		}
		if (! S_ISDIR(stself.st_mode)) {
			zprtx("filecpy: self is not a directory '%s'\n",
				dirtop);
			err = -1;
			break;
		}

		/* insure destination exists (and it's a directory) */
		err = stat(dirdst,&stdst);
		if (err < 0) {
			zprtx("filecpy: unable to stat destination '%s' -- %s\n",
				dirdst,strerror(errno));
			break;
		}
		if (! S_ISDIR(stdst.st_mode)) {
			zprtx("filecpy: destination is not a directory '%s'\n",
				dirdst);
			err = -1;
			break;
		}

		/* insure self/destination directories are different */
		if ((stdst.st_dev == stself.st_dev) &&
			(stdst.st_ino == stself.st_ino)) {
			zprtx("filecpy: self '%s' and destination '%s' are the same\n",
				dirtop,dirdst);
			err = -1;
			break;
		}

		if (dirsrc != NULL) {
			/* insure source exists (and it's a directory) */
			err = stat(dirsrc,&stsrc);
			if (err < 0) {
				zprtx("filecpy: unable to stat source '%s' -- %s\n",
					dirsrc,strerror(errno));
				break;
			}
			if (! S_ISDIR(stsrc.st_mode)) {
				zprtx("filecpy: source is not a directory '%s'\n",
					dirdst);
				err = -1;
				break;
			}

			/* insure source/destination directories are different */
			if ((stdst.st_dev == stsrc.st_dev) &&
				(stdst.st_ino == stsrc.st_ino)) {
				zprtx("filecpy: source '%s' and destination '%s' are the same\n",
					dirsrc,dirdst);
				err = -1;
				break;
			}
		}
	} while (0);

	dstunit = -1;
	srcunit = -1;
	do {
		/* bug out if error */
		if (err < 0)
			break;

		/* bug out if nogo mode */
		if (! goflg)
			break;

		/* show files we're copying */
		sprintf(dstfile,"%s/%s",dirdst,tail);
		zprtx("filecpy: %s",dstfile);
		if (dirsrc != NULL) {
			sprintf(srcfile,"%s/%s",dirsrc,tail);
			zprtx(" <-- %s",srcfile);
		}
		zprtx(" ...\n");

		/* get permissions (mimic source if possible) */
		perm = 0755;
		if (dirsrc != NULL)
			perm = stsrc.st_mode & 0777;

		/* open/create the destination file */
		dstunit = open(dstfile,O_WRONLY | O_CREAT | O_TRUNC,perm);
		if (dstunit < 0) {
			zprtx("filecpy: unable to open destination '%s' -- %s\n",
				dstfile,strerror(errno));
			err = -1;
			break;
		}

		/* force correct permissions (in case of a prior bogus install) */
		err = fchmod(dstunit,perm);
		if (err < 0) {
			zprtx("install: unable change permissions on '%s' -- %s\n",
				dstfile,strerror(errno));
			break;
		}

		/* just open the destination for caller */
		if (dirsrc == NULL)
			break;

		/* open the source */
		srcunit = open(srcfile,O_RDONLY);
		if (srcunit < 0) {
			zprtx("filecpy: unable to open source '%s' -- %s\n",
				srcfile,strerror(errno));
			break;
		}

		/* copy the data */
		err = 0;
		while (1) {
			/* read a chunk */
			rlen = read(srcunit,bigbf,sizeof(bigbf));
			if (rlen < 0) {
				zprtx("filecpy: read error on '%s' -- %s\n",
					srcfile,strerror(errno));
				err = -1;
				break;
			}
			if (rlen == 0)
				break;

			/* write a chunk */
			err = write(dstunit,bigbf,rlen);
			if (err < 0) {
				zprtx("filecpy: write error on '%s' -- %s\n",
					dstfile,strerror(errno));
				break;
			}
		}
	} while (0);

	/* close units */
	if ((dirsrc != NULL) || (err < 0)) {
		if (dstunit >= 0)
			close(dstunit);
		if (srcunit >= 0)
			close(srcunit);
	}

	/* return open destination unit to caller */
	else
		err = dstunit;

	return err;
}

/* filehead -- get head of name */
/* RETURNS: pointer to head */
char *
filehead(char *name)
/* name -- name to get head of */
{
	char *head;

	head = strrchr(name,'/');
	if (head != NULL)
		*head = 0;

	return name;
}

/* filetail -- get tail of name */
/* RETURNS: pointer to tail */
char *
filetail(char *name)
/* name -- name to get tail of */
{
	char *tail;

	tail = strrchr(name,'/');
	if (tail != NULL)
		++tail;
	else
		tail = name;

	return tail;
}

/* pathdir -- insure a directory is in $PATH */
/* RETURNS: pointer to $PATH match (or NULL) */
char *
pathdir(const char *dir)
{
	char *bp;
	char *cp;

	/* get path value */
	cp = getenv("PATH");
	strcpy(pathtmp,cp);

	bp = pathtmp;
	while (1) {
		/* get next element of the path */
		cp = strtok(bp,":");
		if (cp == NULL)
			break;
		bp = NULL;

		/* bug out if we found it */
		if (strcmp(dir,cp) == 0)
			break;
	}

	/* add directory to path (this is the probable place for insmod et. al.) */
	if (cp == NULL) {
		cp = getenv("PATH");
		sprintf(pathtmp,"PATH=%s:%s",cp,dir);

		/* NOTE: putenv won't strdup (e.g. pathbf must stay stable) */
		strcpy(pathbf,pathtmp);
		putenv(pathbf);
	}

	return cp;
}

/* pathlook -- lookup file in $PATH */
/* RETURNS: pointer to directory name (or NULL) */
char *
pathlook(const char *tail,const char *dir)
/* tail -- file to locate */
{
	char *cp;
	char *bp;
	int err;
	struct stat mystat;
	char file[1024];

	/* create path to program */
	if (dir == NULL)
		dir = getenv("PATH");
	strcpy(pathtmp,dir);

	bp = pathtmp;
	while (1) {
		/* get next element of the path */
		cp = strtok(bp,":");
		if (cp == NULL)
			break;
		bp = NULL;

		/* create file name */
		sprintf(file,"%s/%s",cp,tail);

		/* ignore nonexistent file */
		err = stat(file,&mystat);
		if (err < 0)
			continue;

		/* only want regular files */
		if (! S_ISREG(mystat.st_mode))
			continue;

		/* it must be executable by us */
		err = access(file,X_OK);
		if (err >= 0)
			break;
	}

	/* copy result to stable area */
	if (cp != NULL) {
		strcpy(bigbf,cp);
		cp = bigbf;
	}

	return cp;
}

