/**********************************************************************
$Id: g-scan.c,v 1.1 1999/05/30 01:51:00 linas Exp $
Copyright (C) 1997,1998 Christopher Lee
 
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with this software; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#define MAXARGS  100
#define MAXDECLS 1000

typedef char * pchar;

typedef struct {
  int nargs;
  pchar rettype, fnname, *args, *argtypes, flags, description;
} fndec_st;
typedef fndec_st * pfndec_st;

static pchar export_header = "\n/*[EXPORT";
static pchar export_tail   = "*/";

pchar get_string(int size) {
  return (pchar)malloc(((size+20)/8) * 8);
}

static int cur_line = 1;
static char *filename = NULL;

int get_char(FILE *file) {
  int ret = fgetc(file);
  if ( ret == '\n' ) { cur_line += 1; }
  return ret;
}

int unget_char(int ch, FILE *file) {
  if ( ch == '\n' ) { cur_line -= 1; }
  return ungetc(ch,file);
}

void quit_with_error(char *msg)
{
  char *errsym = "%%%error";
  printf("(%s \"%s: %d: %s\")\n",errsym,filename,cur_line,msg);
  exit(1);
}

int position_after_string(FILE *file, pchar str)
{
  int ch;
  int i = 0;
  int init_line, init_pos, failpos, fail_line;

  init_pos = ftell(file); init_line = cur_line;
  while ( str[i] ) {
    ch = get_char(file);
    if ( EOF == ch ) { 
      cur_line = init_line; fseek(file,init_pos,SEEK_SET);
      return 0;
    }
    if ( str[i] == ch ) {
      if ( 0 == i ) { failpos = ftell(file); fail_line = cur_line; }
      i++;
    } else {
      if ( i > 0 ) { 
	unget_char(ch,stdout);
	fseek(file,failpos,SEEK_SET);
	cur_line = fail_line;
      }
      i = 0;
    }
  }
  return 1;
}

int skip_ws(FILE *file)
{
  int ch;
  for ( ch = get_char(file); EOF != ch; ch = get_char(file) ) {
    switch (ch) {
    case ' ': case '\t': case '\n': case '\r':
      break;
    default:
      unget_char(ch,file);
      return 1;
    }
  }
  return 0; /*EOF*/
}

pchar get_name(FILE *file)
{
  char buffer[512];
  pchar pret;
  int ch, i;
  skip_ws(file);
  for ( i = 0, ch = get_char(file); 
	i<511 && EOF!=ch; 
	i++, ch = get_char(file) ) {
    buffer[i] = ch;
    switch (ch) {
    case '*':
      if ( 0 == i ) {
	buffer[1] = (char)0;
	pret = get_string(2);
	strcpy(pret,buffer);
	return pret;
      }
      /*else fallthrough*/
    case ' ': case '\t': case '\n': case '\r': case ',': case '(': case ')':
      if ( 0 == i ) { return NULL; }
      unget_char(ch,file);
      buffer[i] = (char)0;
      pret = get_string(i+1);
      strcpy(pret,buffer);
      return pret;
    default:
      break;
    }
  }
  return NULL; /*EOF*/
}

int get_token(FILE *file)
{
  int ch;
  if ( !skip_ws(file) ) { return 0; }
  ch = get_char(file);
  switch (ch) {
  case '(':  case ')':  case ',':
    return (char)ch;
  default:
    unget_char(ch,file);
    return (char)0;
  }
}

/* Prints description, skipping up to 3 characters of spaces and '*' at the
 * beginning of each line
 */
void display_description(char *desc)
{
  char *loc, *lastloc;
  int i;
  loc = desc;
  lastloc = loc;
  while ( *loc ) {
    for ( i = 0; i < 3; i++ ) {
      switch (*loc) {
      case 0:
	return;
      case ' ': case '*':
	loc++;
      default:
	break;
      }
    }
    lastloc = loc;
    for ( ; *loc && *loc != '\n'; loc++ ) { /* nothing */ }
    fwrite(lastloc,loc-lastloc,1,stdout);
    fputc('\n',stdout);
    lastloc = loc;
    if ( *loc ) { loc++; }
  }
}

void pointify_type(char* type)
{
  int size = strlen(type);
  type[size] = '*';
  type[size+1] = (char)0;
}

int get_name_and_type(FILE *file, pchar *pname, pchar *ptype) {
  *ptype = get_name(file);
  if ( !*ptype ) { return 0; }
  *pname = get_name(file);
  if ( !*pname ) { free(*ptype); return 0; }
  while ( '*' == (*pname)[0] ) {
    pointify_type(*ptype);
    free(*pname);
    *pname = get_name(file);
    if ( !*pname ) { free(*ptype); return 0; }
  }
  return 1;
}

int check_void(FILE *file) {
  int  loc  = ftell(file);
  int  nlines = cur_line;
  char *tmpstr = get_name(file);
  int  ret  = !strcmp(tmpstr,"void");
  fseek(file,loc,SEEK_SET);
  cur_line = nlines;
  free(tmpstr);
  return ret;
}

fndec_st* get_fndeclaration(FILE *file)
{
  pchar rettype, fnname, args[MAXARGS], argtypes[MAXARGS];
  int   i, nargs = 0;
  fndec_st *fndec;

  if ( !get_name_and_type(file,&fnname,&rettype) ) { return 0; }
  if ( '(' != get_token(file) ) { goto cleanup; }
  if ( check_void(file) ) { nargs = -1; goto done; }
  for ( nargs = 0; nargs < MAXARGS; nargs++ ) {
    if ( !get_name_and_type(file,args+nargs,argtypes+nargs) ) { goto cleanup; }
    switch ( get_token(file) ) {
    case ',':
      continue;
    case ')':
      goto done;
    }
  }
cleanup:
  if ( rettype ) { free(rettype); }
  if ( fnname )  { free(fnname); }
  for ( i = 0; i < nargs; i++ ) {
    free(args[i]);
    free(argtypes[i]);
  }
  return 0;
done:
  fndec = (fndec_st*)malloc(sizeof(fndec_st));
  fndec->rettype  = rettype;
  fndec->fnname   = fnname;
  fndec->nargs    = nargs+1;
  fndec->args     = (pchar*)malloc(fndec->nargs*sizeof(pchar));
  fndec->argtypes = (pchar*)malloc(fndec->nargs*sizeof(pchar));
  memcpy(fndec->args,args,fndec->nargs*sizeof(pchar));
  memcpy(fndec->argtypes,argtypes,fndec->nargs*sizeof(pchar));
  return fndec;
}

char* grab_text_between(FILE *file, long begloc, long endloc, int endsize)
{
  pchar retstr;
  int   strsize = endloc - begloc - endsize;
  retstr = get_string(strsize);
  fseek(file,begloc,SEEK_SET);
  fread(retstr,strsize,1,file);
  retstr[strsize] = 0;
  fseek(file,endloc,SEEK_SET);
  return retstr;
}

void free_fndecl(fndec_st *fndecl) {
  int i;
  free(fndecl->rettype);
  free(fndecl->fnname);
  free(fndecl->flags);
  free(fndecl->description);
  for ( i = 0; i < fndecl->nargs; i++ ) {
    free(fndecl->args[i]);
    free(fndecl->argtypes[i]);
  }
  free(fndecl->args);
  free(fndecl->argtypes);
  free(fndecl);
}

int get_declarations(char *filename, pfndec_st **ppret)
{
  FILE *file = fopen(filename, "r");
  int ndecls;
  long begloc, endloc;
  pchar flags, description;
  pfndec_st fndecls[MAXDECLS];
  
  if ( !file ) { return 0; }
  for ( ndecls = 0; ndecls < MAXDECLS; ndecls++ ) {
    if ( !position_after_string(file,export_header) ) {
      /* No more functions to export -- return the ones we found */
      *ppret = (fndec_st**)malloc(ndecls*sizeof(fndec_st*));
      memcpy(*ppret,fndecls,ndecls*sizeof(fndec_st*));
      return ndecls;
    }
    /* Found another function to export */
    skip_ws(file);
    begloc = ftell(file);
    if ( !position_after_string(file,"]") ) { 
      quit_with_error("could not locate ']' to close flags region");
    }
    endloc = ftell(file);
    flags = grab_text_between(file, begloc, endloc, 1);
    skip_ws(file);
    begloc = ftell(file);
    if ( !position_after_string(file,export_tail)  ) {
      quit_with_error("could not locate end of export comment");
    }
    endloc = ftell(file);
    description = grab_text_between(file, begloc, endloc, strlen(export_tail));
    fndecls[ndecls] = get_fndeclaration(file);
    if ( !fndecls[ndecls] ) {
      quit_with_error("error parsing function declaration");
    }
    fndecls[ndecls]->flags = flags;
    fndecls[ndecls]->description = description;
  }
  quit_with_error("Out of room for more function declarations");
  return 0;
}

void schemeify_name(pchar buff, pchar name)
{
  int   i, len = strlen(name);
  for ( i = 0; i < len; i++ ) {
    buff[i] = ((name[i]=='_') ? '-' : name[i]);
  }
  buff[i] = (char)0;
}

/*
 * (new-function 'symbol 'rettype "name" '((type1 fn1) ...) "description)
 */
void write_fn_info(pfndec_st pdecl)
{
  int i;
  char buffer[256];

  schemeify_name(buffer,pdecl->fnname);
  printf("(%s\n"
	 " (%s)\n"
	 " %s \"%s\" (", buffer, pdecl->flags,pdecl->rettype, pdecl->fnname);
  for ( i = 0; i < pdecl->nargs; i++ ) {
    printf("(%s %s) ", pdecl->argtypes[i], pdecl->args[i]);
  }
  printf(")\n\"");
  display_description(pdecl->description);
  printf("\")\n");
}

void output_information(int ndecls, pfndec_st *pdecls)
{
  int i;
  for ( i = 0; i < ndecls; i++ ) {
    write_fn_info(pdecls[i]);
  }
}

int main(int argc, char** argv)
{
  pfndec_st *pfndecs;
  int ndecls;
  if ( argc != 2 ) {
    fprintf(stderr,"usage: %s <c-filename>\n",argv[0]);
    exit(1);
  }
  filename = argv[1];
#ifdef sgi
  printf("\n");
#endif
  ndecls = get_declarations(argv[1],&pfndecs);
  output_information(ndecls,pfndecs);
  return 0;
}


/* Here is an example of the format of a statement to export: */
#if 0
/*[EXPORT (name my-string-append)]
 * Returns the result of appending strings st1 and str2.
 */
char * string_append(char* str1, char *str2)
{
  char *ret = get_string(strlen(str1)+strlen(str2));
  strcpy(ret,str1);
  strcat(ret,str2);
  return ret;
}
/*[EXPORT]
 * Prints 'hi' to standard output.
 */
void print_hi(void) { printf("hi\n"); }
#endif
