/*
 * GUI misc module
 * This module is for generic GUI functions.
 * Independent from a specific data structure.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <gnome.h>
#include "gdiff.h"
#include "gui.h"
#include "misc.h"
#include "guimisc.h"
#include "fileview.h" /* for MARK_LENGTH */


/**
 * draw_text_lines:
 * Draw "nl" lines of buffer on GtkText widget.
 * Input:
 * GtkText *text;
 * const FontProp *fprop;
 * const BufInfo *binfo; buffer.
 * int nl; the number of lines to draw.
 * Output:
 * Return value; The drawn bytes.
 **/
int
draw_text_lines(GtkText *text, const FontProp *fprop, const BufInfo *binfo, int nl)
{
	int lenb;
	const char *endp;

	endp = skip_n_lines(binfo->buf, nl, binfo->lenb);
	lenb = endp ? endp - binfo->buf : binfo->lenb;
	gtk_text_insert(text, fprop->font, fprop->fg, fprop->bg, binfo->buf, lenb);
	return lenb;
}


/**
 * insert_remove_line_numbers:
 * Insert or remove line numbers at the head of each line.
 * Input:
 * GtkText *text;
 * int *pos; Position in the text widget to insert numbers.
 * gboolean b_in; if TRUE insert numbers. if FALSE remove numbers. 
 * const FontProp *fprop;
 * const BufInfo *binfo; Used to calculate the length of each line.
 * int begin; The line number to start with.
 * int nline; The number of lines to insert. (e.g. begin=10,nline=3 implies "10","11","12" are inserted.)
 * const LineFormat *lformat;
 * Output:
 * int *pos; The last position after inserting or removing numbers.
 * Return value; The length of processed part of the buffer.
 **/
int insert_remove_line_numbers(GtkText *text, gboolean b_in, int *pos, const FontProp *fprop, const BufInfo *binfo, int begin, int nline, const LineFormat *lformat)
{
	int n;
	int end = begin + nline;
	const char *beginp = binfo->buf;
	const char *endp = beginp;
	gboolean b_sbcs = !text->use_wchar;/* Single byte character set. */
	int num_char;/* number of characters */
	int max_lenb = binfo->lenb;
	
	if (nline <= 0)
		return 0;

	/* Skip to the begin point */
	gtk_text_set_point(text, *pos);
	for (n = begin; n < end; n++) {
		int lenb;
		char buf[32];

		endp = skip_n_lines(beginp, 1, max_lenb);
		lenb = endp ? endp - beginp : max_lenb;
		max_lenb -= lenb;
		if (b_sbcs) {
			num_char = lenb;/* I can assume char=byte. It's a special case! */
		} else {
			num_char = get_num_chars(beginp, lenb);
		}
		beginp = endp;

		if (b_in) {
			sprintf(buf, lformat->format, n);
			gtk_text_insert(text, fprop->font, fprop->fg, fprop->bg, buf, lformat->n_col + MARK_LENGTH);
			*pos += ((lformat->n_col + MARK_LENGTH) + num_char);
		} else {
			gtk_text_forward_delete(text, lformat->n_col + MARK_LENGTH);
			*pos += num_char;
		}
		gtk_text_set_point(text, *pos);
	}
	return (endp - binfo->buf);
}


/**
 * make_label_for_notebook:
 * From two file names, make a GtkLabel widget for adding to GtkNotebook widget.
 * Input:
 * const char *f1;
 * const char *f2;
 * CurrentView cv;
 * Output:
 * Return value; Created GtkLabel widget.
 **/
GtkWidget*
make_label_for_notebook(const char *f1, const char *f2, CurrentView cv)
{
	char *buf;
	GtkWidget *label;
	
	buf = g_malloc(strlen(f1) + strlen(f2) + 8);
	switch (cv) {
	case ONE_PANE_VIEW:
		sprintf(buf, "%s : %s [1]", f1, f2);
		break;
	case TWO_PANE_VIEW:
		sprintf(buf, "%s : %s [2]", f1, f2);
		break;
	case DIR_VIEW:
		sprintf(buf, "[%s : %s]", f1, f2);
		break;
	}
	label = gtk_label_new(buf);
	g_free(buf);

	return label;
}


/**
 * make_current_info_msg:
 * Using file names and line numbers, make a message.
 * Currently, it's used for status-bar.
 * Returned buffer should be freed by the caller.
 * If the line number is -1, ignore it.
 **/
char*
make_current_info_msg(const char *fname1, const char *fname2, int f1n1, int f1n2, int f2n1, int f2n2)
{
	char *msg;

	if (f1n1 == -1)
		msg = g_strdup_printf("%s : %s", fname1, fname2);
	else
		msg = g_strdup_printf("%s [%d - %d] : %s [%d - %d]", fname1, f1n1, f1n2, fname2, f2n1, f2n2);

	return msg;
}

/**
 * ask_yes_no:
 **/
QDialogReply
ask_yes_no(const char *msg)
{
	GnomeMessageBox *mbox;

	mbox = GNOME_MESSAGE_BOX(gnome_message_box_new(msg, GNOME_MESSAGE_BOX_QUESTION,
												   GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, GNOME_STOCK_BUTTON_CANCEL,
												   NULL));
	gnome_dialog_set_default(GNOME_DIALOG(mbox), 2);
	gtk_window_set_modal(GTK_WINDOW(mbox), TRUE);
	gtk_window_set_position(GTK_WINDOW(mbox), GTK_WIN_POS_MOUSE);
	
	return gnome_dialog_run_and_close(GNOME_DIALOG(mbox));
}

/**
 * is_files_different:
 * Check two files are different, i.e. they will be displayed.
 * If b_show_msg == TRUE, show an appropriate message-box.
 * Input:
 * (const)DiffFiles *files;
 * gboolean b_show_msg; TRUE to show message-box.
 * Output:
 * Return value; TRUE if they are different.
 **/
gboolean
is_files_different(DiffFiles *files, gboolean b_show_msg)
{
	FilesSpecialStatus fsst;

	fsst = dfiles_get_special_status(files);
	if (fsst == DIFFERENT_FILES) {
		return TRUE;
	} 
	switch (fsst) {
	case BINARY_FILES:
		if (b_show_msg)
			g_message(_("Binary files\n"));
		break;
	case ONLY_FILE1_EXISTS:
	case ONLY_FILE2_EXISTS:
		break;
	case IDENTICAL_FILES:
		if (b_show_msg)
			g_message(_("Identical files\n"));
		break;
	case DIRECTORIES:
	default:
		g_assert_not_reached();
		break;
	}
	
	return FALSE;
}

/* Guessing routines, which are used when moving next(or previous) diff part. */
/**
 * guess_visible_top_line:
 * Guess the top line number of visible part of text widget, and return it.
 * This can work only when text widget is line-wrap-off.
 **/
int
guess_visible_top_line(GtkAdjustment *adj, int total)
{
	gdouble max = adj->upper - adj->lower;
	int line = 0;
	
	if (max) {
		line = adj->value / max * total;
	}
	return line + 1;
}

/**
 * guess_visible_center_line:
 * Guess the center line number of visible part of text widget, and return it.
 * This can work only when text widget is line-wrap-off.
 **/
int
guess_visible_center_line(GtkAdjustment *adj, int total)
{
	gdouble max = adj->upper - adj->lower;
	int line = 0;
	
	if (max) {
		line = (adj->value + adj->page_size / 2) / max * total;
	}
	return line + 1;
}

/**
 * guess_visible_bottom_line:
 * Guess the bottom line number of visible part of text widget, and return it.
 * This can work only when text widget is line-wrap-off.
 **/
int
guess_visible_bottom_line(GtkAdjustment *adj, int total)
{
	gdouble max = adj->upper - adj->lower;
	int line = 0;
	
	if (max) {
		line = (adj->value + adj->page_size) / max * total;
	}
	return line + 1;
}
