/*
 * GUI file operation module
 * Mainly, handling GtkFileSelection widget.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gnome.h>
#include "gdiff.h"
#include "misc.h"
#include "gui.h"
#include "guiwin.h"
#include "dirview.h"


/* Note:
 * File dialog box has its own state, because it has to accept two file names. */

/* Constant strings */
static const char FIRST_FILE_SEL_CAPTION[] = N_("First file selection");
static const char SECOND_FILE_SEL_CAPTION[]	= N_("Second file selection");


/* Data structure definitions */
/* Internal data kept in GtkFileSelection instance.
 * Kept by gtk_object_set_user_data() and gtk_object_get_user_data(). */
typedef struct {
	/* For reference */
	GDiffWindow *gdwin;

	/* Dialog box state */
	gboolean is_directory;
	char *fname1;
	GtkLabel *fname1_label;
	WhichFile file_count;
} FileDialogKeep;


/* Private function declarations */
static void files_selection_dialog(GDiffWindow *gdwin);
static void file_selection_ok(GtkWidget *w, gpointer data);
static void file_selection_cancel(GtkWidget *w, gpointer data);
static void dir_toggle(GtkWidget *w, gpointer data);
static FileDialogKeep* filedialogkeep_new(GDiffWindow *gdwin, GtkLabel *label);
static void filedialogkeep_reset(FileDialogKeep *fdk);
static void filedialogkeep_update(FileDialogKeep *fdk, const char *fname1);


/**
 * files_open:
 * An entry point to open two files(directories).
 * At present, called only from [menu]-[File]-[Open].
 **/
void
files_open(GDiffWindow *gdwin)
{
	files_selection_dialog(gdwin);
}


/* ---The followings are private functions--- */
/**
 * files_selection_dialog:
 * If the instance of dialog has not created yet, create it and show it.
 * If it exists and has been hidden, just show it.
 * If it has been shown, just raise it.
 * Input:
 * GDiffWindow *gdwin;
 * Output:
 * None;
 **/
static void
files_selection_dialog(GDiffWindow *gdwin)
{
	FileDialogKeep *fdk;
	GtkWidget *filesel;
	GtkWidget *checkbutton;
	GtkWidget *label;

	/* The dialog has already been created */
	if (gdwin->filesel) {
		if (!GTK_WIDGET_VISIBLE(gdwin->filesel)) {
			fdk = gtk_object_get_user_data(GTK_OBJECT(gdwin->filesel));
			
			if (fdk->file_count == FIRST_FILE)
				gtk_window_set_title(GTK_WINDOW(gdwin->filesel),
									 _(FIRST_FILE_SEL_CAPTION));
			/* Not necessary, because I have reset all states at cancellation.
			else if (fdk->file_count == SECOND_FILE)
				gtk_window_set_title(GTK_WINDOW(fs), SECOND_FILE_SEL_CAPTION);
				*/
			gtk_widget_show(GTK_WIDGET(gdwin->filesel));
		} else {
			gdk_window_raise(GTK_WIDGET(gdwin->filesel)->window);
		}
		return;
	}

	/* Create GtkFileSelection dialog */
	filesel = gtk_file_selection_new(_(FIRST_FILE_SEL_CAPTION));
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(filesel));
	gtk_window_set_position(GTK_WINDOW(filesel), GTK_WIN_POS_MOUSE);
	gdwin->filesel = GTK_FILE_SELECTION(filesel);

	/* Label to show the first file name, while you're selecting a second file. */
	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(GTK_FILE_SELECTION(filesel)->action_area), 
					   label, FALSE, FALSE, 0);

	/* Set FileDialogKeep as user_data of file dialog widget */
	fdk = filedialogkeep_new(gdwin, GTK_LABEL(label));
	gtk_object_set_user_data(GTK_OBJECT(filesel), fdk);
	
	/*XXX: this causes an error.
	gtk_signal_connect(GTK_OBJECT(filesel), "destroy",
					   GTK_SIGNAL_FUNC(filesel_destroyed),
					   filesel);*/
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
					   "clicked", GTK_SIGNAL_FUNC(file_selection_ok), filesel);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
					   "clicked", GTK_SIGNAL_FUNC(file_selection_cancel), filesel);

	/* Checkbutton to specify directory selection */
	checkbutton = gtk_check_button_new_with_label(_("Compare directories"));
	/* XXX:what effect? gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(checkbutton), TRUE);*/
	gtk_signal_connect(GTK_OBJECT(checkbutton),
					   "toggled", GTK_SIGNAL_FUNC(dir_toggle), filesel);
	gtk_box_pack_start(GTK_BOX(GTK_FILE_SELECTION(filesel)->action_area), 
					   checkbutton, FALSE, FALSE, 0);
	gtk_widget_show(checkbutton);
	
    gtk_widget_show(filesel);
}

/**
 * file_selection_ok:
 * The behavior depends on the internal state.
 * In case first file selected, change its state for second file.
 * In case second file selected, do the job.
 **/
static void
file_selection_ok(GtkWidget *w, gpointer data)
{
	GtkFileSelection *fs = data;
	FileDialogKeep *fdk;
	char *fname;
	FileType ft;

	fdk = gtk_object_get_user_data(GTK_OBJECT(fs));
	fname = gtk_file_selection_get_filename(fs);
	ft = check_filetype(fname);
	if (ft == OTHERFILE) {
		g_message(_("Wrong file type\n"));
		return;
	} else if (ft == REGULARFILE && fdk->is_directory) {
		g_message(_("You have to select a directory. Or don't check the option button.\n"));
		return;
	} else if (ft == DIRECTORY && !(fdk->is_directory)) {
		g_message(_("You have to select a (regular)file. Or check the option button.\n"));
		return;
	}
	
	if (fdk->file_count == FIRST_FILE) {
		filedialogkeep_update(fdk, fname);
		gtk_window_set_title(GTK_WINDOW(fs), _(SECOND_FILE_SEL_CAPTION));
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), "");
	} else if (fdk->file_count == SECOND_FILE) {
		GDiffWindow *gdwin;
		GDiffDirViews *gdirviews;
		char *fname1;
		char *fname2;
		
		gdwin = fdk->gdwin;
		fname1 = fdk->fname1;
		fname2 = fname;

		gdirviews = gdiff_dirviews_new(gdwin, fname1, fname2, fdk->is_directory);
		
		filedialogkeep_reset(fdk);
		gtk_widget_hide(GTK_WIDGET(fs));

		/* Do the job */
		dirview_display(gdirviews);
	} else {
#ifdef DEBUG		
		g_error("XXX");
#endif
	}
}

/**
 * file_selection_cancel:
 * Reset all internal states.
 **/
static void
file_selection_cancel(GtkWidget *w, gpointer data)
{
	GtkFileSelection *fs = data;
	FileDialogKeep *fdk;

	fdk = gtk_object_get_user_data(GTK_OBJECT(fs));
	filedialogkeep_reset(fdk);
	gtk_widget_hide(GTK_WIDGET(fs));
}

/**
 * dir_toggle:
 * Toggle whether to select two directories or two files.
 * If a user has already selected the first one, don't change the mode.
 **/
static void
dir_toggle(GtkWidget *w, gpointer data)
{
	GtkFileSelection *fs = data;
	FileDialogKeep *fdk;

	fdk = gtk_object_get_user_data(GTK_OBJECT(fs));
		
	if (fdk->file_count == FIRST_FILE) {
		fdk->is_directory = !(fdk->is_directory);
		gtk_widget_set_sensitive(fs->file_list, !(fdk->is_directory));
	} else /* Don't change */
		GTK_TOGGLE_BUTTON(w)->active = !GTK_TOGGLE_BUTTON(w)->active;/* Is this a proper way ?*/
}

/* Routines for dialog box's own private data, i.e. FileDialogKeep */
/**
 * filedialogkeep_new:
 * Allocate FileDialogKeep, intialize it and return its pointer.
 * Input:
 * GDiffWindow *gdwin; kept by FileDialogKeep.
 * GtkLabel *label; kept by FileDialogKeep.
 * Output:
 * Return value; allocated FileDialogKeep pointer.
 **/
static FileDialogKeep*
filedialogkeep_new(GDiffWindow *gdwin, GtkLabel *label)
{
	FileDialogKeep *fdk;

	fdk = g_new(FileDialogKeep, 1);

	fdk->gdwin = gdwin;
	fdk->is_directory = FALSE;
	fdk->fname1 = NULL;
	fdk->fname1_label = label;
	fdk->file_count = FIRST_FILE;

	return fdk;
}

/**
 * filedialogkeep_reset:
 * Reset FileDialogKeep.
 * Called when a user closes a dialog while he has selected a first file,
 * but hasn't selected a second file.
 * Input:
 * FileDialogKeep *fdk;
 * Output:
 * FileDialogKeep *fdk; updated.
 **/
static void
filedialogkeep_reset(FileDialogKeep *fdk)
{
	if (fdk->fname1 == NULL)
		return;
	/* Not change: fdk->is_directory */
	g_free(fdk->fname1);
	fdk->fname1 = NULL;
	gtk_widget_hide(GTK_WIDGET(fdk->fname1_label));
	fdk->file_count = FIRST_FILE;
}

/**
 * filedialogkeep_update:
 * Update FileDialogKeep.
 * Called when a user selects a first file.
 * Input:
 * FileDialogKeep *fdk;
 * const char *fname1; Selected first file name.
 * Output:
 * FileDialogKeep *fdk; updated.
 **/
static void
filedialogkeep_update(FileDialogKeep *fdk, const char *fname1)
{
	fdk->fname1 = g_strdup(fname1);
	gtk_label_set_text(fdk->fname1_label, fname1);
	gtk_widget_show(GTK_WIDGET(fdk->fname1_label));
	fdk->file_count = SECOND_FILE;
}

