#!/usr/bin/perl
# Script perl to generate a LaTeX file from the configuration file 
# (set with -config=...), then run latex, dvips and gv on this file.
# (use -psviewer=... to set another postscript visualizer)

sub usage {
  print STDERR "Usage : [perl] $0 -config=file [-psviewer=prog] [OPTIONS]

 'file' : config file generated by tkcdlayout or written manually.
 'prog' : the postscript visualizer you want to use.
            It can be \"gv -landscape\", kghostview, pageview...
 -update : no ps visualizer will be launched after the ps is generated.
            Use 'Update' button in the visualizer to see the new ps.
 -help : display this information
 -version : display version and author information
 NOTE : '--' options are supported too (ex: --help).
";
  exit 1;
}

sub version {
  print STDERR "$0 version 0.5. --help for usage.
Written by David Faure, faure\@kde.org, 1998. Licensed under GPL.
";
  exit 1;
}

sub fix {
  ($_) = @_;
  # Fixes LaTeX special handing of some special characters.
  s/_/\\_/g;			#  _ -> \_
  s/\#/\\\#/g; 			#  # -> \#
  s/&/\\&/g;			#  & -> \&
  return $_;
}

sub GetFiles {
  # List files ($maxdepth levels from $cddir)
  my ($cddir, $maxdepth, $showdirs, $showfiles, $exclude) = @_;
  my ($maxdepthOPT, $typeOPT, @lines);
  
  @lines=();
  if ($cddir ne "") { 
      $maxdepthOPT = ($maxdepth > 0) ? "-maxdepth " . $maxdepth : "";
        # -links 2 -> no directory with subdirectories.
      $typeOPT = ($showdirs eq "yes") ? "-type d -links 2" : "";
      $typeOPT .= " -o " if ($showdirs eq "yes" && $showfiles eq "yes");
      $typeOPT .= "-type f" if ($showfiles eq "yes");
      # Run a find command
      chdir $cddir or die "Can't cd to $cddir : $!";
      open (FIND,"find . $maxdepthOPT $typeOPT |");
      while (<FIND>)
	{
	  # Exclude "." and $exclude (if set)
	  unless (($exclude ne "" && /$exclude/) || /^.$/)
	    {
	      s/^.\///;		#  Remove ./ at the beginning of each line
	      print ;
	      &fix($_);
	      chomp;		# Remove \n
	      # Add space at the beginning and the LaTeX sign '\\' at the end
	      $_ = "        " . $_ . '\\\\' . "\n"; 
	      push @lines, $_;
	    }
	}
      close (FIND);
      @lines = sort @lines;		# Do we always want to sort ?
    }
  return @lines;
}

sub BeginLaTeX {
  my($Title, $rules) = @_;
  my($LeftRule, $RightRule);
  
  $LeftRule="";
  $RightRule="";
  if ($rules eq "yes") {
    if (length $Title > 21) { print "Title too big for left & right rules\n"; }
    else {
      $LeftRule='\rule{1cm}{0.1mm} ';
      $RightRule=' \rule{2cm}{0.1mm}';
    }
  }
  print LATEXFILE 
    '\documentclass[a4paper] {article}

\usepackage{t1enc}     % for 8bit chars (optional)
\usepackage{lscape}    % for landscape
\usepackage{newcent}   % PS police with serif=New Century (optional)
\usepackage{rotate}    % for rotate :)
\usepackage{vmargin}   % for setmarginsrb

\pagestyle{empty}      % no page numbers

\newcommand{\cdtitle}{' . &fix($Title) . '}
\newcommand{\leftrule}{' . $LeftRule . '}
\newcommand{\rightrule}{' . $RightRule . '}

\setmarginsrb{2cm}{2cm}{2cm}{2cm}{0pt}{0pt}{0pt}{1cm} % page margins
%            Left  Top Right Bot.

\begin{document}
\landscape{
';
}

sub beginframebox {
  my ($h, $w, $contents) = @_;
  print LATEXFILE '\framebox{\parbox[c]['.$h.'][t]{'.$w.'}{'.$contents;
}

sub endframebox {
  print LATEXFILE "}}";		# No \n here ! (Otherwise there is a space between the frames)
}

sub MakeTitle {
  my ($size) = @_;
  print LATEXFILE '    \begin{center}
      '.$size.'{\leftrule{}\rmfamily\itshape\cdtitle{}\rightrule{}\\\\}
    \end{center}'."\n";
  
}
sub DoLayout {
  # Do the LaTeX output for the layouts.
  # $page can be front, back, or both
  # $front_layout_type and $back_layout_type can be taf or btas
  # The rest is passed to GetFiles()
  my ($page, $front_layout_type, $back_layout_type, $subtitle, @getfiles_args) = @_;
  my (@lines, @pages, $p, @remaininglines);
  if ($page eq "both") { @pages=("front","back"); } else { @pages=($page); }
  if (($front_layout_type eq "taf") || ($back_layout_type eq "taf")) { @lines = &GetFiles(@getfiles_args); }
  my $height = "11.6cm";	# or 12 ???
  
  foreach $p (@pages) {		# Loop over the pages
    
    if ($p eq "back") {		# back : vertical title on the left
      $layout_type = $back_layout_type;
      &beginframebox($height,"1cm",'\rotate[l]{\cdtitle{}}');
      &endframebox();
      &beginframebox($height,"13.6cm","\n");
    } else {			# front : only a big frame
      $layout_type = $front_layout_type;
      &beginframebox($height,"12.2cm","\n");
    }
    if ($layout_type eq "taf")	# Title and files
      {
	print LATEXFILE '    \vspace{2mm}'."\n".'    \raggedright'."\n";
	&MakeTitle('\Huge');	# or \LARGE if \Huge is too big...
	# Write the first 24 lines, cut them if back (only 1 page)
	@remaininglines = &WriteLines(24, ($p eq "back") ? "cut" : "no", @lines);
	# TODO : if the title is really big and goes on 2 lines, then this
	#        should be 21 instead of 24. But how to check that ? What's
	#        more, if a line is too long, it wraps, eating another line !
      }
    elsif ($layout_type eq "btas") # Big title and subtitle
      {
	print LATEXFILE '\vspace{4cm} \raggedright'."\n";
	&MakeTitle('\Huge');	# Maximum size
	&Subtitle($subtitle);
      }
    if ($p eq "front")		# front : two pages. Let's do the second one.
      {
	&endframebox();
	&beginframebox($height,"12.2cm","\n");
	&WriteLines(28, "cut", @remaininglines) if ($layout_type eq "taf");
      }
    &endframebox();
    if ($p eq "back") {		# back : vertical title on the right
      &beginframebox($height,"1cm",'\rotate[r]{\cdtitle{}}');
      &endframebox();
    }
    if (($page eq "both") && ($p eq "front")) {
      print LATEXFILE '\newpage'."\n"; 
    }
  }				# foreach
}

sub WriteLines {
# Writes up to $N lines from @lines to the LaTeX file, in a box
# If more than $N lines, either cut them (if $cut)
# or send them back, for the next page
  my ($N, $cut, @lines) = @_;
  my $count = $#lines+1;
  if (($cut eq "cut") && ($count > $N)) { # Too many lines : remove some
    @lines = (@lines[0..$N-2], ' \ldots\\\\' . "\n");	  # And add '...'
    $count = $#lines+1; # should be == $N
  }
  print STDERR "N : $N  lines count : $count \n"; # debug output
  print STDERR "Last one is ".$lines[$#lines]; # debug output
  print LATEXFILE '    \vspace{5mm}
    \hspace{1cm}\parbox[c][10.9cm][t]{11cm}{\sffamily\raggedright\small{'."\n";
  my $lasttoprint = ($count < $N) ? $count-1 : $N-1 ;
  print LATEXFILE @lines[0..$lasttoprint] if ($lasttoprint >= 0);
  print STDERR "Lasttoprint : $lasttoprint\n"; # debug output
  print LATEXFILE '    }}'."\n"; # ends small and parbox
  if ($#lines >= $lasttoprint+1) { return @lines[$lasttoprint+1..$#lines]; } else { return (); }
}

sub Subtitle {
  my ($subtitle) = @_;
  # the \hspace doesn't work, but I left it because it allows two spaces
  # to be taken into account (the one before and the one after)    !!
  print LATEXFILE '   \vspace{3cm} \begin{flushright}\sffamily\LARGE{';
  print LATEXFILE &fix($subtitle).'} \hspace{4cm} \end{flushright}'."\n";
}

##############################################################################
## Main program
##############################################################################

$home=$ENV{'HOME'};
$psviewer="gv -landscape";
$config="";
$updateonly = "no";
## Parse command line
foreach $_ (@ARGV) {
  $psviewer = $1 if (/^--*psviewer\s*=\s*\"*\s*([^\"]*)\s*$/i);
  $config = $1 if (/^--*config\s*=\s*(.*)$/i);
  $updateonly = "yes" if (/^--*update$/i);
  &usage if (/^--*help$/i);
  &version if (/^--*version$/i);
}
&usage if ($config eq "");
$config =~ s/\~/$home/;		# ~ -> $HOME

## Default option values, in order of use below
$cdn="";
$lay="~/.tkcdlayout/layouts";
$Title="";
$rules="no";
$cddir="";
$maxdepth=0;
$showdirs="no";
$showfiles="no";
$exclude="";
$page="front";
$front_layout_type="taf";
$back_layout_type="btas";
$subtitle="";

## Parse config file
open (CONFIG, $config) || die "File not found : ".$config;
while (<CONFIG>) {
  unless (/^\#/ || /^\s*$/) 	# Skip comments and empty lines
    {
      s/\~/$home/;		# ~ -> $HOME
      eval "\$".$_;		# Set the variable. ex: $page=front
    }
}
close (CONFIG);

## Create LaTeX file
die "cdn option not set in ".$config if ($cdn eq "");
$cdn = $lay . "/" . $cdn;

open (LATEXFILE, ">$cdn.latex");

&BeginLaTeX($Title, $rules);
@getfiles_args = ($cddir, $maxdepth, $showdirs, $showfiles, $exclude);
&DoLayout($page, $front_layout_type, $back_layout_type, $subtitle, @getfiles_args);
print LATEXFILE "}\n";		# ends \landscape
print LATEXFILE '\end{document}' . "\n";
close (LATEXFILE);

## Process LaTeX file and view result
chdir $lay or die "Can't cd to $lay : $!";
system ("latex $cdn.latex") == 0 or die "latex command failed";
system ("dvips $cdn.dvi -o") == 0 or die "dvips command failed";
if ($updateonly eq "no") {
  print STDERR "*** Running $psviewer $cdn".".ps ***\n";
  $SIG{CHLD} = sub { wait };	# Avoid zombies
  unless (fork)			# Run the psviewer in a fork().
    {
      exec "$psviewer $cdn".".ps";
      die "$psviewer failed";
    }
}				# otherwise, just update .ps. User will have to press "Update" in gv.
exit 0;
