# This file is part of Malaga, a system for Natural Language Analysis.
# Copyright (C) 1995-1998 Bjoern Beutel
#
# Bjoern Beutel
# Universitaet Erlangen-Nuernberg
# Abteilung fuer Computerlinguistik
# Bismarckstrasse 12
# D-91054 Erlangen
# e-mail: malaga@linguistik.uni-erlangen.de
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# description =================================================================

# Module to draw the computation tree for a malaga analysis.
#
# Global variables:
# $tree_mode -- tree mode ("result_paths", "no_dead_ends", "complete_tree")
# $final_nodes -- list of end nodes of result paths
# $tree_input -- the word form or sentence that has been analysed
# $selected_start_node -- first node of selected path
# $selected_end_node -- last node of selected path
#
# Malaga analysis tree:
# $state($node,children) -- descendants of $node
# $state($node,type) -- type of $node ("final", "inter", "break", "pruned")
# $state($node,ancestor) -- ancestor of $node
# $state($node,rule) -- rule that has been invoked in $node
# $state($node,next_surf) -- next surface of $node
# $state($node,next_cat) -- next category of $node
# $state($node,result_surf) -- result surface of $node
# $state($node,result_cat) -- result category of $node
# $state($node,rule_set) -- result rule set of $node
# $state($node,result_node) -- "1" if $node is part of a successful path
#
# Nodes for all widgets of the analysis tree:
# node_desc($widget_id) -- the node that $widget_id belongs to
# edge_widget($node) -- edge-widget to be highlighted when shown in path
# node_widget($node) -- node-widget to be highlighted when shown in path
# surf_widget($node) -- text-widget to be framed when shown in path

# functions ===================================================================

proc tree_define_bindings {} {
# Define bindings to navigate in the Malaga analysis tree 
# and to show the paths or node contents.
  
  global node_desc selected_start_node
  
  # Binding to display result-category
  .tree.frame.canvas bind tag_node <Button-1> {
    set node $node_desc([.tree.frame.canvas find withtag current])
    if {$state($node,result_cat) != ""} {path_display $node $node}
  }
  
  # Binding to display path section
  .tree.frame.canvas bind tag_node <Button-3> {
    if [info exists selected_start_node] {
      set end_node $node_desc([.tree.frame.canvas find withtag current])
      set start_node $end_node
      while {$start_node >= 0} {
	if {$start_node == $selected_start_node} {
	  path_display $start_node $end_node
	  break
	}
	set start_node $state($start_node,ancestor)
      }
    }
  }
  
  # Binding to display next-category
  .tree.frame.canvas bind tag_next_surf <Button-1> {
    set node $node_desc([.tree.frame.canvas find withtag current])
    path_display -1 $node
  }
  
  # Binding to display single derivation
  .tree.frame.canvas bind tag_rule_name <Button-1> {
    set node $node_desc([.tree.frame.canvas find withtag current])
    path_display $state($node,ancestor) $node
  }   
}

#------------------------------------------------------------------------------

proc tree_mark_path {highlight} {
# If "highlight == 1", highlight selected path.
# Else unhighlight selected path.

  global state node_desc selected_start_node selected_end_node \
  node_widget edge_widget surf_widget highlight_frame

  if {! [info exists selected_start_node]} {return}

  if {$highlight} {set width 3} else {set width 1}

  set node $selected_end_node
  
  if {$selected_start_node != -1} {
    while {$node != $selected_start_node} {
      .tree.frame.canvas itemconfigure $node_widget($node) -width $width
      .tree.frame.canvas itemconfigure $edge_widget($node) -width $width
      set node $state($node,ancestor)
    }
    
    .tree.frame.canvas itemconfigure $node_widget($node) -width $width
    
  } else {
    if {$highlight} {
      set highlight_frame \
      [.tree.frame.canvas create rectangle \
       [widget_left .tree.frame.canvas $surf_widget($node)] \
       [widget_top .tree.frame.canvas $surf_widget($node)] \
       [widget_right .tree.frame.canvas $surf_widget($node)] \
       [widget_bottom .tree.frame.canvas $surf_widget($node)] \
       -width $width]
    } else {
      .tree.frame.canvas delete $highlight_frame
      unset highlight_frame
    }
  }
}

#------------------------------------------------------------------------------

proc tree_create_view_menu {} {
# Allow printing of successful paths only, all but dead ends or complete tree

  global menu_font

  # View settings:
  menubutton .tree.menu.view \
  -text " View " \
  -font $menu_font \
  -menu .tree.menu.view.m
  menu .tree.menu.view.m

  foreach view_entry {
    { "result_paths" " Result paths only " r}
    { "no_dead_ends" " All but dead ends " d}
    { "complete_tree" " All nodes " a} 
  } {
    .tree.menu.view.m add radio \
    -variable tree_mode \
    -value [lindex $view_entry 0] \
    -label [lindex $view_entry 1] \
    -font $menu_font \
    -command "tree_redraw new_mode" \
    -accelerator Control+[lindex $view_entry 2]
  }
  bind .tree <Control-r> {
    set tree_mode "result_paths"
    tree_redraw new_mode
  }
  bind .tree <Control-d> {
    set tree_mode "no_dead_ends"
    tree_redraw new_mode
  }
  bind .tree <Control-a> {
    set tree_mode "complete_tree"
    tree_redraw new_mode
  }

  pack .tree.menu.view -side left
}

#------------------------------------------------------------------------------

proc tree_open_window {} {
# Create a tree window

  global tree_geometry

  if [winfo exists .tree] {return}

  if {! [info exists tree_geometry]} {set tree_geometry 600x600}

  toplevel .tree
  wm minsize .tree 100 100
  wm geometry .tree $tree_geometry
  wm protocol .tree WM_DELETE_WINDOW tree_close_window
  wm title .tree "Malaga Analysis Tree"
  wm iconname .tree "Tree"
  wm focusmodel .tree active

  frame .tree.menu -relief raised -borderwidth 1
  pack .tree.menu -side top -fill x
  create_window_menu tree
  create_font_menu tree
  tree_create_view_menu
  create_result_menu tree
  tk_menuBar .tree.menu \
  .tree.menu.window .tree.menu.font .tree.menu.path .tree.menu.result
  
  create_scroll_frame tree
  tree_define_bindings
  focus .tree.frame.canvas
}

#------------------------------------------------------------------------------

proc tree_close_window {} {
# Close the tree window

  global tree_geometry
  
  if [winfo exists .path] {path_close_window}

  if [winfo exists .tree] {
    set tree_geometry [wm geometry .tree]
    destroy .tree
  }
}

#------------------------------------------------------------------------------

proc tree_draw_final_node {node x1 y1} {
# Draw a final node consisting of two circles.

  global node_widget font_size node_desc

  set radius [expr $font_size(tree) / 2]

  # Draw a big circle.
  set widget_id [.tree.frame.canvas create oval \
		 [expr $x1 - $radius] [expr $y1 - $radius] \
		 [expr $x1 + $radius] [expr $y1 + $radius] \
		 -outline black \
		 -fill white \
		 -tags {tag_tree tag_node}]
  set node_widget($node) $widget_id
  set node_desc($widget_id) $node

  # Draw a small circle.
  set widget_id [.tree.frame.canvas create oval \
		 [expr $x1 - $radius * 0.60] [expr $y1 - $radius * 0.60] \
		 [expr $x1 + $radius * 0.60] [expr $y1 + $radius * 0.60] \
		 -outline black \
		 -fill white \
		 -tags {tag_tree tag_node}]
  set node_desc($widget_id) $node
}

#------------------------------------------------------------------------------

proc tree_draw_break_node {node x1 y1} {
# Draw a break node as a filled rectangle.

  global node_desc node_widget font_size
  
  set radius [expr $font_size(tree) / 3]
  
  set widget_id [.tree.frame.canvas create rectangle \
		 [expr $x1 - $radius] [expr $y1 - $radius] \
		 [expr $x1 + $radius] [expr $y1 + $radius] \
		 -outline black \
		 -fill black \
		 -tags {tag_tree tag_node}]
  
  set node_widget($node) $widget_id
  set node_desc($widget_id) $node
}

#------------------------------------------------------------------------------

proc tree_draw_inter_node {node x1 y1} {
# Draw an intermediate node as a circle.
  
  global node_desc font_size node_widget
  
  set radius [expr $font_size(tree) / 2]
  
  set widget_id [.tree.frame.canvas create oval \
		 [expr $x1 - $radius] [expr $y1 - $radius] \
		 [expr $x1 + $radius] [expr $y1 + $radius] \
		 -outline black \
		 -fill white \
		 -tags {tag_tree tag_node}]
  set node_widget($node) $widget_id
  set node_desc($widget_id) $node
}

#------------------------------------------------------------------------------

proc tree_draw_pruned_node {node x1 y1} {
# Draw an intermediate node as a circle.
  
  global node_desc font_size node_widget
  
  set radius [expr $font_size(tree) / 2]
  
  # Draw a circle.
  set widget_id [.tree.frame.canvas create oval \
		 [expr $x1 - $radius] [expr $y1 - $radius] \
		 [expr $x1 + $radius] [expr $y1 + $radius] \
		 -outline black \
		 -fill white \
		 -tags {tag_tree tag_node}]
  set node_widget($node) $widget_id
  set node_desc($widget_id) $node

  # Draw a line across the circle.
  set widget_id [.tree.frame.canvas create line \
		 [expr $x1 - $radius] [expr $y1 + $radius] \
		 [expr $x1 + $radius] [expr $y1 - $radius] \
		 -width 2 \
		 -tags {tag_tree tag_node}]
  set node_desc($widget_id) $node

  # Draw another line across the circle.
  set widget_id [.tree.frame.canvas create line \
		 [expr $x1 - $radius] [expr $y1 - $radius] \
		 [expr $x1 + $radius] [expr $y1 + $radius] \
		 -width 2 \
		 -tags {tag_tree tag_node}]
  set node_desc($widget_id) $node
}

#------------------------------------------------------------------------------

proc tree_draw_edge {node x y1 y2} {
# Draw an edge with next-surface and rule name, if they exist

  global state node_desc font_size font char_set surf_widget edge_widget

  # Move to beginning of text
  set x1 [expr $x + $font_size(tree) / 2]
  set x2 $x1

  # Draw surface
  set surface $state($node,next_surf)
  if {$surface != ""} {
    set surf_id [.tree.frame.canvas create matrix \
		 $x1 $y2 "\"$surface\"" \
		 -font $font(tree) \
		 -char_set $char_set \
		 -tags {tag_tree tag_next_surf}]
    set surf_widget($node) $surf_id
    set node_desc($surf_id) $node
    set x2 [maximum $x2 [widget_right .tree.frame.canvas $surf_id]]
  }

  # Draw rule name
  set rule_name $state($node,rule)
  if {$rule_name != ""} {
    set rule_id [.tree.frame.canvas create matrix \
		 $x1 $y2 $rule_name \
		 -font $font(tree) \
		 -char_set $char_set \
		 -tags {tag_tree tag_rule_name}]
    set node_desc($rule_id) $node
    set x2 [maximum $x2 [widget_right .tree.frame.canvas $rule_id]]
  }

  # Move surface
  if {$surface != ""} {
    .tree.frame.canvas move $surf_id \
    [expr ($x2 - [widget_right .tree.frame.canvas $surf_id]) / 2] \
    [expr ($y2 - [widget_bottom .tree.frame.canvas $surf_id])]
  }
  
  # Move rule name
  if {$rule_name != ""} {
    .tree.frame.canvas move $rule_id \
    [expr ($x2 - [widget_right .tree.frame.canvas $rule_id]) / 2] 0
  }

  # Move to the center of the next node
  set x2 [expr $x2 + $font_size(tree) / 2]

  # Draw edge
  set new_edge [.tree.frame.canvas create line \
		$x $y1 $x $y2 $x2 $y2 \
		-width 1 \
		-tags {tag_tree tag_edge}]
  set edge_widget($node) $new_edge

  return $x2
}

#------------------------------------------------------------------------------

proc tree_draw_nodes {nodes x1 y1} {
# Draw $nodes and all their successor nodes of the analysis tree at $x1/$y1.

  global state tree_mode font_size

  set max_x $x1
  set y2 $y1
  foreach node $nodes {

    set node_state $state($node,type)

    # Only show result nodes if appropriate tree mode is switched on.
    switch $tree_mode {
      "result_paths" {
	if {! $state($node,result_node)} {continue}
      }
      "no_dead_ends" {
	if {$node_state == "break"} {continue}
      }
    }
    
    # Calculate vertical position of this node.
    if [info exists sister] {
      set y2 [expr $y2 + 2.5 * $font_size(tree)]
    } else {set sister 1}

    # Draw an edge to this node
    if {$state($node,ancestor) != -1} {
      set x2 [tree_draw_edge $node $x1 $y1 $y2]
    } else {set x2 $x1}

    switch -exact $node_state {
      
      "inter" {
	# intermediate node of analysis
	tree_draw_inter_node $node $x2 $y2
	set x_y [tree_draw_nodes $state($node,children) $x2 $y2]
	set max_x [maximum $max_x [lindex $x_y 0]]
	set y2 [lindex $x_y 1]
      }

      "break" {
	# node for failing analysis step
	tree_draw_break_node $node $x2 $y2
	set max_x [maximum $max_x $x2]
      }

      "final" {
	# final node of analysis
	tree_draw_final_node $node $x2 $y2
	set max_x [maximum $max_x $x2]
      }

      "pruned" {
	# pruned node of analysis
	tree_draw_pruned_node $node $x2 $y2
	set max_x [maximum $max_x $x2]
      }
    }
  }

  return [list $max_x $y2]
}

#------------------------------------------------------------------------------

proc tree_draw {} {
# Draw the analysis tree and configure the canvas scrollbars.

  global font font_size char_set tree_input
  
  .tree.frame.canvas configure -cursor watch
  
  # Delete old tree
  .tree.frame.canvas delete tag_tree
  if [info exists node_desc] {unset node_desc}
  
  set start_x 15
  set current_y 15
  set max_x $start_x
  
  set input_id [.tree.frame.canvas create matrix \
		$start_x $current_y $tree_input \
		-font $font(tree) \
		-char_set $char_set \
		-tags {tag_tree}]
  set max_x [maximum $max_x [widget_right .tree.frame.canvas $input_id]]
  set current_y [expr ([widget_bottom .tree.frame.canvas $input_id] \
		       + 2 * $font_size(tree))]

  set x_y [tree_draw_nodes {0} $start_x $current_y]
  set max_x [maximum $max_x [lindex $x_y 0]]
  set current_y [lindex $x_y 1]
  
  .tree.frame.canvas lower tag_edge
  .tree.frame.canvas configure \
  -scrollregion \
  "0 0 [expr $max_x+15+$font_size(tree)] [expr $current_y+15+$font_size(tree)]"
  
  .tree.frame.canvas configure -cursor ""
}

#------------------------------------------------------------------------------

proc tree_mark_result_nodes {} {
# Visit all nodes of a successful path from final nodes til start
# node and mark all result nodes.
  
  global state final_nodes
  
  foreach final_node $final_nodes {
    set ancestor $final_node
    while {$ancestor > 0} {
      set state($ancestor,result_node) 1
      set ancestor $state($ancestor,ancestor)
    }
    set state($ancestor,result_node) 1
  }
}

#------------------------------------------------------------------------------

proc tree_read {} {
# Read the description of a computation tree generated by malaga.
# The trace nodes are stored in "state".
  
  global state tree_input final_nodes
  
  set tree_input [read_line stdin]
  set final_nodes {}

  while {[set line [read_line stdin]] != "end"} {
    set node [lindex $line 0]
    set state($node,type) [lindex $line 1]
    set state($node,ancestor) [lindex $line 2]
    set state($node,rule) [lindex $line 3]
    set state($node,next_surf) [lindex $line 4]
    set state($node,next_cat) [lindex $line 5]
    set state($node,result_surf) [lindex $line 6]
    set state($node,result_cat) [lindex $line 7]
    set state($node,rule_set) [lindex $line 8]
    set state($node,result_node) 0
    set state($node,children) {}
    
    # Mark this node as a child of its ancestor
    if {$state($node,ancestor) != -1} {
      lappend state($state($node,ancestor),children) $node
    }
    
    # Collect final nodes
    if {$state($node,type) == "final"} {lappend final_nodes $node}
  }
  
  # Collect all nodes that are part of a result path.
  tree_mark_result_nodes
}

#------------------------------------------------------------------------------

proc tree_redraw {mode} {
# Redraw the current tree, where $mode may be one of
# "new_fontsize", "new_mode" or "reload".
  
  global selected_start_node
  
  tree_open_window
  
  switch $mode {
    
    "new_fontsize" {
      tree_mark_path false
      tree_draw
      tree_mark_path true
    }
    
    "new_mode" {
      path_close_window
      tree_draw
    }
    
    "reload" {
      # Delete old path, but leave the window.
      tree_mark_path false
      if [info exists selected_start_node] {unset selected_start_node}
      if [info exists state] {unset state}
      path_close_window
      
      tree_read
      tree_draw
      if {[wm state .tree] == "normal"} {raise .tree}
    }
  }
}

#------------------------------------------------------------------------------

proc tree_init {} {
# Initialize the tree variables.
  
  global font font_size tree_mode final_nodes

  set font(tree) $font($font_size(tree))
  set font(path) $font($font_size(path))
  set tree_mode "complete_tree"
  set final_nodes {}
}

# end of file =================================================================
