/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This program takes a symbol file and compiles it to internal format. */

/* includes =================================================================*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "scanner.h"
#include "files.h"
#include "sym_compiler.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL

/* variables ================================================================*/

LOCAL string_t program_name; /* this is set to argv[0] by main */

/* functions ================================================================*/

LOCAL void error_handler (string_t format, ...) NO_RETURN;
LOCAL void error_handler (string_t format, ...)
/* Print an error message and exit. */
{
  va_list arg;
  
  fprintf (stderr, "%s: ", program_name);

  if (current_line_number () != -1)
    fprintf (stderr, "file \"%s\", line %ld, column %ld: ", 
	     name_in_path (current_file_name ()), current_line_number (), 
	     current_column ());

  va_start (arg, format);
  vfprintf (stderr, format, arg);
  va_end (arg);

  fprintf (stderr, "\n");

  if (getenv ("MALAGA_MODE") != NULL && current_line_number () != -1)
    printf ("SHOW \"%s\":%ld:%ld\n", 
	    current_file_name (), current_line_number (), current_column ());

  exit (1);
}

/*---------------------------------------------------------------------------*/

LOCAL void program_message (void)
/* Print some information about the program. */
{
  printf ("%s (%s) - Copyright (C) 1995-1999 Bjoern Beutel\n",
	  program_name, MALAGA_VERSION);
  printf ("It is part of Malaga, a system for Natural Language Analysis.\n");
  printf ("This program comes with ABSOLUTELY NO WARRANTY.\n");
  printf ("This is free software which you may redistribute "
	  "under certain conditions.\n");
  printf ("For details, refer to the GNU General Public License.\n");
}

/*---------------------------------------------------------------------------*/

GLOBAL int main (int argc, string_t argv[])
/* The main function of "malsym". */
{
  string_t symbol_file, object_file, old_symbol_file;
  int_t i;

  error = error_handler;
  program_name = name_in_path (argv[0]);

#ifdef HANGUL
  init_hangul ();
#endif

  if (argc == 2 && (strcmp_no_case (argv[1], "-version") == 0
		    || strcmp_no_case (argv[1], "-v") == 0))
  {
    program_message ();
    exit (0);
  }
  
  symbol_file = object_file = old_symbol_file = NULL;
  for (i = 1; i < argc; i++) 
  {
    string_t argument;

    if (*argv[i] == '-') 
      argument = new_string (argv[i], NULL);
    else
      argument = absolute_path (argv[i], NULL);
    
    if (has_extension (argument, "sym")) 
    {
      set_file_name (&object_file, argument, "sym_c");
      symbol_file = new_string (argument, NULL);
    } 
    else if (has_extension (argument, "esym")) 
    {
      set_file_name (&object_file, argument, "esym_c");
      symbol_file = new_string (argument, NULL);
    } 
    else if (has_extension (argument, "sym_c")) 
      set_file_name (&old_symbol_file, argument, NULL);
    else
      error ("illegal argument \"%s\"", argument);

    free_mem (&argument);
  }
  
  if (object_file == NULL)
    error ("missing symbol file name");

  init_values ();

  compile_symbol_file (symbol_file, object_file, old_symbol_file);

  terminate_values ();

  free_mem (&object_file);
  free_mem (&symbol_file);
  free_mem (&old_symbol_file);

#ifdef HANGUL
  terminate_hangul ();
#endif

  return 0;
}

/* end of file ==============================================================*/
