//=======================================================================
//	videcnv.cpp:	Source for vedTextEditor class
//  Copyright (C) 1995-1999  Bruce E. Wampler
//
//  This program is part of the V C++ GUI Framework example programs.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  (see COPYING) along with this program; if not, write to the Free
//  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//=======================================================================

#include <v/vnotice.h>
#include <v/vutil.h>
#include <ctype.h>	// for tolower
#include <v/vos.h>	// for rename

#define VCmdWindow videCmdWindow
#include "videapp.h"
#include "videcnv.h"
#include "videcmdw.h"

    const int maxBuff = 1000;

//===================>>> vedTextEditor::vedTextEditor <<<====================
  vedTextEditor::vedTextEditor(VCmdWindow* parent) : 
      vTextEditor((vCmdWindow*)parent)
  {
  }

//===================>>> vedTextEditor::~vedTextEditor <<<====================
  vedTextEditor::~vedTextEditor()
  {
  }

#define isSpace(x) (x == ' ' || x == '\t')

#define isOp(x) (x=='+' || x=='-' || x=='*' || x=='/' || x=='<' || x=='>' \
	|| x=='?' || x==':' || x==';' || x=='!' || x=='%' || x=='^' || x=='&' \
        || x=='\\' || x=='|' || x == '=' || x==',' || x=='\'' || x=='\"' \
        || x==')' || x=='(' || x=='[' || x==']' || x=='{' || x=='}')


// ======================>>> vTextEditor::paintLine <<<=====================
  void vedTextEditor::paintLine(char* linout, int lineStart,
        int hiStart, int hiLast, long lineNum)
  {
    // paint a line.
    // linout: the line to output with tabs converted to spaces, etc.
    // lineStart: where to begin printing the line (for hoiz. scrolling)
    // hiStart, hiLast: reverse text attribute
    // lineNum: the real line number in the buffer this is.
    // This version overrides the original to handle syntax highlighting

    ChrAttr attrs[MAX_LINE+1];	// for attributes
    int wasComment = 0;

    int linlen = strlen(linout);
    if (linlen <= 0)             // only draw if there!
        return;

    for (int ix = 0 ; ix <= MAX_LINE ; ++ix)	// assume normal
        attrs[ix] = ChNormal;

    // Parse the line for syntax

    if (GetFileType() == CPP)			// if a C file, parse
        wasComment = parseC(linout,attrs,lineNum,CPP);
    else if (GetFileType() == Java)			// if a C file, parse
        wasComment = parseC(linout,attrs,lineNum,Java);
    else if (GetFileType() == HTML)			// if a C file, parse
        wasComment = parseHTML(linout,attrs,lineNum);
    else if (GetFileType() == gccError)
      {
        int ig = 0;
        while (isSpace(linout[ig]))
          ++ig;
        if (linout[ig] == '>' || linout[ig] == '+')
          {
	    if (linout[ig] == '+')
		attrs[ig++] = ChGreen | ChDimColor;
            while (linout[ig])
                attrs[ig++] = ChBlue;
          }
        else if (linout[ig] == '!' || linout[ig] == '*')
          {
	    if (linout[ig] == '*')
		attrs[ig++] = ChGreen | ChDimColor;
            while (linout[ig])
                attrs[ig++] = ChRed;
          }
      }

    // Now fill in highlight attributes
    for (int ih = 0 ; linout[ih] != 0 ; ++ih)
      {
        if (ih >= hiStart && ih < hiLast)
            attrs[ih] = getHighlight();		// override syntax colors
      }


    for (int ixx = lineStart ; linout[ixx] != 0 ; ++ixx)
	DrawChar(linout[ixx],attrs[ixx]);

  }

//===================>>> vedTextEditor::TextMouseDown <<<====================
  void vedTextEditor::TextMouseDown(int row, int col, int button)
  {
    int btn = (GetFileType() == gccError) ? 1 : button;

    vTextEditor::TextMouseDown(row, col, btn);	// translate to left
  }

//===================>>> vedTextEditor::TextMouseUP <<<====================
  void vedTextEditor::TextMouseUp(int row, int col, int button)
  {
    int btn = (GetFileType() == gccError) ? 1 : button;
    vTextEditor::TextMouseUp(row, col, btn);
    if (button == 3 && GetFileType() == gccError)  // open error file
      {
        ((VCmdWindow*)_parent)->GotoErrorLine();
      }
  }

//===================>>> vedTextEditor::parseC <<<====================
  int vedTextEditor::parseC(char* linout, ChrAttr* attrs, 
	long lineNum, FileType ft)
  {
    // FileType will be either CPP or Java

    int ix = 0;
    int wasComment = 0;
    char token[MAX_LINE+1];

    for (ix = 0 ; linout[ix] != 0 ; ++ix)
        if (!isSpace(linout[ix]))
            break;			// skip leading spaces

    if (linout[ix] == '*')		// probably a * comment
      {
	for (long ln = lineNum + 1 ;
	     ln < lineNum + 200 && ln <= lastLineBF() ; ++ln)
	  {
	    if (strstr((char*)GLine(ln),"//")!=0)
		break;
	    else if (strstr((char*)GLine(ln),"/*")!=0)
		break;
	    else if (strstr((char*)GLine(ln),"*/")!=0)	// is part of comment
	      {
                /* A comment...
                 * of this form, widely used in Java
                 */
                int ij;
                for (ij = ix ; linout[ij] ; ++ij)
                  {
                    attrs[ij] = ChYellow | ChDimColor;
                    if (linout[ij] =='/' && linout[ij-1] == '*') // "*/"
                        break;
                  }
                return 1;
	      }
	  }

      }

    while (linout[ix] != 0)
      {
        // gather a token
        int iy;
        for (iy = ix ; linout[iy] != 0 ; ++iy)
          {
            if (isOp(linout[iy]) || isSpace(linout[iy]) )
                break;
          }
        // token goes from ix to iy
        int from = ix;
        if (ix == iy)
            ++iy;		// single char tokens

        char *tp;
        for (tp = token ; from < iy ; ++from)
          *tp++ = linout[from];
        *tp = 0;
        // OK, now highlight the token appropriately
        if (isOp(*token))
          {
            if (*token == '"')
              {
                // A String...
                int ij;
                for (ij = ix ; linout[ij] ; ++ij)
                  {
                    attrs[ij] = ChRed;
                    if (linout[ij] == '\\')
                      {
		        if (linout[ij+1])
			  {
                            attrs[ij+1] = ChRed;
                            ++ij;
			  }
                      }
                    else if (linout[ij] == '"' && ij != ix)
                        break;
                  }
                if (linout[ij] == 0)
		    break;
		else
		    iy = ij+1;
              }
            else if (*token == '/' && linout[ix+1] == '/')
              {
                // A comment...
                int ij;
                for (ij = ix ; linout[ij] ; ++ij)
                    attrs[ij] = ChGreen | ChDimColor;
                iy = ij;
                wasComment = 1;
              }
            else if (*token == '/' && linout[ix+1] == '*')
              {
                /* A comment... */
                int ij;
                for (ij = ix ; linout[ij] ; ++ij)
                  {
                    attrs[ij] = ChYellow | ChDimColor;
                    if (linout[ij] =='/' && linout[ij-1] == '*') // "*/"
                        break;
                  }
                wasComment = 1;
                if (linout[ij] == 0)
                    break;
                else
                    iy = ij+1;
              }
            else if (*token == '*' && linout[ix+1] == '/' && linout[ix+2] != '*')
              {
                /* A comment...  with trailing * / */
                int ij;
                for (ij = 0 ; linout[ij] && ij <= ix+1 ; ++ij)
                  {
                    attrs[ij] = ChYellow | ChDimColor;
                  }
                wasComment = 1;
                if (linout[ij] == 0)
                    break;
                else
                    iy = ij+1;
              }
            else
                attrs[ix] |= ChBlue | ChDimColor;
          }
        else if (*token == '#' && ft == CPP)
          {
            for (int ij = ix ; ij < iy ; ++ij)
                attrs[ij] |= ChCyan | ChDimColor;
          }
        else if (*token >= '0' && *token <= '9')
          {
            for (int ij = ix ; ij < iy ; ++ij)
                attrs[ij] |= ChRed;
          }
        else if (ft == CPP && isCKeyWord(token))
          {
            for (int ij = ix ; ij < iy ; ++ij)
                attrs[ij] |= ChBlue;
          }
        else if (ft == Java && isJavaKeyWord(token))
          {
            for (int ij = ix ; ij < iy ; ++ij)
                attrs[ij] |= ChBlue;
          }

        ix = iy;
      }
    return wasComment;
  }

//===================>>> vedTextEditor::parseHTML <<<====================
  int vedTextEditor::parseHTML(char* linout, ChrAttr* attrs, long lineNum)
  {
    char prevtok;		// track it here!
    int ix = 0;
    int wasComment = 0;
    char token[MAX_LINE+1];

    for (ix = 0 ; linout[ix] != 0 ; ++ix)
        if (!isSpace(linout[ix]))
            break;			// skip leading spaces
    prevtok = 0;

    while (linout[ix] != 0)
      {
        // gather a token
        int iy;
        for (iy = ix ; linout[iy] != 0 ; ++iy)
          {
            if (linout[iy] == '<' || linout[iy] == '>' ||
	      linout[iy] == '\"' || linout[iy] == '/' ||
              linout[iy] == '&' || isSpace(linout[iy]) )
                break;
          }
        // token goes from ix to iy

        int from = ix;
        if (ix == iy)
            ++iy;		// single char tokens

        char *tp;
        for (tp = token ; from < iy ; ++from)
          *tp++ = linout[from];
        *tp = 0;
        // OK, now highlight the token appropriately


        if (*token == '<' || (*token == '/' && prevtok == '<'))
          {
	    attrs[ix] = ChBlue;
	    prevtok = *token;
	  }
	else if (*token == '>')
	  {
	    attrs[ix] = ChBlue;
	    prevtok = '>';
	  }
        else if (*token == '&')
          {
            attrs[ix] = ChRed;
            prevtok = '&';
          }
        else if (*token == '"')
	      {
		// A String...
                prevtok = *token;
		int ij;
		for (ij = ix ; linout[ij] ; ++ij)
		  {
		    attrs[ij] = ChRed;
		    if (linout[ij] == '\\')
		      {
			attrs[ij+1] = ChRed;
			++ij;
		      }
		    else if (linout[ij] == '"' && ij != ix)
			break;
		  }
		iy = ij+1;
	      }
	    else if (*token >= '0' && *token <= '9')
	      {
                prevtok = *token;
		for (int ij = ix ; ij < iy ; ++ij)
		    attrs[ij] |= ChRed;
	      }
	    else if ((prevtok == '<' || prevtok == '/') && isHTMLKeyWord(token))
	      {
                prevtok = *token;
		for (int ij = ix ; ij < iy ; ++ij)
		    attrs[ij] |= ChBlue;
	      }
            else
            	prevtok = *token;

	ix = iy;
      }

    return wasComment;
  }

//===================>>> vedTextEditor::isCKeyWord <<<====================
  int vedTextEditor::isCKeyWord(char* token)
  {
    static char* keywords[] =
     {
	"FALSE", "NULL", "TRUE",
	"asm", "auto", "bool", "break", 
	"case", "catch", "char", "class", "const", "const_cast", 
	"continue", "default", "delete", "do", "double", 
	"dynamic_cast", "else", "enum", "explicit", "extern", 
	"false", "float", "for", "friend", "goto", "if", 
	"inline", "int", "long", "mutable", "namespace", "new",
	"operator", "private", "protected", "public", "register", 
	"reinterpret_cast", "return", "short", "signed", "sizeof", 
	"static", "static_cast", "struct", "switch", "template", 
	"this", "throw", "true", "try", "typedef", "typeid", 
	"typename", "union", "unsigned", "using", "virtual", "void", 
	"volatile", "wchar_t", "while", ""
     };

     for (int ix = 0 ; *keywords[ix] ; ++ix)
       {
         if (strcmp(token,keywords[ix]) == 0)
             return 1;
       }
     return 0;
  }

//===================>>> vedTextEditor::isJavaKeyWord <<<====================
  int vedTextEditor::isJavaKeyWord(char* token)
  {
    static char* keywords[] =
    {
	"abstract", "boolean", "break", "byte",
	"case", "catch", "char", "class", "const",
	"continue", "default", "do", "double", 
	"else", "extends",
	"false", "final", "finally", "float", "for", "future",
	"generic", "goto", "if", 
	"implements", "import", "inner", "instanceof", "int",
	"interface", "long", "native", "new", "null",
	"operator", "outer", 
	"private", "protected", "public", "rest", 
	"return", "short",
	"static", "super", "switch", "synchronized",
	"this", "throw", "throws", "true", "try",
	"var", "void", 
	"volatile", "while", ""
     };

     for (int ix = 0 ; *keywords[ix] ; ++ix)
       {
         if (strcmp(token,keywords[ix]) == 0)
             return 1;
       }
     return 0;
  }

//===================>>> vedTextEditor::isHTMLKeyWord <<<====================
  int vedTextEditor::isHTMLKeyWord(char* token)
  {
    static char* keywords[] =
     {
	"a", "address", "applet", "area", "b", "base", "basefont",
	"bdo", "bgsound", "big", "blink", "blockquote", "body",
	"br", "button", "caption", "center", "cite", "code", "col",
	"colgroup", "dd", "del", "dfn", "dir", "div", "dl",
	"dt", "em", "embed", "filedset", "font", "form", "frame",
	"frameset", "h1", "h2", "h3", "h4", "h5", "h6", "head",
	"hr", "html", "i", "iframe", "ilayer", "img", "input",
	"ins", "isindex", "kbd", "label", "layer", "legend",
	"li", "link", "listing", "map", "marquee", "menu", "meta",
	"multicol", "nobr", "noframes", "nolayer", "noscript",
	"object", "ol", "option", "p", "param", "plaintext",
	"pre", "q", "s", "samp", "script", "select", "small",
	"spacer", "span", "strike", "strong", "style", "sub",
	"sup", "table", "tbody", "td", "textarea", "tfoot",
	"th", "thead", "title", "tr", "tt", "u", "ul", "var",
	"wbr", "xmp", ""
     };
    char tryword[80];
    // char* cp;
    int tk;

    for (tk = 0 ; tk < 79 && *token ; )
	tryword[tk++] = tolower(*token++);
    tryword[tk] = 0;

    for (int ix = 0 ; *keywords[ix] ; ++ix)
       {
         if (strcmp(tryword,keywords[ix]) == 0)
             return 1;
       }
     return 0;
  }

//===================>>> vedTextEditor::ChangeLoc <<<====================
  void vedTextEditor::ChangeLoc(long line, int col)
  {
    ((VCmdWindow*)_parent)->ChangeLoc(line,col);
  }

//===================>>> vedTextEditor::ChangeInsMode <<<===================
  void vedTextEditor::ChangeInsMode(int IsInsMode, char* msg)
  {
    ((VCmdWindow*)_parent)->ChangeInsMode(IsInsMode, msg);
  }

//===================>>> vedTextEditor::StatusMessage <<<====================
  void vedTextEditor::StatusMessage(char *msg)
  {
    if (state.echof)
	((VCmdWindow*)_parent)->StatusMessage(msg);
  }

//===================>>> vedTextEditor::ErrorMsg <<<====================
  void vedTextEditor::ErrorMsg(char *str)
  {
    if (state.echof)
      {
	vBeep();
	((VCmdWindow*)_parent)->StatusMessage(str);
      }
  }

//===================>>> vedTextEditor::ReadFile <<<====================
  int vedTextEditor::ReadFile(char* name, int paintIt)
  {

    char buff[maxBuff+2];

    if (!name || !*name)
	return 0;

    ifstream inFile(name);

    if (!inFile)
	return 0;		// file not there

    resetBuff();		// this buffer is empty

    while (inFile.getline(buff,maxBuff))
      {
	if (!addLine(buff))
	  {
	    vNoticeDialog note(theApp);
	    note.Notice("File too big -- only paritally read.");
	    break;
	  }
      }

    inFile.close();

    if ( ((videApp*)theApp)->getBackup())
      {
	char outname[maxBuff+2];
	vOS os;
	if (strstr(name,".bak") == 0 && strlen(name) < maxBuff - 4)
	  {
	    strcpy(outname,name); strcat(outname,".bak");
	    os.vDeleteFile(outname);	// delete old .bak first
	    if (os.vRenameFile(name,outname))
		SaveFile(name);
	  }
      }

    displayBuff(1,paintIt);		// Now, display the buffer , don't paint
    return 1;
  }

//===================>>> vedTextEditor::SaveFile <<<====================
  int vedTextEditor::SaveFile(char* name)
  {
    char buff[maxBuff+2];

    char* prefix = "Saved ";

    ofstream ofs(name);

    if (!ofs)
      {
	StatusMessage("Can't save file");
	return 0;
      }

    for (long lx = 1 ; lx <= GetLines() ; ++lx)
      {
	getLine(buff,maxBuff,lx);
	ofs << buff << "\n";
      }

    ofs.close();

    strcpy(buff,prefix);

    int ix;
    for (ix = strlen(prefix) ; ix < 40 && *name != 0 ; ++ix)
	buff[ix] = *name++;
    buff[ix] = 0;
    StatusMessage(buff);

    state.changes = 0;
    return 1;
  }
