/* File "rule_code.h":
 * Manages the emission of instructions and keeps track of the stack index.
 * Holds buffers for the compiled code. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

GLOBAL struct /* internal structure that contains compiled Malaga rule code. */
{
  short_t file_type;       /* ALLO_RULE_FILE, MORPHO_RULE_FILE, 
			    * SYNTAX_RULE_FILE or FILTER_RULE_FILE. */
  long_t rule_number;      /* the current rule number */
  long_t stack_index;      /* the current stack index */
  long_t next_instr_index; /* the index of the instruction to be emitted */

  /* the following values must be copied to the rule file */
  long_t initial_rule_set; /* index into <rule_set_pool> */
  long_t initial_cat;      /* index into <value_pool> */
  long_t robust_rule;      /* index into <rule_pool> */
  long_t pruning_rule;     /* index into <rule_pool> */
  long_t allo_rule;        /* index into <rule_pool> */
  long_t filter_rule;      /* index into <rule_pool> */
  long_t input_rule;       /* index into <rule_pool> */
  pool_t rule_pool;        /* the pool of all rules */
  pool_t rule_set_pool;    /* the pool of rule sets */
  pool_t instr_pool;       /* the pool of all instructions */
  pool_t value_pool;       /* the pool of all constant Malaga values */
  pool_t src_line_pool;    /* the pool of all correspondences
                            * between source lines and rule code */
  pool_t var_pool;         /* the pool of all variables */
  pool_t var_scope_pool;   /* the pool of all variable scopes */
  pool_t string_pool;      /* the pool of all strings
                            * (variable and rule names, patterns) */
} code;

/* possible values of file_type */
#define ALLO_RULE_FILE 0
#define MORPHO_RULE_FILE 1
#define SYNTAX_RULE_FILE 2
#define FILTER_RULE_FILE 3

extern void init_code (short_t file_type);
/* Initialise <code> to contain compilation data for a file of <file_type>.
 * <file_type> may be ALLO_RULE_FILE, MORPHO_RULE_FILE,
 * SYNTAX_RULE_FILE, or FILTER_RULE_FILE. */

extern void free_code (void);
/* Free all memory used by <code>. */

extern void write_code (string_t file_name);
/* Write <code> to <file_name>. */

/* functions for constant folding ===========================================*/

extern void buffer_instr (u_byte_t opcode, long_t info);
/* Buffer the instructions BUILD_LIST, BUILD_RECORD, PUSH_SYMBOL,
 * and PUSH_CONST for constant folding. */

extern void buffer_push_const_instr (value_t value);
/* Buffer the instruction PUSH_CONST with constant <value>. */

extern void flush_buffer (void);
/* Emit the instructions that are still in the buffer. */

extern value_t get_buffer_top_value (void);
/* Test if the buffer contains a value and return the top value. 
 * The value is valid until "clear_value_heap" is called again. */

extern instr_t *emit_instr (u_byte_t opcode, long_t info);
/* Emit an instruction to the instruction pool (flushes buffer before)
 * and return the address of the instruction in the pool. */
