/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#define BVMBugEntry			(0xF8000404)

#define BB_PUTCHAR			0
#define BB_PUTSTR			1
#define BB_PRINT			2
#define BB_FORMAT			3
#define BB_GETCHAR			4
#define BB_GETCHAR_NW		5
#define BB_GETSTR			6
#define BB_ATOI				7
#define BB_MALLOC			8
#define BB_FREE				9
#define BB_MEMCMP			10
#define BB_MEMSET			11
#define BB_MEMMOVE			12
#define BB_DISK_OPEN		13
#define BB_DISK_READ		14
#define BB_DISK_CLOSE		15
#define BB_FILE_OPEN		16
#define BB_FILE_READ		17
#define BB_FILE_CLOSE		18
#define BB_GETTIME			19
#define BB_DEFINE_SYM		20
#define BB_DELETE_SYM		21
#define BB_ENTER			22

/*--------------------------------------------------------------------------*/
/* BVMBug error codes
 */

#define DISK_READ_ERROR		1
#define OUT_OF_MEMORY		2
#define INVALID_FILE_MAP	3
#define ILLEGAL_SYMBOL_NAME	4

/*--------------------------------------------------------------------------*/

typedef struct {
	char		boot_id[8];				/* id string "BVM LTD"				*/
#define BOOT_BLOCK_ID  "BVM LTD"

	unsigned long boot_start_block;		/* linear bootfile start block addr	*/
										/* if zero boot_data[] holds user	*/
										/* code or file map of non-linear	*/
										/* bootfile							*/

	unsigned long boot_block_count;		/* number of blocks in linear and	*/
										/* non-linear bootfiles		 		*/
										/* zero indicates boot_data[] holds	*/
										/* machine instructions	which are	*/
										/* moved to the load address and	*/
										/* executed							*/
										
	unsigned long boot_load_address;	/* address to load bootfile	or user	*/
										/* code, if -1 the data is stored	*/
										/* in memory allocated from BVMBug's*/
										/* local memory pool				*/
#define BOOT_ALLOC_MEM		(0xffffffff)

	unsigned long boot_run_offset;		/* offset from load address to		*/
										/* begin execution					*/
										/* if -1 take SP and PC from start	*/
										/*       of loaded data				*/
#define BOOT_RUN_VECTORED	(0xffffffff)

	char		boot_message[32];		/* message to display				*/

#define BOOT_DATA_SZ		(390)
	unsigned char boot_data[BOOT_DATA_SZ];
										/* boot data area:					*/
										/* may contain a file map describing*/
										/* the location of the bootfile or	*/
										/* it may contain user startup code */

} BOOTBLOCK;

/*--------------------------------------------------------------------------*/
/* Call BVMBug putchar subroutine.
 */

static
__inline__
void
BVMBug_putchar
(	int		c
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_PUTCHAR;
	register unsigned long	d1 asm("d1") = (unsigned long) c;

	asm volatile (	"jsr (%0)"
					: /* no outputs */
					: "a" (BVMBugEntry), "r" (d0), "r" (d1)
					: "d0", "d1"
	);
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug putstr subroutine.
 */

static
__inline__
void
BVMBug_putstr
(	const char		*str
)
{	register unsigned long	 d0 asm("d0") = (unsigned long) BB_PUTSTR;
	register void			*a0 asm("a0") = (void *) str;

	asm volatile (	"jsr (%0)"
					: /* no outputs */
					: "a" (BVMBugEntry), "r" (d0), "r" (a0)
					: "d0", "a0"
	);
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug print subroutine.
 */

static
__inline__
int
BVMBug_print
(	const char		*str,
	void			*vap
)
{	register unsigned long	 d0 asm("d0") = (unsigned long) BB_PRINT;
	register void			*a0 asm("a0") = (void *) str;
	register void			*a1 asm("a1") = (void *) vap;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (a0), "r" (a1)
					: "d0", "a0", "a1"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug format subroutine.
 */

static
__inline__
int
BVMBug_format
(	char			*buf,
	const char		*str,
	void			*vap
)
{	register unsigned long	 d0 asm("d0") = (unsigned long) BB_FORMAT;
	register void			*a0 asm("a0") = (void *) str;
	register void			*a1 asm("a1") = (void *) vap;
	register void			*a2 asm("a2") = (void *) buf;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (a0),
										 "r" (a1), "r" (a2)
					: "d0", "a0", "a1", "a2", "memory"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug getchar subroutine.
 */

static
__inline__
int
BVMBug_getchar
(	void
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_GETCHAR;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0)
					: "d0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug getchar_nowait subroutine.
 */

static
__inline__
int
BVMBug_getchar_nowait
(	void
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_GETCHAR_NW;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0)
					: "d0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug getstr subroutine.
 */

static
__inline__
int
BVMBug_getstr
(	void	*buffer,
	int		maxlen
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_GETSTR;
	register unsigned long	d1 asm("d1") = (unsigned long) maxlen;
	register void*			a0 asm("a0") = (void *) buffer;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (d1), "r" (a0)
					: "d0", "d1", "a0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug atoi subroutine.
 */

static
__inline__
int
BVMBug_atoi
(	char	*string,
	int		base,
	unsigned long	*result
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_ATOI;
	register unsigned long	d1 asm("d1");
	register unsigned long	d2 asm("d2") = (unsigned long) base;
	register void *			a0 asm("a0") = (void *) string;

	asm volatile (	"jsr (%2)"
					: "=r" (d0), "=r" (d1)
					: "a" (BVMBugEntry), "r" (d0), "r" (d2), "r" (a0)
					: "d0", "d1", "d2", "a0"
	);

	*result = d1;
	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug malloc subroutine.
 */

static
__inline__
void *
BVMBug_malloc
(	unsigned long	count
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_MALLOC;
	register unsigned long	d1 asm("d1") = (unsigned long) count;
	register void *			a0 asm("a0");

	asm volatile (	"jsr (%1)"
					: "=r" (a0)
					: "a" (BVMBugEntry), "r" (d0), "r" (d1)
					: "d0", "d1", "a0"
	);

	return a0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug free subroutine.
 */

static
__inline__
void
BVMBug_free
(	void	*memptr
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_FREE;
	register void *			a0 asm("a0") = (void *) memptr;

	asm volatile (	"jsr (%0)"
					: /* no outputs */
					: "a" (BVMBugEntry), "r" (d0), "r" (a0)
					: "d0", "a0"
	);
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug memcmp subroutine.
 */

static
__inline__
int
BVMBug_memcmp
(	void			*ptr1,
	void			*ptr2,
	unsigned long	siz
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_MEMCMP;
	register unsigned long	d1 asm("d1") = (unsigned long) siz;
	register void *			a0 asm("a0") = (void *) ptr1;
	register void *			a1 asm("a1") = (void *) ptr2;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (d1), "r" (a0), "r" (a1)
					: "d0", "d1", "a0", "a1", "memory"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug memset subroutine.
 */

static
__inline__
void
BVMBug_memset
(	void	*dst,
	int		val,
	unsigned long	siz
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_MEMSET;
	register unsigned long	d1 asm("d1") = (unsigned long) val;
	register unsigned long	d2 asm("d2") = (unsigned long) siz;
	register void *			a0 asm("a0") = (void *) dst;

	asm volatile (	"jsr (%0)"
					: /* no outputs */
					: "a" (BVMBugEntry), "r" (d0), "r" (d1), "r" (d2), "r" (a0)
					: "d0", "d1", "d2", "a0", "memory"
	);
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug memmove subroutine.
 */

static
__inline__
void
BVMBug_memmove
(	void		*dst,
	const void	*src,
	unsigned long		siz
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_MEMMOVE;
	register unsigned long	d1 asm("d1") = (unsigned long) siz;
	register void *			a0 asm("a0") = (void *) dst;
	register void *			a1 asm("a1") = (void *) src;

	asm volatile (	"jsr (%0)"
					: /* no outputs */
					: "a" (BVMBugEntry), "r" (d0), "r" (d1), "r" (a0), "r" (a1)
					: "d0", "d1", "a0", "a1", "memory"
	);
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug disk_open subroutine.
 */

static
__inline__
int
BVMBug_disk_open
(	void
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_DISK_OPEN;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0)
					: "d0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug disk_read subroutine.
 */

static
__inline__
int
BVMBug_disk_read
(	void			*buffer,
	unsigned long	block,
	unsigned long	count
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_DISK_READ;
	register unsigned long	d1 asm("d1") = (unsigned long) count;
	register unsigned long	d2 asm("d2") = (unsigned long) block;
	register void *			a0 asm("a0") = (void *) buffer;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (d1), "r" (d2), "r" (a0)
					: "d0", "d1", "d2", "a0", "memory"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug disk_close subroutine.
 */

static
__inline__
void
BVMBug_disk_close
(	void
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_DISK_CLOSE;

	asm volatile (	"jsr (%0)"
					: /* no outputs */
					: "a" (BVMBugEntry), "r" (d0)
					: "d0"
	);
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug file_open subroutine.
 */

static
__inline__
int
BVMBug_file_open
(	unsigned long	file_size,
	unsigned long	map_entries,
	const void		*map
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_FILE_OPEN;
	register unsigned long	d1 asm("d1") = (unsigned long) file_size;
	register unsigned long	d2 asm("d2") = (unsigned long) map_entries;
	register void * 		a0 asm("a0") = (void *) map;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (d1), "r" (d2), "r" (a0)
					: "d0", "d1", "d2", "a0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug file_read subroutine.
 */

static
__inline__
int
BVMBug_file_read
(	void			*buffer,
	unsigned long	offset,
	unsigned long	count
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_FILE_READ;
	register unsigned long	d1 asm("d1") = (unsigned long) count;
	register unsigned long	d2 asm("d2") = (unsigned long) offset;
	register void *			a0 asm("a0") = (void *) buffer;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (d1), "r" (d2), "r" (a0)
					: "d0", "d1", "d2", "a0", "memory"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug file_close subroutine.
 */

static
__inline__
void
BVMBug_file_close
(	void
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_FILE_CLOSE;

	asm volatile (	"jsr (%0)"
					: /* no outputs */
					: "a" (BVMBugEntry), "r" (d0)
					: "d0"
	);
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug gettime subroutine.
 */

static
__inline__
void
BVMBug_gettime
(	unsigned long	*time,
	unsigned long	*date
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_GETTIME;
	register unsigned long	d1 asm("d1");

	asm volatile (	"jsr (%2)"
					: "=r" (d0), "=r" (d1)
					: "a" (BVMBugEntry), "r" (d0)
					: "d0", "d1"
	);

	if (time) *time = d0;
	if (date) *date = d1;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug define symbol subroutine.
 */

static
__inline__
int
BVMBug_define_symbol
(	const char		*name,
	unsigned long	value,
	unsigned long	type	/* 0 = code, 1 = data		*/
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_DEFINE_SYM;
	register unsigned long	d1 asm("d1") = (unsigned long) value;
	register unsigned long	d2 asm("d2") = (unsigned long) type;
	register void * 		a0 asm("a0") = (void *) name;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (d1), "r" (d2), "r" (a0)
					: "d0", "d1", "d2", "a0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug delete symbol subroutine.
 */

static
__inline__
int
BVMBug_delete_symbol
(	const char	*name		/* symbol name or NULL for all symbols			*/
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_DELETE_SYM;
	register void * 		a0 asm("a0") = (void *) name;

	asm volatile (	"jsr (%1)"
					: "=r" (d0)
					: "a" (BVMBugEntry), "r" (d0), "r" (a0)
					: "d0", "a0"
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug.
 */

static
__inline__
void
BVMBug_enter
(	void
)
{	register unsigned long	d0 asm("d0") = (unsigned long) BB_ENTER;

	asm volatile (	"jsr (%0)"
					: /* no outputs */
					: "a" (BVMBugEntry), "r" (d0)
					: "d0", "d1", "d2", "d3", "d4", "d5", "d6",
					  "a0", "a1", "a2", "a3", "a4", "memory"
	);
}

/*--------------------------------------------------------------------------*/
/* Call BVMBug reset entry.
 */

static __inline__ void BVMBug_reset(void) __attribute((noreturn));

static
__inline__ 
void
BVMBug_reset
(	void
)
{
	asm volatile (
	"
		move.w	#0x2700,%%sr	/* mask interrupts							*/
		move.l	4(%0),%%a0		/* get reset PC								*/
		move.l	(%0),%%a7		/* get reset SP								*/
		moveq	#0,%%d0			/* clear cache control register				*/
		movec	%%d0,%%cacr
		cinva	%%bc			/* invalidate data and instruction caches	*/
		jmp		(%%a0)
	"
	: /* no outputs */
	: "a" (BVMBugEntry & 0xff000000)
	: "a0", "d0"
	);

	/* stop compiler complaining that we will return */
	while (1)
		;
}

/*-----------------------------< end of file >------------------------------*/
