# copyright (C) 1997-1999 Jean-Luc Fontaine (mailto:jfontain@multimania.com)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: config.tcl,v 1.6 1999/08/09 20:25:27 jfontain Exp $}

# general
#     main window size
#     save directory
# canvas
#     size
#     background
# viewers
#     color series
#     graphs
#         samples
#     pies
#         labeling
#     freetext
#         font


namespace eval configuration {

    variable rcFileName ~/.moodssrc
    variable container
    variable interface
    variable hierarchy {canvas canvas.size canvas.colors viewers}

    proc load {arrayList} {
        foreach {name value} $arrayList {
            set ::global::$name $value
        }
    }

    proc edit {preferencesMode} {                                                                        ;# preferences mode boolean
        variable hierarchy
        variable container
        variable interface
        variable tree
        variable preferences

        set preferences $preferencesMode                                                                ;# store mode for sub sheets

        set dialog [new dialogBox .\
            -buttons oc -default o -title {moodss: Configuration} -x [winfo pointerx .] -y [winfo pointery .]\
            -command configuration::done\
        ]
        set frame [frame $widget::($dialog,path).frame]

        set tree [blt::hierbox $frame.tree\
            -font $font::(mediumBold) -separator . -selectmode single -selectbackground lightgray -hideroot 1 -borderwidth 1\
            -highlightthickness 0 -takefocus 0 -width 150\
        ]
        set container [frame $frame.container -borderwidth 1 -relief sunken]

        if {$preferences} {
            set helpMessage \
{This is the preferences dialog box for
application-wide settings.

On UNIX systems, preferences data is saved
in each user home directory under the rc
file named .moodssrc.

On Windows, data is saved in C:\.moodssrc
(by default, but depends on the HOME variable).

After selection of the category from the tree
on the left, a related dialog interface
replaces this message.

Contextual help is provided for each interface,
which may or may not allow immediately applying
new data values.

Clicking on the OK button results in the
preferences data to be written to the rc file.}
        } else {
            set helpMessage \
{This is the configuration dialog box for the
current view.

After selection of the category from the tree
on the left, a related dialog interface
replaces this message.

Contextual help is provided for each interface,
which may or may not allow immediately applying
new data values.

Clicking on the OK button results in the current
configuration data to be saved in memory. It will
be stored in the save file when requested (see
File / Save / Save as... menus).}
        }

        set message [createMessage $container.message -text $helpMessage]
        pack $message -fill both -expand 1                                                   ;# initially display help message above

        bindtags $tree [list $tree [winfo toplevel $tree] all]                                              ;# ignore class bindtags
        $tree bind all <Double-ButtonPress-1> {}                                                           ;# and selection bindings
        $tree bind all <Shift-ButtonPress-1> {}
        $tree bind all <Control-ButtonPress-1> {}
        $tree bind all <B1-Motion> {}
        # only keep opening binding (no longer on double click)
### check before toggling ###
        $tree bind all <ButtonRelease-1> "$tree toggle current; $tree toggle current"

        catch {unset interface(current)}                                                               ;# currently opened interface

        foreach entry $hierarchy {
            set index [$tree insert end $entry]                                         ;# used generated tree index as unique index
            regsub -all {\.} $entry :: interface($index,class)
            $interface($index,class)::initialize
### eventually use close command for checking ###
            $tree entry configure $index -opencommand "configuration::open $index"
        }

        pack $tree -side left -fill y -padx 2
        pack $container -fill both -expand 1 -padx 2

        dialogBox::display $dialog $frame

        wm geometry $widget::($dialog,path) 500x400                                                        ;# maintain constant size
    }

    proc open {index} {
        variable container
        variable interface

        # check validity before moving to next interface
        if {[info exists interface(current)]&&![$interface($interface(current),class)::check]} return

        destroyChildren $container
        $interface($index,class)::edit $container
        set interface(current) $index
    }

    proc done {} {
        variable interface
        variable preferences
        variable variables

        foreach name [array names interface *,class] {
            $interface($name)::apply                                                                             ;# apply new values
        }
        if {$preferences} {
            preferences::save $variables
        }
    }

    proc createMessage {path args} {                  ;# create a generic message widget with eventual option / value pairs argument
        message $path -width [winfo screenwidth .] -font $font::(mediumNormal) -justify left
        if {[llength $args]>0} {
            eval $path configure $args
        }
        return $path
    }

    proc initialize {name} {                  ;# for use by folder classes, in order to initialize local variables depending on mode
        variable preferences

        if {$preferences} {
            if {![info exists ::preferences::$name]} {                       ;# initialize from global value if rc file was not read
                set ::preferences::$name [set ::global::$name]
            }
            return [set ::preferences::$name]
        } else {
            return [set ::global::$name]
        }
    }

    proc apply {name value} {                    ;# for use by folder classes, in order to update global variables depending on mode
        variable preferences

        set namespaces ::global                                                                       ;# always update global values
        if {$preferences} {
            lappend namespaces ::preferences                                                           ;# and eventually preferences
        }
        foreach namespace $namespaces {
            if {![info exists ${namespace}::$name]||($value!=[set ${namespace}::$name])} {
                # update only when necessary in order not to activate a trace needlessly
                set ${namespace}::$name $value
            }
        }
    }

    proc variables {} {
        variable variables

        return $variables
    }

    namespace eval canvas {

        proc initialize {} {}

        proc edit {parentPath} {
            set message [configuration::createMessage $parentPath.message\
                -text "The canvas is the data viewers background area.\nYou can set its characteristics from this node down."
            ]
            pack $message -fill both -expand 1                                               ;# initially display help message above
        }

        proc check {} {return 1}

        proc apply {} {}

        proc variables {} {return {}}

        namespace eval size {

            proc variables {} {                                       ;# list of unqualified global variables handled by this folder
                return {canvasHeight canvasWidth}
            }

            proc initialize {} {
                variable height [configuration::initialize canvasHeight]
                variable width [configuration::initialize canvasWidth]
            }

            proc edit {parentPath} {
                variable height
                variable width

                set message [configuration::createMessage $parentPath.message -text {Enter size (in pixels):}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                    ;# occupy whole width
                grid rowconfigure $parentPath 0 -weight 1

                grid columnconfigure $parentPath 0 -weight 1

                set widthEntry [new spinEntry $parentPath -font $font::(mediumBold) -width 4 -list {640 800 1024 1280 1600}]
                set path $composite::($widthEntry,entry,path)
                $path configure -textvariable configuration::canvas::size::width
                # filter on positive integers and limit entry length
                wcb::callback $path before insert wcb::checkStrForNum {wcb::checkEntryLen 4}
                spinEntry::set $widthEntry $width
                grid [label $parentPath.width -font $font::(mediumBold) -text width:] -row 1 -column 1 -padx 2
                grid $widget::($widthEntry,path) -row 1 -column 2

                grid columnconfigure $parentPath 3 -weight 1

                set heightEntry [new spinEntry $parentPath -font $font::(mediumBold) -width 4 -list {400 480 600 768 1024 1280}]
                set path $composite::($heightEntry,entry,path)
                $path configure -textvariable configuration::canvas::size::height
                # filter on positive integers and limit entry length
                wcb::callback $path before insert wcb::checkStrForNum {wcb::checkEntryLen 4}
                spinEntry::set $heightEntry $height
                grid [label $parentPath.height -font $font::(mediumBold) -text height:] -row 1 -column 4 -padx 2
                grid $widget::($heightEntry,path) -row 1 -column 5

                grid [button $parentPath.apply -text Apply -command configuration::canvas::size::apply] -row 1 -column 6
                grid columnconfigure $parentPath 6 -weight 2

                set message [configuration::createMessage $parentPath.help\
                    -text "The size is immediately updated when clicking\non the Apply button."
                ]
                grid $message -sticky nsew -row 2 -column 0 -columnspan 100
                grid rowconfigure $parentPath 2 -weight 1

                bind $message <Destroy> "delete $widthEntry $heightEntry"                   ;# delete inner objects upon destruction
            }

            proc check {} {return 1}

            proc apply {} {
                variable height
                variable width

                if {![check]} return
                configuration::apply canvasHeight $height
                configuration::apply canvasWidth $width
            }

        }

        namespace eval colors {

            proc variables {} {
                return canvasBackground
            }

            proc initialize {} {
                variable background [configuration::initialize canvasBackground]
            }

            proc edit {parentPath} {
                variable background
                variable colorViewer

                set message [configuration::createMessage $parentPath.message -text {Background color:}]
                grid $message -sticky nsew -row 0 -column 0 -columnspan 100                                    ;# occupy whole width
                grid rowconfigure $parentPath 0 -weight 1

                grid columnconfigure $parentPath 0 -weight 1

                set colorViewer\
                    [button $parentPath.choose -text Choose... -command "configuration::canvas::colors::choose $parentPath"]
                updateColorViewer
                grid $colorViewer -row 1 -column 1
                grid [button $parentPath.apply -text Apply -command configuration::canvas::colors::apply] -row 1 -column 2

                grid columnconfigure $parentPath 1 -weight 1
                grid columnconfigure $parentPath 2 -weight 1
                grid columnconfigure $parentPath 3 -weight 1

                set message [configuration::createMessage $parentPath.help\
                    -text "The color is immediately updated when clicking\non the Apply button."
                ]
                grid $message -sticky nsew -row 2 -column 0 -columnspan 100
                grid rowconfigure $parentPath 2 -weight 1
            }

            proc check {} {return 1}

            proc apply {} {
                variable background

                if {![check]} return
                configuration::apply canvasBackground $background
            }

            proc updateColorViewer {} {
                variable colorViewer
                variable background

                foreach {red green blue} [winfo rgb $colorViewer $background] {}
                if {($red+$green+$blue)>=(32768*3)} {                                                            ;# light background
                    $colorViewer configure -foreground black
                } else {                                                                                          ;# dark background
                    $colorViewer configure -foreground white
                }
                $colorViewer configure -background $background
            }

            proc choose {parentPath} {
                variable background

                set choice [tk_chooseColor -initialcolor $background -title {Choose color:} -parent $parentPath]
                if {[string length $choice]>0} {
                    set background $choice
                    updateColorViewer
                }
            }

        }

    }


    namespace eval viewers {

        proc variables {} {
            return graphNumberOfIntervals
        }

        proc initialize {} {
            variable numberOfSamples [configuration::initialize graphNumberOfIntervals]
        }

        proc edit {parentPath} {
            variable numberOfSamples

            set message [configuration::createMessage $parentPath.message -text {Enter number of samples for data graphs:}]
            grid $message -sticky nsew -row 0 -column 0 -columnspan 100
            grid rowconfigure $parentPath 0 -weight 1

            grid columnconfigure $parentPath 0 -weight 1

            set entry [new spinEntry $parentPath -font $font::(mediumBold) -width 4 -list {20 50 100 150 200 300 500 1000}]
            set path $composite::($entry,entry,path)
            $path configure -textvariable configuration::viewers::numberOfSamples
            # filter on positive integers and limit entry length
            wcb::callback $path before insert wcb::checkStrForNum {wcb::checkEntryLen 4}
            spinEntry::set $entry $numberOfSamples
            grid [label $parentPath.width -font $font::(mediumBold) -text samples:] -row 1 -column 1 -padx 2
            grid $widget::($entry,path) -row 1 -column 2

            grid columnconfigure $parentPath 3 -weight 1

            set message [configuration::createMessage $parentPath.help\
                -text \
{This is the number of samples (on the X axis)
for data graph viewers. The specified value will
not be used for existing graphs but for newly
created ones.}
            ]
            grid $message -sticky nsew -row 2 -column 0 -columnspan 100
            grid rowconfigure $parentPath 2 -weight 1

            bind $message <Destroy> "delete $entry"                                         ;# delete inner objects upon destruction
        }

        proc check {} {return 1}

        proc apply {} {
            variable numberOfSamples

            if {![check]} return
            configuration::apply graphNumberOfIntervals $numberOfSamples
        }

    }

    # contruct configuration variables list which can be used at any time (i.e. when saving configuration to file)
    variable variables {}
    foreach entry $hierarchy {
        regsub -all {\.} $entry :: class
        set variables [concat $variables [${class}::variables]]
    }

}
