#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <confdb.h>
#include <fviews.h>
#include <subsys.h>
#include <configf.h>
#include "suse.h"

#define SUSE_DEBUG 0


static CONFIG_FILE f_rcconfig("/etc/rc.config",help_nil
	,CONFIGF_MANAGED|CONFIGF_OPTIONNAL
	,"root","root",0644);

enum SUSE_FILE { SUSE_rcconfig, SUSE_dummy };


enum SUSE_TYPE {SUSE_STR,SUSE_YESNO,SUSE_ONOFF,SUSE_VAL0, SUSE_VALU, SUSE_NET,SUSE_NM};

struct SUSELK{
	const char *key1;
	const char *key2;
	const char *var;
	SUSE_FILE file;
	SUSE_TYPE type;
        // some parts of rc.config are dynamic, e.g. NETDEV_x, so we do a special handling here
        // in key2 there will only be the prefix of key2 and var, e.g. device, NETDEV_ for NETDEV_x handling
        int devno;
};



static SUSELK tblk[]={
	{"linuxconf",	"keymap",	"KEYTABLE",		SUSE_rcconfig,	SUSE_STR,	-1},
	{"datetime",	"universal",	"GMT",			SUSE_rcconfig,	SUSE_VALU,	-1},

	{"network",	"hostname",	"FQHOSTNAME",		SUSE_rcconfig,	SUSE_STR,	-1},
	/* entries for the net devices */
	{"IINFO",	"device",	"NETDEV_",		SUSE_rcconfig,	SUSE_STR,	-1},
	/* entries for the ip address of the device */
	{"ipaddr",	"adaptor",	"IPADDR_",		SUSE_rcconfig,	SUSE_STR,	-1},

	/* entries for the activity of a device; have a look at SUSE_NET cases in
	   getvalfromrh, updaterh */
	{"netenable",	"device",	"NETCONFIG",		SUSE_rcconfig,	SUSE_NET,	-1},


	/* Special dummy entry. We are using pseudo entry in conf.linuxconf */
	/* to handle distribution "preferences" */
	{"--distrib--",	"sysvrestart",	"",			SUSE_dummy,	SUSE_STR,	-1},
	{"--distrib--",	"privgroup",	"",			SUSE_dummy,	SUSE_STR,	-1},
	{"--distrib--",	"sysvenh",	"",			SUSE_dummy,	SUSE_STR,	-1},
	{"--distrib--",	"rpmvendor",	"",			SUSE_dummy,	SUSE_STR,	-1},
	{"--distrib--",	"rpmdistribution",	"",		SUSE_dummy,	SUSE_STR,	-1},
	{"--distrib--",	"release",			"",		SUSE_dummy,	SUSE_STR,	-1},
};
#define NBLK (sizeof(tblk)/sizeof(tblk[0]))


class CONFDB_SUSE: public CONFDB{
	CONFDB *orig;	// Original CONFDB object managing conf.linuxconf
	struct {
		VIEWITEMS *rcconfig;
		bool rcconfig_modified;
	} sys;
	SSTRINGS tbstr;		// Old temporary strings returned
						// to caller of getval
	/*~PROTOBEG~ CONFDB_SUSE */
public:
	CONFDB_SUSE (CONFDB *_orig);
	void add (const char *prefix,
		 const char *key,
		 const char *val);
	int archive (SSTREAM&ss, const char *_sys);
	int extract (SSTREAM&ss, const char *_sys);
	int getall (const char *prefix,
		 const char *key,
		 SSTRINGS&lst,
		 bool copy);
	const char *getval (const char *prefix,
		 const char *key,
		 const char *defval);
private:
	const char *getvalfromrh (SUSELK *rh);
public:
	void removeall (const char *prefix,
		 const char *key);
private:
	void reset_modified (void);
public:
	int save (PRIVILEGE *priv);
	void setcursys (const char *_subsys);
#if SUSE_DEBUG
private:
        void suse_debug(const char *str,const char *str2) { FILE *fout = fopen ("/var/log/linuxconf_suse_module.log","a"); fprintf (fout,"%s> %s\n",str,str2); fclose (fout); }
        void suse_debug(const char *str,char * str2) { suse_debug(str,(const char*)str2); }
        void suse_debug(const char *str,int val) { char str2[200]; sprintf(str2,"%d",val); suse_debug(str,str2); }
#else
#define suse_debug(x,y) ;
#endif
private:
	void updaterh (SUSELK *rh, const char *val);
public:
	~CONFDB_SUSE (void);
	/*~PROTOEND~ CONFDB_SUSE */
};

PRIVATE void CONFDB_SUSE::reset_modified()
{
	sys.rcconfig_modified = false;
}

PUBLIC CONFDB_SUSE::CONFDB_SUSE(CONFDB *_orig)
{
	orig = _orig;
	sys.rcconfig = NULL;
	reset_modified();
	suse_debug("SUSE_VERSION",suse_release());
}

PUBLIC CONFDB_SUSE::~CONFDB_SUSE()
{
	delete orig;
	delete sys.rcconfig;
}

PUBLIC void CONFDB_SUSE::setcursys (const char *_subsys)
{
	orig->setcursys (_subsys);
}

static SUSELK *filter_lookup (const char *key1, const char *key2)
{
	SUSELK *ret = NULL;
	SUSELK *pt = tblk;
	for (unsigned i=0; i<NBLK; i++, pt++){
	        if (strcmp(pt->key1,key1)==0) {
		        if (strcmp(pt->key2,key2)==0){
			        ret = pt;
				ret->devno = -1;
				break;
			}
		        // for the dynamic stuff we need a different strategy
			char formstr[200];
			int devno;
			sprintf(formstr,"%s%%d",pt->key2);
			if (sscanf(key2,formstr,&devno)==1) {
			        ret = pt;
				ret->devno = devno;
				break;
			}
		}
	}
	return ret;
}


static void saveconf (
	VIEWITEMS *&items,
	CONFIG_FILE &cf,
	PRIVILEGE *priv,
	bool modified)
{
	if (items != NULL){
		if (modified) items->write (cf,priv);
		delete items;
		items = NULL;
	}
}


PUBLIC int CONFDB_SUSE::save(PRIVILEGE *priv)
{
	int ret = orig->save(priv);
	if (ret != -1){
		saveconf (sys.rcconfig,f_rcconfig,priv,sys.rcconfig_modified);
		reset_modified();
	}
	tbstr.remove_all();
	return ret;
}


static CONFIG_FILE f_release("/var/lib/YaST/update.inf",help_nil
	,CONFIGF_NOARCH);

/*
	Return the release of the SuSE distribution
	release 5.2 becomes 502
*/
int suse_release()
{
        static int ret = 0;
	FILE *fin = f_release.fopen ("r");
	if (fin != NULL){
	        char line[200];
		while (fgets (line,sizeof(line)-1,fin) != NULL){
		        int a,b,c;
			if (sscanf(line,"Distribution_Version:%d.%d.%d",&a,&b,&c) == 3){
			        ret = a*100 + b;
			        break;
			}
		}
		fclose (fin);
	}
	return ret;
}


PRIVATE const char *CONFDB_SUSE::getvalfromrh(SUSELK *rh)
{
	const char *ret = NULL;
	VIEWITEMS **rhf = NULL;
	CONFIG_FILE *cf = NULL;

	char rh_key2[200];
	char rh_var[200];
	if (rh->devno == -1){
	        strcpy(rh_key2,rh->key2);
		strcpy(rh_var,rh->var);
	}else {
		sprintf(rh_key2,"%s%d",rh->key2,rh->devno);
		if(*(rh->var+strlen(rh->var)-1) == '_') {
		        sprintf(rh_var,"%s%d",rh->var,rh->devno);
		}else {
		        strcpy(rh_var,rh->var);
		}
	}

	if (rh->file == SUSE_dummy){
		int release = suse_release();
		if (strcmp(rh_key2,"sysvrestart")==0){
			ret = "0";
		}else if (strcmp(rh_key2,"privgroup")==0){
			ret = "0";
		}else if (strcmp(rh_key2,"sysvenh")==0){
			ret = "0";
		}else if (strcmp(rh_key2,"rpmvendor")==0){
			ret = "S.u.S.E. GmbH, Fuerth, Germany";
		}else if (strcmp(rh_key2,"rpmdistribution")==0){
			if (release == 503){
				ret = "S.u.S.E. Linux 5.3 (i386)";
			}else if (release == 502){
				ret = "S.u.S.E. Linux 5.2 (i386)";
			}else if (release == 501){
				ret = "S.u.S.E. Linux 5.1 (i386)";
			}else{
				ret = "";
			}
		}else if (strcmp(rh_key2,"release")==0){
			static char release_str[10];
			sprintf (release_str,"%d.%d",release/100,release%100);
			ret = release_str;
		}
	}else if (rh->file == SUSE_rcconfig){
		rhf = &sys.rcconfig;
		cf = &f_rcconfig;
	}
	if (rhf != NULL){
		if (*rhf == NULL){
			*rhf = new VIEWITEMS;
			(*rhf)->read (*cf);
		}
		char tmp[1000];
		const char *val = (**rhf).locateval (rh_var,tmp);
		if (val != NULL){
		        /* some strange suse special...*/
		        if(strcmp(val,"YAST_ASK") == 0) {
			        val = "";
			}
			if (rh->type == SUSE_YESNO
				|| rh->type == SUSE_ONOFF){
				ret = stricmp(val,"yes")==0
					|| stricmp(val,"true")==0
					|| stricmp(val,"on")==0
					? "1" : "0";
			}else if (rh->type == SUSE_VAL0 && val[0] == '\0'){
				ret = "0";
			}else if (rh->type == SUSE_VALU){
				ret = strcmp(val,"-u")==0 ? "1" : "0";
			}else if (rh->type == SUSE_NET){
			        char dev[6];
				sprintf(dev,"_%d",rh->devno);
				
				SSTRINGS *netenabled = new SSTRINGS();
				str_splitline(val,' ',*netenabled);
				ret = (netenabled->lookup(dev) != -1) ? "1" : "0";

#if 0
			}else if (rh->type == SUSE_NM){
			        ;
				
				static int device_ifaceinfo ("",INTER_INFO &itf,IFACE_NUMS &nums,false);
{

#endif
			}else{
				SSTRING *n = new SSTRING (val);
				tbstr.add (n);
				ret = n->get();
			}
		}
	}
	return ret;
}		

PRIVATE void CONFDB_SUSE::updaterh(SUSELK *rh, const char *val)
{
	VIEWITEMS **rhf = NULL;
	CONFIG_FILE *cf = NULL;

	char rh_var[200];
	if (rh->devno == -1){
		strcpy(rh_var,rh->var);
	}else {
		if(*(rh->var+strlen(rh->var)-1) == '_') {
		        sprintf(rh_var,"%s%d",rh->var,rh->devno);
		}else {
		        strcpy(rh_var,rh->var);
		}
	}

	if (rh->type == SUSE_YESNO){
		val = strcmp(val,"1")==0 ? "yes" : "no";
	}else if (rh->type == SUSE_ONOFF){
		val = strcmp(val,"1")==0 ? "on" : "off";
	}else if (rh->type == SUSE_VAL0){
		if (strcmp(val,"0")==0) val = "";
	}else if (rh->type == SUSE_VALU){
		if (strcmp(val,"1")==0){
			val = "-u";
		}else{
			val = "";
		}
	}
	if (rh->file == SUSE_rcconfig){
		rhf = &sys.rcconfig;
		cf = &f_rcconfig;
		sys.rcconfig_modified = true;
	}
	if (*rhf == NULL){
		*rhf = new VIEWITEMS;
		(*rhf)->read (*cf);
	}
	if (rh->type == SUSE_NET){
	        char dev[6];
		sprintf(dev,"_%d",rh->devno);
		char netconfig[1000];
		(*rhf)->locateval(rh_var,netconfig);

		suse_debug("updatefromrh NETCONFIG BEFORE",netconfig);
		suse_debug("updatefromrh VAL   BEFORE    ",val);

		SSTRINGS *netenabled = new SSTRINGS();
		str_splitline(netconfig,' ',*netenabled);
		if(strcmp(netconfig,"YAST_ASK") == 0) {
		        strcpy(netconfig,"");
		}
		if (strcmp(val,"1") == 0){
		        ;
			/* we add a value: we dont care but add the value */
			netenabled->add(new SSTRING (dev));
		} else {
		        ;
			int pos = netenabled->lookup(dev);
			if(pos > -1) {
			        netenabled->remove_del(pos);
			}
			/* we remove a value */
		}

		netenabled->sort();
		netenabled->remove_empty();
		netenabled->remove_dups();
		strcpy(netconfig,"");
		int nb = netenabled->getnb();
		for (int i=0; i<nb;i++){
		        if (i > 0){
			        strcat(netconfig," ");
			}
			char tmp[100];
			(*(netenabled->getitem(i))).copy(tmp);
			strcat(netconfig,tmp);
		}
		val = netconfig;
		suse_debug("updatefromrh NETCONFIG AFTER ",netconfig);
		suse_debug("updatefromrh VAL   AFTER     ",val);

	}
	if(rh->type == SUSE_STR && strcmp(val,"") == 0 && (*rhf)->locateassign(rh_var) == NULL) {
	        /* do nothing, if value has not been in list, string type and new value is ""*/
	}else {
	        (*rhf)->update(rh_var,val);
	}
}		
	

PUBLIC const char *CONFDB_SUSE::getval(
	const char *prefix,
	const char *key,
	const char *defval)
{
	suse_debug("\ngetval: prefix",prefix);
	suse_debug("getval: key   ",key);
	suse_debug("getval: defval",defval);

	SUSELK *rh = filter_lookup (prefix,key);
	const char *ret = defval;
	if (rh != NULL){
		/* #Specification: SuSE files and conf.linuxconf / strategy
			A migration of some stuff stored in /etc/conf.linuxconf
			is done. The goal is to use from now on the redhat specific
			config file (in /etc/sysconfig) directly. A second goal
			is to let linuxconf users migrate to this strategy painlessly.

			To achieve that, we will use a strategy giving priority
			to the content of /etc/conf.linuxconf. Normally
			for all new linuxconf installation, /etc/conf.linuxconf
			won't have anything conflicting with the sysconfig file, so
			giving priority to it means nothing. For old linuxconf
			installation, conf.linuxconf does contain the more accurate
			(more recent) data.

			So the strategy is for reading

			#
			-read in conf.linuxconf first.
			-if there is something there, use it
			-if there is nothing, read the appropriate SuSE file
			#

			And for writing

			#
			-erase all relevant entries in conf.linuxconf
			-update the appropriate SuSE file
			#

			This strategy should provide a completly transparent
			migration.
		*/
		ret = orig->getval (prefix,key,NULL);
		if (ret == NULL){
			ret = getvalfromrh (rh);
			if (ret == NULL) ret = defval;
		}
	}else{
		ret = orig->getval (prefix,key,defval);
	}
	suse_debug("getval: ret   ",ret);

	return ret;
}

PUBLIC int CONFDB_SUSE::getall (
	const char *prefix,
	const char *key,
	SSTRINGS &lst,
	bool copy)	// Take a copy of the values
{
	return orig->getall (prefix,key,lst,copy);
}

PUBLIC void CONFDB_SUSE::removeall (const char *prefix, const char *key)
{
	orig->removeall (prefix,key);
}
PUBLIC void CONFDB_SUSE::add (
	const char *prefix,
	const char *key,
	const char *val)
{
	/* #specification: Cleaning conf.linuxconf / strategy
		All updates in conf.linuxconf are done using the same
		sequence

		#
		removeall
		add
		#

		The "removeall"s in the RedHat conf.linuxconf filter are going
		through normally. The "add"s are filtered. Some special entries
		are rerouted to the RedHat sysconfig files.

		This strategy, combined with the one for the getval() function
		provide the following functionnality:

		#
		-conf.linuxconf has priority to provide information
		-conf.linuxconf lose information at the first upgrade of this
		 information as the sysconfig file are updated
		#
	*/
	SUSELK *rh = filter_lookup (prefix,key);
	if (rh != NULL){
		updaterh (rh,val);
	}else{
		orig->add (prefix,key,val);
	}
}

PUBLIC int CONFDB_SUSE::archive (SSTREAM &ss, const char *_sys)
{
	return orig->archive (ss,_sys);
}


PUBLIC int CONFDB_SUSE::extract (SSTREAM &ss, const char *_sys)
{
	return orig->extract (ss,_sys);
}

CONFDB *filter_fctnew (CONFDB *orig)
{
	return new CONFDB_SUSE(orig);
}











