#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <features.h>
#ifdef __GLIBC__
	#include <netinet/ip.h>
	#include <netinet/tcp.h>
	#include <netinet/udp.h>
	#include <netinet/ip_icmp.h>
	#include <net/if.h>
	#include "k2.0/ip_fw.h"
#else
	#include <linux/ip.h>
	#include <linux/tcp.h>
	#include <linux/udp.h>
	#include <linux/icmp.h>
	#include <linux/if.h>
	#include <linux/timer.h>
	#include "k2.0/ip_fw.h"
#endif
#include <assert.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include "firewall.h"
#include <netconf.h>
#include <daemoni.h>
#include <misc.h>

#ifndef IP_FW_POLICY_IN
	/* These are just here so it compiles */
	#define IP_FW_POLICY_IN		0
	#define IP_FW_FLUSH_IN		1
	#define IP_FW_APPEND_IN		2
	#define IP_FW_POLICY_OUT	3
	#define IP_FW_FLUSH_OUT		4
	#define IP_FW_F_ACCEPT		5
	#define IP_FW_F_REDIR		6
	#define IP_FW_F_ICMPRPL		7
	#define IP_FW_F_KIND		8

	#define FIREWALL_NONE
#endif

extern HELP_FILE help_ipfw;
static CONFIG_FILE f_script_ipchain ("/var/run/ipchain.sh",help_ipfw
	,CONFIGF_OPTIONAL|CONFIGF_GENERATED
	,"root","root",0700);
static FILE *ipchain_fout = NULL;
static bool ipchain_somecmds = false;

static void ipfw_appendcmd (const char *ctl, ...)
{
	static const char *ipchainpath = NULL;
	if (ipchainpath == NULL){
		DAEMON_INTERNAL *dae = daemon_find ("ipchains");
		if (dae != NULL){
			ipchainpath = dae->getpath();
		}else{
			ipchainpath = "/sbin/ipchains";
		}
	}
	char buf[1000];
	va_list list;
	va_start (list,ctl);
	strcpy (buf,ipchainpath);
	int len = strlen(buf);
	buf[len++] = ' ';
	
	ipchain_somecmds = true;
	vsnprintf (buf+len,sizeof(buf)-1-len,ctl,list);
	va_end (list);
	fprintf (ipchain_fout,"%s\n",buf);
}




static int sockfd = -1;

static const char *ipfw_u2a (
	unsigned long nip,		// Network IP number
	char *buf)				// Will receive the Ascii IP number
{
	int num4[4];
	nip = ntohl(nip);
	num4[0] = (nip >> 24 ) & 0xff;
	num4[1] = (nip >> 16) & 0xff;
	num4[2] = (nip >> 8) & 0xff;
	num4[3] = nip & 0xff;
	sprintf (buf,"%d.%d.%d.%d",num4[0],num4[1],num4[2],num4[3]);
	return buf;
}

void ipfw_heading(
	SSTRING *collect)
{
	if (collect != NULL){
			char buffer[300];
			sprintf(buffer,"%s %5s %15s/%-15s -> %15s/%-15s %-15s %-15s\n"
				,"cmd","proto"
				,"network","netmask"
				,"network","netmask"
				,"iface"
				,"ports ... flags");
			collect->append (buffer);
	}
}

static const char *ipfw_getport (int port, const char *proto, char *buf)
{
	struct servent *ent = getservbyport (ntohs(port),proto);
	if (ent == NULL){
		sprintf (buf,"%d",port);
	}else{
		strcpy (buf,ent->s_name);
	}
	return buf;
}

struct IPCHAIN_INFO{
	const char *proto_cmd;
	const char *proto;
	const char *verb;
	const char *chain;
	char iface_verb[100];
};

static void ipfw_formatcmd (
	struct ip_fw &b,
	IPCHAIN_INFO &inf,
	const char *source_port,
	const char *dest_port)
{
	char b1[LEN_IP_ASC],b2[LEN_IP_ASC],b3[LEN_IP_ASC],b4[LEN_IP_ASC];
	char jump_verb[1000];
	jump_verb[0] = '\0';
	if (inf.verb != NULL){
		snprintf (jump_verb,sizeof(jump_verb)-1,"--jump %s",inf.verb);
	}
	ipfw_appendcmd ("%s%s--append %s %s %s --source %s/%s %s"
		 " --destination %s/%s %s %s"
		,(b.fw_flg & IP_FW_F_PRN) ? "--log " : ""
		,(b.fw_flg & IP_FW_F_TCPSYN) ? "--syn " : ""
		,inf.chain,jump_verb,inf.proto_cmd
		,ipfw_u2a(b.fw_src.s_addr,b1),ipfw_u2a(b.fw_smsk.s_addr,b2)
		,source_port
		,ipfw_u2a(b.fw_dst.s_addr,b3),ipfw_u2a(b.fw_dmsk.s_addr,b4)
		,dest_port
		,inf.iface_verb);

}

static void ipfw_formatdstports (
	struct ip_fw &b,
	IPCHAIN_INFO &inf,
	const char *source_port)
{
	int noport = b.fw_nsp;
	int ndp = b.fw_ndp;
	if (b.fw_flg & IP_FW_F_DRNG){
		char dest_port[1000];
		sprintf(dest_port, "%u:%u",b.fw_pts[noport],b.fw_pts[noport+1]);
		noport += 2;
		ndp -= 2;
		ipfw_formatcmd (b,inf,source_port,dest_port);
	}
	for (int d=0; d<ndp; d++){
		char bufp[1000];
		ipfw_formatcmd (b,inf,source_port
			,ipfw_getport(b.fw_pts[noport++],inf.proto,bufp));
	}
	if (b.fw_ndp == 0){
		ipfw_formatcmd (b,inf,source_port,"");
	}
}


int ipfw_append (
	bool doit,
	SSTRING *collect,
	int command,
	struct ip_fw &b)
{
	int ret = 0;
	if (collect != NULL){
		/* #Specification: firewall / formatting output
			While programming the firewall, we compose in a string
			the same format as will be created by the kernel in one
			of the file /proc/net/ip_input or /proc/net/ip_forward.

			So it become possible to compare this with the current
			content of those file and tell if something as to be done.
		*/
		#ifndef FIREWALL_NONE
			char buffer[300];
			char b1[LEN_IP_ASC],b2[LEN_IP_ASC],b3[LEN_IP_ASC],b4[LEN_IP_ASC];

			const char *verb = "dny";
			if (b.fw_flg & IP_FW_F_MASQ){
				verb = "msq";
			}else if (b.fw_flg & IP_FW_F_REDIR){
				verb = "rdr";
			}else if (b.fw_flg & IP_FW_F_ACCEPT){
				verb = "acc";
			}else if (b.fw_flg & IP_FW_F_ICMPRPL){
				verb = "rjc";
			}

			const char *proto = "All";
			if ((b.fw_flg & IP_FW_F_KIND) == IP_FW_F_TCP){
				proto = "tcp";
			}else if ((b.fw_flg & IP_FW_F_KIND) == IP_FW_F_UDP){
				proto = "udp";
			}else if ((b.fw_flg & IP_FW_F_KIND) == IP_FW_F_ICMP){
				proto = "icmp";
			}

			char iface[LEN_IP_ASC];
			if (b.fw_vianame[0] != '\0'){
				strcpy (iface,b.fw_vianame);
			}else if (b.fw_via.s_addr == 0){
				strcpy (iface,"Any");
			}else{
				ipfw_u2a (b.fw_via.s_addr,iface);
			}

			int len=sprintf(buffer,"%s %-5s %15s/%-15s -> %15s/%-15s %-15s"
				,verb,proto
				,ipfw_u2a(b.fw_src.s_addr,b1),ipfw_u2a(b.fw_smsk.s_addr,b2)
				,ipfw_u2a(b.fw_dst.s_addr,b3),ipfw_u2a(b.fw_dmsk.s_addr,b4)
				,iface);
			//len+=sprintf(buffer+len,"%u %u %-9lu %-9lu"
			//	,b.fw_nsp,b.fw_ndp, b.fw_pcnt,b.fw_bcnt);

			int noport = 0;
			len += sprintf (buffer+len," [");
			int nsp = b.fw_nsp;
			if (b.fw_flg & IP_FW_F_SRNG){
				len+=sprintf(buffer+len, " %u:%u",b.fw_pts[0],b.fw_pts[1]);
				noport = 2;
				nsp -= 2;
			}
			for (int s=0; s<nsp; s++){
				char bufp[100];
				len += sprintf (buffer+len," %s"
					,ipfw_getport(b.fw_pts[noport++],proto,bufp));
			}
			len += sprintf (buffer+len," ] -> [");
			int ndp = b.fw_ndp;
			if (b.fw_flg & IP_FW_F_DRNG){
				len+=sprintf(buffer+len, " %u:%u",b.fw_pts[noport]
					,b.fw_pts[noport+1]);
				noport += 2;
				ndp -= 2;
			}
			for (int d=0; d<ndp; d++){
				char bufp[100];
				len += sprintf (buffer+len," %s"
					,ipfw_getport(b.fw_pts[noport++],proto,bufp));
			}
			len += sprintf (buffer+len," ]");

			if (b.fw_flg & IP_FW_F_REDIR){
				char bufp[100];
				len += sprintf (buffer+len," => %s"
					,ipfw_getport(b.fw_pts[noport++],proto,bufp));
			}
			len += sprintf(buffer+len, " A%02X X%02X", b.fw_tosand, b.fw_tosxor);

			if (b.fw_flg & IP_FW_F_PRN){
				len += sprintf (buffer+len," logging");
			}
			strcpy (buffer+len,"\n");
			collect->append (buffer);
		#endif
	}
	if (doit){
		if (kernel_newer (2,2,0)){
			char redir_verb[100];

			IPCHAIN_INFO inf;
			inf.proto_cmd = "";
			inf.proto = "All";
			if ((b.fw_flg & IP_FW_F_KIND) == IP_FW_F_TCP){
				inf.proto_cmd = "--proto TCP";
				inf.proto = "tcp";
			}else if ((b.fw_flg & IP_FW_F_KIND) == IP_FW_F_UDP){
				inf.proto_cmd = "--proto UDP";
				inf.proto = "udp";
			}else if ((b.fw_flg & IP_FW_F_KIND) == IP_FW_F_ICMP){
				inf.proto_cmd = "--proto ICMP";
				inf.proto = "icmp";
			}
			inf.verb = "DENY";
			if (b.fw_flg & IP_FW_F_MASQ){
				inf.verb = "MASQ";
			}else if (b.fw_flg & IP_FW_F_REDIR){
				char bufp[100];
				int noport = b.fw_nsp + b.fw_ndp;
				snprintf (redir_verb,sizeof(redir_verb)-1,"REDIRECT %s"
					,ipfw_getport(b.fw_pts[noport],inf.proto,bufp));
				inf.verb = redir_verb;
			}else if (b.fw_flg & IP_FW_F_ACCEPT){
				inf.verb = "ACCEPT";
			}else if (b.fw_flg & IP_FW_F_ICMPRPL){
				inf.verb = "REJECT";
			}
			inf.chain = NULL;
			if (command == IP_FW_APPEND_IN){
				inf.chain = "input";
			}else if (command == IP_FW_APPEND_OUT){
				inf.chain = "output";
			}else if (command == IP_FW_APPEND_FWD){
				inf.chain = "forward";
			}else{
				// Accounting rules, no jump needed
				inf.chain = "acct";
				inf.verb = NULL;
			}

			inf.iface_verb[0] = '\0';
			if (b.fw_vianame[0] != '\0'){
				sprintf (inf.iface_verb,"--interface %s",b.fw_vianame);
			}else if (b.fw_via.s_addr != 0){
				char iface[LEN_IP_ASC];
				ipfw_u2a (b.fw_via.s_addr,iface);
				sprintf (inf.iface_verb,"--interface %s",iface);
			}



			int noport = 0;
			int nsp = b.fw_nsp;
			if (b.fw_flg & IP_FW_F_SRNG){
				char source_port[1000];
				sprintf(source_port, "%u:%u",b.fw_pts[0],b.fw_pts[1]);
				noport = 2;
				nsp -= 2;
				ipfw_formatdstports(b,inf,source_port);
			}
			for (int s=0; s<nsp; s++){
				char bufp[100];
				ipfw_formatdstports (b,inf
					,ipfw_getport(b.fw_pts[noport+s],inf.proto,bufp));
			}
			if (b.fw_nsp == 0){
				ipfw_formatdstports (b,inf,"");
			}
			ret = 0;
		}else{
			ret = setsockopt (sockfd, IPPROTO_IP, command
				,&b,sizeof(struct ip_fw));
			if (ret != 0){
				xconf_error ("error append firewall %d(%s)",errno,strerror(errno));
			}
		}
	}
	return ret;
}

/*
	Remove one list (input,forward,output) of firewalling rule from the kernel
*/
int ipfw_flush (
	bool doit,
	SSTRING *,
	int command)
{
	int ret = 0;
	if (doit){
		if (kernel_newer (2,2,0)){
			if (command == IP_ACCT_FLUSH){
				ipfw_appendcmd ("--delete input --jump acct --source 0.0.0.0/0.0.0.0 --destination 0.0.0.0/0.0.0.0 2>/dev/null");
				ipfw_appendcmd ("--flush acct 2>/dev/null");
				ipfw_appendcmd ("--delete-chain acct 2>/dev/null");
			}else{
				const char *chain = NULL;
				if (command == IP_FW_FLUSH_IN){
					chain = "input";
				}else if (command == IP_FW_FLUSH_OUT){
					chain = "output";
				}else if (command == IP_FW_FLUSH_FWD){
					chain = "forward";
				}else{
					assert(0);
				}
				ipfw_appendcmd ("--flush %s",chain);
			}
			ret = 0;
		}else{
			int data = 0;
			ret = setsockopt (sockfd, IPPROTO_IP, command,&data,sizeof(data));
			if (ret != 0){
				xconf_error ("error flush firewall %d(%s)",errno,strerror(errno));
			}
		}
	}
	return ret;
}

/*
	Create the chain for accounting rules.
	They are stored in input rules.
*/
void ipfw_initacct ()
{
	if (kernel_newer (2,2,0)){
		ipfw_appendcmd ("--new-chain acct");
		// Insert at the start of the input chain
		ipfw_appendcmd ("--insert input --jump acct  --source 0.0.0.0/0.0.0.0  --destination 0.0.0.0/0.0.0.0");
	}
}
  
int ipfw_policy (
	bool doit,
	SSTRING *collect,
	int command,
	int policy)
{
	int ret = 0;
	if (collect != NULL){
		char *ctl = NULL;
		switch (command){
		case IP_FW_POLICY_IN:
			ctl = "IP firewall input rules, default %s\n";
			break;
		case IP_FW_POLICY_OUT:
			ctl = "IP firewall output rules, default %s\n";
			break;
		case IP_FW_POLICY_FWD:
			ctl = "IP firewall forward rules, default %s\n";
			break;
		}
		char buf[100];
		const char *pstr = "deny";
		if (policy == IP_FW_F_ACCEPT){
			pstr = "accept";
		}
		sprintf(buf,ctl,pstr);
		collect->append (buf);
	}
	if (doit){
		if (kernel_newer (2,2,0)){
			const char *chain = NULL;
			if (command == IP_FW_POLICY_IN){
				chain = "input";
			}else if (command == IP_FW_POLICY_OUT){
				chain = "output";
			}else if (command == IP_FW_POLICY_FWD){
				chain = "forward";
			}else{
				// No policy for accounting rules
				assert (0);
			}

			const char *policy_str = NULL;
			if (policy == IP_FW_F_ACCEPT){
				policy_str = "ACCEPT";
			}else if (policy == 0){
				policy_str = "DENY";
			}else{
				assert (0);
			}
			ipfw_appendcmd ("--policy %s %s",chain,policy_str);
			ret = 0;
		}else{
			ret = setsockopt (sockfd, IPPROTO_IP, command,&policy,sizeof(policy));
		}
	}
	return ret;
}
  
/*
	Initialise the sockfd needed to program the rules
	Return -1 if any error.
*/
int ipfw_open ()
{
	int ret = -1;
	if (kernel_newer (2,2,0)){
		ipchain_somecmds = false;
		ipchain_fout = f_script_ipchain.fopen ("w");
		if(ipchain_fout != NULL){
			fputs ("#!/bin/sh\n",ipchain_fout);
			ret = 0;
		}
	}else{
		sockfd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
		ret = sockfd;
	}
	return ret;
}


void ipfw_close()
{
	if (kernel_newer (2,2,0)){
		if (ipchain_fout != NULL){
			fclose (ipchain_fout);
			if (ipchain_somecmds){
				netconf_system (15,"/var/run/ipchain.sh");
			}
		}
	}else{
		close (sockfd);
		sockfd = -1;
	}
}

#ifndef FIREWALL_NONE
/*
	Translate and validate an ASCII IP addr.
*/
static int ipfw_a2ip (const char *adr, struct in_addr &ina)
{
	int num4[4];
	int ret = ipnum_aip24 (adr,num4);
	if (ret != -1){
		unsigned long ipa = (num4[0] <<24) | (num4[1] << 16)
			| (num4[2] << 8) | num4[3];
		ina.s_addr = htonl (ipa);
	}
	return ret;
}

static int ipfw_setrange (
	const char *range,
	struct ip_fw &bf,
	int flag,
	int &noport,
	unsigned short &nb)
{
	int ret = 0;
	const char *s = str_skip (range);
	if (s[0] != '\0'){
		bf.fw_flg |= flag;
		bf.fw_pts[noport++] = atoi(s);
		s = str_skipdig (s);
		s = str_skip (s);
		if (*s == ':'){
			s++;
			s = str_skip (s);
			bf.fw_pts[noport++] = atoi(s);
			nb = 2;
		}else{
			ret = -1;
		}
	}
	return ret;
}

/*
	Validation for the dialog
	Return -1 if any errors.
*/
int ipfw_checkrange (const SSTRING &range)
{
	struct ip_fw bf;
	int noport = 0;
	unsigned short nb = 0;
	return ipfw_setrange (range.get(),bf,0,noport,nb);
}



static int ipfw_setports (
	const char *ports,
	struct ip_fw &bf,
	unsigned short &nb,
	int &noport,
	const char *proto)
{
	int ret = 0;
	if (strcmp(proto,"udp")!=0
		&& strcmp(proto,"tcp")!=0){
		ports = str_skip(ports);
		if (ports[0] != '\0'){
			ret = -1;
		}
	}else{
		while (1){
			ports = str_skip (ports);
			if (ports[0] == '\0'){
				break;
			}else{
				char word[200];
				ports = str_copyword(word,ports,sizeof(word));
				int port = -1;
				struct servent *serv = getservbyname (word,proto);
				if (serv != NULL){
					port = ntohs(serv->s_port);
				}else if (str_checkdig(word)){
					port = atoi(word);
				}else{
					ret = -1;
					break;
				}
				bf.fw_pts[noport++] = port;
				nb++;
			}
		}
	}
	return ret;
}

/*
	Validation for the dialog
	Return -1 if any errors.
*/
int ipfw_checkports (
	const SSTRING &ports,
	const SSTRING &proto)
{
	struct ip_fw bf;
	int noport = 0;
	unsigned short nb = 0;
	return ipfw_setports (ports.get(),bf,nb,noport,proto.get());
}
	
/*
	Return -1 if anything is invalid
*/
int ipfw_baseinit (
	int policy,
	const char *iface,
	const char *protocol,
	const char *ip_src,
	const char *msk_src,
	const char *sport_range,
	const char *sports,
	const char *ip_dst,
	const char *msk_dst,
	const char *dport_range,
	const char *dports,
	const char *redirport,
	bool logging,
	bool match_syn,
	struct ip_fw &bf)
{
	memset (&bf,0,sizeof(bf));
	int ret = ipfw_a2ip (ip_src,bf.fw_src);
	ret |= ipfw_a2ip (msk_src,bf.fw_smsk);
	bf.fw_src.s_addr &= bf.fw_smsk.s_addr;
	ret |= ipfw_a2ip (ip_dst,bf.fw_dst);
	ret |= ipfw_a2ip (msk_dst,bf.fw_dmsk);
	bf.fw_dst.s_addr &= bf.fw_dmsk.s_addr;
	/* #Specification: firewall / iface / assumption
		We assume that all network device do begin with a letter.
		This way we differentiate IP number for interface from name
	*/
	if (isalpha(iface[0])){
		strcpy (bf.fw_vianame,iface);
	}else{
		ret |= ipfw_a2ip (iface,bf.fw_via);
	}
	if (stricmp(protocol,"all")==0){
		bf.fw_flg = IP_FW_F_ALL;
	}else if (stricmp(protocol,"tcp")==0){
		bf.fw_flg = IP_FW_F_TCP;
	}else if (stricmp(protocol,"udp")==0){
		bf.fw_flg = IP_FW_F_UDP;
	}else if (stricmp(protocol,"icmp")==0){
		bf.fw_flg = IP_FW_F_ICMP;
	}else{
		ret = -1;
	}
	int noport = 0;
	ret |= ipfw_setrange (sport_range,bf,IP_FW_F_SRNG,noport,bf.fw_nsp);
	ret |= ipfw_setports (sports,bf,bf.fw_nsp,noport,protocol);
	ret |= ipfw_setrange (dport_range,bf,IP_FW_F_DRNG,noport,bf.fw_ndp);
	ret |= ipfw_setports (dports,bf,bf.fw_ndp,noport,protocol);

	bf.fw_tosand = 0xFF;
	bf.fw_tosxor = 0x00;
	if (policy == FW_ACCEPT){
		bf.fw_flg |= IP_FW_F_ACCEPT;
	}else if (policy == FW_REJECT){
		bf.fw_flg |= IP_FW_F_ICMPRPL;
	}
	if (redirport != NULL){
		bf.fw_flg |= IP_FW_F_REDIR;
		unsigned short dummy = 0;
		ret |= ipfw_setports (redirport,bf,dummy,noport,protocol);
	}
	if (logging) bf.fw_flg |= IP_FW_F_PRN;
	if (match_syn) bf.fw_flg |= IP_FW_F_TCPSYN;
	return ret;
}

#else

int ipfw_checkports (
	const SSTRING &,
	const SSTRING &)
{
	return 0;
}
int ipfw_checkrange (const SSTRING &)
{
	return 0;
}

int ipfw_baseinit (
	int,
	const char *,
	const char *,
	const char *,
	const char *,
	const char *,
	const char *,
	const char *,
	const char *,
	const char *,
	const char *,
	const char *,
	bool,
	bool,
	struct ip_fw &)
{
	return -1;
}

#endif
