#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include "dnsconf.h"
#include "internal.h"
#include <netconf.h>
#include "dnsconf.m"
#include <dialog.h>

static DNSCONF_HELP_FILE help_primary("primary");
static DNSCONF_HELP_FILE help_secondary("secondary");


/*
	Used to read back an existing configuration
*/
PUBLIC PRIMARY::PRIMARY(
	const char *_domain,
	const char *_file,	// File used to store the configuration
	const char *named_dir,	// Default directory for configuration files
	bool extract)		// Extract from the archive
{
	domain.setfrom (_domain);
	file.setfrom (_file);
	origins.read (named_dir,_file,_domain,extract);
	domainv.setfrom (_domain);
	isrev = 0;
	zonetype=ZONE_PRIMARY;
	transfered = true;
}

PUBLIC PRIMARY::PRIMARY()
{
	isrev = 0;
	zonetype=ZONE_PRIMARY;
	transfered = true;
	notify = NOTIFY_DEFAULT;
}
/*
	Update the domain name from the visual one.
*/
PUBLIC VIRTUAL void ZONE::setfromv()
{
	domain.setfrom (domainv);
}

/*
	Check if this domaine is a member of the IN-ADDR.ARPA domain
	Return true if this is the case.
*/
static bool primary_is_arpadom (const char *dom)
{
	char buf[strlen(dom)+1];
	strcpy (buf,dom);
	strupr (buf);
	char *pt = strstr(buf,".IN-ADDR.ARPA");
	return pt !=NULL && pt[13] == '\0';
}

/*
	Extract the IP number of a in-addr.arpa domain and reverse it
	x,y.z.in-addr.arpa becomes z.y.x
*/
static void primary_reverse_arpa(const char *arpadom, SSTRING &domainv)
{
	IP_ADDR ipa;
	ipa.setfrom (arpadom);
	ipa.reverse ();
	ipa.shift();
	domainv.setfrom (ipa.get());
}

/*
	Used to read back an existing configuration
*/
PUBLIC PRIMARY_REV::PRIMARY_REV(
	const char *_domain,
	const char *_file,	// File used to store the configuration
	const char *named_dir,	// Default directory for configuration files
	bool extract)
	: PRIMARY(_domain,_file,named_dir,extract)
{
	primary_reverse_arpa (_domain,domainv);
	isrev = 1;
}

PUBLIC PRIMARY_REV::PRIMARY_REV()
{
	isrev = 1;
	zonetype=ZONE_PRIMARY;
}

/*
	Update the domain name from the visual one.
*/
PUBLIC void PRIMARY_REV::setfromv()
{
	IP_ADDR ipa;
	ipa.setfrom (domainv.get());
	char buf[30];
	ipa.setrev (buf);
	domain.setfrom (buf);
}

/*
	Return != if any component of the PRIMARY was modified
*/
PUBLIC int ZONE::was_modified()
{
	int ret = ARRAY_OBJ::was_modified();
	if ((!ret) && (zonetype!=ZONE_SECONDARY)) {
		ret = origins.was_modified();
	}
	return ret;
}

/*
	Return != 0 if the PRIMARY describe the reverse mapping of an
	IP network (x.y,z.in-addr.arpa)
*/
PUBLIC VIRTUAL int ZONE::is_reverse()
{
	return 0;
}
/*
	Return != 0 if the PRIMARY describe the reverse mapping of an
	IP network (x.y,z.in-addr.arpa)
*/
PUBLIC int PRIMARY_REV::is_reverse()
{
	return 1;
}

/*
	Locate all IP number in use in a domain.
	Return the number of IPs added to adrs
*/
PUBLIC int ZONE::getalladr(IP_ADDRS &adrs)
{
	int ret = 0;
	for (int i=0; i<origins.getnb(); i++){
		ret += origins.getitem(i)->getalladr(adrs);
	}
	return ret;
}

/*
	Find the first record of a certain type in the PRIMARY
	Returne NULL if not found.
*/
PRIVATE RECORD *ZONE::getfirst(RECORD_TYPE rtype)
{
	RECORD *ret = NULL;
	for (int i=0; ret == NULL && i<origins.getnb(); i++){
		ORIGIN *ori = origins.getitem(i);
		for (int o=0; o<ori->tbrec.getnb(); o++){
			RECORD *rec = ori->tbrec.getitem(o);
			if (rec->is(rtype)){
				ret = rec;
				break;
			}
		}
	}
	return ret;
}

/*
	Find the (first) soa record of a primary
	Return NULL if it can be found.
*/
PUBLIC RECORD_IN_SOA *ZONE::getsoa()
{
	return (RECORD_IN_SOA*)getfirst(RTYPE_SOA);
}

/*
	Find all the NS records for a name.
	Return the number of record found.
*/
PUBLIC int ZONE::getns(
	SSTRING &dom,
	RECORDS &recs)
{
	FQHOST fq (dom);
	return locate_left (fq,RTYPE_NS,recs);
}

/*
	Find all the NS records for a name.
	Return the number of record found.
*/
PUBLIC int ZONE::getns(
	SSTRING &dom,
	SSTRINGS &strs)
{
	RECORDS recs;
	int nb = getns (dom,recs);
	for (int i=0; i<nb; i++){
		RECORD_IN_NS *ns = (RECORD_IN_NS*)recs.getitem(i);
		strs.add (new SSTRING (ns->ns));
	}
	return nb;
}

/*
	Find all the MX records for name.
	Return the number of record found.
*/
PUBLIC int ZONE::getmx(
	SSTRING &dom,
	RECORDS &recs)
{
	FQHOST fq (dom);
	return locate_left (fq,RTYPE_MX,recs);
}

/*
	Find all the MX records for a name.
	Return the number of record found.
*/
PUBLIC int ZONE::getmx(
	SSTRING &dom,
	SSTRINGS &strs)
{
	RECORDS recs;
	int nb = getmx (dom,recs);
	int tbpri[nb];
	memset (tbpri,-1,sizeof(tbpri));
	for (int i=0; i<nb; i++){
		RECORD_IN_MX *mx = (RECORD_IN_MX*)recs.getitem(i);
		bool found = false;
		for (int m=0; m<i; m++){
			if (tbpri[m] == mx->prefer){
				strs.getitem(m)->appendf (" %s",mx->servname.get());
				found = true;
			}
		}
		tbpri[i] = mx->prefer;
		if (!found) strs.add (new SSTRING (mx->servname));
	}
	return nb;
}

/*
	Find all the A records for a name.
	Return the number of record found.
*/
PUBLIC int ZONE::geta(
	SSTRING &dom,
	RECORDS &recs)
{
	FQHOST fq (dom);
	return locate_left (fq,RTYPE_A,recs);
}

/*
	Find all the A records for a name.
	Return the number of record found.
*/
PUBLIC int ZONE::geta(
	SSTRING &dom,
	IP_ADDRS &adrs)
{
	RECORDS recs;
	int nb = geta (dom,recs);
	for (int i=0; i<nb; i++){
		RECORD_IN_A *a = (RECORD_IN_A*)recs.getitem(i);
		adrs.add (new IP_ADDR (a->addr));
	}
	return nb;
}

/*
	Find the CNAME record for a name.
	Return -1 if not found. cname will be empty.
*/
PUBLIC int ZONE::getcname(
	SSTRING &dom,
	SSTRING &cname)
{
	FQHOST fq (dom);
	RECORDS recs;
	int nb = locate_left (fq,RTYPE_CNAME,recs);
	cname.setfrom ("");
	int ret = -1;
	if (nb > 0){
		RECORD_IN_CNAME *a = (RECORD_IN_CNAME*)recs.getitem(0);
		cname.setfrom (a->name);
		ret = 0;
	}
	return ret;
}

/*
	Increment if needed the serial number of the SOA
	This function may be called several time. The serial number
	will be incremented only once per session though.
*/
PUBLIC void ZONE::updatesoa()
{
	if (origins.was_modified()){
		RECORD_IN_SOA *soa = getsoa();
		if (soa != NULL) soa->update(domain.get());
	}
}

/*
	Format the serial number (revision number) of the zone (selection menu)
	Put an empty string if the zone has no revision so far
*/
PUBLIC void ZONE::format_revision (char *buf)
{
	RECORD_IN_SOA *soa = getsoa();
	buf[0] = '\0';
	if (soa != NULL){
		long serial = soa->new_serial;
		if (serial > 1997000000){
			sprintf (buf,"%04ld/%02ld/%02ld\t%ld"
				,serial/1000000
				,(serial % 1000000)/10000
				,(serial % 10000)/ 100
				,serial % 100);
		}else{
			sprintf (buf,"-\t%ld",serial);
		}
	}
}

PROTECTED void ZONE::writeaccess(FILE *fout) const
{
	if (notify!=NOTIFY_DEFAULT) {
		fprintf (fout,"\t%s %s;\n",K_NOTIFY,notify ? "yes" : "no");
	}
	dns_writeopts (allowtrans,K_ALLOW_TRANSFER,fout);
	dns_writeopts (allowquery,K_ALLOW_QUERY,fout);
	dns_writeopts (allowupdate,K_ALLOW_UPDATE,fout);
	dns_writeopts (alsonotify,K_ALSO_NOTIFY,fout);
}

/*
	Write the records of the domain and the entry in named.boot
	Return -1 if any error.
*/
PUBLIC int PRIMARY::write (bool bind8, FILE *fout, const char *named_dir, const DNS * dns) const
{
	SSTRING file2;
	int ret= origins.save (named_dir,file.get());
	if (bind8){
		fprintf (fout,"%s \"%s\"{\n",K_ZONE,domain.get());
		fprintf (fout,"\t%s %s;\n",K_TYPE,K_MASTER);
		fprintf (fout,"\t%s \"%s\";\n",K_FILE,file.get());
		writeaccess(fout);
		fputs ("};\n",fout);
	}else{
		if (zonetype==ZONE_PRIMARY) {
			fprintf (fout,"primary\t%s\t%s\n",domain.get(),file.get());
		} else {
			fprintf (fout,"secondary\t%s",domain.get());
			for (int i=0; i<4; i++){
				fprintf (fout, "%s",addr[i].get());
			}
			fprintf (fout,"\t%s\n",file2.get());
		}
	}
	return ret;
}


/*
	Add a record in the PRIMARY.
	The record is record relative to the main origin of the primary.
*/
PUBLIC void ZONE::addrec (RECORD *rec)
{
/* do something for secondary here */
	if (origins.getnb()==0){
		ORIGIN *ori = new ORIGIN(domain.get());
		origins.add (ori);
	}
	origins.getitem(0)->tbrec.add (rec);
}

PROTECTED void ZONE::setupzoneaccess (DIALOG &dia, int level)
{
	dia.newf_title (MSG_U(T_ACCESS,"Access control"),level,"",MSG_R(T_ACCESS));
	int level2 = level + 1;
	feature_editlist (dia,level2,MSG_R(F_ALLOWTRANS),allowtrans);
	feature_editlist (dia,level2,MSG_R(F_ALLOWQUERY),allowquery);
	feature_editlist (dia,level2,MSG_U(F_ALLOWUPDATE,"Allow update from"),allowupdate);
	{
		static const char *tbmode[]={
			MSG_U(I_NO,"No"),
			MSG_U(I_YES,"Yes"),
			MSG_U(I_DEFAULT,"Default"),
			NULL
		};
		dia.newf_chkm (MSG_R(F_NOTIFY),notify,tbmode);
	}   
	feature_editlist (dia,level2,MSG_U(F_ALSONOTIFY,"Also notify"),alsonotify);
}

PUBLIC void ZONE::setupdia (
	DIALOG &dia,
	DNS &dns,
	int level,		// Level for notebook pads
	PRIMARY_EDITINFO &info)
{
	THISHOST thost;
	info.soa = getsoa();
	if (getns(domain,info.rns) == 0){
		info.rns.add (new SSTRING (thost.getname1()));
	}
	/* #Specification: dnsconf / main ns records of the domain
		dnsconf allows the specification of up to 3 NS
		record for a primary.
	*/
	while (info.rns.getnb() < 3) info.rns.add (new SSTRING);
	if (getmx(domain,info.rmx) == 0){
		info.rmx.add (new SSTRING (thost.getname1()));
	}
	/* #Specification: dnsconf / main mx records of the domain
		dnsconf allows the specification of up to 3 MX
		records for a primary.
	*/
	while (info.rmx.getnb() < 3) info.rmx.add (new SSTRING);
	if (info.soa == NULL) info.soa = new RECORD_IN_SOA;

	dia.newf_str (MSG_U(F_MAINSERV,"Main server"),info.soa->machine);
	dia.newf_str (MSG_U(F_ADMINMAIL,"Administrator email"),info.soa->admin);
	dia.newf_title (MSG_R(F_DNSADV),level,"",MSG_R(F_DNSADV));
	info.start_ns = dia.getnb();
	int i;
	for (i=0; i<info.rns.getnb(); i++){
		dia.newf_str ("",*(info.rns.getitem(i)));
	}
	info.start_mx = dia.getnb();
	if (!isrev){
		dia.newf_title (MSG_R(F_EMAILADV),level,"",MSG_R(F_EMAILADV));
		for (i=0; i<info.rmx.getnb(); i++){
			dia.newf_str ("",*(info.rmx.getitem(i)));
		}
	}

	if (!isrev) {
		geta(domainv,info.tba);
		while (info.tba.getnb()<4) info.tba.add (new IP_ADDR);
		dns.setip_in_dia (dia,level,info.tba,MSG_U(T_DEFAULTIP,"Default IPs"));
	}


	dia.newf_title (MSG_U(T_FEATURES,"Features"),level,"",MSG_R(T_FEATURES));
	dia.newf_title ("",MSG_U(F_SECREQ,"Secondaries requirements"));
	dia.newf_str (MSG_U(F_REFRESH,"Refresh"),info.soa->refresh);
	dia.last_noempty();
	dia.newf_str (MSG_U(F_RETRY,"Retry"),info.soa->retry);
	dia.last_noempty();
	dia.newf_str (MSG_U(F_EXPIRE,"Expire"),info.soa->expire);
	dia.last_noempty();
	dia.newf_title ("",MSG_U(F_EVERYHOSTS,"Every hosts requirements"));
	dia.newf_str (MSG_U(F_TTL,"Time to live"),info.soa->default_ttl);
	dia.last_noempty();

	if (dns.bind8 && zonetype == ZONE_PRIMARY){
		setupzoneaccess(dia,level);
	}
}
	
/*
	Edit the basic specs of a domain.
	Return -1 if the user abort edition
	Return  0 if the user accepted the changes
	Return  1 if the user wish to delete this domain.
*/
PUBLIC int ZONE::editshow (DNS &dns, bool readonly)
{
	DIALOG dia;
	dia.newf_str (isrev ? MSG_U(F_NETNUM,"Network number")
		: MSG_U(F_DOMAIN,"Domain"),domainv);
	PRIMARY_EDITINFO info;
	setupdia (dia,dns,1,info);
	int ret = -1;
	int nof = 0;
	if (readonly) dia.set_readonly();
	while (1){
		MENU_STATUS status = dia.edit (
			 (zonetype==ZONE_PRIMARY)
				? MSG_U(T_PRIMSPEC,"Primary specification")
				:  MSG_U(T_SECSPEC,"Secondary specification")

			,(zonetype==ZONE_PRIMARY)
				? MSG_U(I_PRIMSPEC
					 ,"You must enter a domain name\n")
				: ""
			,(zonetype==ZONE_PRIMARY) ? help_primary : help_secondary
			,nof
			,readonly
				? MENUBUT_CANCEL
				: MENUBUT_CANCEL|MENUBUT_ACCEPT|MENUBUT_DEL);
		if (status == MENU_CANCEL || status == MENU_ESCAPE){
			break;
		}else if (status == MENU_DEL){
			if (xconf_areyousure(MSG_U(Q_DELPRIMARY
				,"Confirm deletion of a domain"))){
				ret = 1;
				break;
			}
		}else{
			if (domainv.is_empty()){
				xconf_error(MSG_U(E_NODOMAIN,"Fill at least the domain\n"
					"and the first IP address"));
			}else{
				int err = 0;
				dns_lexcheck (domainv,nof,0,err,false);
				dns_lexcheck (info.rns,nof,info.start_ns,err,false);
				if (!isrev) dns_lexcheck (info.rmx,nof,info.start_mx,err,true);
				if (!err){
					if (isrev && primary_is_arpadom(domainv.get())){
						primary_reverse_arpa (domainv.get(),domainv);
					}
					setfromv();
					const char *domname = domain.get();
					PRIMARY *dom = dns.finddomain(domname);
					SECONDARY *sec = dns.findsecond(domname);
					if (zonetype==ZONE_PRIMARY && dom != NULL && dom != this){
						xconf_error (MSG_U(E_DOMEXIST
							,"Domain already exist in this DNS"));
					} else if (zonetype==ZONE_SECONDARY && sec != NULL && sec != this){
						xconf_error (MSG_R(E_DOMEXIST));
					}else if ((sec != NULL) && zonetype==ZONE_PRIMARY) {
						xconf_error (MSG_U(E_ISASECOND
							,"This DNS is already a secondary for this domain.\n"
							 "It can't be both at the same time"));
					}else{
						if (getsoa()==NULL) addrec (info.soa);
						/* #Specification: dnsconf / primary / record file
							dnsconf use the domain name as the file name
							which will contain the record.

							You can change the name in /etc/named.boot and dnsconf
							will use this one instead.
						*/
						if (file.is_empty()) file.setfrom (domainv);
						dns.setmx (domain,info.rmx);
						dns.setns (domain,info.rns);
						if (!isrev){
							const char *tbip[info.tba.getnb()];
							int nbip = dnsrecs_tbip(info.tba,tbip);
							dns.set (domainv.get(),tbip,nbip);
						}
						setmodified();
						ret = 0;
						break;
					}
				}
			}
		}
	}
	if (!readonly){
		if (ret != 0) dia.restore();
		allowtrans.remove_empty();
		allowquery.remove_empty();
		allowupdate.remove_empty();
		alsonotify.remove_empty();
	}
	return ret;
}

/*
	Present the setting of a zone in read-only mode
*/
PUBLIC void ZONE::show(DNS &dns)
{
	editshow (dns,true);
}


PUBLIC int PRIMARY::edit(DNS &dns)
{
	return editshow(dns,false);
}




PUBLIC PRIMARY *PRIMARYS::getitem (int no) const
{
	return (PRIMARY*)ARRAY::getitem(no);
}

PUBLIC PRIMARY *PRIMARYS::getitem (const char *name) const
{
	PRIMARY *ret = NULL;
	int n = getnb();
	for (int i=0; i<n; i++){
		PRIMARY *p = getitem(i);
		if (p->domain.cmp(name)==0){
			ret = p;
			break;
		}
	}
	return ret;
}

/*
	Locate all IP number in use in all domain of this DNS.
	Return the number added to adrs
*/
PUBLIC int PRIMARYS::getalladr(IP_ADDRS &adrs)
{
	int ret = 0;
	for (int i=0; i<getnb(); i++){
		ret += getitem(i)->getalladr(adrs);
	}
	return ret;
}

/*
	Get a PRIMARY for a fully qualified host.
	Get the primary that is the closest to the hostname
	(ie: x.y.z.com will select y.z.com instead of z.com if both
	 domain are defined in this DNS).
	Return NULL if not found.
*/
PUBLIC PRIMARY *PRIMARYS::getitem(
	FQHOST &fq,
	char *hostpart,
	int dontitself)		// if dontitself != 0
				// && fq is itself a domain
				// don't select it
{
	/* #Specification: dnsconf / matching a primary / closest
		When trying to dispatch information about a host
		in the DNS, dnsconf try to find the primary which
		has the closest match. This means that if a DNS
		is a primary for x.y.com and y.com, and dnsconf
		dispatch info about the host host.x.y.com, it will
		select the domain x.y.com.

		On the other end, host.w.y.com will be dispatch
		in the domain y.com.
	*/
	PRIMARY *ret = NULL;
	int minlevel = 100;
	if (hostpart != NULL) hostpart[0] = '\0';
	for (int i=0; i<getnb(); i++){
		PRIMARY *pri = getitem(i);
		char tmp[200];
		int level = fq.is_member(pri->domain.get(),tmp);
		if (level > 0 && level < minlevel){
			if (!dontitself || strcmp(tmp,"@")!=0){
				ret = pri;
				minlevel = level;
				if (hostpart != NULL) strcpy (hostpart,tmp);
			}
		}
	}
	return ret;
}

/*
	Get a PRIMARY for a fully qualified host.
	Get the primary that is the closest to the hostname
	(ie: x.y.z.com will select y.z.com instead of z.com if both
	 domain are defined in this DNS).
	Return NULL if not found.
*/
PUBLIC PRIMARY *PRIMARYS::getitem(FQHOST &fq, char *hostpart)
{
	return getitem (fq,hostpart,0);
}


PUBLIC int PRIMARYS::write (bool bind8, FILE *fout, const char *named_dir, const DNS*   dns) const
{
	int ret = 0;
	for (int i=0; i<getnb(); i++){
		if (getitem(i)->write(bind8,fout,named_dir,dns) == -1) ret = -1;
	}
	return ret;
}

PUBLIC VIRTUAL PRIMARY *PRIMARYS::new_PRIMARY()
{
	return new PRIMARY;
}

PUBLIC PRIMARY *PRIMARYS_REV::new_PRIMARY()
{
	return new PRIMARY_REV;
}



/*
	Present the list of primarys. Show selectivly the
	standard domain primaris or the reverse mapping primarys.
*/
PUBLIC void PRIMARYS::edit(DNS &dns)
{
	int choice=0;
	DIALOG_RECORDS dia;
	while (1){
		setselect (dia);
		dia.addwhat (MSG_U(I_ADDPRIM,"Select [Add] to define a new primary"));
		MENU_STATUS code = dia.editmenu(
			MSG_U(T_PRIMARYS,"Primaries")
			,MSG_U(I_PRIMARYS
				,"You are allowed to edit/add/remove primaries\n")
			,help_primary
			,choice,0);
		if (code == MENU_QUIT || code == MENU_ESCAPE){
			break;
		}else if (code == MENU_ADD){
			PRIMARY *pri = new_PRIMARY();
			add (pri);
			if (pri->edit(dns) != 0){
				remove_del (pri);
			}else{
				if (pri->is_reverse()){
					/* #Specification: dnsconf / new primary / is reverse
						When adding a pseudo domain for reverse mapping
						we walk the DNS to stuff it with currently
						defined IP numbers.
					*/
					dns.preload_rev(pri);
				}
				/* #Specification: dnsconf / new primary / host info
					Whenever we add a new primary, we
					try to update it with the basic host information
					of this host so the DNS will be
					automaticly current.
				*/
				dnsconf_updatedns(true);
				dns.write();
			}
		}else if (choice >= 0 && choice < nb){
			PRIMARY *pri = getitem(choice);
			int ok = pri->edit(dns);
			if (ok >= 0){
				if (ok == 1) remove_del(pri);
				dns.write();
			}
		}
	}

}

/*
	Add a new domain and use a template domain to setup few field
	Return -1 if any errors.
*/
PUBLIC int PRIMARYS::adddomain (
	const char *domain,
	const char *example,
	DNS &dns)
{
	int ret = -1;
	if (getitem(domain)!=NULL){
		// domain exist
		xconf_error (MSG_R(E_DOMEXIST));
	}else{
		PRIMARY *p = new PRIMARY;
		p->domainv.setfrom (domain);
		p->setfromv();
		p->file.setfrom (domain);
		RECORD_IN_SOA *soa = new RECORD_IN_SOA;
		p->addrec (soa);
		add (p);

		PRIMARY *templ = NULL;
		if (example != NULL){
			templ = getitem(example);
			if (templ == NULL){
				xconf_error (MSG_U(E_EXMISSING
					 ,"Example domain %s does not exist")
					,example);
			}else{
				RECORD_IN_SOA *tsoa = templ->getsoa();
				if (tsoa != NULL){
					soa->domain.setfrom (tsoa->domain);
					soa->machine.setfrom (tsoa->machine);
					soa->admin.setfrom (tsoa->admin);
					soa->refresh.setfrom (tsoa->refresh);
					soa->retry.setfrom (tsoa->retry);
					soa->expire.setfrom (tsoa->expire);
					soa->default_ttl.setfrom (tsoa->default_ttl);
				}
				SSTRINGS rmx;
				templ->getmx(templ->domain,rmx);
				dns.setmx (p->domain,rmx);
				SSTRINGS rns;
				templ->getns(templ->domain,rns);
				dns.setns (p->domain,rns);

				// Copy bind8 information
				p->allowquery.append (templ->allowquery);
				p->allowtrans.append (templ->allowtrans);
				p->allowupdate.append (templ->allowupdate);
				p->alsonotify.append (templ->alsonotify);
				p->notify = templ->notify;
				p->option_forward_only = templ->option_forward_only;


				ret = 0;
			}
		}else{
			ret = 0;
		}
		if (ret != 0){
			remove_del (p);
		}
	}
	return ret;
}

PUBLIC int DNS::newdomain(
	const char *domain,
	const char *example)
{
	return primarys.adddomain (domain,example,*this);
}


PUBLIC int DNS::deldomain (const char *domain)
{
	int ret = -1;
	PRIMARY *pri = locate_domain (domain);
	if (pri != NULL){
		primarys.remove_del(pri);
		ret = 0;
	}
	return ret;
}

