// pkg_grouppolicy.cc
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Implementation of stuff that needs implementing.

#include "pkg_grouppolicy.h"
#include "pkg_item.h"
#include "pkg_subtree.h"
#include "pkg_grouppolicy_hash.h"

#include "apt.h"

// The following class is a special policy which is used to terminate a
// policy chain:
class pkg_grouppolicy_end:public pkg_grouppolicy
{
  public:
  pkg_grouppolicy_end(pkg_subtree *_root):pkg_grouppolicy(_root) {}

  virtual void add_package(pkgCache::PkgIterator i)
    {
      get_root()->add_child(new pkg_item(i));
    }
};

pkg_grouppolicy *pkg_grouppolicy_end_factory::instantiate(pkg_subtree *_root)
{
  return new pkg_grouppolicy_end(_root);
}

/*****************************************************************************/

// Groups packages so that the latest version of the package is shown, sorted
// by section.
class pkg_grouppolicy_section:public pkg_grouppolicy
{
  typedef pkg_grouppolicy_hash section_map;
  section_map sections;

  pkg_grouppolicy_factory *chain;

  int split_mode;
public:
  pkg_grouppolicy_section(pkg_subtree *_root, int _split_mode, pkg_grouppolicy_factory *_chain):pkg_grouppolicy(_root), chain(_chain),split_mode(_split_mode) {}

  virtual void add_package(pkgCache::PkgIterator pkg);

  virtual ~pkg_grouppolicy_section()
    {
      for(section_map::iterator i=sections.begin(); i!=sections.end(); i++)
        delete i->second;
    }
};

pkg_grouppolicy *pkg_grouppolicy_section_factory::instantiate(pkg_subtree *_root)
{
  return new pkg_grouppolicy_section(_root, split_mode, chain);
}

void pkg_grouppolicy_section::add_package(pkgCache::PkgIterator pkg)
{
  const char *section=pkg.VersionList().Section();

  if(!section)
    section=split_mode!=pkg_grouppolicy_section_factory::split_none?"Unknown/Unknown":"Unknown";

  if(!*section)
    section=split_mode!=pkg_grouppolicy_section_factory::split_none?"virtual/virtual":"virtual";

  const char *split=strchr(section, '/');
  const char *subdir=split?split+1:section;

  string tag;

  if(split_mode==pkg_grouppolicy_section_factory::split_none)
    tag=section;
  else if(split_mode==pkg_grouppolicy_section_factory::split_topdir)
    tag=(split?string(section,split-section):string("main"));
  else if(split_mode==pkg_grouppolicy_section_factory::split_subdir)
    tag=subdir;

  section_map::iterator found=sections.find(tag);

  if(found==sections.end())
    {
      pkg_subtree *newtree=new pkg_subtree(tag.c_str());
      sections[tag]=chain->instantiate(newtree);

      get_root()->add_child(newtree);
    }

  sections[tag]->add_package(pkg);
}

/*****************************************************************************/

class pkg_grouppolicy_status:public pkg_grouppolicy
{
  pkg_grouppolicy_factory *chain;

  enum states {newpkg, upgradable, installed, not_installed, virtual_pkg};

  static const char * const state_titles[];
  // FIXME: need better titles :)

  pkg_grouppolicy *children[5];
public:
  pkg_grouppolicy_status(pkg_subtree *_root, pkg_grouppolicy_factory *_chain)
    :pkg_grouppolicy(_root), chain(_chain)
    {
      for(int i=0; i<( (int) (sizeof(children)/sizeof(children[0]))); i++)
	children[i]=NULL;
    }

  virtual void add_package(pkgCache::PkgIterator pkg);

  virtual ~pkg_grouppolicy_status()
    {
      for(int i=0; i<4; i++)
	delete children[i];
    }
};

const char * const pkg_grouppolicy_status::state_titles[] = {"New Packages", "Upgradable Packages", "Installed Packages", "Not Installed Packages", "Virtual Packages"};

pkg_grouppolicy *pkg_grouppolicy_status_factory::instantiate(pkg_subtree *_root)
{
  return new pkg_grouppolicy_status(_root, chain);
}

void pkg_grouppolicy_status::add_package(pkgCache::PkgIterator pkg)
{
  states section;
  pkgDepCache::StateCache &state=(*apt_cache_file)[pkg];

  if(pkg.VersionList().end())
    section=virtual_pkg;
  else if((*apt_cache_file)->get_ext_state(pkg).new_package)
    section=newpkg;
  else if(state.Status==2)
    section=not_installed;
  else if(state.Upgradable())
    section=upgradable;
  else
    section=installed;

  if(!children[section])
    {
      pkg_subtree *newtree=new pkg_subtree(state_titles[section]);
      children[section]=chain->instantiate(newtree);
      get_root()->add_child(newtree);
    }

  children[section]->add_package(pkg);
}

/*****************************************************************************/
class pkg_grouppolicy_filter:public pkg_grouppolicy
{
  pkg_filter filter;

  pkg_grouppolicy *chain;
public:
  pkg_grouppolicy_filter(pkg_grouppolicy_factory *_chain, pkg_filter _filter, pkg_subtree *_root)
    :pkg_grouppolicy(_root),
     filter(_filter),
     chain(_chain->instantiate(_root)) {}

  virtual void add_package(pkgCache::PkgIterator pkg)
  {if(filter(pkg)) chain->add_package(pkg);}

  virtual ~pkg_grouppolicy_filter() {delete chain;}
};

pkg_grouppolicy *pkg_grouppolicy_filter_factory::instantiate(pkg_subtree *_root)
{
  return new pkg_grouppolicy_filter(chain, filter, _root);
}

bool pkg_missing_filter(pkgCache::PkgIterator pkg)
{
  return (!pkg.VersionList().end()) || (!pkg.ProvidesList().end());
}

class pkg_grouppolicy_mode:public pkg_grouppolicy
{
  const static int num_children=5;
  enum child_groups {remove_children,upgrade_children,install_children,reinstall_children,hold_children};
  const static char * const child_names[];
  pkg_grouppolicy *children[num_children];
  pkg_grouppolicy_factory *chain;
public:
  pkg_grouppolicy_mode(pkg_subtree *_root,pkg_grouppolicy_factory *_chain)
    :pkg_grouppolicy(_root),chain(_chain)
  {
    for(int i=0; i<num_children; i++)
      children[i]=NULL;
  }

  void add_package(pkgCache::PkgIterator pkg)
  {
    aptitudeDepCache::StateCache state=(*apt_cache_file)[pkg];
    int group=-1;
    if(state.Delete())
      group=remove_children;
    else if(state.Install())
      {
	if(!pkg.CurrentVer().end())
	  {
	    if(state.iFlags&pkgDepCache::ReInstall)
	      group=reinstall_children;
	    else
	      group=upgrade_children;
	  }
	else
	  group=install_children;
      }
    else if(!pkg.CurrentVer().end() && state.Status>0 && state.Keep())
      group=hold_children;
    else if(state.iFlags&pkgDepCache::ReInstall)
      group=reinstall_children;

    if(group!=-1)
      {
	if(!children[group])
	  {
	    pkg_subtree *newtree=new pkg_subtree(child_names[group], true);
	    get_root()->add_child(newtree);
	    children[group]=chain->instantiate(newtree);
	  }

	children[group]->add_package(pkg);
      }
  }

  virtual ~pkg_grouppolicy_mode()
  {
    for(int i=0; i<num_children; i++)
      children[i]=NULL;
  }
};

const char * const pkg_grouppolicy_mode::child_names[]={"Packages to be removed", "Packages to be upgraded", "Packages to be installed", "Packages to be reinstalled", "Packages being held back"};

pkg_grouppolicy *pkg_grouppolicy_mode_factory::instantiate(pkg_subtree *root)
{
  return new pkg_grouppolicy_mode(root, chain);
}
