/* -*-Mode: C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: emit.c 1.10.1.1 Sun, 12 Oct 1997 20:29:34 -0700 jmacd $
 */

#include "xdelta.h"

static XdeltaInstruction*
copy_inst_new (MatchQuery* query, gint from, gint from_seg, gint to)
{
  XdeltaInstruction *inst = g_chunk_new (XdeltaInstruction, query->xdelta_chunk);

  inst->type = CopyInstruction;
  inst->from = from;
  inst->from_seg = from_seg;
  inst->to = to;

  return inst;
}

static XdeltaInstruction*
insert_inst_new (MatchQuery* query, gint from, gint to)
{
  XdeltaInstruction *inst = g_chunk_new (XdeltaInstruction, query->xdelta_chunk);

  inst->type = InsertInstruction;
  inst->from = from;
  inst->to = to;

  return inst;
}

static void
emit_region (MatchQuery   *query,
	     const guint8 *seg,
	     gint          low_to_index,
	     gint          high_to_index)
{
  if (low_to_index == high_to_index)
    return;

  query->xdelta = g_slist_prepend (query->xdelta, insert_inst_new (query, low_to_index, high_to_index));
}

void
emit_rest (MatchQuery* query)
{
  emit_region (query,
	       query->real_to_seg,
	       query->current_output_point,
	       query->real_to_len);

  query->current_output_point = query->real_to_len;

  query->xdelta = g_slist_reverse (query->xdelta);
}

void
emit_match (MatchQuery *query, Match *match)
{
  gint low, high, flow, fseg;

  low  = (* query->index_to_real_offset) (match->to_low, query->multibyte_to_map)     + match->low_neg_leftover;
  high = (* query->index_to_real_offset) (match->to_high, query->multibyte_to_map)    - match->high_pos_leftover;
  flow = match->from_real_offset;
  fseg = match->from_segment_index;

  if (query->current_output_point < low)
    {
      emit_region (query,
		   query->real_to_seg,
		   query->current_output_point,
		   low);
    }
  else if (query->current_output_point > low)
    {
      low = query->current_output_point;
    }

  query->current_output_point = high;

  query->xdelta = g_slist_prepend (query->xdelta, copy_inst_new (query, flow, fseg, flow + high - low));
}

static void
emit_num_bytes (guint num, datum* dat)
{
  do
    {
      gint left = num & 0x7f;
      guint outnum;

      num >>= 7;

      outnum = left | (num ? 0x80 : 0);

      dat->dptr[dat->dsize++] = outnum;
    }
  while (num);
}

datum
xdelta_to_bytes (MatchQuery* query)
{
  GSList *l;
  gint len = 0;
  datum dat;

  for (l=query->xdelta; l; l = l->next)
    {
      XdeltaInstruction* inst = (XdeltaInstruction*)l->data;

      if (inst->type == CopyInstruction)
	len += 64;
      else
	len += 64 + (inst->to - inst->from);
    }

  dat.dsize = 0;
  dat.dptr = g_new (gchar, len);

  emit_num_bytes (query->from_count, &dat);
  emit_num_bytes (query->real_to_len, &dat);

#ifdef DEBUG_XDP
  {
    gint i;

    for (i=0;i<query->from_count;i+=1)
      {
	gchar md5[16];

	md5_buffer (query->from[i]->real_seg, query->from[i]->real_len, md5);

	memcpy (dat.dptr + dat.dsize, md5, 16);

	dat.dsize += 16;
      }
  }
#endif

  for (l=query->xdelta; l; l = l->next)
    {
      XdeltaInstruction* inst = (XdeltaInstruction*)l->data;

      if (inst->type == CopyInstruction)
	{
#ifdef DEBUG_INST
	  g_print ("copy %d,%d,%d\n", inst->to - inst->from, inst->from_seg, inst->from);
#endif
	  dat.dptr[dat.dsize++] = CopyInstruction;
	  emit_num_bytes (inst->to - inst->from, &dat);
	  emit_num_bytes (inst->from_seg, &dat);
	  emit_num_bytes (inst->from, &dat);
	}
      else
	{
#ifdef DEBUG_INST
	  g_print ("insert %d\n", inst->to - inst->from);
#endif
	  dat.dptr[dat.dsize++] = InsertInstruction;
	  emit_num_bytes (inst->to - inst->from, &dat);
	  memcpy (dat.dptr + dat.dsize, query->real_to_seg + inst->from, inst->to - inst->from);
	  dat.dsize += inst->to - inst->from;
	}
    }

  return dat;
}

static gboolean
get_num (guint8 *seg, gint len, gint *index, gint* val)
{
  gint c;
  guint8 arr[16];
  gint i = 0;
  gint donebit = 1;
  gint bits;

  while ((*index) < len)
    {
      c = seg[(*index)++];
      donebit = c & 0x80;
      bits = c & 0x7f;

      arr[i++] = bits;

      if (!donebit)
	break;
    }

  if ((*index) == len && donebit)
    return FALSE;

  *val = 0;

  for (i -= 1; i >= 0; i -= 1)
    {
      *val <<= 7;
      *val |= arr[i];
    }

  return TRUE;
}

gint
xdelta_bytes_seg_count (datum delta)
{
  gint index = 0;
  gint ret;

  if (get_num (delta.dptr, delta.dsize, &index, &ret))
    return ret;
  return -1;
}

datum
xpatch (datum delta, datum* segs, gint nsegs)
{
  datum err;
  datum ret;
  gint index = 0;
  gint nsegs_delta;
  gint length;

  err.dptr = NULL;
  ret.dptr = NULL;

  if (!get_num (delta.dptr, delta.dsize, &index, &nsegs_delta))
    goto bail;

  assert (nsegs == nsegs_delta);

  if (!get_num (delta.dptr, delta.dsize, &index, &length))
    goto bail;

  ret.dptr = g_new (gchar, length);
  ret.dsize = 0;

#ifdef DEBUG_XDP
  {
    gint i;

    for (i=0;i<nsegs;i+=1)
      {
	gchar md5[16];

	md5_buffer (segs[i].dptr, segs[i].dsize, md5);

	assert (memcmp (delta.dptr + index, md5, 16) == 0);

	index += 16;
      }

    g_print ("seg ordering ok\n");
  }
#endif

  while (index < delta.dsize)
    {
      gint from_seg_index;
      gint c = delta.dptr[index++];
      gint copy_index;
      gint inst_length;

      if (!get_num (delta.dptr, delta.dsize, &index, &inst_length))
	goto bail;

      switch (c)
	{
	case CopyInstruction:
	  if (!get_num (delta.dptr, delta.dsize, &index, &from_seg_index))
	    goto bail;
	  if (from_seg_index >= nsegs)
	    goto bail;
	  if (!get_num (delta.dptr, delta.dsize, &index, &copy_index))
	    goto bail;
	  if (copy_index + inst_length > segs[from_seg_index].dsize)
	    goto bail;

#ifdef DEBUG_INST
	  g_print ("copy %d %d %d\n", inst_length, from_seg_index, copy_index);
#endif

	  memcpy (ret.dptr + ret.dsize, segs[from_seg_index].dptr + copy_index, inst_length);
	  ret.dsize += inst_length;

	  break;
	case InsertInstruction:
#ifdef DEBUG_INST
	  g_print ("insert %d\n", inst_length);
#endif
	  memcpy (ret.dptr + ret.dsize, delta.dptr + index, inst_length);
	  index += inst_length;
	  ret.dsize += inst_length;
	}
    }

  if (length != ret.dsize)
    goto bail;

  return ret;

bail:

  if (ret.dptr)
    g_free (ret.dptr);

  return err;
}
