/*
 * GTimer
 *
 * Copyright:
 *	(C) 1998 Craig Knudsen, cknudsen@radix.net
 *	See accompanying file "COPYING".
 * 
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version 2
 *	of the License, or (at your option) any later version.
 * 
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 * 
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the
 *	Free Software Foundation, Inc., 59 Temple Place,
 *	Suite 330, Boston, MA  02111-1307, USA
 *
 * Description:
 *	Helps you keep track of time spent on different tasks.
 *
 * Author:
 *	Craig Knudsen, cknudsen@radix.net, http://www.radix.net/~cknudsen
 *
 * Home Page:
 *	http://www.radix.net/~cknudsen/gtimer/
 *
 * History:
 *	05-Apr-1998	Added splash screen.
 *	01-Apr-1998	Added status bar and total hours for today at
 *			the bottom of the main window.
 *	18-Mar-1998	Reduce flicker in task list when changing sort
 *			code submitted by Zach Beane (xach@mint.net)
 *	18-Mar-1998	Release 0.95
 *	18-Mar-1998	Added calls to gtk_window_set_wmclass so the windows
 *			behave better for window managers.
 *			code submitted by ObiTuarY (Obituary@cybernet.be)
 *	17-Mar-1998	Fixed handing if $HOME is not defined.
 *	16-Mar-1998	Changed name to "GTimer"
 *			updated application icon
 *	15-Mar-1998	Added memory debugging calls (memdebug.h).
 *			(The memdebug library is something I wrote myself
 *			a few years ago for another project.  It keeps
 *			track of all malloc/realloc/free calls and can
 *			show you what's been allocated but not freed with
 *			the md_print_all () function.  Email me if you
 *			would like this library.)
 *	15-Mar-1998	Added annotate icon and ability to add annotations
 *			(dated comments) to tasks.
 *	13-Mar-1998	Click on column header to sort by that column.
 *	13-Mar-1998	Did some code redesign.  Pulldown menus setup
 *			from the TTPulldown structure.
 *	13-Mar-1998	Add pulldown menu for right mouse click on a task.
 *			Added functions to add time and remove time from
 *			a task.
 *	13-Mar-1998	Add pulldown menu for right mouse click on a task.
 *			code submitted by Zach Beane (xach@mint.net)
 *	13-Mar-1998	Double-click on a task brings up the edit window
 *			code submitted by Zach Beane (xach@mint.net)
 *	13-Mar-1998	Fixed handling of date change when time passes
 *			midnight.
 *	11-Mar-1998	Rearranged UI.  Added toolbar with icons.
 *			Added "Report" to menubar.
 *	25-Feb-1998	Created
 *
 ****************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include <time.h>
#include <memory.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <gtk/gtk.h>

#include "task.h"
#include "gtimer.h"
#include "config.h"

#ifdef GTIMER_MEMDEBUG
#include "memdebug/memdebug.h"
#endif

/* splash icon */
#include "icons/splash.xpm"

/* timer icon */
#include "icons/clock1.xpm"
#include "icons/clock2.xpm"
#include "icons/clock3.xpm"
#include "icons/clock4.xpm"
#include "icons/clock5.xpm"
#include "icons/clock6.xpm"
#include "icons/clock7.xpm"
#include "icons/clock8.xpm"
#include "icons/blank.xpm"

/* toolbar icons */
#include "icons/start.xpm"
#include "icons/stop.xpm"
#include "icons/stop_all.xpm"
#include "icons/annotate.xpm"
#include "icons/new.xpm"
#include "icons/edit.xpm"

GtkWidget *main_window = NULL;
GtkWidget *splash_window = NULL;
time_t splash_until;
int splash_seconds = 2;
GtkWidget *task_list = NULL;
GtkWidget *status = NULL;
guint status_id = 0;
GtkWidget *total_label = NULL;
static char total_str[20];
static GdkPixmap *icons[8], *blankicon;
static GdkBitmap *icon_masks[8], *blankicon_mask;

typedef struct {
  char *name;
  int width;
  GtkJustification justify;
  GtkWidget *widget;
} list_column_def;

static int sort_forward = 1;
static int last_sort = 0;
static int rebuilding_list = 0;

int today_year, today_mon, today_mday;
int midnight_offset = 0;

char *taskdir = NULL;
char *config_file = NULL;
int selected_task = -1;

TaskData **tasks;
int num_tasks = 0;

list_column_def task_list_columns[3] = {
  { "Task",	330,	GTK_JUSTIFY_LEFT,	NULL },
  { "Today",	70,	GTK_JUSTIFY_RIGHT,	NULL },
  { "Total",	70,	GTK_JUSTIFY_RIGHT,	NULL }
};

/*
** Local functions
*/
void update_list ();
static void build_list ();
static void about_callback ( GtkWidget *widget, gpointer data );
static void save_callback ( GtkWidget *widget, gpointer data );
static void exit_callback ( GtkWidget *widget, gpointer data );
static void start_callback ( GtkWidget *widget, gpointer data );
static void stop_callback ( GtkWidget *widget, gpointer data );
static void stop_all_callback ( GtkWidget *widget, gpointer data );
static void add_callback ( GtkWidget *widget, gpointer data );
static void edit_callback ( GtkWidget *widget, gpointer data );
static void delete_callback ( GtkWidget *widget, gpointer data );
static void increment_time_callback ( GtkWidget *widget, gpointer data );
static void decrement_time_callback ( GtkWidget *widget, gpointer data );
static void report_callback ( GtkWidget *widget, gpointer data );
static void annotate_callback ( GtkWidget *widget, gpointer data );

/*
** Structure for defining the the pulldown menus.
*/
typedef struct {
  char *label;
  void (*callback)();
  gpointer data;
} TTPulldown;

/* File pulldown menu */
TTPulldown file_menu[] = {
  { "About...", about_callback, NULL },
  { "Save", save_callback, NULL },
  { "", NULL, NULL },
  { "Exit", exit_callback, NULL },
  { NULL, NULL, NULL }
};
/* Task pulldown menu */
TTPulldown task_menu[] = {
  { "Start Timing", start_callback, NULL },
  { "Stop Timing", stop_callback, NULL },
  { "Stop All Timing", stop_all_callback, NULL },
  { "", NULL, NULL },
  { "Add...", add_callback, NULL },
  { "Edit...", edit_callback, NULL },
  { "Annotate...", annotate_callback, NULL },
  { "Delete", delete_callback, NULL },
  { "", NULL, NULL },
  { "Increment 5 Minutes", increment_time_callback, (gpointer) (5*60) },
  { "Increment 30 Minutes", increment_time_callback, (gpointer) (30*60) },
  { "Decrement 5 Minutes", decrement_time_callback, (gpointer) (5*60) },
  { "Decrement 30 Minutes", decrement_time_callback, (gpointer) (30*60) },
  { "Set to Zero", increment_time_callback, (gpointer) 0 },
  { NULL, NULL, NULL }
};
/* Task pulldown menu */
TTPulldown report_menu[] = {
  { "Daily...", report_callback, (gpointer)REPORT_TYPE_DAILY },
  { "Weekly...", report_callback, (gpointer)REPORT_TYPE_WEEKLY },
  { "Monthly...", report_callback, (gpointer)REPORT_TYPE_MONTHLY },
  { "Yearly...", report_callback, (gpointer)REPORT_TYPE_YEARLY },
  { NULL, NULL, NULL }
};

/*
** Structure for defining the toolbar
*/
typedef struct {
  char *label;
  char *tooltip;
  gchar **icon_data;
  void (*callback)();
  gpointer data;
} TTToolButton;
TTToolButton main_toolbar[] = {
  { "Start", "Start Timing the Selected Task", start_xpm,
    start_callback, NULL },
  { "Stop", "Stop Timing the Selected Task", stop_xpm,
    stop_callback, NULL },
  { "Stop All", "Stop Timing All Tasks", stop_all_xpm,
    stop_all_callback, NULL },
  { "Annotate", "Add Annotation to Selected Task", annotate_xpm,
    annotate_callback, NULL },
  { "Add", "Add New Task", new_xpm,
    add_callback, NULL },
  { "Edit", "Edit Name of the Selected Task", edit_xpm,
    edit_callback, NULL },
  { NULL, NULL, NULL, NULL, NULL }
};


static int sort_task_by_name ( td1, td2 )
TaskData **td1;
TaskData **td2;
{
  TaskData *tda = *td1;
  TaskData *tdb = *td2;
  int ret;

  ret = ( strcmp ( tda->task->name, tdb->task->name ) );
  if ( sort_forward )
    return ( ret );
  else
    return ( - ret );
}

static int sort_task_by_today ( td1, td2 )
TaskData **td1;
TaskData **td2;
{
  TaskData *tda = *td1;
  TaskData *tdb = *td2;
  int ret = 0;

  if ( tda->last_today_int > tdb->last_today_int )
    ret = 1;
  else if ( tda->last_today_int < tdb->last_today_int )
    ret = -1;
  else if ( tda->last_today_int == tdb->last_today_int )
    ret = ( (void *)tda < (void *)td2 );

  if ( sort_forward )
    return ( - ret );
  else
    return ( ret );
}


static int sort_task_by_total ( td1, td2 )
TaskData **td1;
TaskData **td2;
{
  TaskData *tda = *td1;
  TaskData *tdb = *td2;
  int ret = 0;

  if ( tda->last_total_int > tdb->last_total_int )
    ret = 1;
  else if ( tda->last_total_int < tdb->last_total_int )
    ret = -1;
  else if ( tda->last_total_int == tdb->last_total_int )
    ret = ( (void *)tda < (void *)td2 );

  if ( sort_forward )
    return ( - ret );
  else
    return ( ret );
}




/*
** Transfer all the time for tasks currently being timed into the
** Task data structure so the reports will have access to it easily.
*/
static void update_tasks ()
{
  int i;
  time_t now, diff;

  time ( &now );
  for ( i = 0; i < num_tasks; i++ ) {
    if ( tasks[i]->timer_on ) {
      diff = now - tasks[i]->on_since;
      tasks[i]->todays_entry->seconds += diff;
      tasks[i]->on_since = now;
    }
  }
}



/*
** Save all the tasks to their files.
*/
void save_all ()
{
  update_tasks ();
  taskSaveAll ( taskdir );
}



/* Delete window handler */
gint delete_event ( widget, event, data )
GtkWidget *widget;
GdkEvent *event;
gpointer data;
{
  save_all ();
  configSaveAttributes ( config_file );
#ifdef GTIMER_MEMDEBUG
  configClear ();
#endif
  return ( TRUE );
}

static void exit_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  save_all ();
  configSaveAttributes ( config_file );
#ifdef GTIMER_MEMDEBUG
  configClear ();
#endif
  gtk_main_quit ();
}


static void save_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  gtk_statusbar_push ( GTK_STATUSBAR ( status ), status_id, "All data saved" );
  save_all ();
}

static void about_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  char text[1024];

  sprintf ( text,
    "GTimer\n%s\nVersion: %s (%s)\n%s\nGTK Version: %d.%d.%d\n\n",
    GTIMER_COPYRIGHT, GTIMER_VERSION, GTIMER_VERSION_DATE,
    GTIMER_URL,
    gtk_major_version, gtk_minor_version, gtk_micro_version );
  sprintf ( text + strlen ( text ),
     "Author:\nCraig Knudsen\ncknudsen@radix.net\n\n" );
  create_confirm_window ( CONFIRM_ABOUT,
    "About: GTimer",
    text,
    "Ok", NULL, NULL, NULL, NULL );
}


static void add_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  create_edit_window ( NULL );
}

static void edit_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  if ( selected_task < 0 ) {
    create_confirm_window ( CONFIRM_ERROR,
      "Error",
      "You have not selected\na task to edit.",
      "Ok", NULL,
      NULL, NULL,
      NULL );
  } else {
    create_edit_window ( tasks[selected_task] );
  }
}



static void delete_confirm_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  TaskData *td = (TaskData *)data;
  int ret, loop, tasknumber;
  char msg[500];

  if ( ( ret = taskDelete ( td->task, taskdir ) ) ) {
    sprintf ( msg, "Error deleting task:\n%s",
      taskErrorString ( ret ) );
    create_confirm_window ( CONFIRM_ERROR,
      "Error",
      msg,
      "Ok", NULL,
      NULL, NULL, NULL );
  }

  tasknumber = -1;
  for ( loop = 0; loop < num_tasks && tasknumber < 0; loop++ ) {
    if ( tasks[loop] == td )
      tasknumber = loop;
  }

  if ( tasknumber >= 0 ) {
    gtk_clist_remove ( GTK_CLIST ( task_list ), tasknumber );
    for ( loop = tasknumber; loop < num_tasks; loop++ ) {
      if ( loop + 1 < num_tasks )
        tasks[loop] = tasks[loop + 1];
    }
  }
  free ( td );
  num_tasks--;
  gtk_statusbar_push ( GTK_STATUSBAR ( status ), status_id, "Task removed" );

  update_list ();
}


static void delete_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  if ( selected_task < 0 ) {
    create_confirm_window ( CONFIRM_ERROR,
      "Error",
      "You have not selected\na task to delete.",
      "Ok", NULL,
      NULL, NULL,
      NULL );
  } else {
    create_confirm_window ( CONFIRM_CONFIRM,
      "Confirm: Delete Task",
      "Are you sure you want\nto delete this task?",
      "Ok", "Cancel",
      delete_confirm_callback, NULL,
      (char *)tasks[selected_task] );
  }
}

static void start_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  TaskData *td;

  if ( selected_task < 0 ) {
    create_confirm_window ( CONFIRM_ERROR,
      "Error",
      "You have not selected\na task to start timing.",
      "Ok", NULL,
      NULL, NULL,
      NULL );
  } else {
    td = tasks[selected_task];
    if ( td->timer_on ) {
      create_confirm_window ( CONFIRM_ERROR,
        "Error",
        "Task is already being timed.",
        "Ok", NULL,
        NULL, NULL,
        NULL );
    } else {
      td->timer_on = 1;
      time ( &td->on_since );
      if ( td->todays_entry == NULL )
        td->todays_entry = taskNewTimeEntry ( td->task, today_year,
          today_mon, today_mday );
      update_list ();
    }
  }
}

static void stop_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  TaskData *td;
  time_t now, diff;

  if ( selected_task < 0 ) {
    create_confirm_window ( CONFIRM_ERROR,
      "Error",
      "You have not selected\na task to stop timing.",
      "Ok", NULL,
      NULL, NULL,
      NULL );
  } else {
    td = tasks[selected_task];
    if ( ! td->timer_on ) {
      create_confirm_window ( CONFIRM_ERROR,
        "Error",
        "Task is not being timed.",
        "Ok", NULL,
        NULL, NULL,
        NULL );
    } else {
      td->timer_on = 0;
      time ( &now );
      diff = now - td->on_since;
      td->todays_entry->seconds += diff;
      td->on_since = 0;
      update_list ();
    }
  }
}




static void annotate_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  if ( selected_task < 0 ) {
    create_confirm_window ( CONFIRM_ERROR,
      "Error",
      "You have not selected\na task to annotate.",
      "Ok", NULL,
      NULL, NULL,
      NULL );
  } else {
    create_annotate_window ( tasks[selected_task] );
  }
}


static void stop_all_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  TaskData *td;
  time_t now, diff;
  int loop;

  for ( loop = 0; loop < num_tasks; loop++ ) {
    td = tasks[loop];
    if ( td->timer_on ) {
      td->timer_on = 0;
      time ( &now );
      diff = now - td->on_since;
      td->todays_entry->seconds += diff;
      td->on_since = 0;
    }
  }
  update_list ();
}


static void report_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  report_type rt = (report_type) data;
  update_tasks ();
  create_report_window ( rt );
}


static void adjust_task_time ( offset )
int offset;
{
  TaskData *td;

  if ( selected_task < 0 ) {
    create_confirm_window ( CONFIRM_ERROR,
      "Error",
      "You have not selected\na task to adjust the time for.",
      "Ok", NULL,
      NULL, NULL,
      NULL );
  } else {
    update_tasks ();
    td = tasks[selected_task];
    if ( td->todays_entry == NULL )
      td->todays_entry = taskNewTimeEntry ( td->task,
        today_year, today_mon, today_mday );
    if ( offset == 0 ) {
      td->todays_entry->seconds = 0;
    } else if ( offset < 0 ) {
      if ( td->todays_entry->seconds < ( 0 - offset ) ) {
        td->todays_entry->seconds = 0;
      } else {
        td->todays_entry->seconds += offset;
      }
    } else {
      td->todays_entry->seconds += offset;
    }
    update_list ();
  }
}



static void increment_time_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  int offset = (int)data;
  adjust_task_time ( offset );
}


static void decrement_time_callback ( widget, data )
GtkWidget *widget;
gpointer data;
{
  int offset = (int)data;
  adjust_task_time ( 0 - offset );
}



/*
** Create a pulldown menu (from the user selecting a task with the
** right mouse button.
*/
static GtkWidget *create_task_pulldown () {
  GtkWidget *menu;
  GtkWidget *menu_item;
  int loop;

  menu = gtk_menu_new ();
  for ( loop = 0; task_menu[loop].label != NULL; loop++ ) {
    if ( strlen ( task_menu[loop].label ) )
      menu_item = gtk_menu_item_new_with_label ( task_menu[loop].label );
    else
      menu_item = gtk_menu_item_new ();
    if ( task_menu[loop].callback )
      gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
        GTK_SIGNAL_FUNC ( task_menu[loop].callback ),
        (gpointer) task_menu[loop].data );
    gtk_menu_append ( GTK_MENU ( menu ), menu_item );
    gtk_widget_show ( menu_item );
  }

  return menu;
}



/*
** Callback for selecting the column header.
** Sort the list by the selected column.
*/
static void column_selected_callback ( widget, col )
GtkWidget *widget;
int col;
{
  int i;
  char temp[20];

  if ( col == last_sort )
    sort_forward = ! sort_forward;
  else
    sort_forward = 1;
  last_sort = col;
  switch ( col ) {
    case 0:
      qsort ( tasks, num_tasks, sizeof ( TaskData * ), sort_task_by_name );
      break;
    case 1:
      qsort ( tasks, num_tasks, sizeof ( TaskData * ), sort_task_by_today );
      break;
    default:
    case 2:
      qsort ( tasks, num_tasks, sizeof ( TaskData * ), sort_task_by_total );
      break;
  }
  rebuilding_list = 1;
  gtk_clist_freeze( GTK_CLIST (task_list) );
  build_list ();
  update_list ();
  rebuilding_list = 0;
  for ( i = 0; i < num_tasks; i++ ) {
    if ( tasks[i]->selected )
      gtk_clist_select_row ( GTK_CLIST (task_list), i, 0 );
  }
  gtk_clist_thaw( GTK_CLIST (task_list) );
  sprintf ( temp, "%d", col );
  configSetAttribute ( CONFIG_SORT, temp );
  sprintf ( temp, "%d", sort_forward );
  configSetAttribute ( CONFIG_SORT_FORWARD, temp );
}





/*
** Callback for user selecting a task (single-click, double-click,
** right-mouse, etc.)
*/
static gint task_selected_callback ( widget, col, row, bevent, user_data )
GtkWidget *widget;
int col;
int row;
GdkEventButton *bevent;
gpointer user_data;
{
  GtkWidget *menu;
  int i;

  if ( rebuilding_list )
    return ( TRUE );

  for ( i = 0; i < num_tasks; i++ )
    tasks[i]->selected = 0;

  selected_task = col;
  tasks[selected_task]->selected = 1;

  if ( bevent != NULL ) {
    /* double-click ? */
    if ( bevent->type == GDK_2BUTTON_PRESS )
      edit_callback ( widget, user_data );

    /* right mouse button ? */
    if ( bevent->button == 3 ) {
      menu = create_task_pulldown ();
      gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL,
         NULL, 3, bevent->time);
    }
  }

  return ( TRUE );
}



/*
** Create the main window's menu bar.
*/
static GtkWidget *create_main_window_menu_bar ()
{
  GtkWidget *menu;
  GtkWidget *root_menu;
  GtkWidget *menu_item;
  GtkWidget *menu_bar;
  int loop;

  menu_bar = gtk_menu_bar_new ();

  /* File menu */
  root_menu = gtk_menu_item_new_with_label ( "File" );
  menu = gtk_menu_new ();
  for ( loop = 0; file_menu[loop].label != NULL; loop++ ) {
    if ( strlen ( file_menu[loop].label ) )
      menu_item = gtk_menu_item_new_with_label ( file_menu[loop].label );
    else
      menu_item = gtk_menu_item_new ();
    if ( file_menu[loop].callback )
      gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
        GTK_SIGNAL_FUNC ( file_menu[loop].callback ),
        (gpointer) file_menu[loop].data );
    gtk_menu_append ( GTK_MENU ( menu ), menu_item );
    gtk_widget_show ( menu_item );
  }

  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  /* Task menu */
  root_menu = gtk_menu_item_new_with_label ( "Task" );
  menu = create_task_pulldown ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  /* Report menu */
  root_menu = gtk_menu_item_new_with_label ( "Report" );
  menu = gtk_menu_new ();
  for ( loop = 0; report_menu[loop].label != NULL; loop++ ) {
    if ( strlen ( report_menu[loop].label ) )
      menu_item = gtk_menu_item_new_with_label ( report_menu[loop].label );
    else
      menu_item = gtk_menu_item_new ();
    if ( report_menu[loop].callback )
      gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
        GTK_SIGNAL_FUNC ( report_menu[loop].callback ),
        (gpointer) report_menu[loop].data );
    gtk_menu_append ( GTK_MENU ( menu ), menu_item );
    gtk_widget_show ( menu_item );
  }

  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  return ( menu_bar );
}


static GtkWidget *create_list_column_def (num, cols)
int num;
list_column_def *cols;
{
  GtkWidget *clist;
  GtkWidget *alignment;
  GtkWidget *label;
  int i;

  clist = gtk_clist_new (num);
  GTK_CLIST_SET_FLAGS (clist, CLIST_SHOW_TITLES);

  for (i = 0; i < num; i++) {
    gtk_clist_set_column_width (GTK_CLIST (clist), i, cols[i].width);
    if (cols[i].justify != GTK_JUSTIFY_LEFT) {
      gtk_clist_set_column_justification (GTK_CLIST (clist), i,
        cols[i].justify);
    }

    alignment = gtk_alignment_new (0.0, 0.5, 0.0, 0.0);

    label = gtk_label_new (cols[i].name);
    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
    gtk_container_add (GTK_CONTAINER (alignment), label);
    gtk_widget_show (label);

    cols[i].widget = label;

    gtk_clist_set_column_widget (GTK_CLIST (clist), i, alignment);
    /*gtk_clist_column_title_passive ( GTK_CLIST(clist), i );*/
    gtk_widget_show (alignment);
  }

  return clist;
}



/*
** Update the time values shown in the list.
*/
void update_list () {
  TaskData *taskdata;
  int i;
  int h, m, s;
  char text[100];
  time_t now, diff, total, today;
  GdkPixmap *icon;
  GdkBitmap *mask;
  char *row[3];
  int total_today = 0;
  char today_test[20];

  time ( &now );
  icon = icons[now%8];
  mask = icon_masks[now%8];

  /*gtk_clist_freeze ( GTK_CLIST(task_list) );*/
  for ( i = 0; i < num_tasks; i++ ) {
    taskdata = tasks[i];
    /* new task ? */
    if ( taskdata->new_task ) {
      taskdata->new_task = 0;
      row[0] = taskdata->task->name;
      row[1] = "00:00:00";
      row[2] = "00:00:00";
      gtk_clist_append ( GTK_CLIST(task_list), row );
      gtk_clist_set_pixtext (GTK_CLIST (task_list), i, 0, 
        taskdata->task->name, 2, blankicon, blankicon_mask);
      continue;
    }
    /* update the name ? */
    if ( taskdata->name_updated || taskdata->moved ) {
      taskdata->name_updated = 0;
      if ( taskdata->timer_on ) {
        gtk_clist_set_pixtext (GTK_CLIST (task_list), i, 0,
          taskdata->task->name, 2, icon, mask);
        taskdata->last_on = 1;
      } else {
        gtk_clist_set_pixtext (GTK_CLIST (task_list), i, 0, 
          taskdata->task->name, 2, blankicon, blankicon_mask);
        taskdata->last_on = 0;
      }
    }
    /* calc total */
    total = taskdata->total;
    if ( taskdata->todays_entry )
      total += taskdata->todays_entry->seconds;
    if ( taskdata->timer_on ) {
      time ( &now );
      diff = now - taskdata->on_since;
      total += diff;
    }
    h = total / 3600;
    m = ( total - h * 3600 ) / 60;
    s = total % 60;
    sprintf ( text, "%d:%02d:%02d", h, m, s );
    if ( strcmp ( text, taskdata->last_total ) || taskdata->moved ) {
      gtk_clist_set_text ( GTK_CLIST(task_list), i, 2, text );
      strcpy ( taskdata->last_total, text );
    }
    taskdata->last_total_int = total;
    today = 0;
    if ( taskdata->todays_entry )
      today = taskdata->todays_entry->seconds;
    if ( taskdata->timer_on ) {
      time ( &now );
      diff = now - taskdata->on_since;
      today += diff;
    } 
    h = today / 3600;
    m = ( today - h * 3600 ) / 60;
    s = today % 60;
    sprintf ( text, "%d:%02d:%02d", h, m, s );
    if ( strcmp ( text, taskdata->last_today ) || taskdata->moved ) {
      gtk_clist_set_text ( GTK_CLIST(task_list), i, 1, text );
      strcpy ( taskdata->last_today, text );
    }
    taskdata->last_today_int = today;
    /* draw the icon ? */
    if ( taskdata->timer_on ) {
      gtk_clist_set_pixtext (GTK_CLIST (task_list), i, 0,
        taskdata->task->name, 2, icon, mask);
      taskdata->last_on = 1;
    } else if ( ! taskdata->timer_on && taskdata->last_on ) {
      gtk_clist_set_pixtext (GTK_CLIST (task_list), i, 0, 
        taskdata->task->name, 2, blankicon, blankicon_mask);
      taskdata->last_on = 0;
    }
    taskdata->moved = 0;
    total_today += today;
  }
  /*gtk_clist_thaw ( GTK_CLIST(task_list) );*/

  h = total_today / 3600;
  m = ( total_today - h * 3600 ) / 60;
  s = total_today % 60;
  sprintf ( today_test, "Today: %d:%02d:%02d", h, m, s );
  if ( strcmp ( today_test, total_str ) ) {
    strcpy ( total_str, today_test );
    gtk_label_set ( GTK_LABEL ( total_label ), total_str );
  }
}


/*
** Create the task list.
*/
static void build_list () {
  Task *task;
  TaskData *taskdata;
  char today_str[100], total_str[100];
  char *row[3];
  int i, j;
  GdkPixmap *icon;
  GdkBitmap *mask;
  time_t now;
  static int first = 1;
  GtkWidget *win;

  if ( splash_window )
    win = splash_window;
  else
    win = main_window;

  /* gtk_clist_freeze ( GTK_CLIST(task_list) ); */
  gtk_clist_clear ( GTK_CLIST(task_list) );

  if ( first ) {
    icons[0] = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &icon_masks[0],
      &win->style->white, clock1_xpm);
    icons[1] = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &icon_masks[1],
      &win->style->white, clock2_xpm);
    icons[2] = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &icon_masks[2],
      &win->style->white, clock3_xpm);
    icons[3] = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &icon_masks[3],
      &win->style->white, clock4_xpm);
    icons[4] = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &icon_masks[4],
      &win->style->white, clock5_xpm);
    icons[5] = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &icon_masks[5],
      &win->style->white, clock6_xpm);
    icons[6] = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &icon_masks[6],
      &win->style->white, clock7_xpm);
    icons[7] = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &icon_masks[7],
      &win->style->white, clock8_xpm);
    blankicon = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( win )->window, &blankicon_mask,
      &win->style->white, blank_xpm);

    tasks = (TaskData **) malloc ( taskCount() * sizeof ( TaskData * ) );
    for ( i = 0, task = taskGetFirst(); task != NULL;
      i++, task = taskGetNext () ) {
      taskdata = (TaskData *) malloc ( sizeof ( TaskData ) );
      memset ( taskdata, '\0', sizeof ( TaskData ) );
      taskdata->task = task;
      taskdata->todays_entry = taskGetTimeEntry ( taskdata->task, today_year,
        today_mon, today_mday );
      for ( j = 0; j < taskdata->task->num_entries; j++ ) {
        if ( taskdata->task->entries[j] != taskdata->todays_entry )
          taskdata->total += taskdata->task->entries[j]->seconds;
      }
      strcpy ( taskdata->last_today, today_str );
      strcpy ( taskdata->last_total, total_str );
      tasks[num_tasks++] = taskdata;
    }
    /* sort the list of tasks */
    qsort ( tasks, num_tasks, sizeof ( TaskData * ), sort_task_by_name );
  }

  time ( &now );
  icon = icons[now%8];
  mask = icon_masks[now%8];
  for ( i = 0; i < num_tasks; i++ ) {
    tasks[i]->moved = 1;
    task = tasks[i]->task;
    row[0] = task->name;
    sprintf ( today_str, "00:00:00" );
    row[1] = today_str;
    sprintf ( total_str, "00:00:00" );
    row[2] = total_str;
    gtk_clist_append ( GTK_CLIST(task_list), row );
    if ( tasks[i]->timer_on )
      gtk_clist_set_pixtext (GTK_CLIST (task_list), i, 0, 
        task->name, 2, icon, mask);
    else
      gtk_clist_set_pixtext (GTK_CLIST (task_list), i, 0, 
        task->name, 2, blankicon, blankicon_mask);
  }
  /* gtk_clist_thaw ( GTK_CLIST(task_list) ); */

  first = 0;
}



/*
** Handle the update.  This gets called every 1 second.
** Notice that, Unlike Xt, you don't have to add the timeout again :-)
*/
static gint timeout_handler ( gpointer data ) {
  time_t now;
  struct tm *tm;
  int loop;

  time ( &now );

  /* remove splash window ? */
  if ( splash_window && now > splash_until ) {
    if ( GTK_IS_WIDGET ( splash_window ) )
      gtk_widget_destroy ( splash_window );
    splash_window = NULL;
    gtk_widget_show ( main_window );
  }

  /* Check to see if the date has changed. */
  now -= midnight_offset;
  tm = localtime ( &now );
  if ( today_mday != tm->tm_mday ) {
    update_tasks ();
    today_year = tm->tm_year + 1900;
    today_mon = tm->tm_mon + 1;
    today_mday = tm->tm_mday;
    for ( loop = 0; loop < num_tasks; loop++ ) {
      if ( tasks[loop]->todays_entry )
        tasks[loop]->total += tasks[loop]->todays_entry->seconds;
      tasks[loop]->todays_entry = taskGetTimeEntry ( tasks[loop]->task,
        today_year, today_mon, today_mday );
      if ( tasks[loop]->timer_on ) {
        if ( ! tasks[loop]->todays_entry )
          tasks[loop]->todays_entry = taskNewTimeEntry ( tasks[loop]->task,
            today_year, today_mon, today_mday );
        time ( &tasks[loop]->on_since );
      }
    }
  }

  /* Update the list */
  update_list ();

  /* return TRUE to so this timeout happens again in 1 second */
  return ( TRUE );
}




void create_splash_window () {
  GtkWidget *table, *pixmap, *label;
  GdkPixmap *icon;
  GdkBitmap *mask;
  char msg[500];
  GtkStyle *style;

  splash_window = gtk_window_new ( GTK_WINDOW_TOPLEVEL );
  gtk_window_set_wmclass ( GTK_WINDOW ( splash_window ), "GTimer", "gtimer" );
  gtk_window_set_title ( GTK_WINDOW ( splash_window ), "GTimer" );
  gtk_widget_set_usize ( splash_window, 400, 200 );
  gtk_window_position ( GTK_WINDOW ( splash_window ), GTK_WIN_POS_CENTER );
  gtk_widget_realize ( splash_window );

  table = gtk_table_new ( 2, 2, FALSE );
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 8);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add ( GTK_CONTAINER ( splash_window ), table );

  icon = gdk_pixmap_create_from_xpm_d (
    GTK_WIDGET ( splash_window )->window, &mask,
    &splash_window->style->white, splash_xpm );
  pixmap = gtk_pixmap_new ( icon, mask );
  gtk_misc_set_alignment (GTK_MISC (pixmap), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), pixmap, 0, 1, 0, 2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show ( pixmap );

  style = gtk_style_new ();
  gdk_font_unref ( style->font );
  style->font = gdk_font_load (
    "-adobe-helvetica-bold-r-normal-*-24-*-*-*-*-*-iso8859-1" );
  if ( style->font == NULL )
    style->font = gdk_font_load ( "fixed" );
  gtk_widget_push_style ( style );

  sprintf ( msg, "GTimer v%s", GTIMER_VERSION );
  label = gtk_label_new ( msg );
  gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show ( label );

  gtk_widget_pop_style ();

  sprintf ( msg, "%s\nGTK Version: %d.%d.%d\n",
    GTIMER_COPYRIGHT,
    gtk_major_version, gtk_minor_version, gtk_micro_version );
  label = gtk_label_new ( msg );
  gtk_table_attach (GTK_TABLE (table), label, 1, 2, 1, 2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show ( label );

  gtk_widget_show ( table );
  gtk_widget_show ( splash_window );

  time ( &splash_until );
  splash_until += splash_seconds;
}


void create_main_window () {
  GtkWidget *vbox;
  GtkWidget *menu_bar, *toolbar, *toolbutton, *iconw, *table;
  GdkPixmap *icon;
  GdkBitmap *mask;
  int loop;
  char msg[50];

  main_window = gtk_window_new ( GTK_WINDOW_TOPLEVEL );
  gtk_window_set_wmclass( GTK_WINDOW ( main_window ), "GTimer", "gtimer" );
  gtk_signal_connect ( GTK_OBJECT ( main_window ), "delete_event",
    GTK_SIGNAL_FUNC ( exit_callback ), NULL );
  gtk_signal_connect ( GTK_OBJECT ( main_window ), "destroy",
    GTK_SIGNAL_FUNC ( exit_callback ), NULL );
  gtk_window_set_title (GTK_WINDOW (main_window), "GTimer" );
  gtk_widget_realize ( main_window );

  vbox = gtk_vbox_new ( FALSE, 0 );
  gtk_container_add ( GTK_CONTAINER ( main_window ), vbox );

  menu_bar = create_main_window_menu_bar ();
  gtk_box_pack_start ( GTK_BOX ( vbox ), menu_bar, FALSE, FALSE, 0 );
  gtk_widget_show ( menu_bar );

  /* create toolbar */
  toolbar = gtk_toolbar_new ( GTK_ORIENTATION_HORIZONTAL,
    GTK_TOOLBAR_ICONS );
  gtk_toolbar_set_space_size ( GTK_TOOLBAR ( toolbar ), 5 );
  gtk_toolbar_append_space ( GTK_TOOLBAR ( toolbar ) );
  gtk_box_pack_start ( GTK_BOX ( vbox ), toolbar, FALSE, FALSE, 5 );

  for ( loop = 0; main_toolbar[loop].label != NULL; loop++ ) {
    icon = gdk_pixmap_create_from_xpm_d (
      GTK_WIDGET ( main_window )->window, &mask,
      &main_window->style->white, main_toolbar[loop].icon_data );
    iconw = gtk_pixmap_new ( icon, mask );
    toolbutton = gtk_toolbar_append_item ( GTK_TOOLBAR ( toolbar ),
      main_toolbar[loop].label, main_toolbar[loop].tooltip, "Private",
      iconw, GTK_SIGNAL_FUNC ( main_toolbar[loop].callback ), NULL );
    gtk_toolbar_append_space ( GTK_TOOLBAR ( toolbar ) );
    gtk_widget_show ( toolbutton );
  }

  gtk_widget_show ( toolbar );

  /* add in list here */
  task_list = create_list_column_def ( 3, task_list_columns );
  gtk_clist_set_selection_mode (GTK_CLIST (task_list), GTK_SELECTION_BROWSE);
  gtk_widget_set_usize (GTK_WIDGET (task_list), 500, 150);
  gtk_clist_set_policy (GTK_CLIST (task_list),
    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (task_list), "click_column",
    GTK_SIGNAL_FUNC (column_selected_callback), NULL);
  gtk_signal_connect_after (GTK_OBJECT (task_list), "select_row",
    GTK_SIGNAL_FUNC (task_selected_callback), NULL);
  gtk_box_pack_start ( GTK_BOX ( vbox ), task_list, TRUE, TRUE, 0 );
  gtk_widget_show (task_list);

  /* add a status area and a place for the total time for today */
  table = gtk_table_new ( 10, 1, FALSE );
  gtk_box_pack_start ( GTK_BOX ( vbox ), table, TRUE, TRUE, 2 );

  status = gtk_statusbar_new ();
  gtk_table_attach_defaults ( GTK_TABLE (table), status, 0, 9, 0, 1 );
  gtk_widget_show (status);
  status_id = gtk_statusbar_get_context_id ( GTK_STATUSBAR ( status ),
    GTIMER_STATUS_ID );
  sprintf ( msg, "Welcome to GTimer %s", GTIMER_VERSION );
  gtk_statusbar_push ( GTK_STATUSBAR ( status ), status_id, msg );

  total_label = gtk_label_new ( "Total: 0:00:00" );
  gtk_table_attach_defaults ( GTK_TABLE (table), total_label, 9, 10, 0, 1 );
  gtk_label_set_justify ( GTK_LABEL ( total_label ), GTK_JUSTIFY_RIGHT );
  gtk_widget_show (total_label);

  gtk_widget_show (table);

  gtk_widget_show (vbox);
}


int main ( int argc, char *argv[] ) {
  char *home = "";
  uid_t uid;
  struct passwd *passwd;
  time_t now;
  struct tm *tm;
  int loop, offset;
  char *ptr;
  struct stat buf;
  int display_splash = 1;

  if ( getenv ( "HOME" ) ) {
    home = getenv ( "HOME" );
  }
  else {
    uid = getuid ();
    passwd = getpwuid ( uid );
    if ( passwd )
      home = passwd->pw_dir;
  }

  taskdir = (char *) malloc ( strlen ( home ) +
    strlen ( TASK_DIRECTORY ) + 10 );
  sprintf ( taskdir, "%s/%s", home, TASK_DIRECTORY );

  if ( stat ( taskdir, &buf ) != 0 ) {
    /* check for ".tasktimer" directory for backwards compatiblity */
    sprintf ( taskdir, "%s/%s", home, ".tasktimer" );
    if ( stat ( taskdir, &buf ) != 0 ) {
      sprintf ( taskdir, "%s/%s", home, TASK_DIRECTORY );
      if ( mkdir ( taskdir, 0777 ) ) {
        fprintf ( stderr, "Error: unable to create directory %s\n", taskdir );
        exit ( 1 );
      }
    }
  }

  /* Init GTK */
  gtk_init ( &argc, &argv );

  /* Examine command line args */
  for ( loop = 1; loop < argc; loop++ ) {
    if ( strcmp ( argv[loop], "-dir" ) == 0 ) {
      if ( ! argv[loop+1] ) {
        fprintf ( stderr, "Error: -dir requires an argument.\n" );
        exit ( 1 );
      }
      taskdir = argv[++loop];
    } else if ( strcmp ( argv[loop], "-nosplash" ) == 0 ) {
      display_splash = 0;
    } else if ( strcmp ( argv[loop], "-midnight" ) == 0 ) {
      if ( ! argv[loop+1] ) {
        fprintf ( stderr, "Error: -midnight requires an argument.\n" );
        exit ( 1 );
      }
      for ( ptr = argv[++loop]; *ptr != '\0'; ptr++ ) {
        if ( ! isdigit ( *ptr ) && *ptr != '-' ) {
          fprintf ( stderr, "Error: -midnight requires a number (not %s)\n",
            argv[loop] );
          exit ( 1 );
        }
      }
      ptr = argv[loop];
      if ( *ptr == '-' )
        offset = atoi ( ptr + 1 );
      else
        offset = atoi ( ptr );
      if ( offset > 2359 ) {
        fprintf ( stderr, "Invalid offset for -midnight: %s\n", argv[loop] );
        fprintf ( stderr, "Should be HHMM (<2359)\n" );
        exit ( 1 );
      }
      midnight_offset = ( offset / 100 * 3600 ) + ( offset % 100 * 60 );
      if ( *ptr == '-' )
        midnight_offset *= -1;
    } else {
      fprintf ( stderr, "Ingoring unknown option: %s\n", argv[loop] );
    }
  }

  /* read config values */
  config_file = (char *) malloc ( strlen ( taskdir ) + 
    strlen ( CONFIG_DEFAULT_FILE ) + 2 );
  sprintf ( config_file, "%s/%s", taskdir, CONFIG_DEFAULT_FILE );
  configReadAttributes ( config_file );

  /* in the future check version number and pop up license and/or
  ** release notes if a new version
  */
  configSetAttribute ( CONFIG_VERSION, GTIMER_VERSION );

  /* load all tasks */
  taskLoadAll ( taskdir );

  /* Create splash window */
  if ( display_splash )
    create_splash_window ();

  /* Create window */
  create_main_window ();
  if ( ! splash_window )
    gtk_widget_show ( main_window );

  /* set the current date */
  time ( &now );
  now -= midnight_offset;
  tm = localtime ( &now );
  today_year = tm->tm_year + 1900;
  today_mon = tm->tm_mon + 1;
  today_mday = tm->tm_mday;

  /* build and update the task list */
  build_list ();
  update_list ();

  /* sort list like it was last time */
  if ( configGetAttribute ( CONFIG_SORT, &ptr ) == 0 ) {
    last_sort = atoi ( ptr );
    if ( configGetAttribute ( CONFIG_SORT_FORWARD, &ptr ) == 0 )
      sort_forward = ! atoi ( ptr );
    else
      sort_forward = 0;
    column_selected_callback ( NULL, last_sort );
  } else {
    sort_forward = 0;
    column_selected_callback ( NULL, 0 );
  }

  /* Add a timeout to update the display once a second */
  gtk_timeout_add ( 1000, timeout_handler, NULL );

  /* Loop endlessly */
  gtk_main ();

#ifdef GTIMER_MEMDEBUG
  /* memory debugging... make sure md_print_all gets linked in so we can
  ** call it from gdb.
  */
  for ( loop = 0; loop < num_tasks; loop++ ) {
    taskFree ( tasks[loop]->task );
    free ( tasks[loop] );
  }
  free ( tasks );
  md_print_all ();
#endif

  return ( 0 );
}

