/*****************************************************************************
* ppipemon.c	PPP Monitor.
*
* Author:	Jaspreet Singh
*		Farhan Thawar
*
* Copyright:	(c) 1995-1997 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Nov 24, 1997  Jaspreet Singh  Added new stats for driver statistics
* Oct 20, 1997	Jaspreet Singh	Added new commands for Driver statistics and
*				Router Up time.
* Jul 28, 1997	Jaspreet Singh	Added a new command for running line trace 
*				displaying RAW data. 
* Jun 24, 1997  Jaspreet Singh	S508/FT1 test commands
* Apr 25, 1997	Farhan Thawar	Initial version based on ppipemon for WinNT.
*****************************************************************************/

#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <string.h>
#include "ppipemon.h"
#include <linux/config.h>

#define TIMEOUT 1
#define TRUE 1
#define FALSE 0
#define MDATALEN 2024

//
// Structures and Types
//
#pragma pack(1)
typedef struct {
	unsigned char signature[8];
	unsigned char requestreply;
	unsigned char id;
	unsigned char reserved2[6];
	unsigned char opp_flag;
        unsigned char command;
	unsigned short buffer_length;
        unsigned char return_code;
	unsigned char reserved[11];
        unsigned char data[MDATALEN];
} CBLOCK;
#pragma pack()


// Structures for tracing
//
// The one frame trace;
typedef struct {
   unsigned char status;			// equals to 0x00 if the frame is incoming, 0x01 if outgoing
   // for the first frame of the bunch of frames sent to user the 
   // bits 2-7 of this byte tells number of frames passed, and bit 1 tell
   // if there are more frames already available on the board

   unsigned char passed_length;	// 0 if no data passed. 1 - if real_length bytes passed
   unsigned short  real_length;	// real(original) frame length
   unsigned short  time_stamp;		// the frame's time stamp
   unsigned char data[1];		// first byte of data
} FRAME_DATA, *PFRAME_DATA;


/* Prototypes */
void ResetCB(CBLOCK *c1);
int MakeConnection( void );
int ObtainConfiguration( void );
unsigned char DoCommand( CBLOCK *cb );
void SaveCB( CBLOCK *dest, CBLOCK *src );
void init( int argv, char *argc[]);

/* global for now */
int sock;
struct sockaddr_in soin;
CBLOCK cb, cbconfig;
char codeversion[10];
unsigned int frame_count;
int is_508;
int raw_data = FALSE;
/* for S508/FT1 test commands */
int fail;
unsigned char par_port_A_byte, par_port_B_byte;
int off_counter, green_counter, red_counter;
int loop_counter;


/* defines for now */
char ipaddress[16];
int udp_port = 9000;

void ResetCB(CBLOCK *c1)
{
	memset((void *)&c1->opp_flag, 0, sizeof(CBLOCK)-16);
}; //ResetCB

int MakeConnection( void ) {
   sock = socket( PF_INET, SOCK_DGRAM, 0 );
   if( sock < 0 ) {
      perror("Unable to open socket!");
      return( FALSE );
   } /* if */
   soin.sin_family = AF_INET;
   soin.sin_addr.s_addr = inet_addr(ipaddress);
   soin.sin_port = htons((u_short)udp_port);
   if( !connect( sock, (struct sockaddr *)&soin, sizeof(soin)) == 0 ) {
      perror("Cannot make connection!");
      return( FALSE );
   } /* if */
   if( !ObtainConfiguration() ) {
      perror("Unable to obtain PPP information.  Make sure the IP and UDP port are correct.");
      close( sock );
      return( FALSE );
   } /* if */   
   return( TRUE );
}; /* MakeConnection */

int ObtainConfiguration( void ) {
   unsigned char x;
   
   x = 0;
   cb.command = READ_CONFIGURATION;
   cb.buffer_length = 0;
   while (DoCommand(&cb) != 0 && ++x < 4) {
      if (cb.return_code == 0xaa) {
	 perror("Response timeout occurred"); 
	 return(FALSE);
      }
      if (cb.return_code == 0xCC ) return(FALSE);
   }
   if (x >= 4) return(FALSE);
   if( cb.buffer_length == 0x73 ) {
      is_508 = TRUE;
   } else {
      is_508 = FALSE;
   } //if
   
   strcpy(codeversion, "?.??");
   
   cb.command = READ_CODE_VERSION;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if (cb.return_code == 0) {
      cb.data[cb.buffer_length] = 0;
      strcpy(codeversion, cb.data);
   }
   return(TRUE);
}; /* ObtainConfiguration */

unsigned char DoCommand( CBLOCK *cb ) {
   static unsigned char id = 0;
   fd_set ready;
   struct timeval to;
   int x;

   for( x = 0; x < 4; x += 1 ) {
      cb->opp_flag = 0;
      cb->requestreply = 0x01;
      cb->id = id;
      cb->return_code = 0xaa;	// 0xAA is our special return code indicating packet timeout
      send(sock, cb->signature, 32+cb->buffer_length,0);
      FD_ZERO(&ready);
      FD_SET(sock,&ready);
      to.tv_sec = 5;
      to.tv_usec = 0;
      if(select(sock + 1,&ready, NULL, NULL, &to)) {
	 recv(sock, cb->signature, 32+MDATALEN,0);
	break;
      } else {
	 perror("Response timeout occurred."); 
      } /* if */
   } //for
   // make sure the id is correct (returning results from a previous
   // call may be disastrous if not recognized)
   // also make sure it is a reply
   if (cb->id != id || cb->requestreply != 0x02) cb->return_code = 0xbb;
   id++;
   if (cb->return_code == 0xCC) {
      perror("Code is not running on the board!");
      exit(1);
   }
   return(cb->return_code);
   
}; /* DoCommand */

void init( int argv, char *argc[]){
   cb.id = 0;
   /* Different signature for Driver Statistics */
   if( argc[1][0] == 'd' )
	strcpy( cb.signature, "DRVSTATS");
   else
   	strcpy( cb.signature, "PTPIPEAB");
	
   if( (argc[1][0] == 'f')  || (argc[1][0] == 't')) {
      strcpy(ipaddress,argc[3]);
      if( argv > 4 ) {
	 udp_port = atoi(argc[4]);
      } //if
   } 
   else {
      strcpy(ipaddress,argc[2]);
      if( argv > 3 ) {
	 udp_port = atoi(argc[3]);
      } //if
   }
};

unsigned long decode_bps( unsigned char bps ) {
   unsigned long number;
   
   switch( bps ) {
   case 0x00:
      number = 0;
      break;
   case 0x01:
      number = 1200;
      break;
   case 0x02:
      number = 2400;
      break;
   case 0x03:
      number = 4800;
      break;
   case 0x04:
      number = 9600;
      break;
   case 0x05:
      number = 19200;
      break;
   case 0x06:
      number = 38400;
      break;
   case 0x07:
      number = 45000;
      break;
   case 0x08:
      number = 56000;
      break;
   case 0x09:
      number = 64000;
      break;
   case 0x0A:
      number = 74000;
      break;
   case 0x0B:
      number = 112000;
      break;
   case 0x0C:
      number = 128000;
      break;
   case 0x0D:
      number = 156000;
      break;
   } //switch
   return number;
}; //decode_bps

void error( void ) {
   printf("Command returned non-zero value!\n");
}; //error

void general_conf( void ) {
   unsigned long baud;
   unsigned short s_number, mtu, mru;
   unsigned char misc;
   
   cb.command = READ_CONFIGURATION;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      if( is_508 == TRUE ) {
	 printf("508 Board\n");
	 memcpy(&baud,&cb.data[0],4);
	 memcpy(&s_number,&cb.data[4],2);
	 printf("Transmit buffer allocation in percent: %d\n",s_number);
	 misc = cb.data[6];
	 printf("Use alternate adapter frequency: ");
	 ( misc & 0x10 ) ? printf("Yes\n") : printf("No\n");
	 printf("Set interface type to RS-232: ");
	 ( misc & 0x20 ) ? printf("Yes\n") : printf("No\n");
	 memcpy(&mtu, &cb.data[8],2);
	 memcpy(&mru, &cb.data[10],2);
      } else {
	 printf("502 Board\n");
	 baud = decode_bps(cb.data[0]);
	 misc = cb.data[3];
	 printf("Discard transmit-aborted frames: ");
	 ( misc & 0x01 ) ? printf("Yes\n") : printf("No\n");
	 memcpy(&mtu, &cb.data[5],2);
	 memcpy(&mru, &cb.data[7],2);
      } //if
      printf("Baud rate in bps: %lu\n",baud);
      printf("Update transmit statistics( user data ): ");
      ( misc & 0x02 ) ? printf("Yes\n") : printf("No\n");
      printf("Update receive statistics( user data ): ");
      ( misc & 0x04 ) ? printf("Yes\n") : printf("No\n");
      printf("Timestamp received packets: ");
      ( misc & 0x08 ) ? printf("Yes\n") : printf("No\n");
      printf("Maximum Receive/Transmit Unit(MRU/MTU): %d\n",mtu);
      printf("Minimum remote MRU required on connection: %d\n",mtu);
   } else {
      error();
   } //if
}; //general_conf

void timers( void ) {
   int i;
   unsigned short tmp;
   
   cb.command = READ_CONFIGURATION;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      if( is_508 == TRUE ) {
	 i = 12;
      } else {
	 i = 9;
      } //if
      memcpy(&tmp,&cb.data[i],2);
      printf("Restart timer: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+2],2);
      printf("Authentication restart timer: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+4],2);
      printf("Authentication wait timer: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+6],2);
      printf("DCD/CTS failure detection timer: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+8],2);
      printf("Drop DTR duration timer: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+10],2);
      printf("Connection attempt timeout: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+12],2);
      printf("Max-Configure counter: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+14],2);
      printf("Max-Terminate counter: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+16],2);
      printf("Max-Failure counter: %d\n",tmp);
      memcpy(&tmp,&cb.data[i+18],2);
      printf("Max-Authenticate counter: %d\n",tmp);
   } else {
      error();
   } //if
}; //timers

void authent( void ) {
   unsigned char tmp;
   
   cb.command = READ_CONFIGURATION;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      if( is_508 == TRUE ) {
	 tmp = cb.data[32];
      } else {
	 tmp = cb.data[29];
      } //else
      if( tmp & 0x01 ) {
	 printf("Allow the use of PAP for inbound/outbound: Yes\n");
	 if( tmp & 0x80 ) {
	    printf("Using inbound authentication.\n");
	 } else {
	    printf("Using outbound authentication.\n");
	 } //if
      } else {
	 printf("Allow the use of PAP for inbound/outbound: No\n");
      } //if
      if( tmp & 0x02 ) {
	 printf("Allow the use of CHAP for inbound/outbound: Yes\n");
	 if( tmp & 0x80 ) {
	    printf("Using inbound authentication.\n");
	 } else {
	    printf("Using outbound authentication.\n");
	 } //if
      } else {
	 printf("Allow the use of CHAP for inbound/outbound: No\n");
      } //if
   } else {
      error();
   } //if
}; //authent

void ip_config( void ) {
   int i;
   
   cb.command = READ_CONFIGURATION;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      if( is_508 == TRUE ) {
	 i = 33;
      } else {
	 i = 30;
      } //if
      printf("Enable the use of IP: ");
      ( cb.data[i] & 0x80 ) ? printf("Yes\n") : printf("No\n");
      printf("Notify remote of locally-configure address: ");
      ( cb.data[i] & 0x01 ) ? printf("Yes\n") : printf("No\n");
      printf("Local IP address( 0.0.0.0 = request ): %d.%d.%d.%d\n",cb.data[i+1],cb.data[i+2],cb.data[i+3],cb.data[i+4]);
      printf("Request remote to provide local address: ");
      ( cb.data[i] & 0x02 ) ? printf("Yes\n") : printf("No\n");
      printf("Provide remote with pre-configured address: ");
      ( cb.data[i] & 0x04 ) ? printf("Yes\n") : printf("No\n");
      printf("Remote IP address: %d.%d.%d.%d.\n",cb.data[i+5],cb.data[i+6],cb.data[i+7],cb.data[i+8]);
      printf("Require that remote provide an address: ");
      ( cb.data[i] & 0x08 ) ? printf("Yes\n") : printf("No\n");
   } else {
      error();
   } //if
}; //ip_config

void ipx_config( void ) {
   int i;
   
   cb.command = READ_CONFIGURATION;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      if( is_508 == TRUE ) {
	 i = 42;
      } else {
	 i = 39;
      } //if
      printf("Enable the use of IPX: ");
      ( cb.data[i] & 0x80 ) ? printf("Yes\n") : printf("No\n");
      printf("Include network number in Config-Request: ");
      ( cb.data[i] & 0x01 ) ? printf("Yes\n") : printf("No\n");
      printf("Network number( 00000000 = request ): %02X %02X %02X %02X\n",cb.data[i+1],cb.data[i+2],cb.data[i+3],cb.data[i+4]);
      printf("Include local node # in Config-Request: ");
      ( cb.data[i] & 0x02 ) ? printf("Yes\n") : printf("No\n");
      printf("Local node number( 000000000000 = request ): %02X %02X %02X %02X %02X %02X\n",cb.data[i+5],cb.data[i+6],cb.data[i+7],cb.data[i+8],cb.data[i+9],cb.data[i+10]);
      printf("Force remote to accept remote node number: ");
      ( cb.data[i] & 0x04 ) ? printf("Yes\n") : printf("No\n");
      printf("Remote node number: %02X %02X %02X %02X %02X %02X\n",cb.data[i+11],cb.data[i+12],cb.data[i+13],cb.data[i+14],cb.data[i+15],cb.data[i+16]);
      printf("Include config-complete in Config-Request: ");
      ( cb.data[i] & 0x40 ) ? printf("Yes\n") : printf("No\n");
      if( cb.data[i] & 0x20 ) {
	 printf("Routing protocol: Request default( normally RIP/SAP )\n");
      } else if( cb.data[i] & 0x18 ){
	 printf("Routing protocol: Use either RIP/SAP or NLSP\n");
      } else if( cb.data[i] & 0x08 ){
	 printf("Routing protocol: Use RIP/SAP only\n");
      } else if( cb.data[i] & 0x10 ){
	 printf("Routing protocol: Use NLSP only\n");
      } else {
	 printf("Routing protocol: No routing protocol\n");
      } //if
      printf("Local router name( max. 47 characters ): %s\n",&cb.data[i+17]);
   } else {
      error();
   } //if
}; //ipx_config

void set_state( unsigned char tmp ) {
   switch( tmp ) {
   case 0:
      printf("Initial\n");
      break;
   case 1:
      printf("Starting\n");
      break;
   case 2:
      printf("Closed\n");
      break;
   case 3:
      printf("Stopped\n");
      break;
   case 4:
      printf("Closing\n");
      break;
   case 5:
      printf("Stopping\n");
      break;
   case 6:
      printf("Request Sent\n");
      break;
   case 7:
      printf("Ack Received\n");
      break;
   case 8:
      printf("Ack Sent\n");
      break;
   case 9:
      printf("Opened\n");
      break;
   default:
      printf("Unknown\n");
   } //switch
}; //set_state

void state( void ) {
   cb.command = PPIPE_GET_IBA_DATA;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      switch( cb.data[2] ) {
      case 0:
	 printf("PPP Phase: Link Dead\n");
	 break;
      case 1:
	 printf("PPP Phase: Establishment( LCP )\n");
	 break;
      case 2:
	 printf("PPP Phase: Authentication\n");
	 break;
      case 3:
	 printf("PPP Phase: Network Layer\n");
	 break;
      case 4:
	 printf("PPP Phase: Link Termination\n");
	 break;
      default:
	 printf("PPP Phase: Unknown\n");
      } //switch
      printf("LCP State: ");
      set_state( cb.data[1] );
      printf("IPCP State: ");
      set_state( cb.data[3] );
      printf("IPXCP State: ");
      set_state( cb.data[4] );
      switch( cb.data[5] ) {
      case 0:
	 printf("PAP State: Initial( Inactive )\n");
	 break;
      case 1:
	 printf("PAP State: Failed\n");
	 break;
      case 2:
	 printf("PAP State: Request Sent\n");
	 break;
      case 3:
	 printf("PAP State: Waiting\n");
	 break;
      case 4:
	 printf("PAP State: Opened( Success )\n");
	 break;
      default:
	 printf("PAP State: Unknown\n");
      } //switch
      switch( cb.data[5] ) {
      case 0:
	 printf("CHAP State: Initial( Inactive )\n");
	 break;
      case 1:
	 printf("CHAP State: Failed\n");
	 break;
      case 2:
	 printf("CHAP State: Challenge Sent\n");
	 break;
      case 3:
	 printf("CHAP State: Waiting for Challenge\n");
	 break;
      case 4:
	 printf("CHAP State: Response Sent\n");
	 break;
      case 5:
	 printf("CHAP State: Opened( Success )\n");
	 break;
      default:
	 printf("CHAP State: Unknown\n");
	 break;
      } //switch
   } else {
      error();
   } //if
}; //state

void negot( void ) {
   unsigned short mru;
   
   cb.command = READ_CONNECTION_INFORMATION;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&mru,&cb.data[0],2);
      printf("Remote Maximum Receive Unit: %d\n",mru);
      printf("Negotiated IP options: %02X",cb.data[2]);
      ( cb.data[2] & 0x80 ) ? printf("( IP Enabled )\n") : printf("( IP Disabled )\n");
      printf("Local IP address: %d.%d.%d.%d\n",cb.data[3],cb.data[4],cb.data[5],cb.data[6]);
      printf("Remote IP address: %d.%d.%d.%d\n",cb.data[7],cb.data[8],cb.data[9],cb.data[10]);
      printf("Negotiated IPX options: %02X",cb.data[11]);
      ( cb.data[11] & 0x80 ) ? printf("( IPX Enabled )\n") : printf("( IPX Disabled )\n");
      printf("IPX network number: %02X %02X %02X %02X\n",cb.data[12],cb.data[13],cb.data[14],cb.data[15]);
      printf("Local IPX node number: %02X %02X %02X %02X %02X %02X\n",cb.data[16],cb.data[17],cb.data[18],cb.data[19],cb.data[20],cb.data[21]);
      printf("Remote IPX node number: %02X %02X %02X %02X %02X %02X\n",cb.data[22],cb.data[23],cb.data[24],cb.data[25],cb.data[26],cb.data[27]);
      printf("Remote IPX router name: %s\n",&cb.data[28]);
      switch( cb.data[76] ) {
      case 0:
	 printf("Authentication status: No inbound authentication negotiated\n");
	 break;
      case 1:
	 printf("Authentication status: PeerID valid, Password Incorrect\n");
	 break;
      case 2:
	 printf("Authentication status: PeerID was invalid\n");
	 break;
      case 3:
	 printf("Authentication status: PeerID and Password were correct\n");
	 break;
      default:
	 printf("Authentication status: Unknown\n");
      } //switch
      printf("Inbound PeerID: %s\n",&cb.data[77]);
   } else {
      error();
   } //if
}; //negot

void cause( void ) {
   unsigned short disc;
   
   cb.command = PPIPE_GET_IBA_DATA;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&disc,&cb.data[7],2);
      printf("Local request by termination phase: ");
      (disc & 0x0100) ? printf("Yes\n") : printf("No\n");
      printf("DCD and/or CTS dropped: ");
      (disc & 0x0200) ? printf("Yes\n") : printf("No\n");
      printf("Disabled communications locally: ");
      (disc & 0x0400) ? printf("Yes\n") : printf("No\n");
      printf("Inbound/Outbound authentication failed: ");
      (disc & 0x0800) ? printf("Yes\n") : printf("No\n");
      printf("Failed to negotiate inbound auth. protocol with peer: ");
      (disc & 0x1000) ? printf("Yes\n") : printf("No\n");
      printf("Rejected peer's request for authentication: ");
      (disc & 0x2000) ? printf("Yes\n") : printf("No\n");
      printf("Peer rejected MRU option of config-request: ");
      (disc & 0x4000) ? printf("Yes\n") : printf("No\n");
      printf("MRU of peer was below required minumum: ");
      (disc & 0x8000) ? printf("Yes\n") : printf("No\n");
      printf("Rejected peer's LCP option(s) too many times: ");
      (disc & 0x0001) ? printf("Yes\n") : printf("No\n");
      printf("Rejected peer's IPCP option(s) too many times: ");
      (disc & 0x0002) ? printf("Yes\n") : printf("No\n");
      printf("Rejected peer's IPXCP option(s) too many times: ");
      (disc & 0x0004) ? printf("Yes\n") : printf("No\n");
   } else {
      error();
   } //if
}; //cause

void modem( void ) {
   unsigned char cts_dcd;
   
   cb.command = PPIPE_GET_IBA_DATA;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&cts_dcd,&cb.data[0],1);
      printf("DCD: ");
      (cts_dcd & 0x08) ? printf("High\n") : printf("Low\n");
      printf("CTS: ");
      (cts_dcd & 0x20) ? printf("High\n") : printf("Low\n");
   } else {
      error();
   } //if
}; //modem

void general_stats( void ) {
   unsigned short tmp;
   unsigned long l_tmp;
   
   cb.command = READ_OPERATIONAL_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      if( is_508 == TRUE ) {
	 memcpy(&tmp,&cb.data[2],2);
	 printf("Number of received frames discarded due to bad length: %d\n",tmp);
      } else {
	 memcpy(&tmp,&cb.data[0],2);
	 printf("Number of received frames discarded at the interrupt level: %d\n",tmp);
	 memcpy(&tmp,&cb.data[2],2);
	 printf("Number of received frames discarded at the application level: %d\n",tmp);
	 memcpy(&tmp,&cb.data[4],2);
	 printf("Number of received retransmitted due to aborts: %d\n",tmp);
      } //if
      memcpy(&l_tmp,&cb.data[6],4);
      printf("Number of user frames transmitted: %lu\n",l_tmp);
      memcpy(&l_tmp,&cb.data[10],4);
      printf("Number of user bytes transmitted: %lu\n",l_tmp);
      memcpy(&l_tmp,&cb.data[14],4);
      printf("Number of user frames received: %lu\n",l_tmp);
      memcpy(&l_tmp,&cb.data[18],4);
      printf("Number of user bytes received: %lu\n",l_tmp);
   } else {
      error();
   } //if
}; //general_stats

void flush_general_stats( void ) {
   cb.command = FLUSH_OPERATIONAL_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_general_stats

void comm_err( void ) {
   cb.command = READ_COMMS_ERR_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      if( is_508 == TRUE ) {
	 printf("Number of times receiver was halted due to full buffers: %d\n",cb.data[3]);
      } else {
	 printf("Number of frames discarded at the interrupt level due to frame being too long: %d\n",cb.data[3]);
	 printf("Number of transmit underruns: %d\n",cb.data[5]);
      } //if
      printf("Number of receiver overrun errors: %d\n",cb.data[0]);
      printf("Number of receiver CRC errors: %d\n",cb.data[1]);
      printf("Number of abort frames received: %d\n",cb.data[2]);
      printf("Number of abort frames transmitted: %d\n",cb.data[4]);
      printf("Number of missed transmit underrun interrupts: %d\n",cb.data[6]);
      printf("Number of times DCD changed state: %d\n",cb.data[8]);
      printf("Number of times CTS changed state: %d\n",cb.data[9]);
   } else {
      error();
   } //if
}; //comm_err

void flush_comm_err( void ) {
   cb.command = FLUSH_COMMS_ERR_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_general_stats

void packet( void ) {
   unsigned short tmp, tmp2;
   
   cb.command = READ_PACKET_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&tmp,&cb.data[0],2);
      printf("Number discards( bad header ): %d\n",tmp);
      memcpy(&tmp,&cb.data[2],2);
      printf("Number discards( unknown/unsupported protocol ): %d\n",tmp);
      memcpy(&tmp,&cb.data[4],2);
      printf("Number discards(unknown/unsupported protocol+too large for Protocol-Reject): %d\n",tmp);
      printf("\n\t\t\tReceived\tTransmitted\n");
      memcpy(&tmp,&cb.data[6],2);
      memcpy(&tmp2,&cb.data[8],2);
      printf("Number of LCP packets: %d\t\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[10],2);
      memcpy(&tmp2,&cb.data[12],2);
      printf("Number of IPCP packets: %d\t\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[14],2);
      memcpy(&tmp2,&cb.data[16],2);
      printf("Number of IPXCP packets: %d\t\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[18],2);
      memcpy(&tmp2,&cb.data[20],2);
      printf("Number of PAP packets: %d\t\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[22],2);
      memcpy(&tmp2,&cb.data[24],2);
      printf("Number of CHAP packets: %d\t\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[26],2);
      memcpy(&tmp2,&cb.data[28],2);
      printf("Number of LQR packets: %d\t\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[30],2);
      memcpy(&tmp2,&cb.data[32],2);
      printf("Number of IP packets:  %d\t\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[34],2);
      memcpy(&tmp2,&cb.data[36],2);
      printf("Number of IPX packets: %d\t\t%d\n",tmp,tmp2);
   } else {
      error();
   } //if
}; //packet

void flush_packet( void ) {
   cb.command = FLUSH_PACKET_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_packet

void lcp( void ) {
   unsigned short tmp, tmp2;
   
   cb.command = READ_LCP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&tmp,&cb.data[0],2);
      printf("Packets discarded (unknown LCP code): %d\n",tmp);
      memcpy(&tmp,&cb.data[48],2);
      printf("Received LCP packets too large: %d\n",tmp);
      memcpy(&tmp,&cb.data[50],2);
      printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
      memcpy(&tmp,&cb.data[52],2);
      printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
      memcpy(&tmp,&cb.data[54],2);
      printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
      printf("\n\t\t\tReceived\tTransmitted\n");
      memcpy(&tmp,&cb.data[2],2);
      memcpy(&tmp2,&cb.data[26],2);
      printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[4],2);
      memcpy(&tmp2,&cb.data[28],2);
      printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[6],2);
      memcpy(&tmp2,&cb.data[30],2);
      printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[8],2);
      memcpy(&tmp2,&cb.data[32],2);
      printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[10],2);
      memcpy(&tmp2,&cb.data[34],2);
      printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[12],2);
      memcpy(&tmp2,&cb.data[36],2);
      printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[14],2);
      memcpy(&tmp2,&cb.data[38],2);
      printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[16],2);
      memcpy(&tmp2,&cb.data[40],2);
      printf("Number of Protocol-Rej packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[18],2);
      memcpy(&tmp2,&cb.data[42],2);
      printf("Number of Echo-Request packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[20],2);
      memcpy(&tmp2,&cb.data[44],2);
      printf("Number of Echo-Reply packets : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[22],2);
      memcpy(&tmp2,&cb.data[46],2);
      printf("Number of Discard-Request packets: %d\t%d\n",tmp,tmp2);
   } else {
      error();
   } //if
}; //lcp

void flush_lcp( void ) {
   cb.command = FLUSH_LCP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_packet

void loopback( void ) {
   unsigned short tmp;
   
   cb.command = READ_LCP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&tmp,&cb.data[0],2);
      printf("Looped-back link possible given Config-Req/Nak/Rej: %d\n",tmp);
      memcpy(&tmp,&cb.data[2],2);
      printf("Looped-back link detected with Echo-Request: %d\n",tmp);
      memcpy(&tmp,&cb.data[4],2);
      printf("Echo-Request received from bad source: %d\n",tmp);
      memcpy(&tmp,&cb.data[6],2);
      printf("Looped-back link detected with Echo-Reply: %d\n",tmp);
      memcpy(&tmp,&cb.data[8],2);
      printf("Echo-Reply received from bad source: %d\n",tmp);
      memcpy(&tmp,&cb.data[10],2);
      printf("Looped-back link detected with Discard-Request: %d\n",tmp);
      memcpy(&tmp,&cb.data[12],2);
      printf("Discard-Request received from bad source: %d\n",tmp);
      memcpy(&tmp,&cb.data[14],2);
      printf("Echo-Reply discarded( transmit collision ): %d\n",tmp);
      memcpy(&tmp,&cb.data[16],2);
      printf("Echo-Reply discarded( receive collision ): %d\n",tmp);
   } else {
      error();
   } //if
}; //loopback

void flush_loopback( void ) {
   cb.command = FLUSH_LOOPBACK_ERR_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_loopback

void ipcp( void ) {
   unsigned short tmp, tmp2;
   
   cb.command = READ_IPCP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&tmp,&cb.data[0],2);
      printf("Packets discarded (unknown IPCP code): %d\n",tmp);
      memcpy(&tmp,&cb.data[32],2);
      printf("Received IPCP packets too large: %d\n",tmp);
      memcpy(&tmp,&cb.data[34],2);
      printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
      memcpy(&tmp,&cb.data[36],2);
      printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
      memcpy(&tmp,&cb.data[38],2);
      printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
      printf("\n\t\t\tReceived\tTransmitted\n");
      memcpy(&tmp,&cb.data[2],2);
      memcpy(&tmp2,&cb.data[18],2);
      printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[4],2);
      memcpy(&tmp2,&cb.data[20],2);
      printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[6],2);
      memcpy(&tmp2,&cb.data[22],2);
      printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[8],2);
      memcpy(&tmp2,&cb.data[24],2);
      printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[10],2);
      memcpy(&tmp2,&cb.data[26],2);
      printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[12],2);
      memcpy(&tmp2,&cb.data[28],2);
      printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[14],2);
      memcpy(&tmp2,&cb.data[30],2);
      printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
   } else {
      error();
   } //if
}; //ipcp

void flush_ipcp( void ) {
   cb.command = FLUSH_IPCP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_ipcp

void ipxcp( void ) {
   unsigned short tmp, tmp2;
   
   cb.command = READ_IPXCP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&tmp,&cb.data[0],2);
      printf("Packets discarded (unknown IPXCP code): %d\n",tmp);
      memcpy(&tmp,&cb.data[32],2);
      printf("Received IPXCP packets too large: %d\n",tmp);
      memcpy(&tmp,&cb.data[34],2);
      printf("Received packets invalid or out-of-sequence Configure-Acks: %d\n",tmp);
      memcpy(&tmp,&cb.data[36],2);
      printf("Received packets invalid Configure-Naks or Configure-Rejects: %d\n",tmp);
      memcpy(&tmp,&cb.data[38],2);
      printf("Configure-Naks or Configure-Rejects with bad Identifier: %d\n",tmp);
      printf("\n\t\t\tReceived\tTransmitted\n");
      memcpy(&tmp,&cb.data[2],2);
      memcpy(&tmp2,&cb.data[18],2);
      printf("Number of Config-Request pkts: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[4],2);
      memcpy(&tmp2,&cb.data[20],2);
      printf("Number of Config-Ack packets : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[6],2);
      memcpy(&tmp2,&cb.data[22],2);
      printf("Number of Config-Nack packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[8],2);
      memcpy(&tmp2,&cb.data[24],2);
      printf("Number of Config-Reject packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[10],2);
      memcpy(&tmp2,&cb.data[26],2);
      printf("Number of Term-Reqst packets : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[12],2);
      memcpy(&tmp2,&cb.data[28],2);
      printf("Number of Terminate-Ack packets: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[14],2);
      memcpy(&tmp2,&cb.data[30],2);
      printf("Number of Code-Reject packets: %d\t%d\n",tmp,tmp2);
   } else {
      error();
   } //if
}; //ipxcp

void flush_ipxcp( void ) {
   cb.command = FLUSH_IPXCP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_ipxcp

void pap( void ) {
   unsigned short tmp, tmp2;
   
   cb.command = READ_PAP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&tmp,&cb.data[0],2);
      printf("Packets discarded (unknown PAP code): %d\n",tmp);
      memcpy(&tmp,&cb.data[16],2);
      printf("Received PAP packets too large: %d\n",tmp);
      memcpy(&tmp,&cb.data[18],2);
      printf("Received packets invalid inbound PeerID: %d\n",tmp);
      memcpy(&tmp,&cb.data[20],2);
      printf("Received packets invalid inbound Password: %d\n",tmp);
      printf("\n\t\t\tReceived\tTransmitted\n");
      memcpy(&tmp,&cb.data[2],2);
      memcpy(&tmp2,&cb.data[10],2);
      printf("Number of Authent-Request pkts: %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[4],2);
      memcpy(&tmp2,&cb.data[12],2);
      printf("Number of Authent-Ack packets : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[6],2);
      memcpy(&tmp2,&cb.data[14],2);
      printf("Number of Authent-Nack packets: %d\t%d\n",tmp,tmp2);
   } else {
      error();
   } //if
}; //pap

void flush_pap( void ) {
   cb.command = FLUSH_PAP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_pap

void chap( void ) {
   unsigned short tmp, tmp2;
   
   cb.command = READ_CHAP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) {
      memcpy(&tmp,&cb.data[0],2);
      printf("Packets discarded (unknown CHAP code): %d\n",tmp);
      memcpy(&tmp,&cb.data[20],2);
      printf("Received CHAP packets too large: %d\n",tmp);
      memcpy(&tmp,&cb.data[22],2);
      printf("Received packets invalid inbound PeerID: %d\n",tmp);
      memcpy(&tmp,&cb.data[24],2);
      printf("Received packets invalid inbound Password/Secret: %d\n",tmp);
      memcpy(&tmp,&cb.data[26],2);
      printf("Received packets invalid inbound MD5 message digest format: %d\n",tmp);
      memcpy(&tmp,&cb.data[28],2);
      printf("Invalid inbound ID or out-of-order or unelicited responses: %d\n",tmp);
      printf("\n\t\t\tReceived\tTransmitted\n");
      memcpy(&tmp,&cb.data[2],2);
      memcpy(&tmp2,&cb.data[12],2);
      printf("Number of Challenge packets  : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[4],2);
      memcpy(&tmp2,&cb.data[14],2);
      printf("Number of Response packets   : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[6],2);
      memcpy(&tmp2,&cb.data[16],2);
      printf("Number of Success packets    : %d\t%d\n",tmp,tmp2);
      memcpy(&tmp,&cb.data[8],2);
      memcpy(&tmp2,&cb.data[18],2);
      printf("Number of Failure packets    : %d\t%d\n",tmp,tmp2);
   } else {
      error();
   } //if
}; //chap

void flush_chap( void ) {
   cb.command = FLUSH_CHAP_STATISTICS;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code != 0 ) {
       error();
   } //if
}; //flush_chap

void usage( void ) {
   printf("ppipemon [f](lush)/ [t](1 features) option ip-address [port]\n\n");
   printf("One of the following:\n");
   printf("\tg  PPP General Configuration\n");
   printf("\tu  PPP Timers and Counters\n");
   printf("\ta  Authentication Configuration\n");
   printf("\ti  IP Configuration\n");
   printf("\tx  IPX Configuration\n");
   printf("\ts  PPP FSM Current State\n");
   printf("\tn  Parameters Negotiated on Last Connection or Attempt\n");
   printf("\tc  Cause for Last Disconnection\n");
   printf("\tm  Modem Status\n");
   printf("\tq  Line Trace: display Interpreted data\n");
   printf("\tqr Line Trace: display RAW data\n");
   printf("\tds Display If_Send Driver Statistics\n");
   printf("\tdi Display Interrrupt Driver Statistics\n");
   printf("\tdg Display General Driver Statistics\n"); 
   printf("\tj  Display Router UP time\n");
   printf("\nOptions that can be flushed:\n");
   printf("\tl General Statistics\n");
   printf("\te Communication Error Statistics\n");
   printf("\tp Packet Statistics\n");
   printf("\tr LCP Statistics\n");
   printf("\tb Loopback Detection / LCP Error Statistics\n");
   printf("\to IP Control Protocol( IPCP )Statistics\n");
   printf("\ty IPX Control Protocol( IPXCP )Statistics\n");
   printf("\tw Password Authentication Protocol( PAP )Statistics\n");
   printf("\th Challenge-Handshake Auth. Protocol( CHAP )Statistics\n");
   printf("\td Flush all Driver Statistics\n");
   printf("\nOptions for S508/FT1 only\n");
   printf("\tv View Status\n");
   printf("\ts Self Test\n");
   printf("\tl Line Loop Test\n");
   printf("\td Digital Loop Test\n");
   printf("\tr Remote Test\n");
   printf("\to Operational Mode\n");
}; //usage

void line_trace() {
   unsigned char num_frames, num_chars;
   unsigned short curr_pos = 0;
   PFRAME_DATA frame_info;
   unsigned int i, j;
   unsigned char outstr[2000];
   int recv_buff = sizeof(CBLOCK) + 100;
   fd_set ready;
   struct timeval to;
   unsigned short tmp;
   
   setsockopt( sock, SOL_SOCKET, SO_RCVBUF, &recv_buff, sizeof(int) );
   cb.command = PPIPE_ENABLE_TRACING;
   cb.buffer_length = 0;
   DoCommand(&cb);
   if( cb.return_code == 0 ) { 
      printf("Starting trace...(Press ENTER to exit)\n");
   } else if( cb.return_code == 0xCD ) {
      printf("Cannot Enable Line Tracing from Underneath.\n");
      return;
   } else if( cb.return_code == 0x01 ) {
      printf("Starting trace...(although it's already enabled!)\n");
      printf("Press ENTER to exit.\n");
   } else {
      printf("Failed to Enable Line Tracing. Return code: 0x%02X\n", cb.return_code );
      return;
   }
   for(;;) {
      FD_ZERO(&ready);
      FD_SET(0,&ready);
      to.tv_sec = 1;
      to.tv_usec = 0;
      if(select(1,&ready, NULL, NULL, &to)) {
	 break;
      } /* if */
      cb.command = PPIPE_GET_TRACE_INFO;
      cb.buffer_length = 0;
      DoCommand(&cb);
      if (cb.return_code == 0 && cb.buffer_length) { // if we got something back
	 // get number of frames
	 num_frames = cb.data[0] >> 2;

	 for ( i=0; i<num_frames; i++) {
	    frame_info = (PFRAME_DATA)(cb.data + curr_pos);
	    // frame type
	    if (frame_info->status & 0x01) {
	       sprintf(outstr,"OUTGOING\t\0");
	    } else {
	       sprintf(outstr,"INCOMING\t\0");
	    }
	    //real length and time stamp
	    sprintf(outstr+strlen(outstr), "%d\t%d\t\0", frame_info->real_length, frame_info->time_stamp);
	    // first update curr_pos
	    curr_pos += sizeof(FRAME_DATA);
	    // the actual data
//	    for( j=0; j<cb.buffer_length; j++ ) {
//	       sprintf(outstr+strlen(outstr), "%02X \0", (unsigned char)frame_info->data[j]);
//	    }
	    if (frame_info->passed_length == 0) {
	       sprintf( outstr+strlen(outstr), "the frame data is not available" );
	    } else {
	       // update curr_pos again
	       curr_pos += (frame_info->real_length-1); 
	       num_chars = (unsigned char)((frame_info->real_length <= 25)?frame_info->real_length:25);
	       if (raw_data) { // show raw data
		  for( j=0; j<num_chars; j++ ) {
		     sprintf(outstr+strlen(outstr), "%02X \0", (unsigned char)frame_info->data[j]);
		  }
		  outstr[strlen(outstr)-1] = '\0';
	       } else { // show int data
		  memcpy(&tmp,&frame_info->data[2],2);
		  switch(tmp) {
		  case 0x2100: //IP Packet
		     sprintf(outstr+strlen(outstr),"IP data packet from %d.%d.%d.%d to %d.%d.%d.%d",frame_info->data[16],frame_info->data[17],frame_info->data[18],frame_info->data[19],frame_info->data[20],frame_info->data[21],frame_info->data[22],frame_info->data[23]);
		     break;
		  case 0x2B00: //IPX Packet
		     sprintf(outstr+strlen(outstr),"IPX Data packet");
		     break;
		  case 0x2180: //IPCP Packet
		  case 0x2B80: //IPXCP Packet
		  case 0x21C0: //LCP Packet
		     switch(tmp) {
		     case 0x2180:
			sprintf(outstr+strlen(outstr),"IPCP packet - ");
			break;
		     case 0x2B80:
			sprintf(outstr+strlen(outstr),"IPXCP packet - ");
			break;
		     case 0x21C0:
			sprintf(outstr+strlen(outstr),"LCP packet - ");
			break;
		     } //switch
		     switch(frame_info->data[4]){
		     case 1:
			sprintf(outstr+strlen(outstr),"Configure REQUEST");
			break;
		     case 2:
			sprintf(outstr+strlen(outstr),"Configure ACK");
			break;
		     case 3:
			sprintf(outstr+strlen(outstr),"Configure NACK");
			break;
		     case 4:
			sprintf(outstr+strlen(outstr),"Configure REJECT");
			break;
		     case 5:
			sprintf(outstr+strlen(outstr),"Terminate REQUEST");
			break;
		     case 6:
			sprintf(outstr+strlen(outstr),"Terminate ACK");
			break;
		     case 7:
			sprintf(outstr+strlen(outstr),"Code REJECT");
			break;
		     case 8:
			sprintf(outstr+strlen(outstr),"Protocol REJECT");
			break;
		     case 9:
			sprintf(outstr+strlen(outstr),"Echo REQUEST");
			break;
		     case 10:
			sprintf(outstr+strlen(outstr),"Echo REPLY");
			break;
		     case 11:
			sprintf(outstr+strlen(outstr),"Discard REQUEST");
			break;
		     default:
			sprintf(outstr+strlen(outstr),"Unknown type");
		     } //switch
		     break;
		  case 0x23C0: //PAP
		     sprintf(outstr+strlen(outstr),"PAP packet - ");
		     switch(frame_info->data[4]){
		     case 1:
			sprintf(outstr+strlen(outstr),"Authenticate REQUEST");
			break;
		     case 2:
			sprintf(outstr+strlen(outstr),"Authenticate ACK");
			break;
		     case 3:
			sprintf(outstr+strlen(outstr),"Authenticate NACK");
			break;
		     default:
			sprintf(outstr+strlen(outstr),"Unknown type");
		     } //switch
		     break;
		  case 0x25C0: //LQR
		     sprintf(outstr+strlen(outstr),"Link Quality Report");
		     break;
		  case 0x23C2: //PAP
		     sprintf(outstr+strlen(outstr),"CPAP packet - ");
		     switch(frame_info->data[4]){
		     case 1:
			sprintf(outstr+strlen(outstr),"Challenge");
			break;
		     case 2:
			sprintf(outstr+strlen(outstr),"Response");
			break;
		     case 3:
			sprintf(outstr+strlen(outstr),"Success");
			break;
		     case 4:
			sprintf(outstr+strlen(outstr),"Failure");
			break;
		     default:
			sprintf(outstr+strlen(outstr),"Unknown type");
		     } //switch
		     break;
		  default:
		     sprintf(outstr+strlen(outstr),"Unknown type");
		  } //switch
	       } //if
	    } //if

	    // we have to stay on even boundary here, so update the curr_pos
	    // if needed. It's done exactly the same way in driver. Si, it's cool
	    if (curr_pos & 0x0001) curr_pos++;
	    printf("%s\n",outstr);
	 } //for
      } //if
      curr_pos = 0;
      if (!(cb.data[0] &= 0x02)) { 
	 sleep(TIMEOUT);
      }
   }
   cb.command = PPIPE_DISABLE_TRACING;
   cb.buffer_length = 0;
   DoCommand(&cb);
}; //line_trace

void set_FT1_monitor_status( unsigned char status) {
      fail = 0;
      cb.command = FT1_MONITOR_STATUS_CTRL;
      cb.buffer_length = 1;
      cb.data[0] = status; 	
      DoCommand(&cb);
      if( cb.return_code != 0 ){
	fail = 1;
	printf("This command is only possible with S508/FT1 board!");
      }
} /* set_FT1_monitor_status */


void set_FT1_mode( void ){
     
     for(;;){ 
       cb.command = SET_FT1_MODE;
       cb.buffer_length = 0;
       DoCommand(&cb);
       if(cb.return_code == 0){
 	 break;
       }
     }
} /* set_FT1_mode */

void read_FT1_status( void ){
     int i;
     unsigned long delay;
     struct timeval tv;

     i = gettimeofday(&tv,NULL);
     delay = tv.tv_usec;

     for(;;){
      	i = gettimeofday(&tv,NULL);
      	if(abs((tv.tv_usec - delay)) > 90000 )
		break;  
     }

     cb.command = PPIPE_FT1_READ_STATUS;
     cb.buffer_length = 0;
     DoCommand(&cb); 
     if( cb.return_code == 0 ){
	par_port_A_byte = cb.data[0];
	par_port_B_byte = cb.data[1];
     }
} /* read_FT1_status */

void view_FT1_status( void ){
     
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     loop_counter = 0;
     /* check for INS light */
     for(;;){
     	read_FT1_status();
     	if((par_port_B_byte & 0x30) == 0x30)
 		off_counter++;
     	if((par_port_B_byte & 0x10) == 0x00)
		red_counter++;
     	if((par_port_B_byte & 0x20) == 0x00)
		green_counter++;
	if(red_counter != 0 && off_counter != 0 && loop_counter == 30 ) {
		printf("Unit is not IN SERVICE\n");
		break;
	}
	if(green_counter != 0 && off_counter == 0 && loop_counter == 30){
		printf("Unit is IN SERVICE\n");
		break;
	}
	if(green_counter != 0 && off_counter != 0 && loop_counter == 30){
		printf("INS is flasing green\n");
		break;
	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("INS is off\n");
		break;
	}
	loop_counter++;
     }	
     /* check for ERR light */
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     loop_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x0C) == 0x0C)
		off_counter++;
	if((par_port_B_byte & 0x08) == 0x00)
		red_counter++;
	if((par_port_B_byte & 0x04) == 0x00)
		green_counter++;
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("No Line ERROR being received or Valid Line\n");
		break;
	}
	if(red_counter != 0 && off_counter == 0 && green_counter == 0 && loop_counter == 30){
		printf("Line ERROR being received or Invalid Line\n");
		break;
	}

	loop_counter++;
     }

     /* check TXD light */
     loop_counter = 0;
     off_counter = 0;
     green_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x02) == 0x02)
		off_counter++;
	if((par_port_B_byte & 0x02) == 0x00)
		green_counter++;
	if(off_counter != 0 && green_counter == 0 && loop_counter == 20){
		printf("Transmit data is not present\n");	
		break;
	}
	if(off_counter != 0 && green_counter != 0 && loop_counter == 20){
		printf("Transmit data is present \n");
		break;
	}
	loop_counter++;
     }

     /* check RXD light */
     loop_counter = 0;
     off_counter = 0;
     green_counter = 0;
     for(;;){
	read_FT1_status();
	if((par_port_B_byte & 0x01) == 0x01)
		off_counter++;
	if((par_port_B_byte & 0x01) == 0x00)
		green_counter++;
	if(off_counter != 0 && green_counter == 0 && loop_counter == 20){
		printf("Receive data is not present\n");	
		break;
	}
	if(off_counter != 0 && green_counter != 0 && loop_counter == 20){
		printf("Receive data is present\n");
		break;
	}
	loop_counter++;
     }

}/* view_FT1_status */


void FT1_operational_mode(void){
    
     printf("Operational Mode has been selected\n");
     printf("Putting S508/FT1 in operational mode....");
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
     	read_FT1_status();
     	/* ST light is OFF */
     	if((par_port_B_byte & 0xc0) == 0xc0 ){
		off_counter++;
     	}
     	/* ST light is GREEN */
     	if((par_port_B_byte & 0x40) == 0x00){
		green_counter++;
		red_counter = 0;
     	}
     	/* ST light is RED */
     	if((par_port_B_byte & 0x80) == 0x00){
		red_counter++;
		green_counter = 0;
   	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 20){
		break;
	}
	if((red_counter != 0 || green_counter != 0) && loop_counter == 20){
		set_FT1_mode();
		break;
	}

   	loop_counter++;
     } /* end of for */
       
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
   	/* DL light is OFF */
   	if((par_port_A_byte & 0x08) == 0x08){
		off_counter++;
	}
        /* DL light is RED */
   	if((par_port_A_byte & 0x08) == 0x00){
		red_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
		break;
	}
	if(red_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		break;
	}
        loop_counter++; 
     } 
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
   	/* LL light is off */
   	if((par_port_A_byte & 0x04) == 0x04){
		off_counter++;
	}
   	if((par_port_A_byte & 0x04) == 0x00){
		red_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && loop_counter == 20){
		break;
	}
	if(red_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		break;
	}
        loop_counter++;
     }
     loop_counter = 0;
     off_counter = 0;
     red_counter = 0;
     green_counter = 0;
     for(;;){
   	read_FT1_status();
	/* RT light is OFF */
   	if((par_port_A_byte & 0x03) == 0x03){
		off_counter++;
	}
	/* RT light is RED */
   	if((par_port_A_byte & 0x01) == 0x00){
		red_counter++;
	}
	/* RT light is GREEN */
   	if((par_port_A_byte & 0x02) == 0x00){
   		green_counter++;
	}
	if(off_counter != 0 && red_counter == 0 && green_counter == 0 && loop_counter == 20){
		printf("Done!\n");
		break;
	}
	if((red_counter != 0 || green_counter != 0) && off_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		printf("Done!\n");
		break;
	}
	if(off_counter == 0 && green_counter != 0 && loop_counter == 20){
		printf("Failed!\n");
		printf("Remote End is running Remote Test\n");
		printf("Exit Remote Test at remote end\n");
		break;
	}
	loop_counter++;
     }        	

} /* FT1_operational_mode */


void FT1_self_test(void){
     int started = 0; 
     int selftest = 0;
 
     set_FT1_mode();
     off_counter = 0;
     green_counter = 0;
     red_counter = 0;
     loop_counter = 0;
     printf("Self Test has been selected\n");
     for(;;){
	  read_FT1_status();
	  /* ST light is OFF */
	  if((par_port_B_byte & 0xc0) == 0xc0){
		off_counter++;
	  }
	  /* ST light is GREEN */
	  if((par_port_B_byte & 0x40) == 0x00){
		green_counter++;
	  }
	  /* ST light is RED */
	  if((par_port_B_byte & 0x80) == 0x00){
		red_counter++;
	  }
          if(red_counter == 0 && loop_counter == 3){
		set_FT1_mode();    
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = -1;
     		printf("Selecting Self Test....\r");
		selftest++;
	  	if( selftest == 10){
		  printf("\t\t\tFailed!\n");
		  printf("Self Test will not operate while connected to a valid line\n");
		  FT1_operational_mode();
		  break;
	        }
	  }
	  if(red_counter != 0 && off_counter != 0 && (loop_counter % 2)){
		printf("Performing Self Test....\r");
	        started = 1;
	  }
	  if(green_counter != 0 && off_counter != 0 && started){
	   	printf("\t\t\tDone!\n");
		FT1_operational_mode();
		break;
	  }
          loop_counter++;	  
     }/* end of for */     
} /* FT1_self_test */


void FT1_dl_test( void ){
     int dl_test=0;

     set_FT1_mode();
     off_counter = 0;
     red_counter = 0;
     loop_counter = 0;
     printf("Bi-directional Digital Loop has been selected\n"); 
     for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x08) == 0x08){
		off_counter++;  
	}
	if((par_port_A_byte & 0x08) == 0x00){
		red_counter++;
	}
	if(red_counter == 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		loop_counter = -1;
		printf("Selecting Digital Loop Test....\r");
		dl_test++;
		if(dl_test==10){
	 		printf("\t\t\t\tFailed\n");
			printf("Remote End might be running Remote Test\n");
			break;
		}
	}
	if(red_counter != 0){
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = 0;
		printf("Performing Digital Loop Test....\r");
		for(;;){
			read_FT1_status();
			printf("Performing Digital Loop Test....\r");
			/* check INS light */
			if((par_port_B_byte & 0x30) == 0x30)
				off_counter++;
			if((par_port_B_byte & 0x10) == 0x00){
				red_counter++;
				green_counter = 0;
			}
			if((par_port_B_byte & 0x20) == 0x00){
				green_counter++;
				red_counter = 0;
			}
			if(red_counter != 0 && (par_port_B_byte & 0x08) == 0x00 ){
				printf("\t\t\t\tFailed!\n");
				printf("Bi-directional Digital Loop test has failed\n");
				printf("Either the unit is not connected or the line is Invalid\n");
				break;
			}
			if(green_counter != 0  && (par_port_B_byte & 0x0C) == 0x0C && loop_counter == 100 ){
				printf("\t\t\t\tDone!\n");
				printf("Bi-directional Digital Loop test has been successfully\n");
				printf("completed\n");
				break;
		
			}
			loop_counter++;
		} /* end of for */
		break;	
	} 
	loop_counter++;
     } /* end of for */

} /* FT1_dl_test */

void FT1_ll_test( void ){
    int ll_test = 0;

    set_FT1_mode();	
    off_counter = 0;
    red_counter = 0;
    loop_counter = 0;
    printf("Line Loop Test has been selected\n");
    for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x04) == 0x04){
		off_counter++;
	}
	if((par_port_A_byte & 0x04) == 0x00){
		red_counter++;
	}
	if(red_counter == 0 && off_counter != 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		loop_counter = -1;
		printf("Selecting Line Loop Test....\r");
		ll_test++;
		if(ll_test == 10){
			printf("\t\t\t\tFailed!\n");
			printf("Line Loop Test will not operate while connected to a valid line\n");
			FT1_operational_mode();
		 	break;	
		}
	}
	if(red_counter != 0){	
		off_counter = 0;
		red_counter = 0;
		loop_counter = 0;
		for(;;){
			printf("Performing Line Loop Test....\r");
			read_FT1_status();	
			/* check INS light */
			if((par_port_B_byte & 0x30) == 0x30)
				off_counter++;
			if((par_port_B_byte & 0x10) == 0x00){
				red_counter++;
				green_counter = 0;
			}
			if((par_port_B_byte & 0x20) == 0x00){
				green_counter++;
				red_counter = 0;
			}
			if(green_counter != 0 && red_counter == 0){
				printf("\t\t\t\tDone!\n");
				printf("Line Loop Test has been successfully completed\n");
				break;
			}
			if(red_counter != 0 && green_counter == 0 && loop_counter == 100){
				printf("\t\t\t\tFailed!\n");
				break;
			}
			loop_counter++;
		} /* end of for */
		break;
	} /* end of if */
	loop_counter++;
    } /* end of for */

} /* FT1_ll_test */

void FT1_rt_test( void ){
 
    off_counter = 0;
    red_counter = 0;
    green_counter = 0;	  	
    loop_counter = 0;
    printf("Remote Test has been selected\n");
    for(;;){
	read_FT1_status();
	if((par_port_A_byte & 0x03) == 0x03)
		off_counter++;
	if((par_port_A_byte & 0x01) == 0x00)
		red_counter++;
	if((par_port_A_byte & 0x02) == 0x00)
		green_counter++;
	if(red_counter == 0 && loop_counter == 20){
		set_FT1_mode();
		off_counter = 0;
		red_counter = 0;
		green_counter = 0;
		loop_counter = -1;
		printf("Selecting Remote Test....\r");		
	}
	if(green_counter != 0 && loop_counter == 10){
	   loop_counter = 0;
	   off_counter = 0;
	   for(;;) {
		read_FT1_status();
		if((par_port_A_byte & 0x03) == 0x03)
			off_counter++;
	        if(off_counter == 0 && loop_counter == 20){
		   printf("Remote End is currently sending Remote Test data\n");
		   printf("Exit from the Remote Test mode at the remote end\n");
		   break;
		}
		if(off_counter != 0 && loop_counter == 20) {			
		   printf("This unit is currently in Remote Testing Mode\n");
		   break;
		}
	        loop_counter++;
	   }
	   break;
	}
	if(red_counter != 0){
		printf("Waiting for a valid response from remote end....\r");
		loop_counter = 0;
		for(;;){
			read_FT1_status();
			if((par_port_A_byte & 0x02) == 0x00){
				printf("\t\t\t\t\t\tDone!\n");
				printf("Valid response has been received from remote end\n");
				printf("Remote Test has been successfully completed\n");
				break;	
			}
			if((par_port_B_byte & 0x08) == 0x00){
				printf("\t\t\t\t\t\tFailed!\n");
				printf("The test can only be successful between two Sangoma\n");
				printf("S508/FT1 units configured the SAME WAY\n");

				break;
			}
		} /* end of for */
		break;
	}
    	loop_counter++;	
    } /* end of for */	
} /* FT1_rt_test */

void ppp_driver_stat_ifsend( void )
{
      cb.command = PPIPE_DRIVER_STAT_IFSEND;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
      

      printf("                                 Total number of If_Send entries:  %d\n", cb.data[0] + (cb.data[1]*256) + (cb.data[2]*6553) + (cb.data[3]*16777216) );
      printf("                       Number of If_Send entries with SKB = NULL:  %d\n", cb.data[4] + (cb.data[5]*256) + (cb.data[6]*6553) + (cb.data[7]*16777216) );
      printf("Number of If_Send entries with broadcast addressed pkt discarded:  %d\n", cb.data[8] + (cb.data[9]*256) + (cb.data[10]*6553) + (cb.data[11]*16777216) );
      printf("Number of If_Send entries with multicast addressed pkt discarded:  %d\n", cb.data[12] + (cb.data[13]*256) + (cb.data[14]*6553) + (cb.data[15]*16777216) );
      printf("             Number of If_Send entries with CRITICAL_RX_INTR set:  %d\n", cb.data[16] + (cb.data[17]*256) + (cb.data[18]*6553) + (cb.data[19]*16777216) );
      printf("   Number of If_Send entries with Critical set and pkt discarded:  %d\n", cb.data[20] + (cb.data[21]*256) + (cb.data[22]*6553) + (cb.data[23]*16777216) );
      printf("                  Number of If_Send entries with Device Busy set:  %d\n", cb.data[24] + (cb.data[25]*256) + (cb.data[26]*6553) + (cb.data[27]*16777216) );
      printf("              Number of If_Send entries with Device Busy Timeout:  %d\n", cb.data[28] + (cb.data[29]*256) + (cb.data[30]*6553) + (cb.data[31]*16777216) );
      printf("             Number of If_Send entries with DRIVER STATS Request:  %d\n", cb.data[32] + (cb.data[33]*256) + (cb.data[34]*6553) + (cb.data[35]*16777216) );
      printf("           Number of If_Send entries with PTPIPE MONITOR Request:  %d\n", cb.data[36] + (cb.data[37]*256) + (cb.data[38]*6553) + (cb.data[39]*16777216) );
      printf("                 Number of If_Send entries with WAN Disconnected:  %d\n", cb.data[40] + (cb.data[41]*256) + (cb.data[42]*6553) + (cb.data[43]*16777216) );
      printf("                   Number of If_Send entries with Protocol Error:  %d\n", cb.data[48] + (cb.data[49]*256) + (cb.data[50]*6553) + (cb.data[51]*16777216) );
      printf("              Number of If_Send entries with Adapter Send Failed:  %d\n", cb.data[44] + (cb.data[45]*256) + (cb.data[46]*6553) + (cb.data[47]*16777216) );
      printf("       Number of If_Send entries with Transmit Interrupt Enabled:  %d\n", cb.data[52] + (cb.data[53]*256) + (cb.data[54]*6553) + (cb.data[55]*16777216) );
      printf("              Number of If_Send entries with Adapter Send passed:  %d\n", cb.data[56] + (cb.data[57]*256) + (cb.data[58]*6553) + (cb.data[59]*16777216) );
      printf("               Number of times host irq left disabled in If_Send:  %d\n", cb.data[60] + (cb.data[61]*256) ); 

} /* ppp_driver_stat_ifsend */

void ppp_driver_stat_intr( void )
{
      cb.command = PPIPE_DRIVER_STAT_INTR;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
      

      printf("                                    Number of ISR entries:   %d\n", cb.data[0] + (cb.data[1]*256) + (cb.data[2]*6553) + (cb.data[3]*16777216) );
      
      printf("                  Number of ISR entries with Critical Set:   %d\n", cb.data[4] + (cb.data[5]*256) + (cb.data[6]*6553) + (cb.data[7]*16777216) );
      
      printf("                              Number of Receive Interrupt:   %d\n", cb.data[8] + (cb.data[9]*256) + (cb.data[10]*6553) + (cb.data[11]*16777216) );
      printf("                             Number of Transmit Interrupt:   %d\n", cb.data[12] + (cb.data[13]*256) + (cb.data[14]*6553) + (cb.data[15]*16777216) );
      printf("              Number of ISR entries for Interrupt Testing:   %d\n", cb.data[16] + (cb.data[17]*256) + (cb.data[18]*6553) + (cb.data[19]*16777216) );
      printf("                             Number of Spurious Interrupt:   %d\n", cb.data[20] + (cb.data[21]*256) + (cb.data[22]*6553) + (cb.data[23]*16777216) );
      printf("       Number of times Transmit Interrupts Enabled in ISR:   %d\n", cb.data[24] + (cb.data[25]*256) + (cb.data[26]*6553) + (cb.data[27]*16777216) );
      
      printf("         Number of Receive Interrupts with Corrupt Buffer:   %d\n", cb.data[28] + (cb.data[29]*256) + (cb.data[30]*6553) + (cb.data[31]*16777216) );
      
      printf("              Number of Receive Interrupts with No socket:   %d\n", cb.data[32] + (cb.data[33]*256) + (cb.data[34]*6553) + (cb.data[35]*16777216) );
      printf("     Number of Receive Interrupts for DRIVER STAT Request:   %d\n", cb.data[36] + (cb.data[37]*256) + (cb.data[38]*6553) + (cb.data[39]*16777216) );
      printf("  Number of Receive Interrupts for PTPIPE MONITOR Request:   %d\n", cb.data[40] + (cb.data[41]*256) + (cb.data[42]*6553) + (cb.data[43]*16777216) );
      printf(" Number of Receive Interrupts with Buffer Passed to Stack:   %d\n", cb.data[44] + (cb.data[45]*256) + (cb.data[46]*6553) + (cb.data[47]*16777216) );
      printf("     Number of Receive Interrupts with Device not started:   %d\n", cb.data[48] + (cb.data[49]*256) + (cb.data[50]*6553) + (cb.data[51]*16777216) );
      printf("    Number of Transmit Interrupts with Device not started:   %d\n", cb.data[52] + (cb.data[53]*256) + (cb.data[54]*6553) + (cb.data[55]*16777216) );
     
}


void ppp_driver_stat_gen( void )
{
      cb.command = PPIPE_DRIVER_STAT_GEN;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);
     
	 
      printf("           Number of PTPIPE Monitor call with Kmalloc error:  %d\n", cb.data[0] + (cb.data[1]*256) + (cb.data[2]*6553) + (cb.data[3]*16777216) );

      printf("      Number of PTPIPE Monitor call with Adapter Type error:  %d\n", cb.data[4] + (cb.data[5]*256) + (cb.data[6]*6553) + (cb.data[7]*16777216) );

      printf("         Number of PTPIPE Monitor call with Direction Error:  %d\n", cb.data[8] + (cb.data[9]*256) + (cb.data[10]*6553) + (cb.data[11]*16777216) );

      printf(" Number of PTPIPE Monitor call with Adapter Command Timeout:  %d\n", cb.data[12] + (cb.data[13]*256) + (cb.data[14]*6553) + (cb.data[15]*16777216) );
      printf("      Number of PTPIPE Monitor call with Adapter Command OK:  %d\n", cb.data[16] + (cb.data[17]*256) + (cb.data[18]*6553) + (cb.data[19]*16777216) );
      printf("  Number of PTPIPE Monitor call with pkt passed to Adapter:   %d\n", cb.data[20] + (cb.data[21]*256) + (cb.data[22]*6553) + (cb.data[23]*16777216) );
      printf("    Number of PTPIPE Monitor call with pkt passed to Stack:   %d\n", cb.data[24] + (cb.data[25]*256) + (cb.data[26]*6553) + (cb.data[27]*16777216) );
      printf("              Number of PTPIPE Monitor call with no socket:   %d\n", cb.data[28] + (cb.data[29]*256) + (cb.data[30]*6553) + (cb.data[31]*16777216) );

      printf("             Number of DRIVER STAT call with Kmalloc error:   %d\n", cb.data[32] + (cb.data[33]*256) + (cb.data[34]*6553) + (cb.data[35]*16777216) );
      printf("   Number of DRIVER STAT call with Adapter Command Timeout:   %d\n", cb.data[36] + (cb.data[37]*256) + (cb.data[38]*6553) + (cb.data[39]*16777216) );
      printf("        Number of DRIVER STAT call with Adapter Command OK:   %d\n", cb.data[40] + (cb.data[41]*256) + (cb.data[42]*6553) + (cb.data[43]*16777216) );
      printf("     Number of DRIVER STAT call with pkt passed to Adapter:   %d\n", cb.data[44] + (cb.data[45]*256) + (cb.data[46]*6553) + (cb.data[47]*16777216) );
      printf("       Number of DRIVER STAT call with pkt passed to Stack:   %d\n", cb.data[48] + (cb.data[49]*93) + (cb.data[50]*6553) + (cb.data[51]*16777216) );
      printf("                 Number of DRIVER STAT call with no socket:   %d\n", cb.data[52] + (cb.data[53]*256) + (cb.data[54]*6553) + (cb.data[55]*16777216) );
      printf("                                    Number of Poll Entries:   %d\n", cb.data[56] + (cb.data[57]*256) + (cb.data[58]*6553) + (cb.data[59]*16777216) );
      printf("                  Number of Poll Entries with Critical set:   %d\n", cb.data[60] + (cb.data[61]*256) + (cb.data[62]*6553) + (cb.data[63]*16777216) );
      printf("                          Number of Poll Entries Processed:   %d\n", cb.data[64] + (cb.data[65]*256) + (cb.data[66]*6553) + (cb.data[67]*16777216) );
      printf("            Number of times host irq left disabled in Poll:   %d\n", cb.data[68] + (cb.data[69]*256) );

} /* ppp_driver_stat_ifsend */

void flush_driver_stats( void )
{
      cb.command = PPIPE_FLUSH_DRIVER_STATS;
      cb.buffer_length = 0;
      cb.data[0] = 0;
      DoCommand(&cb);

      printf("All Driver Statistics are Flushed.\n");

}

void ppp_router_up_time( void )
{
     unsigned long time;
     cb.command = PPIPE_ROUTER_UP_TIME;
     cb.buffer_length = 0;
     cb.data[0] = 0;
     DoCommand(&cb);
     
     time = cb.data[0] + (cb.data[1]*256) + (cb.data[2]*6553) + 
					(cb.data[3]*16777216);
     if (time < 3600) {
	if (time<60) 
     		printf("    Router UP Time:  %ld seconds\n", time);
	else
     		printf("    Router UP Time:  %ld minute(s)\n", (time/60));
     }else
     		printf("    Router UP Time:  %ld hour(s)\n", (time/3600));
			
      
}

void main(int argv, char* argc[]){
   int proceed;

   printf("\n");
   if( argv > 2 ) {
     init( argv, argc);
     proceed = MakeConnection();
     if(proceed == TRUE){
      switch( argc[1][0] ) {
      case 'g':
	 general_conf();
	 break;
      case 'u':
	 timers();
	 break;
      case 'a':
	 authent();
	 break;
      case 'i':
	 ip_config();
	 break;
      case 'x':
	 ipx_config();
	 break;
      case 's':
	 state();
	 break;
      case 'n':
	 negot();
	 break;
      case 'c':
	 cause();
	 break;
      case 'm':
	 modem();
	 break;
      case 'l':
	 general_stats();
	 break;
      case 'e':
	 comm_err();
	 break;
      case 'p':
	 packet();
	 break;
      case 'r':
	 lcp();
	 break;
      case 'b':
	 loopback();
	 break;
      case 'o':
	 ipcp();
	 break;
      case 'y':
	 ipxcp();
	 break;
      case 'w':
	 pap();
	 break;
      case 'h':
	 chap();
	 break;
      case 'q':
	 if( argc[1][1] == 'r' )
	   raw_data = TRUE;
	 line_trace();
	 break;
      case 'd':
	 if (argc[1][1] == 's')
	 	ppp_driver_stat_ifsend();
	 if (argc[1][1] == 'i')
		ppp_driver_stat_intr();	 
	 if (argc[1][1] == 'g')
		ppp_driver_stat_gen();	 
	 break;
      case 'j':
	 ppp_router_up_time();
	 break;
      case 'f':
	 if( argv > 2 ) {
	    switch( argc[2][0] ) {
	    case 'l':
	       flush_general_stats();
	       general_stats();
	       break;
	    case 'e':
	       flush_comm_err();
	       comm_err();
	       break;
	    case 'p':
	       flush_packet();
	       packet();
	       break;
	    case 'r':
	       flush_lcp();
	       lcp();
	       break;
	    case 'b':
	       flush_loopback();
	       loopback();
	       break;
	    case 'o':
	       flush_ipcp();
	       ipcp();
	       break;
	    case 'y':
	       flush_ipxcp();
	       ipxcp();
	       break;
	    case 'w':
	       flush_pap();
	       pap();
	       break;
	    case 'h':
	       flush_chap();
	       chap();
	       break;
	    case 'd':
	       flush_driver_stats();
	       break;
	    default:
	       usage();
	    } //switch
	 } else {
	    usage();
	 }//if
	 break;
      case 't':
      	if( argv > 2){
           switch( argc[2][0] ){
	   case 'v':
	     set_FT1_monitor_status(0x01);
	     if(!fail){
		view_FT1_status();
		set_FT1_monitor_status(0x00);
	     }
	     break;
	   case 's':
	     set_FT1_monitor_status(0x01);
	     if(!fail){	 	
		FT1_self_test();
		set_FT1_monitor_status(0x00);
	     }
	     break;
           case 'l':
	     set_FT1_monitor_status(0x01);
	     if(!fail){
	    	FT1_ll_test();
		set_FT1_monitor_status(0x00);
             }
             break;
           case 'd':
             set_FT1_monitor_status(0x01);
	     if(!fail){
		FT1_dl_test();
		set_FT1_monitor_status(0x00);
	     }
             break;
           case 'r':
	     set_FT1_monitor_status(0x01);
	     if(!fail){
		FT1_rt_test();
		set_FT1_monitor_status(0x00);
	     }
	     break;
	   case 'o':
	     set_FT1_monitor_status(0x01);
	     if(!fail){
		FT1_operational_mode();
		set_FT1_monitor_status(0x00);
	     }
	     break;
	   default:
	     usage();
	   }
	} else{
  	   usage();
	} 
        break; 
      default:
	 usage();
      } //switch
     } 
     close( sock );
   } else {
      usage();
   } //if
   printf("\n");
}; //main

/*
 * EOF ppipemon.c
 */
