/* filename: rlpr-msg.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlpr-net.c,v 1.19 1997/01/14 01:10:18 meem Exp meem $
 * contents: rlpr's messaging facility implementation
 *
 * Time-stamp: <1997/04/27 14:44:20 cdt -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996, 1997 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#include <syslog.h>                   /* for syslog(), LOG_* */

#include <stdarg.h>                   /* for varargs          */
#include <stdio.h>                    /* for FILE*, etc...    */
#include <stdlib.h>                   /* for EXIT_* constants */
#include <string.h>                   /* for strerror()       */
#include <errno.h>

#include "getopt.h"

#include "rlpr-msg.h"

static rlpr_msg_info rlpr_msg_props = { 0, /* all args */ };

void rlpr_msg(enum rlpr_msg_level l, enum rlpr_msg_errno e, char *fmt, ...)
{
  va_list ap;
  FILE * stream = (l == INFO) ? stdout : stderr;

  if (l == DEBUG && !rlpr_msg_props.debug) return;
  if (l <  FATAL &&  rlpr_msg_props.quiet) return;
  va_start(ap, fmt);

  if (rlpr_msg_props.syslog) {
    static char buf[BUFSIZ];
    vsnprintf(buf, BUFSIZ, fmt, ap);

    if (e == ERRNO) strcat(buf, ": %m");
#ifdef SYSLOG_USABLE
    syslog(l == DEBUG ? LOG_DEBUG :
           l == INFO  ? LOG_INFO  :
           l == WARNING ? LOG_WARNING : LOG_ERR, buf);
#endif /* SYSLOG_USABLE */
  } else {
    fprintf(stream, "%s: %s", program_name,
                                  (l == DEBUG)   ? "debug: "   :
                                  (l == INFO)    ? ""          : 
                                  (l == WARNING) ? "warning: " : "fatal: ");

    vfprintf(stream, fmt, ap);
    if (e == ERRNO) fprintf(stream, ": %s", strerror(errno));
    fputc('\n', stream);
  }
  va_end(ap);
  if (l == FATAL) exit(EXIT_FAILURE);
}

void rlpr_msg_args(int argc, char * const argv[])
{
  extern int                 opterr, optind;
  extern const char   *      rlpr_opts;
  extern const struct option rlpr_long_opts[];
  int c;                              /* the number of arguments read */

  if (argv == NULL) return;

  opterr = 0;                         /* do not report errors         */
  optind = 0;			      /* reset to start of argv       */

  while ((c = getopt_long(argc, argv, rlpr_opts, rlpr_long_opts, NULL)) != EOF)
    /* ignore any options we don't recognize */

    switch(c) {
    case 'q':                         /* be quiet! */
      rlpr_msg_props.quiet = 1;
      break;
    case -101:
      rlpr_msg_props.debug = 1;
      break;
    case -102:		   	     /* verbose flag */
      rlpr_msg_props.quiet = 0;
    }
}


void * rlpr_msg_get(enum rlpr_msg_field field) {
  switch(field) {                     
  case RLPR_MSG_DEBUG:                /* the only visible field */
    return &rlpr_msg_props.debug;
  case RLPR_MSG_QUIET:
    return &rlpr_msg_props.quiet;
  default:                            /* just to supress warning messages */
  }
  return NULL;
}


int rlpr_msg_set(enum rlpr_msg_field field, void * value) {
  switch (field) {
  default:
    return 0;
  case RLPR_MSG_QUIET:
    rlpr_msg_props.quiet  = *(char *)value;
    break;
  case RLPR_MSG_DEBUG:
    rlpr_msg_props.debug  = *(char *)value;
    break;
  case RLPR_MSG_SYSLOG:
    rlpr_msg_props.syslog = *(char *)value;
  }
  return 1;
}

