/***

utfilter.c  - UDP/TCP display filter module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***/

#include <curses.h>
#include <panel.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <netdb.h>
#include "dirs.h"
#include "deskman.h"
#include "attrs.h"
#include "menurt.h"
#include "utfdefs.h"
#include "utfilter.h"
#include "input.h"
#include "error.h"

/*
 * Generate a string representation of a number to be used as a name.
 */
 
void genname(unsigned long n, char *m)
{
    sprintf(m, "%lu", n);
}

void listfileerr(int code)
{
    int resp;

    if (code == 1)
	errbox("Error opening filter list file", ANYKEY_MSG, &resp);
    else
	errbox("Error writing filter list file", ANYKEY_MSG, &resp);
}

void gethostparams(struct hostparams *hp, int *aborted)
{
    struct FIELDLIST fieldlist;
    struct FIELD *dlist;
    WINDOW *dlgwin;
    PANEL *dlgpanel;

    dlgwin = newwin(10, 80, 7, 0);
    dlgpanel = new_panel(dlgwin);

    wattrset(dlgwin, BOXATTR);
    colorwin(dlgwin);
    box(dlgwin, ACS_VLINE, ACS_HLINE);
    wmove(dlgwin, 0, 26);
    wprintw(dlgwin, " First ");

    wmove(dlgwin, 0, 52);
    wprintw(dlgwin, " Second ");

    wattrset(dlgwin, STDATTR);
    wmove(dlgwin, 2, 2);
    wprintw(dlgwin, "Host name/IP address:");
    wmove(dlgwin, 4, 2);
    wprintw(dlgwin, "Wildcard mask:");
    wmove(dlgwin, 6, 2);
    wprintw(dlgwin, "Port:");
    wmove(dlgwin, 8, 2);
    tabkeyhelp(dlgwin);
    wmove(dlgwin, 8, 20);
    stdkeyhelp(dlgwin);
    update_panels();
    doupdate();

    initfields(&fieldlist, 7, 52, 8, 27);

    addfield(&fieldlist, 25, 1, 0, "");
    addfield(&fieldlist, 25, 3, 0, "");
    addfield(&fieldlist, 5, 5, 0, "");
    addfield(&fieldlist, 25, 1, 26, "0.0.0.0");
    addfield(&fieldlist, 25, 3, 26, "0.0.0.0");
    addfield(&fieldlist, 5, 5, 26, "0");

    dlist = fieldlist.list->nextfield->nextfield->nextfield;

    fillfields(&fieldlist, aborted);
    strcpy(hp->s_fqdn, fieldlist.list->buf);
    strcpy(hp->s_mask, fieldlist.list->nextfield->buf);
    hp->sport = atoi(fieldlist.list->nextfield->nextfield->buf);

    strcpy(hp->d_fqdn, dlist->buf);
    strcpy(hp->d_mask, dlist->nextfield->buf);
    hp->dport = atoi(dlist->nextfield->nextfield->buf);

    destroyfields(&fieldlist);
    del_panel(dlgpanel);
    delwin(dlgwin);
    update_panels();
    doupdate();

    if (*aborted)
	return;

}

void definefilter(int protocol, int *aborted)
{
    struct FIELDLIST descfield;
    struct filterfileent ffile;
    struct hostparams hp;
    char ffilename[40] = WORKDIR;
    char fntemp[14];

    int pfd;
    int bw;
    int resp;

    WINDOW *dlgwin;
    PANEL *dlgpanel;

    dlgwin = newwin(7, 42, 9, 9);
    dlgpanel = new_panel(dlgwin);
    wattrset(dlgwin, BOXATTR);
    colorwin(dlgwin);
    box(dlgwin, ACS_VLINE, ACS_HLINE);
    wattrset(dlgwin, STDATTR);
    wmove(dlgwin, 2, 2);
    wprintw(dlgwin, "Enter a description for this filter");
    wmove(dlgwin, 5, 2);
    stdkeyhelp(dlgwin);
    update_panels();
    doupdate();

    initfields(&descfield, 1, 35, 12, 11);
    addfield(&descfield, 33, 0, 0, "");
    fillfields(&descfield, aborted);
    strcpy(ffile.desc, descfield.list->buf);
    destroyfields(&descfield);
    del_panel(dlgpanel);
    delwin(dlgwin);
    update_panels();
    doupdate();

    if (*aborted)
	return;

    genname(time((time_t *) NULL), fntemp);
    strcat(ffilename, fntemp);

    pfd = open(ffilename, O_CREAT | O_WRONLY | O_TRUNC, S_IRUSR | S_IWUSR);
    if (pfd < 0) {
	errbox("Cannot create filter record file", ANYKEY_MSG, &resp);
	*aborted = 1;
	return;
    }
    do {
	gethostparams(&hp, aborted);
	if (!(*aborted)) {
	    bw = write(pfd, &hp, sizeof(struct hostparams));

	    if (bw < 0) {
		errbox("Unable to write filter record", ANYKEY_MSG, &resp);
		close(pfd);
		return;
	    }
	}
    } while (!(*aborted));

    close(pfd);

    if (protocol == F_TCP)
	pfd = open(TCPFLNAME, O_CREAT | O_WRONLY | O_APPEND, S_IRUSR | S_IWUSR);
    else
	pfd = open(UDPFLNAME, O_CREAT | O_WRONLY | O_APPEND, S_IRUSR | S_IWUSR);

    if (pfd < 0) {
	listfileerr(1);
	return;
    }
    strcpy(ffile.filename, fntemp);
    bw = write(pfd, &ffile, sizeof(struct filterfileent));
    if (bw < 0)
	listfileerr(2);

    close(pfd);
}

void init_filter_table(struct filterlist *fl)
{
    fl->head = fl->tail = NULL;
}

void displayfilters(struct ffnode *ffiles, WINDOW * win)
{
    unsigned int row = 0;
    struct ffnode *pptr;

    pptr = ffiles;

    wattrset(win, STDATTR);
    do {
	wmove(win, row, 2);
	wprintw(win, pptr->ffe.desc);
	row++;
	pptr = pptr->next_entry;
    } while ((row < 8) && (pptr != NULL));

    update_panels();
    doupdate();
}

void operate_select(struct ffnode *ffiles, WINDOW * win, struct ffnode **item,
		    int *aborted)
{
    unsigned int row = 0;
    struct ffnode *pptr;
    int ch;
    int exitloop = 0;

    listkeyhelp();
    update_panels();
    doupdate();
    pptr = ffiles;
    do {
	wattrset(win, PTRATTR);
	wmove(win, row, 1);
	waddch(win, ACS_RARROW);
	ch = wgetch(win);
	wmove(win, row, 1);
	wprintw(win, " ");
	wattrset(win, STDATTR);

	switch (ch) {
	case KEY_UP:
	    if (pptr->prev_entry != NULL) {
		if (row > 0)
		    row--;
		else {
		    wscrl(win, -1);
		    wmove(win, 0, 0);
		    wprintw(win, "%58c", ' ');
		    wmove(win, 0, 2);
		    wprintw(win, pptr->prev_entry->ffe.desc);
		}
		pptr = pptr->prev_entry;
	    }
	    break;
	case KEY_DOWN:
	    if (pptr->next_entry != NULL) {
		if (row < 7)
		    row++;
		else {
		    wscrl(win, 1);
		    scrollok(win, 0);
		    wmove(win, 7, 0);
		    wprintw(win, "%58c", ' ');
		    scrollok(win, 1);
		    wmove(win, 7, 2);
		    wprintw(win, pptr->next_entry->ffe.desc);
		}
		pptr = pptr->next_entry;
	    }
	    break;
	case 13:
	    exitloop = 1;
	    *aborted = 0;
	    break;
	case 27:
	case 24:
	case 'X':
	case 'x':
	case 'Q':
	case 'q':
	    exitloop = 1;
	    *aborted = 1;
	    break;
	}
	update_panels();
	doupdate();
    } while (!exitloop);

    *item = pptr;
}

void loadfilterlist(unsigned int protocol, struct ffnode **fltfile)
{
    int pfd;

    struct ffnode *ffiles = NULL;
    struct ffnode *ptemp;
    struct ffnode *tail = NULL;

    int br;

    if (protocol == F_TCP)
	pfd = open(TCPFLNAME, O_RDONLY);
    else
	pfd = open(UDPFLNAME, O_RDONLY);

    if (pfd < 0) {
	listfileerr(1);
	*fltfile = NULL;
	return;
    }
    do {
	ptemp = malloc(sizeof(struct ffnode));
	br = read(pfd, &(ptemp->ffe), sizeof(struct filterfileent));

	if (br > 0) {
	    if (ffiles == NULL) {
		ffiles = ptemp;
		ffiles->prev_entry = NULL;
	    } else {
		tail->next_entry = ptemp;
		ptemp->prev_entry = tail;
	    }

	    ptemp->next_entry = NULL;
	    tail = ptemp;
	} else
	    free(ptemp);
    } while (br > 0);

    close(pfd);
    *fltfile = ffiles;
}

void pickafilter(struct ffnode *ffiles,
		 struct ffnode **fltfile, int *aborted)
{
    WINDOW *borderwin;
    PANEL *borderpanel;
    WINDOW *selectwin;
    PANEL *selectpanel;

    borderwin = newwin(10, 60, 6, 8);
    borderpanel = new_panel(borderwin);
    wattrset(borderwin, BOXATTR);
    box(borderwin, ACS_VLINE, ACS_HLINE);

    selectwin = newwin(8, 58, 7, 9);
    selectpanel = new_panel(selectwin);
    wattrset(selectwin, STDATTR);
    colorwin(selectwin);
    stdwinset(selectwin);
    wtimeout(selectwin, -1);

    displayfilters(ffiles, selectwin);
    operate_select(ffiles, selectwin, fltfile, aborted);

    del_panel(borderpanel);
    delwin(borderwin);
    del_panel(selectpanel);
    delwin(selectwin);
    update_panels();
    doupdate();
}

void destroyfilterlist(struct ffnode *fltlist)
{
    struct ffnode *fftemp;

    if (fltlist != NULL) {
	fftemp = fltlist->next_entry;

	do {
	    free(fltlist);
	    fltlist = fftemp;
	    if (fftemp != NULL)
		fftemp = fftemp->next_entry;
	} while (fltlist != NULL);
    }
}

void selectfilter(unsigned int protocol,
		  struct filterfileent *ffe, int *aborted)
{
    struct ffnode *fltfile;
    struct ffnode *ffiles;

    loadfilterlist(protocol, &ffiles);

    if (ffiles == NULL) {
	*aborted = 1;
	return;
    }
    pickafilter(ffiles, &fltfile, aborted);

    if (!(*aborted))
	*ffe = fltfile->ffe;

    destroyfilterlist(ffiles);
}

unsigned long int nametoaddr(char *ascname)
{
    unsigned long int result;
    struct hostent *he;
    char imsg[45];
    int resp;
    
    result = inet_addr(ascname);
    if (result == -1) {
        bzero(imsg, 45);
        strcpy(imsg, "Resolving ");    
        strncat(imsg, ascname, 35);
        indicate(imsg);
        he = gethostbyname(ascname);
        if (he != NULL)
            bcopy((he->h_addr_list)[0], &result, he->h_length);
        else {
            bzero(imsg,45);
            strcpy(imsg, "Unable to resolve ");
            strncat(imsg, ascname, 22);
            errbox(imsg, ANYKEY_MSG, &resp);
            result = -1;
        }
    }
    
    return result;
}
    
/* loads the filter from the filter file */

void loadfilter(char *filename, struct filterlist *fl)
{
    struct filterent *fe;
    int pfd;
    unsigned int idx = 0;
    int br;
    int resp;

    init_filter_table(fl);

    pfd = open(filename, O_RDONLY);

    if (pfd < 0) {
	errbox("Error opening filter file", ANYKEY_MSG, &resp);
	fl->head = NULL;
	return;
    }
    do {
	fe = malloc(sizeof(struct filterent));
	br = read(pfd, &(fe->hp), sizeof(struct hostparams));

	if (br > 0) {
	    fe->index = idx;
	    fe->saddr = nametoaddr(fe->hp.s_fqdn);	    
	    fe->daddr = nametoaddr(fe->hp.d_fqdn);
	    
	    if ((fe->saddr == -1) || (fe->daddr == -1)) {
	        free(fe);
	        continue;
	    }
	    
	    fe->smask = inet_addr(fe->hp.s_mask);
	    fe->dmask = inet_addr(fe->hp.d_mask);

	    if (fl->head == NULL) {
		fl->head = fe;
		fe->prev_entry = NULL;
	    } else {
		fl->tail->next_entry = fe;
		fe->prev_entry = fl->tail;
	    }
	    fe->next_entry = NULL;
	    fl->tail = fe;
	    idx++;
	} else
	    free(fe);
    } while (br > 0);

    if (br == 0)
	close(pfd);
}


/* the display filter */

int utfilter(struct filterlist *fl,
	     unsigned long source, unsigned long dest,
	     unsigned int sport, unsigned int dport)
{
    struct filterent *fe;
    unsigned long fsaddr, fdaddr;
    unsigned long csaddr, cdaddr;
    unsigned long crsaddr, crdaddr;

    fe = fl->head;

    while (fe != NULL) {
	fsaddr = fe->saddr & fe->smask;
	fdaddr = fe->daddr & fe->dmask;
	csaddr = source & fe->smask;
	cdaddr = dest & fe->dmask;
	crsaddr = source & fe->dmask;
	crdaddr = dest & fe->smask;

	if (((csaddr == fsaddr) && ((fe->hp.sport == sport) || (fe->hp.sport == 0)) &&
	     (cdaddr == fdaddr) && ((fe->hp.dport == dport) || (fe->hp.dport == 0))) ||
	    ((crsaddr == fdaddr) && ((fe->hp.dport == sport) || (fe->hp.dport == 0)) &&
	     (crdaddr == fsaddr) && ((fe->hp.sport == dport) || (fe->hp.sport == 0))))
	    return 1;

	fe = fe->next_entry;
    }

    return 0;
}

/* remove a currently applied filter from memory */

void destroyfilter(struct filterlist *fl)
{
    struct filterent *fe;
    struct filterent *cfe;

    if (fl->head != NULL) {
	fe = fl->head;
	cfe = fl->head->next_entry;

	do {
	    free(fe);
	    fe = cfe;
	    if (cfe != NULL)
		cfe = cfe->next_entry;
	} while (fe != NULL);

	fl->head = fl->tail = NULL;
    }
}

/* delete a filter record from the disk */

void delfilter(unsigned int protocol, int *aborted)
{
    struct ffnode *fltfile;
    struct ffnode *fltlist;
    struct ffnode *ffntemp;
    char fntemp[40] = WORKDIR;
    int fd;
    int bw;

    loadfilterlist(protocol, &fltlist);

    if (fltlist == NULL)
	return;

    pickafilter(fltlist, &fltfile, aborted);

    if (*aborted)
	return;

    strcat(fntemp, fltfile->ffe.filename);
    unlink(fntemp);

    if (fltfile->prev_entry == NULL) {
	fltlist = fltlist->next_entry;
	if (fltlist != NULL)
	    fltlist->prev_entry = NULL;
    } else {
	fltfile->prev_entry->next_entry = fltfile->next_entry;

	if (fltfile->next_entry != NULL)
	    fltfile->next_entry->prev_entry = fltfile->prev_entry;
    }

    free(fltfile);

    if (protocol == F_TCP)
	fd = open(TCPFLNAME, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
    else
	fd = open(UDPFLNAME, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);

    if (fd < 0) {
	listfileerr(1);
	return;
    }
    fltfile = fltlist;
    while (fltfile != NULL) {
	bw = write(fd, &(fltfile->ffe), sizeof(struct filterfileent));

	if (bw < 0) {
	    listfileerr(2);
	    return;
	}
	ffntemp = fltfile;
	fltfile = fltfile->next_entry;
	free(ffntemp);
    }

    close(fd);
}

void makefiltermenu(struct MENU *menu)
{
    initmenu(menu, 7, 31, 10, 30);
    additem(menu, " ^D^efine new filter...", "Defines a new set of filter parameters");
    additem(menu, " ^A^pply filter...", "Applies filter to be used for TCP display");
    additem(menu, " Detac^h^ filter", "Removes the currently applied filter");
    additem(menu, " Dele^t^e filter...", "Removes a filter from the filter list");
    additem(menu, " E^x^it menu", "Returns to the main menu");
}

void makeudpfiltermenu(struct MENU *menu)
{
    initmenu(menu, 8, 31, 10, 30);
    additem(menu, " Show ^a^ll UDP packets", "Permits display of all UDP packets");
    additem(menu, " Show ^n^o UDP packets", "Omits all UDP packets from display");
    additem(menu, " ^D^efine custom UDP filter...", "Defines a custom UDP filter");
    additem(menu, " Apply ^c^ustom UDP filter...", "Applies a custom UDP filter");
    additem(menu, " Dele^t^e custom UDP filter...", "Removes a custom UDP filter");
    additem(menu, " E^x^it menu", "Returns to the main menu");
}


/* display a menu and perform appropriate filter action */

void udpfilterselect(struct filterlist *fl,
		     unsigned int *filtercode,
		     char *filename, int *faborted)
{
    struct MENU fmenu;
    struct filterfileent ffe;
    unsigned int frow;

    makeudpfiltermenu(&fmenu);

    frow = 1;

    do {
	showmenu(&fmenu);
	operatemenu(&fmenu, &frow, faborted);

	switch (frow) {
	case 1:
	    *filtercode = 1;
	    break;
	case 2:
	    if (*filtercode == 2)
		destroyfilter(fl);

	    *filtercode = 0;
	    break;
	case 3:
	    definefilter(F_UDP, faborted);
	    break;
	case 4:
	    selectfilter(F_UDP, &ffe, faborted);
	    if (!(*faborted)) {
		strcpy(filename, WORKDIR);
		strcat(filename, ffe.filename);
		loadfilter(filename, fl);
		*filtercode = 2;
	    }
	    break;
	case 5:
	    delfilter(F_UDP, faborted);
	    break;
	}
    } while (frow != 6);

    destroymenu(&fmenu);
    update_panels();
    doupdate();
}


/* display a menu and perform appropriate filter operation */

void tcpfilterselect(struct filterlist *fl,
		     unsigned int *filtered, int *faborted)
{
    struct MENU fmenu;
    unsigned int frow;
    struct filterfileent ffe;
    char filename[30];

    makefiltermenu(&fmenu);

    frow = 1;
    do {
	showmenu(&fmenu);
	operatemenu(&fmenu, &frow, faborted);

	switch (frow) {
	case 1:
	    definefilter(F_TCP, faborted);
	    break;
	case 2:
	    selectfilter(F_TCP, &ffe, faborted);
	    if (!(*faborted)) {
		strcpy(filename, WORKDIR);
		strcat(filename, ffe.filename);
		loadfilter(filename, fl);
		*filtered = 1;
	    }
	    break;
	case 3:
	    if (*filtered) {
		destroyfilter(fl);
		*filtered = 0;
	    }
	    break;
	case 4:
	    delfilter(F_TCP, faborted);
	    break;
	}
    } while (frow != 5);

    destroymenu(&fmenu);
    update_panels();
    doupdate();
}
