/***

hostmon.c - Host traffic monitor
Discovers hosts and displays packet statistics for them
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***/

#include <curses.h>
#include <panel.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <linux/if_packet.h>
#include <net/if_arp.h>
#include <stdlib.h>
#include <sys/time.h>
#include "if_ether.h"
#include "deskman.h"
#include "packet.h"
#include "hostmon.h"
#include "stdwinset.h"
#include "attrs.h"
#include "log.h"
#include "timer.h"

#ifndef _I386_TYPES_H
#include <asm/types.h>
#endif

#define SCROLLUP 0
#define SCROLLDOWN 1

/* 
 * from log.c, applicable only to this module 
 */

extern void writeethlog(struct ethtabent *list, unsigned long nsecs, FILE * logfile);

void initethtab(struct ethtab *table)
{
    table->head = table->tail = NULL;
    table->firstvisible = table->lastvisible = NULL;
    table->count = table->entcount = 0;

    table->borderwin = newwin(23, 80, 1, 0);
    table->borderpanel = new_panel(table->borderwin);

    table->tabwin = newwin(21, 78, 2, 1);
    table->tabpanel = new_panel(table->tabwin);

    wattrset(table->borderwin, BOXATTR);
    box(table->borderwin, ACS_VLINE, ACS_HLINE);
    wmove(table->borderwin, 0, 4);
    wprintw(table->borderwin, " PktsIn ");
    wmove(table->borderwin, 0, 14);
    wprintw(table->borderwin, " IP In ");
    wmove(table->borderwin, 0, 22);
    wprintw(table->borderwin, " BytesIn ");
    wmove(table->borderwin, 0, 32);
    wprintw(table->borderwin, " InRate ");

    wmove(table->borderwin, 0, 39);
    wprintw(table->borderwin, " PktsOut ");
    wmove(table->borderwin, 0, 49);
    wprintw(table->borderwin, " IP Out ");
    wmove(table->borderwin, 0, 57);
    wprintw(table->borderwin, " BytesOut ");
    wmove(table->borderwin, 0, 68);
    wprintw(table->borderwin, " OutRate ");

    wmove(table->borderwin, 22, 40);
    wprintw(table->borderwin, " InRate and OutRate are in kbits/sec ");

    wattrset(table->tabwin, STDATTR);
    colorwin(table->tabwin);
    stdwinset(table->tabwin);
    wtimeout(table->tabwin, -1);
    update_panels();
    doupdate();
}

struct ethtabent *addethnode(struct ethtab *table, int *nomem)
{
    struct ethtabent *ptemp;

    ptemp = malloc(sizeof(struct ethtabent));

    if (ptemp == NULL) {
	printnomem();
	*nomem = 1;
	return NULL;
    }
    if (table->head == NULL) {
	ptemp->prev_entry = NULL;
	table->head = ptemp;
	table->firstvisible = ptemp;
    } else {
	ptemp->prev_entry = table->tail;
	table->tail->next_entry = ptemp;
    }

    table->tail = ptemp;
    ptemp->next_entry = NULL;

    table->count++;
    ptemp->index = table->count;

    if (table->count <= 21)
	table->lastvisible = ptemp;

    return ptemp;
}

void convethaddr(char *addr, char *result)
{
    unsigned int i;
    __u8 *ptmp = addr;
    char hexbyte[3];

    strcpy(result, "");
    for (i = 0; i <= 5; i++) {
	sprintf(hexbyte, "%02x", *ptmp);
	strcat(result, hexbyte);
	ptmp++;
    }
}

struct ethtabent *addethentry(struct ethtab *table, char *addr, int *nomem)
{
    struct ethtabent *ptemp;

    ptemp = addethnode(table, nomem);

    if (ptemp == NULL)
	return NULL;

    ptemp->type = 0;
    memcpy(&(ptemp->un.desc.eth_addr), addr, ETH_ALEN);
    convethaddr(addr, ptemp->un.desc.ascaddr);
    ptemp->un.desc.printed = 0;

    ptemp = addethnode(table, nomem);

    if (ptemp == NULL)
	return NULL;

    ptemp->type = 1;
    ptemp->un.figs.inpcount = ptemp->un.figs.inpktact = 0;
    ptemp->un.figs.outpcount = ptemp->un.figs.outpktact = 0;
    ptemp->un.figs.inspanbr = ptemp->un.figs.outspanbr = 0;
    ptemp->un.figs.inippcount = ptemp->un.figs.outippcount = 0;
    ptemp->un.figs.inbcount = ptemp->un.figs.outbcount = 0;
    ptemp->un.figs.inrate = ptemp->un.figs.outrate = 0;

    table->entcount++;

    wmove(table->borderwin, 22, 1);
    wprintw(table->borderwin, " %u entries ", table->entcount);

    return ptemp;
}

struct ethtabent *in_ethtable(struct ethtab *table, char *addr)
{
    struct ethtabent *ptemp = table->head;

    while (ptemp != NULL) {
	if ((ptemp->type == 0) &&
	    (memcmp(addr, ptemp->un.desc.eth_addr, ETH_ALEN) == 0))
	    return ptemp->next_entry;

	ptemp = ptemp->next_entry;
    }

    return NULL;
}

void updateethent(struct ethtabent *entry, unsigned int pktsize,
		  int is_ip, int inout)
{
    if (inout == 0) {
	entry->un.figs.inpcount++;
	entry->un.figs.inbcount += pktsize;
	entry->un.figs.inspanbr += pktsize;
	if (is_ip)
	    entry->un.figs.inippcount++;
    } else {
	entry->un.figs.outpcount++;
	entry->un.figs.outbcount += pktsize;
	entry->un.figs.outspanbr += pktsize;
	if (is_ip)
	    entry->un.figs.outippcount++;
    }
}

void printethent(struct ethtab *table, struct ethtabent *entry,
		 unsigned int idx)
{
    unsigned int target_row;

    if ((entry->index < idx) || (entry->index > idx + 20))
	return;

    target_row = entry->index - idx;

    if (entry->type == 0) {
	wmove(table->tabwin, target_row, 1);
	wattrset(table->tabwin, STDATTR);
	wprintw(table->tabwin, "HW addr: %s", entry->un.desc.ascaddr);
	entry->un.desc.printed = 1;
    } else {
	wattrset(table->tabwin, PTRATTR);
	wmove(table->tabwin, target_row, 1);
	waddch(table->tabwin, ACS_LLCORNER);

	wattrset(table->tabwin, HIGHATTR);

	wmove(table->tabwin, target_row, 2);
	wprintw(table->tabwin, "%8lu %8lu %9lu",
		entry->un.figs.inpcount,
		entry->un.figs.inippcount,
		entry->un.figs.inbcount);

	wmove(table->tabwin, target_row, 38);
	wprintw(table->tabwin, "%8lu %8lu %9lu",
		entry->un.figs.outpcount,
		entry->un.figs.outippcount,
		entry->un.figs.outbcount);
    }
}

void destroyethtab(struct ethtab *table)
{
    struct ethtabent *ptemp = table->head;
    struct ethtabent *cnext = NULL;

    if (table->head != NULL)
	cnext = table->head->next_entry;

    while (ptemp != NULL) {
	free(ptemp);
	ptemp = cnext;

	if (cnext != NULL)
	    cnext = cnext->next_entry;
    }
}

void hostmonhelp()
{
    move(24, 1);
    printkeyhelp("Up/Down/PgUp/PgDn", "-scroll window  ", stdscr);
    stdexitkeyhelp();
}

void updateethrates(struct ethtab *table, time_t starttime, time_t now,
		    unsigned int idx)
{
    struct ethtabent *ptmp = table->firstvisible;
    unsigned int target_row = 0;

    if (table->lastvisible == NULL)
        return;
        
    while (ptmp != table->lastvisible->next_entry) {
	if (ptmp->type == 1) {
	    ptmp->un.figs.inrate =
		((float) (ptmp->un.figs.inspanbr * 8 / 1000)) / ((float) (now - starttime));
	    ptmp->un.figs.outrate =
		((float) (ptmp->un.figs.outspanbr * 8 / 1000)) / ((float) (now - starttime));
	    if ((ptmp->index >= idx) && (ptmp->index <= idx + 20)) {
		wattrset(table->tabwin, HIGHATTR);
		target_row = ptmp->index - idx;
		wmove(table->tabwin, target_row, 30);
		wprintw(table->tabwin, "%8.2f", ptmp->un.figs.inrate);
		wmove(table->tabwin, target_row, 67);
		wprintw(table->tabwin, "%8.2f", ptmp->un.figs.outrate);
	    }
	    ptmp->un.figs.inspanbr = ptmp->un.figs.outspanbr = 0;
	}
	ptmp = ptmp->next_entry;
    }
}

void scrollethwin(struct ethtab *table, int direction, int *idx)
{
    wattrset(table->tabwin, STDATTR);
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->tabwin, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    (*idx)++;
	    wmove(table->tabwin, 20, 0);
	    scrollok(table->tabwin, 0);
	    wprintw(table->tabwin, "%78c", 32);
	    scrollok(table->tabwin, 1);
	    printethent(table, table->lastvisible, *idx);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->tabwin, -1);
	    table->lastvisible = table->lastvisible->prev_entry;
	    table->firstvisible = table->firstvisible->prev_entry;
	    (*idx)--;
	    wmove(table->tabwin, 0, 0);
	    scrollok(table->tabwin, 0);
	    wprintw(table->tabwin, "%78c", 32);
	    scrollok(table->tabwin, 1);
	    printethent(table, table->firstvisible, *idx);
	}
    }
}

void pageethwin(struct ethtab *table, int direction, int *idx)
{
    int i = 1;

    if (direction == SCROLLUP) {
	while ((i <= 18) && (table->lastvisible != table->tail)) {
	    i++;
	    scrollethwin(table, direction, idx);
	}
    } else {
	while ((i <= 18) && (table->firstvisible != table->head)) {
	    i++;
	    scrollethwin(table, direction, idx);
	}
    }
}


void hostmon(int logging, time_t logspan)
{
    int fd;
    struct ethtab table;
    struct ethtabent *entry;
    struct sockaddr_pkt fromaddr;

    int br;
    char buf[8192];
    unsigned int idx = 1;
    int is_ip;
    int ch;
    int endloop = 0;
    time_t starttime;
    time_t now = 0;
    time_t statbegin = 0, startlog = 0;

    FILE *logfile = NULL;

    int nomem = 0;

    hostmonhelp();

    initethtab(&table);

    if (logging) {
	logfile = opentlog();

	if (logfile == NULL)
	    logging = 0;
    }
    writelog(logging, logfile, "******** Ethernet traffic monitor started");
    open_socket(&fd);

    starttime = statbegin = startlog = time((time_t *) NULL);

    do {
	getpacket(fd, buf, &fromaddr, &ch, &br, 5, table.tabwin);

	if (ch != ERR)
	    switch (ch) {
	    case KEY_UP:
		scrollethwin(&table, SCROLLDOWN, &idx);
		break;
	    case KEY_DOWN:
		scrollethwin(&table, SCROLLUP, &idx);
		break;
	    case KEY_PPAGE:
		pageethwin(&table, SCROLLDOWN, &idx);
		break;
	    case KEY_NPAGE:
		pageethwin(&table, SCROLLUP, &idx);
		break;
	    case 'q':
	    case 'Q':
	    case 'x':
	    case 'X':
	    case 24:
		endloop = 1;
	    }
	if (br > 0) {
	    if (fromaddr.spkt_family == ARPHRD_ETHER) {
		if (fromaddr.spkt_protocol == htons(ETH_P_IP))
		    is_ip = 1;
		else
		    is_ip = 0;

		/*
		 * Check source address entry
		 */

		entry = in_ethtable(&table, ((struct ethhdr *) buf)->h_source);

		if ((entry == NULL) && (!nomem))
		    entry = addethentry(&table, ((struct ethhdr *) buf)->h_source, &nomem);

		if (entry != NULL) {
		    updateethent(entry, br, is_ip, 1);
		    if (!entry->prev_entry->un.desc.printed)
			printethent(&table, entry->prev_entry, idx);

		    printethent(&table, entry, idx);
		}
		/*
		 * Check destination address entry
		 */

		entry = in_ethtable(&table, ((struct ethhdr *) buf)->h_dest);
		if ((entry == NULL) && (!nomem))
		    entry = addethentry(&table, ((struct ethhdr *) buf)->h_dest, &nomem);

		if (entry != NULL) {
		    updateethent(entry, br, is_ip, 0);
		    if (!entry->prev_entry->un.desc.printed)
			printethent(&table, entry->prev_entry, idx);

		    printethent(&table, entry, idx);
		}
	    }
	}
	now = time((time_t *) NULL);

	if ((now - starttime) >= 5) {
	    printelapsedtime(statbegin, now, 22, 15, table.borderwin);
	    updateethrates(&table, starttime, now, idx);
	    starttime = now;
	}
	if (((now - startlog) >= logspan) && (logging)) {
	    writeethlog(table.head, now - statbegin, logfile);
	    startlog = now;
	}
	update_panels();
	doupdate();
    } while (!endloop);

    if (logging) {
	writeethlog(table.head, time((time_t *) NULL) - statbegin, logfile);
	writelog(logging, logfile, "******** Ethernet traffic monitor stopped");
	fclose(logfile);
    }
    del_panel(table.tabpanel);
    delwin(table.tabwin);
    del_panel(table.borderpanel);
    delwin(table.borderwin);
    update_panels();
    doupdate();
    destroyethtab(&table);
}
