#
# 1.0 (Feb 1995)
#
# $Id: Canvas.w,v 1.2 1996-11-29 09:41:21+01 mho Exp $

@class XfwfCanvas (XfwfBoard) @file=Canvas

@ The Canvas widget is used as like the |core| widget. It adds only a
handling for the |backing_store| window attribute.

@public

@ Using |backingStore| it is possible to change the |backing_store| window
attribute of the widget.

	@var int backingStore = NotUseful

@ |scrollDelay| specifies how long scrolling done by |XfwfScrollCanvas|
should be delayed. This is especially intresting, if the redrawing takes
a lot of time.

	@var int scrollDelay = 200

@ |compressScrolling| specifieshow much calls to |XfwfScrollCanvas| should be
compressed to one call to the |expose| method.

	@var int compressScrolling = 10

@private

@ Since I compress the scrolling, I must store, which part of the canvas
was exposed. But I do it the other way: I store, which part of the canvas is
not to be exposed and subtract the region from the whole canvas. This is
easier to process and needs less memory.

	@var int vx
	@var int vy
	@var int vw
	@var int vh

@ XCopyArea generates |graphics_expose| events. I don't like them to be
delivered during the compression of scrolling events. For this I have to
keep them until the scrolling is done.

	@var Region graphicsExposeRegion

@ |noScrolls| counts the calls to |XfwfScrollCanvas|.

	@var float noScrolls

@ |expose| is called indirectly by |XfwfScrollCanvas| to allow the accumulation
of scrollevents to make redrawing faster. The id of the time out function is
stored in |timer|. 

	@var XtIntervalId timer

@classvars

@ The Core variable |compress_exposure| is OR'ed with
|XtExposeGraphicsExpose|, in order to get graphics expose events delivered
to the |expose| method.

	@var compress_exposure = XtExposeCompressMaximal | XtExposeGraphicsExpose

@utilities

@ For debugging purposes only.

@def visible_area(s) =
{
    printf("%s Visible area %3d %3d size %3d %3d\n", s, $vx, $vy, $vw, $vh);
}

@def cleared_rectangles(s) =
{
    while (i > 0) {
	--i;
	printf("%s      cleared %3d %3d size %3d %3d\n", s, r[i].x, r[i].y, r[i].width, r[i].height);
    }
}

@def print_region(s, REG) =
{
    XClipBox(REG, &r);
    printf("  %s: %3d %3d size %3d %3d\n", s, r.x, r.y, r.width, r.height);
   
}

@exports

@ The |XfwfScrollCanvas| function is used to move the canvas by a specified
amount |(dx,dy)|. The obscured part of the canvas is cleared using the |GC bg|.
|expose| is called automatically.

@proc XfwfScrollCanvas($, int dx, int dy, GC bg)
{
    int        oldx, oldy, oldw, oldh, srcx, srcy;
    XRectangle r[2];
    int        i = 0;

    if (! XtIsSubclass($, xfwfCanvasWidgetClass))
	XtError("XfwfScrollCanvas called with incorrect widget type");

    /* old position and size of the region */
    srcx = oldx = $vx; srcy = oldy = $vy; oldw = $vw; oldh = $vh;
    /* no scrolling to be done? */
    if ( !(dx || dy) || !oldw || !oldh )
	return;
    /* increase numbers of done scrolls weighted with delta */
    $noScrolls += 1.0 + (fabs(dx) * 10.0 / $width) + (fabs(dy) * 10.0 / $height);
    /* compute unobscured rectangle after scrolling and rectangles to clear */
    if (dx > 0) {
	$vx = min($vx + dx, $width); $vw = max($vw - dx, 0);
	r[i].x = oldx; r[i].y = oldy; r[i].width = dx; r[i].height = oldh;
	++i;
    } else if (dx < 0) {
        $vw = max($vw + dx, 0); srcx -= dx;
	r[i].x = $vx+$vw; r[i].y = oldy; r[i].width = -dx; r[i].height = oldh;
	++i;
    }
    if (dy > 0) {
	$vy = min($vy + dy, $height); $vh = max($vh - dy, 0);
	r[i].x = oldx; r[i].y = oldy; r[i].width = oldw; r[i].height = dy;
	++i;
    } else if (dy < 0) {
	$vh = max($vh + dy, 0); srcy -= dy;
	r[i].x = oldx; r[i].y = $vy+$vh; r[i].width = oldw; r[i].height = -dy;
	++i;
    }
    /* copy region and clear obscured rectangles */
    if ($vw && $vh)
	XCopyArea(XtDisplay($), XtWindow($), XtWindow($), DefaultGCOfScreen(XtScreen($)),
		  srcx, srcy, $vw, $vh, $vx, $vy);
    XFillRectangles(XtDisplay($), XtWindow($), bg, r, i);
    /* the $graphicsExposeRegion has to shifted by delta */
    if ( !XEmptyRegion($graphicsExposeRegion) )
	XOffsetRegion($graphicsExposeRegion, dx, dy);
    /* redraw exposed region */
    if ($noScrolls < $compressScrolling) {
	/* delay exposing */
	if ($timer) XtRemoveTimeOut($timer);
	$timer = XtAppAddTimeOut(XtWidgetToApplicationContext($), $scrollDelay, do_expose, $);
    } else {
	/* expose immediately */
	if ($timer) XtRemoveTimeOut($timer);
	do_expose($, NULL);
    }
}

@utilities

@ Store rectangle inside newly created region.

@def rect_to_region(X, Y, WIDTH, HEIGHT, REG) =
{
    r.x      = X;
    r.y      = Y;
    r.width  = WIDTH;
    r.height = HEIGHT;
    REG      = XCreateRegion();
    XUnionRectWithRegion(&r, REG, REG);
}

@ |do_expose| is either called as a time out function or immediately by
|XfwfScrollCanvas|. The exposing is delayed to make cumulative scrolling faster.

@proc do_expose(XtPointer client_data, XtIntervalId *id)
{
    Widget       $ = (Widget)client_data;
    XExposeEvent dummy_event;
    Region       expose_region, visible_region;
    XRectangle   r;

    /* re-initialize $timer and $noScrolls */
    $timer     = 0;
    $noScrolls = 0;
    /* compute region to be exposed: canvas - visible + graphics_expose */
    rect_to_region(0, 0, $width, $height, expose_region);
    rect_to_region($vx, $vy, $vw, $vh, visible_region);
    XSubtractRegion(expose_region, visible_region, expose_region);
    XUnionRegion(expose_region, $graphicsExposeRegion, expose_region);
    dummy_event.type	  = Expose;
    dummy_event.display	  = XtDisplay($);
    dummy_event.send_event = True;
    dummy_event.window	  = XtWindow($);
    dummy_event.count	  = 0;
    dummy_event.x	  = 0;
    dummy_event.y	  = 0;
    dummy_event.width	  = $width;
    dummy_event.height	  = $height;
    $expose($, (XEvent*)&dummy_event, expose_region);
    /* destroy/clear region structures */
    XDestroyRegion(expose_region);
    XDestroyRegion(visible_region);
    XDestroyRegion($graphicsExposeRegion);
    $graphicsExposeRegion = XCreateRegion();
    /* now everything is visible again */
    $vx = $vy = 0; $vw = $width; $vh = $height;
}

@methods

@ |initialize| and |destroy| handle the private variables of the canvas widget.

@proc initialize
{
    $noScrolls = 0;
    $timer = 0;
    $vx = 0;
    $vy = 0;
    $vw = $width;
    $vh = $height;
    $graphicsExposeRegion = XCreateRegion();
}

@proc destroy
{
    /* destroy current custom expose region */
    if ($graphicsExposeRegion)
	XDestroyRegion($graphicsExposeRegion);
    $graphicsExposeRegion = 0;
    if ($timer)
	XtRemoveTimeOut($timer);
    $timer = 0;
}

@ If |expose| is called during the canvas is scrolled and the exposing is
delayed, add the region - if there is any - to |graphicsExposeRegion|. This is
especially for catching GraphicsExpose events. NoExpose events are NOT caught!

@proc expose
{
    if ($noScrolls) {
	/* I'm scrolling */
	if (region)
	    /* I have a region */
	    XUnionRegion(region, $graphicsExposeRegion, $graphicsExposeRegion);
	else
	    /* take region from event */
	    XtAddExposureToRegion(event, $graphicsExposeRegion);
    } else {
	/* "normal" expose */
	#expose($, event, region);
    }
}

@ The |realize| changes the |backing_store| attribute of the realized window.

@proc realize
{
    if (($backingStore == Always)
    ||  ($backingStore == NotUseful)
    ||  ($backingStore == WhenMapped)) {
	*mask |= CWBackingStore;
	attributes->backing_store = $backingStore;
    } else {
	*mask &= ~CWBackingStore;
    }
    /* chain to parent method */
    #realize($, mask, attributes);
}

@ |resize| is needed to keep the visible size of the canvas. It is improbable
that a resize will occure during the scrolling is processed.

@proc resize
{
    $vx = 0;
    $vy = 0;
    $vw = $width;
    $vh = $height;
}

@ The |set_values| method has to deal with changes in |backing_store|.

@proc set_values
{
    if ($old$backingStore != $backingStore) {
	if (($backingStore == Always)
	||  ($backingStore == NotUseful)
	||  ($backingStore == WhenMapped)) {
	    XSetWindowAttributes attributes;
	    unsigned long	 mask = CWBackingStore;

	    attributes.backing_store = $backingStore;
	    XChangeWindowAttributes(XtDisplay($), XtWindow($), mask, &attributes);
	}
    }
    return FALSE; /* there is no need to redraw */
}

@imports

@incl <stdio.h>
@incl <math.h>
