// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _Connection_h_
#define _Connection_h_

#include <qobject.h>
#include "Url.h"

//: The Connection class is the abstract base class for any type of data retrieval class.
//. The Connection class provides the interface, and some common functionality,
//. for all of the classes that retrieve data.
class Connection : public QObject {
    Q_OBJECT
protected:
    Url  _url;
    int  _fd;
    int  _size;
    char _bytes[1024];
public:
    //. Create a new connection.
    Connection( const Url& url );

    //. Abort the connection, if it is still active.
    virtual ~Connection();

    //. Establish a new connection.
    virtual bool open() = 0;

    void guessFileType( QString& mediaType, QString& mediaSubtype );
public slots:
    //. Close the current connection immediately.
    virtual void abort();

    //. Process an exception on the file descriptor.
    virtual void exception();

    //. Process notification from the Poller that there is data to be read
    //. from the file descriptor.
    virtual void readable();

    //. Process notification from the Poller that data can be written to the
    //. file descriptor.
    virtual void writable();
signals:
    //. This signal is emitted whenever any data is received.  There can be
    //. many data signals emitted between the startOfData and endOfData
    //. signals.
    virtual void data( const char* bytes, int length ) = 0;

    //. This signal is emitted after all data has been received, and the
    //. connection is closed.
    virtual void endOfData() = 0;

    //. This signal is emitted after the connection has determined the Internet
    //. media type and subtype of the data stream.  The total expected size of
    //. the data stream is also provided, but this may not always be known.
    virtual void startOfData( QString mediaType, QString mediaSubtype, int totalSize ) = 0;

    //. This signal is emitted by the connection to update the browser window
    //. with changes in the connection status.  The types of status messages
    //. are data being recieved, the connection being opened, etc.
    virtual void status( QString msg );

    //. This signal is emitted when the URL that is being retrieved changes.
    //. The URL may change as the Connection follows HTTP redirects.
    virtual void urlChanged( const Url& url );
};

#endif
