// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
extern "C" {
#include <stdio.h>
}
#include <qapp.h>
#include <qbitmap.h>
#include <qframe.h>
#include <qkeycode.h>
#include <qlabel.h>
#include <qlined.h>
#include <qmenubar.h>
#include <qmsgbox.h>
#include <qpixmap.h>
#include <qpopmenu.h>
#include <qpushbt.h>
#include <qtooltip.h>
#include "qweb.h"
#include "Bookmarks.h"
#include "Browser.h"
#include "BrowserList.h"
#include "Cache.h"
#include "Canvas.h"
#include "ConsoleWindow.h"
#include "DownloadRenderer.h"
#include "DtdManager.h"
#include "History.h"
#include "Options.h"
#include "PrefsWindow.h"
#include "SgmlCatalog.h"
#include "SgmlParser.h"
#include "SgmlRenderer.h"
#include "StyleSelect.h"
#include "StyleEditor.h"
#include "StyleSheetManager.h"
#include "TransferLogo.h"

#include "back.h"
#include "backmask.xbm"
#include "forw.h"
#include "forwmask.xbm"
#include "home.h"
#include "homemask.xbm"
#include "reld.h"
#include "reldmask.xbm"
#include "stop.h"
#include "stopmask.xbm"

//=============================================================================
// Public methods.
//-----------------------------------------------------------------------------
Browser::Browser( QWidget* parent, const char* name )
        : QWidget( parent, name ),
          _baseUrl( "" ),
          _statusTimerId( 0 )
{
    setCaption( "QWeb" );
    setIconText( "QWeb" );
    
    setFocus();
    setFocusPolicy( QWidget::StrongFocus );
    
    QPixmap p;

    QFont menuFont( options->menuFontName(),
                    options->menuFontSize(),
                    options->menuFontWeight(),
                    options->menuFontItalic() );
    
    _file = new QPopupMenu;
    _file->setFont( menuFont );
    _file->insertItem( "New Browser...", this, SLOT( doNewBrowser() ) );
    _file->insertItem( "Save As..."    , this, SLOT( doSaveAs() ) );
    _file->insertSeparator();
    _file->insertItem( "Close"         , this, SLOT( doClose() ) );
    _file->insertItem( "Exit"          , qApp, SLOT( quit() ), ALT+Key_Q );

    _go = new QPopupMenu;
    _go->setFont( menuFont );
    connect( _go, SIGNAL( activated( int ) ), this, SLOT( goActivated( int ) ) );
    _go->insertItem( "&Back"   , this, SLOT( back() )   , ALT+Key_Left );
    _go->insertItem( "&Forward", this, SLOT( forward() ), ALT+Key_Right );
    _go->insertItem( "&Home"   , this, SLOT( home() ) );
    _go->insertSeparator();

    QPopupMenu* opts = new QPopupMenu;
    opts->setFont( menuFont );
    opts->insertItem( "Clear Document Cache"  , this, SLOT( optionsClearDocumentCache() ) );
    opts->insertItem( "Clear DTD Cache"       , this, SLOT( optionsClearDtdCache() ) );
    opts->insertSeparator();
    opts->insertItem( "Save Preferences"      , this, SLOT( optionsSavePreferences() ) );
    opts->insertItem( "Save SGML Catalog"     , this, SLOT( optionsSaveSgmlCatalog() ) );

    QPopupMenu* windows = new QPopupMenu;
    windows->setFont( menuFont );
    windows->insertItem( "Preferences" , this, SLOT( windowsPreferences() ) );
    windows->insertItem( "Style Editor", this, SLOT( windowsStyleEditor() ) );
    windows->insertItem( "SGML Catalog", this, SLOT( windowsSgmlCatalog() ) );
    windows->insertItem( "Console"     , this, SLOT( windowsConsole() ) );

    QPopupMenu* about = new QPopupMenu;
    about->setFont( menuFont );
    about->insertItem( "About QWeb", this, SLOT( aboutAboutQweb() ) );

    _menu = new QMenuBar( this );
    _menu->setFont( menuFont );
    _menu->insertItem( "&File"     , _file );
    _menu->insertItem( "&Go"       , _go );
    _menu->insertItem( "&Bookmarks", bookmarks->menu() );
    _menu->insertItem( "&Options"  , opts );
    _menu->insertItem( "&Windows"  , windows );
    _menu->insertSeparator();
    _menu->insertItem( "&Help"    , about );

    _back = new QPushButton( this );
    p.loadFromData( back_bmp_data, back_bmp_len );
    p.setMask( QBitmap( backmask_width, backmask_height, backmask_bits, TRUE ) );
    _back->setPixmap( p );
    connect( _back, SIGNAL( clicked() ), this, SLOT( back() ) );
    QToolTip::add( _back, "Visit the Previous Document" );

    _forward = new QPushButton( this );
    p.loadFromData( forw_bmp_data, forw_bmp_len );
    p.setMask( QBitmap( forwmask_width, forwmask_height, forwmask_bits, TRUE ) );
    _forward->setPixmap( p );
    connect( _forward, SIGNAL( clicked() ), this, SLOT( forward() ) );
    QToolTip::add( _forward, "Visit the Next Document" );

    _home = new QPushButton( this );
    p.loadFromData( home_bmp_data, home_bmp_len );
    p.setMask( QBitmap( homemask_width, homemask_height, homemask_bits, TRUE ) );
    _home->setPixmap( p );
    connect( _home, SIGNAL( clicked() ), this, SLOT( home() ) );
    QToolTip::add( _home, "Visit the Home Document" );

    _reload = new QPushButton( this );
    p.loadFromData( reld_bmp_data, reld_bmp_len );
    p.setMask( QBitmap( reldmask_width, reldmask_height, reldmask_bits, TRUE ) );
    _reload->setPixmap( p );
    connect( _reload, SIGNAL( clicked() ), this, SLOT( doReload() ) );
    QToolTip::add( _reload, "Reload the Current Document" );

    _logo = new TransferLogo( this );
    
    _url = new QLineEdit( this );
    _url->setText( options->homeUrl() );

    _stop = new QPushButton( this );
    p.loadFromData( stop_bmp_data, stop_bmp_len );
    p.setMask( QBitmap( stopmask_width, stopmask_height, stopmask_bits, TRUE ) );
    _stop->setPixmap( p );
    _stop->setEnabled( FALSE );
    QToolTip::add( _stop, "Abort the Current Transfer" );

    _separator1 = new QFrame( this );
    _separator1->setFrameStyle( QFrame::HLine | QFrame::Sunken );
	
    _canvas = new Canvas( this );
	
    _separator2 = new QFrame( this );
    _separator2->setFrameStyle( QFrame::HLine | QFrame::Sunken );
	
    _status = new QLabel( "Status:", this );
    _status->setFont( QFont( "helvetica", 12 ) );

    connect( _url , SIGNAL( returnPressed() ), this   , SLOT( open() ) );
    connect( _stop, SIGNAL( clicked() )      , _canvas, SLOT( stop() ) );

    _history = new History( _go );
    canGoBack( FALSE );
    canGoForward( FALSE );
    connect( _history, SIGNAL( canGoBack( bool ) )   , this, SLOT( canGoBack( bool ) ) );
    connect( _history, SIGNAL( canGoForward( bool ) ), this, SLOT( canGoForward( bool ) ) );

    connect( bookmarks, SIGNAL( bookmarksChanged() ), this, SLOT( bookmarksChanged() ) );
    
    resize( 650, 650 );
}

Browser::~Browser()
{
    _canvas->stop();
    delete _canvas;
}

void Browser::setTitle( QString title )
{
    // Trim the title string to 64 chars max.
    QString foo = title.copy();
    if ( foo.length() > 64 ) {
        foo.truncate( 61 );
        foo += "...";
    }
    
    _history->setTitle( foo );
    QString tmp( "QWeb: " );
    tmp += foo;
    setCaption( tmp );
    setIconText( tmp );
}

void Browser::urlChanged( const Url& url )
{
    setBaseUrl( url );
    _url->setText( url.url() );
}

void Browser::setBaseUrl( const Url& url )
{
    _baseUrl = url;
}

//=============================================================================
// Public slots.
//-----------------------------------------------------------------------------
void Browser::open()
{
    _canvas->stop();
    Url url( _url->text() );
    open( url );
}

void Browser::open( const Url& url, bool reload )
{
    _reloadFlag = reload;
    _dtdName = "";
    setCaption( "QWeb: <Untitled>" );
    setIconText( "QWeb: <Untitled>" );

    if ( !reload &&
         ( _baseUrl.method()     == url.method()     ) &&
         ( _baseUrl.user()       == url.user()       ) &&
         ( _baseUrl.password()   == url.password()   ) &&
         ( _baseUrl.hostname()   == url.hostname()   ) &&
         ( _baseUrl.path()       == url.path()       ) &&
         ( _baseUrl.query()      == url.query()      ) &&
         ( _baseUrl.parameters() == url.parameters() ) &&
         ( _baseUrl.port()       == url.port()       ) ) {
        // Same document, possibly different fragment.
        urlChanged( url );
        _canvas->redraw();
        return;
    }

    urlChanged( url );
    Request* request = new Request( _baseUrl, _canvas );
    connect( request, SIGNAL( startOfData( Request *, QString, QString, int ) ), _canvas , SLOT( startOfData( Request*, QString, QString, int ) ) );
    connect( request, SIGNAL( status( QString ) )                              , this    , SLOT( status( QString ) ) );
    connect( request, SIGNAL( urlChanged( const Url& ) )                       , this    , SLOT( urlChanged( const Url& ) ) );
    
    if ( !request->open() ) {
        delete request;
    }
}

void Browser::startOfData()
{
    if ( ( !_history->current() ) || ( _history->current()->url() != _baseUrl.url() ) ) {
        _history->open( _baseUrl );
    }
}

void Browser::status( QString msg )
{
    _status->setText( msg );
    _status->repaint();
    qApp->flushX();
    if ( _statusTimerId ) {
        killTimer( _statusTimerId );
    }
    _statusTimerId = startTimer( 5000 );
}

void Browser::setTransfer( bool b )
{
    _stop->setEnabled( b );
    _logo->setTransfer( b );
}

void Browser::stop()
{
    _canvas->stop();
}

void Browser::doReload()
{
    const Url* url = _history->current();
    if ( url ) {
        open( *url, TRUE );
    }
}

void Browser::home()
{
    open( options->homeUrl() );
}

void Browser::back()
{
    const Url* url = _history->back();
    if ( url ) {
        open( *url );
    }
}

void Browser::forward()
{
    const Url* url = _history->forward();
    if ( url ) {
        open( *url );
    }
}

void Browser::goActivated( int id )
{
    const Url* url = _history->go( id - 4 );
    if ( url ) {
        open( *url );
    }
}

void Browser::canGoBack( bool b )
{
    _go->setItemEnabled( 0, b );
    _back->setEnabled( b );
}

void Browser::canGoForward( bool b )
{
    _go->setItemEnabled( 1, FALSE );
    _forward->setEnabled( b );
}

void Browser::closeEvent( QCloseEvent* )
{
    delete this;
}

void Browser::resizeEvent( QResizeEvent* )
{
    int menuHeight   = _menu->height();
    int pad          = 4;
    int height       = 24;
    int buttonHeight = 40;
    int buttonWidth  = 40;
    int urlWidth     = width() - 2 * pad;
    int renderHeight = this->height() - 2 * height - 5 * pad - buttonHeight - menuHeight;

    _back->setGeometry( pad, menuHeight + pad, buttonWidth, buttonHeight );
    _forward->setGeometry( 2*pad + buttonWidth, menuHeight + pad, buttonWidth, buttonHeight );
    _home->setGeometry( 3*pad + 2*buttonWidth, menuHeight + pad, buttonWidth, buttonHeight );
    _reload->setGeometry( 4*pad + 3*buttonWidth, menuHeight + pad, buttonWidth, buttonHeight );
    _stop->setGeometry( 7*pad + 4*buttonWidth, menuHeight + pad, buttonWidth, buttonHeight );

    _logo->setGeometry( width() - pad - 100, menuHeight + pad, 100, 40 );

    _url->setGeometry( pad, menuHeight + pad + buttonHeight + pad, urlWidth, height );

    _separator1->setGeometry( 0, menuHeight + height + pad + buttonHeight + pad, width(), pad );

    _canvas->setGeometry( pad, menuHeight + height + 2 * pad + buttonHeight + pad, width() - 2 * pad, renderHeight );

    _separator2->setGeometry( 0, menuHeight + height + 2 * pad + renderHeight + buttonHeight + pad, width(), pad );

    _status->setGeometry( pad, menuHeight + 3 * pad + height + renderHeight + buttonHeight + pad, width() - 2 * pad, height );
}

void Browser::timerEvent( QTimerEvent* e )
{
    if ( e->timerId() == _statusTimerId ) {
        QString blank;
        status( blank );
        killTimer( _statusTimerId );
        _statusTimerId = 0;
    }
}

void Browser::enterEvent( QEvent* )
{
    bookmarks->activeBrowser( this );
}

void Browser::doNewBrowser()
{
    Browser* b = new Browser;
    browserList->browserOpened( b );
    b->show();
}

void Browser::doClose()
{
    delete this;
}

void Browser::canClose( bool b )
{
    _file->setItemEnabled( 2, b );
}

void Browser::errDialog( QString error )
{
    QMessageBox msgBox( this );
    msgBox.setCaption( "QWeb: Error" );
    msgBox.setText( error );
    msgBox.show();
}

void Browser::optionsClearDtdCache()
{
    dtdManager->clear();
}

void Browser::optionsClearDocumentCache()
{
    cache->clear();
}

void Browser::optionsSavePreferences()
{
    options->save();
}

void Browser::optionsSaveSgmlCatalog()
{
    sgmlCatalog->save();
}

void Browser::dtdSelected( QString dtdName )
{
    _dtdName = dtdName.copy();
}

void Browser::bookmarksChanged()
{
    _menu->removeItemAt( 2 );
    _menu->insertItem( "&Bookmarks", bookmarks->menu(), -1, 2 );
}

void Browser::doSaveAs()
{
    DownloadRenderer* dr = new DownloadRenderer( baseUrl() );
    dr->show();
}

void Browser::aboutAboutQweb()
{
    QMessageBox about;
    about.setCaption( "QWeb: About QWeb" );
    about.setIconText( "QWeb: About QWeb" );
    QString text;
    text.sprintf( "QWeb %s Copyright 1997\nSean Vyain\n\nsvyain@mail.tds.net, smvyain@softart.com\n\n\nContributors\n\nMarkus Demleitner", QWEB_VERSION_STRING );
    about.setText( text );
    about.show();
}

void Browser::windowsPreferences()
{
    prefsWindow->show();
}

void Browser::windowsStyleEditor()
{
    styleEditor->show();
}

void Browser::windowsSgmlCatalog()
{
    sgmlCatalog->show();
}

void Browser::windowsConsole()
{
    console->show();
}
