/*--------------------------------*-C-*---------------------------------*
 * Copyright 1992 John Bovey, University of Kent at Canterbury.
 *
 * You can do what you like with this source code as long as
 * you don't try to make money out of it and you include an
 * unaltered copy of this message (including the copyright).
 *
 * This module has been very heavily modified by R. Nation
 * (nation@rocket.sanders.lockheed.com).
 * No additional restrictions are applied
 *
 * Additional modification by Garrett D'Amore (garrett@netcom.com) to
 * allow vt100 printing.  No additional restrictions are applied.
 *
 * Integrated modifications by Steven Hirsch (hirsch@emba.uvm.edu) to
 * properly support X11 mouse report mode and support for DEC
 * "private mode" save/restore functions.
 *
 * Integrated key-related changes by Jakub Jelinek (jj@gnu.ai.mit.edu)
 * to handle Shift+function keys properly.
 * Should be used with enclosed termcap / terminfo database.
 *
 * Additional modifications by mj olesen <olesen@me.queensu.ca>
 * No additional restrictions.
 *
 * Further modification and cleanups for Solaris 2.x and Linux 1.2.x
 * by Raul Garcia Garcia (rgg@tid.es). No additional restrictions.
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 *----------------------------------------------------------------------*/
#include "rxvt.h"

#include <ctype.h>
#include <stdarg.h>
#include <signal.h>
#include <unistd.h>		/*#define HAVE_UNISTD_H*/
#include <fcntl.h>		/*#define HAVE_FCNTL_H*/
#include <grp.h>
#include <time.h>		/*#define TIME_WITH_SYS_TIME*/
#include <sys/time.h>		/*#define HAVE_SYS_TIME_H*/
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#ifdef SVR4
/*# define _NEW_TTY_CTRL*/
# include <sys/stropts.h>
# include <sys/resource.h>
#endif
#ifdef AIXV3
# include <sys/select.h>
# include <sys/ioctl.h>		/*#define HAVE_SYS_IOCTL_H*/
#endif /* AIXV3 */

#include <termios.h>		/*#define HAVE_TERMIOS_H*/

#ifdef __alpha
# define FREEBSD
#endif
#ifdef FREEBSD
# include <sys/ioctl.h>
#endif

#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "screen.h"
#include "sbar.h"
#include "xsetup.h"
#include "command.h"
#include "graphics.h"
#include "grkelot.h"
#include "rxvtkeys.h"

/* the fastest possible baudrate */
#ifdef B38400
# define FAST_BAUD	B38400
#else
# define FAST_BAUD	B9600
#endif

#ifndef MAX_REFRESH_PERIOD
# define MAX_REFRESH_PERIOD	10
#endif
#if defined (MULTIPLE_CLICKS) && !defined (MULTICLICK_TIME)
# define MULTICLICK_TIME	500
#endif

#define PAGE_AMOUNT	0.8	/* amount for page scrolling */

/*#define DEBUG_TTY*/
/*#define DEBUG_KEY*/
/*----------------------------------------------------------------------*
 * extern variables referenced
 */

/*----------------------------------------------------------------------*
 * extern variables declared here
 */
#ifdef NO_7BIT_MODE
#define	hibit_mask	0xFF	/* an appropriate definition */
#else
unsigned char hibit_mask = 0xFF;
#endif
#ifdef NO_REFRESH_RESOURCE
#define refresh_period	MAX_REFRESH_PERIOD
#else
int refresh_period = MAX_REFRESH_PERIOD;
#endif

#ifdef KEYSYM_RESOURCE
unsigned char * KeySym_map [256];	/* probably mostly empty */
#endif

/*----------------------------------------------------------------------*
 * local variables
 */
static int comm_fd = -1;	/* file descriptor connected to the command */
static int comm_pid = -1;	/* process id if child */
static int x_fd;		/* file descriptor of X server connection */
static int fd_width;		/* width of file descriptors being used */
static struct stat ttyfd_stat;	/* original status of the tty we will use */
#ifdef BACKSPACE_AS_DELETE
static int bs_is_bs = 0;	/* backspace is backspace */
#else
static int bs_is_bs = 1;
#endif

static char *ttynam;		/* tty name */

static int appCUR = 0;		/* application cursor keys */
static int appKP = 1;		/* application keypad */
static int refresh_nl_count = 0;
static int refresh_nl_limit = 1;
static int refresh_type = SLOW_REFRESH;

static Atom wm_del_win;

#ifndef NO_MOUSE_REPORT
#define X10_MOUSE	1
#define X11_MOUSE	2
/* preliminary code for highlight tracking, but it's too annoying */
/*#define X11_TRACKING	4*/
/*#define X11_HILITE	8*/
static int Mouse_Report = 0;
/*static int wait_for_reply = 0;*/
#endif
/* buffered command input */
static unsigned char com_buf [2048], *com_buf_next, *com_buf_top;

/*----------------------------------------------------------------------*
 * extern functions referenced
 */

/*----------------------------------------------------------------------*
 * local functions
 */
static void	catch_child (int);
static void	catch_sig (int);
static void	set_ttymodes (void);
static char *	get_ptty (int *pfd);
static int	get_tty (void);
static int	run_command (char **argv);
static void	lookup_key (XEvent *);
static unsigned char get_com_char (void);
static void	get_X_event (void);
/*static void	process_string (int);*/
#ifdef PRINT_PIPE
static void	process_print_pipe (void);
#endif
static void	process_escape_seq (void);
static void	process_csi_seq (void);
static void	process_xterm_seq (void);
static void	process_terminal_mode (unsigned char mode, unsigned char priv,
				       int nargs, int arg[]);
static void	process_sgr_mode (int nargs, int arg[]);
static void	process_robs_seq (void);

static void	stty_size (int fd, int width, int height);

/*----------------------------------------------------------------------*/
#define VT100_ANS	"\033[?1;2c"	/* vt100 answerback */
#ifdef ESCZ_RXVT_ANSWER		/* what to do with obsolete ESC-Z */
# define RXVT_ANS	"\033[?1;2C"	/* almost the same */
#else
# define RXVT_ANS	VT100_ANS	/* obsolete ANSI ESC[c */
#endif
/*
 * ESC-Z processing:
 *
 * By stealing a sequence to which other xterms respond, and sending the
 * same number of characters, but having a distinguishable sequence,
 * we can avoid having a timeout (when not under an rxvt) for every login
 * shell to auto-set its DISPLAY.
 *
 * This particular sequence is even explicitly stated as obsolete since
 * about 1985, so only very old software is likely to be confused, a
 * confusion which can likely be remedied through termcap or TERM.
 * Frankly, I doubt anyone will even notice.  We provide a #ifdef just
 * in case they don't care about auto-display setting.  Just in case the
 * ancient software in question is broken enough to be case insensitive
 * to the 'c' character in the answerback string, we make the
 * distinguishing characteristic be capitalization of that character.
 * The length of the two strings should be the same so that identical
 * read(2) calls may be used.
 */

#define KBUFSZ		8	/* size of keyboard mapping buffer */
#define STRING_MAX	128	/* max string size for process_xterm_seq() */
#define ESC_ARGS	20	/* max # of args for esc sequences */

/* Terminal mode structures. */

#if !defined (FREEBSD) && !defined (CINTR)
# ifndef _POSIX_VDISABLE
#  define _POSIX_VDISABLE 0
# endif
# ifdef SVR4
#  define CINTR		'\177'		/* ^? */
#  define CQUIT		'\025'		/* ^U */
#  define CERASE	'\010'		/* ^H */
# else
#  define CINTR		'\003'		/* ^C */
#  define CQUIT		'\034'		/* ^\ */
#  define CERASE	'\177'		/* ^? */
# endif
# define CEOF		'\004'		/* ^D */
# define CEOL		_POSIX_VDISABLE
# define CEOL2		_POSIX_VDISABLE
# define CNSWTCH	_POSIX_VDISABLE
# define CDSUSP		_POSIX_VDISABLE
# define CFLUSH		'\017'		/* ^O */
# define CRPRNT		'\022'		/* ^R */
# define CSTART		'\021'		/* ^Q */
# define CSTOP		'\023'		/* ^S */

# define CKILL		'\025'		/* ^U */
# define CLNEXT		'\026'		/* ^V */
# define CWERASE	'\027'		/* ^W */
# define CSUSP		'\032'		/* ^Z */
#endif

#ifndef CEOL
# define CEOL _POSIX_VDISABLE
#endif

#ifdef SVR4
static int
getdtablesize (void)
{
#ifdef HPUX
   return sysconf (_SC_OPEN_MAX);
#else
   struct rlimit rlp;
   getrlimit (RLIMIT_NOFILE, &rlp);
   return rlp.rlim_cur;
#endif
}
#endif	/* SVR4 */

/* Catch a SIGCHLD signal and exit if the direct child has died */
static void
catch_child (int unused)
{
   if (wait (NULL) == comm_pid)
     exit (EXIT_SUCCESS);
   signal (SIGCHLD, catch_child);	/* rgg */
}

/* Catch a fatal signal and tidy up before quitting */
static void
catch_sig (int sig)
{
   signal (sig, SIG_DFL);
   cleanutent ();
   setuid (getuid ());
   kill (getpid (), sig);
}

/* Exit gracefully, clearing the utmp entry and restoring tty attributes */
static void
clean_exit (void)
{
   /* #ifndef SVR4 -- generalized -- rgg 24/10/95 */
#ifdef DEBUG_TTY
   fprintf (stderr, "Changing %s to mode %o, uid %d, gid %d\n",
	    ttynam, ttyfd_stat.st_mode, ttyfd_stat.st_uid, ttyfd_stat.st_gid);
#endif
   chmod (ttynam, ttyfd_stat.st_mode);
   chown (ttynam, ttyfd_stat.st_uid, ttyfd_stat.st_gid);
   /* #endif */

   cleanutent ();
}

/*
 * Acquire a pseudo-teletype from the system.
 * The return value is the name of the slave part of the pair or
 * NULL if unsucsessful.  If successful, the file descriptor is
 * returned via the argument.
 */
static char *
get_ptty (int *pfd)
{
   int ptyfd = -1;
#ifdef SVR4
   extern char *ptsname();
   char *tty;

   ptyfd = open ("/dev/ptmx",O_RDWR);
   if (ptyfd < 0)
     {
	print_error ("can't open a pseudo teletype");
	return NULL;
     }
   grantpt (ptyfd);
   unlockpt (ptyfd);
   tty = ptsname (ptyfd);
#else
   static char tty[] = "/dev/tty??";
   char *c8, *c9, pty[] = "/dev/pty??";

   for (c8 = "pqrstuvwxyz"; *c8; c8++) {
      for (c9 = "0123456789abcdef"; *c9; c9++) {
	 pty [8] = tty [8] = *c8;
	 pty [9] = tty [9] = *c9;
	 if ((ptyfd = open (pty, O_RDWR)) >= 0) {
	    if (geteuid () == 0 || access (tty, R_OK|W_OK) == 0)
	      goto Found_pty;
	    close (ptyfd);
	    ptyfd = -1;
	 }
      }
   }

   if (ptyfd < 0)
     {
	print_error ("can't open a pseudo teletype");
	return NULL;
     }
#endif
   Found_pty:
   fcntl (ptyfd, F_SETFL, O_NDELAY);

   *pfd = ptyfd;
   return tty;
}

static int
get_tty (void)
{
   int i, uid, gid, ttyfd;
   struct group *gr;

   uid = getuid ();
   if ((ttyfd = open (ttynam, O_RDWR)) < 0)
     {
	print_error ("could not open slave tty %s",ttynam);
	exit (EXIT_FAILURE);
     }
#ifdef SVR4
   ioctl (ttyfd, I_PUSH, "ptem");
   ioctl (ttyfd, I_PUSH, "ldterm");
#else
   gid = ((gr = getgrnam ("tty")) == NULL) ? -1 : gr->gr_gid;

   fchown (ttyfd, uid, gid);
   fchmod (ttyfd, 0600);
   setuid (uid);
   setgid (getgid());
#endif	/* SVR4 */
#ifdef TIOCCONS			/* become virtual console */
    if (console)
      {
 	int on = 1;
	if (ioctl (ttyfd, TIOCCONS, (unsigned char *)&on) == -1)
	  print_error ("can't open console");
     }
#endif	/* TIOCCONS */

   /*
    * Redirect critical fd's and close the rest.
    *
    * If only stdin, stdout, and stderr are closed, all children
    * processes remain alive upon deletion of the window, so it
    * is necessary to close fd's not only < 3, but all of them,
    * that is, fd's < fd_width -- rgg 10/12/95
    */
   /* for (i = 0; i < 3; i++)*/
   for (i = 0; i < fd_width; i++)
     if (i != ttyfd)
       close (i);

   dup (ttyfd);
   dup (ttyfd);
   dup (ttyfd);

   if (ttyfd > 2)
     close (ttyfd);

#ifdef ultrix
   if ((ttyfd = open ("/dev/tty", O_RDONLY)) >= 0 )
     {
	ioctl (ttyfd, TIOCNOTTY, 0);
	close (ttyfd);
     }
   else
     setpgrp (0, 0);
   /* no error, we could run with no tty to begin with */
#else	/* ultrix */
#ifdef NO_SETSID
   if ((gid = setpgrp (0, 0)) < 0)
     print_error ("failed to set process group");
#else
   if ((gid = setsid ()) < 0)
     print_error ("failed to start session");
#endif

#ifdef TIOCSCTTY
   ioctl (0, TIOCSCTTY, 0);
#endif
     {
	int pid = getpid ();
	ioctl (0, TIOCSPGRP, (char *)&pid);	/* set process group */
	close (open (ttynam, O_WRONLY, 0));
     }
#endif	/* ultrix */
   setgid (getgid ());
   setuid (uid);

   return ttyfd;
}

/*
 * Initialise the terminal attributes.
 */
static void
set_ttymodes (void)
{
#ifndef BSD
   /* Set the terminal using the standard System V termios interface */
   struct termios ttmode;

   /* ways to deal with getting/setting termios structure */

   /* HPUX uses POSIX terminal I/O */
#ifdef TCSANOW                  /* POSIX */
# define GET_TERMIOS(fd,x)      tcgetattr (fd, x)
# define SET_TERMIOS(fd,x)      tcsetattr (fd, TCSANOW, x)
#else
# ifdef TIOCSETA                /* FREEBSD */
#  define GET_TERMIOS(fd,x)     ioctl (fd, TIOCGETA, (char *)(x))
#  define SET_TERMIOS(fd,x)     ioctl (fd, TIOCSETA, (char *)(x))
# else
#  define GET_TERMIOS(fd,x)     ioctl (fd, TCGETS, (char *)(x))
#  define SET_TERMIOS(fd,x)     ioctl (fd, TCSETS, (char *)(x))
# endif
#endif

   GET_TERMIOS (0, &ttmode);	/* init termios structure */

   ttmode.c_iflag = BRKINT | IGNPAR | ICRNL| IXON;
   ttmode.c_lflag = ISIG|IEXTEN|ICANON|ECHO|ECHOE|ECHOK;

#ifdef IMAXBEL
   ttmode.c_iflag |= IMAXBEL;
#endif
#if defined ECHOCTL && defined ECHOKE
   ttmode.c_lflag |= ECHOCTL|ECHOKE;
#endif
   ttmode.c_oflag = OPOST | ONLCR;
   ttmode.c_cflag = FAST_BAUD | CS8 | CREAD;

   ttmode.c_cc [VEOF] = CEOF;
   ttmode.c_cc [VEOL] = CEOL;

   ttmode.c_cc [VINTR] = CINTR;
   ttmode.c_cc [VQUIT] = CQUIT;
   ttmode.c_cc [VERASE] = CERASE;
   ttmode.c_cc [VKILL] = CKILL;
#ifdef HPUX
   ttmode.c_cc [VSUSP] = CSWTCH;
#else
   ttmode.c_cc [VSUSP] = CSUSP;
#endif
#ifdef VDSUSP
   ttmode.c_cc [VDSUSP] = CDSUSP;
#endif
   ttmode.c_cc [VSTART] = CSTART;
   ttmode.c_cc [VSTOP] = CSTOP;
#ifdef VREPRINT
   ttmode.c_cc [VREPRINT] = CRPRNT;
#endif
#if defined (VDISCRD) && !defined (VDISCARD)
# define VDISCARD	VDISCRD
#endif
#ifdef VDISCARD
   ttmode.c_cc [VDISCARD] = CFLUSH;
#endif
#if defined (VWERSE) && !defined (VWERASE)
# define VWERASE	VWERSE
#endif
#ifdef VWERASE
   ttmode.c_cc [VWERASE] = CWERASE;
#endif
#ifdef VLNEXT
   ttmode.c_cc [VLNEXT] = CLNEXT;
#endif
#ifdef VSWTC
   ttmode.c_cc [VSWTC] = 0;
#endif

#ifdef VSWTCH
   ttmode.c_cc [VSWTCH] = 0;
#endif
#if VMIN != VEOF
   ttmode.c_cc [VMIN] = 1;
#endif
#if VTIME != VEOL
   ttmode.c_cc [VTIME] = 0;
#endif

#ifndef NO_7BIT_MODE
   if (hibit_mask == 0x7F)
     {
	ttmode.c_cflag &= ~CS8;
	ttmode.c_cflag |= CS7 | PARENB;
     }
#endif

   SET_TERMIOS (0, &ttmode);

#else /* BSD */

   /*
    * Use sgtty rather than termios interface to configure the terminal.
    */
   int ldisc, lmode;
   struct sgttyb tty;
   struct tchars tc;
   struct ltchars ltc;

#ifdef NTTYDISC
   ldisc = NTTYDISC;
   (void)ioctl(0, TIOCSETD, &ldisc);
#endif /* NTTYDISC */
   tty.sg_ispeed = B9600;
   tty.sg_ospeed = B9600;
   tty.sg_erase = CINTR;
   tty.sg_kill = CKILL;		/* ^U */
   tty.sg_flags = CRMOD | ECHO | EVENP | ODDP;
   (void)ioctl(0, TIOCSETP, &tty);

   tc.t_intrc = CINTR;		/* ^C */
   tc.t_quitc = CQUIT;		/* ^\ */
   tc.t_startc = CSTART;		/* ^Q */
   tc.t_stopc = CSTOP;		/* ^S */
   tc.t_eofc = CEOF;		/* ^D */
   tc.t_brkc = -1;
   (void)ioctl(0,TIOCSETC,&tc);

   ltc.t_suspc = CTRLZ;		/* ^Z */
   ltc.t_dsuspc = CDSUSP;		/* ^Y */
   ltc.t_rprntc = CRPRNT;		/* ^R */
   ltc.t_flushc = CFLUSH;		/* ^O */
   ltc.t_werasc = CWERASE;		/* ^W */
   ltc.t_lnextc = CLNEXT;		/* ^V */
   (void)ioctl(0,TIOCSLTC,&ltc);

   lmode = LCRTBS | LCRTERA | LCTLECH | LPASS8 | LCRTKIL;
   (void)ioctl(0, TIOCLSET, &lmode);

#endif /* BSD */

   /* common part: set window size */
   stty_size (0, RxvtWin.cols, RxvtWin.rows);

   /* become virtual console, if required and possible */
#ifdef TIOCCONS
   if (console)
     {
	int on = 1;
	/* if (ioctl (ttyfd, TIOCCONS, (unsigned char *)&on) == -1) */
	if (ioctl (0, TIOCCONS, (unsigned char *)&on) == -1)
	  {
	     print_error ("can't open console");
	     perror ("rxvt");
	  }
     }
#endif	/* TIOCCONS */
}

/*
 * Run the command in a subprocess and return a file descriptor for the
 * master end of the pseudo-teletype pair with the command talking to
 * the slave.
 */
static int
run_command (char **argv)
{
   int i, ptyfd;
   char *cmd, argv0 [256];

   if ((ttynam = get_ptty (&ptyfd)) == NULL)
     return -1;

   /* store original tty status for restoration clean_exit() -- rgg 04/12/95 */
   lstat (ttynam, &ttyfd_stat);
#ifdef DEBUG_TTY
   fprintf (stderr, "Original settings of %s are mode %o, uid %d, gid %d\n",
	    ttynam, ttyfd_stat.st_mode, ttyfd_stat.st_uid, ttyfd_stat.st_gid);
#endif

   atexit (clean_exit);		/* exit handler with cleanup */

   /* get number of available file descriptors */
   fd_width = getdtablesize();

    /* spin off the command interpreter */
#if 0
   for (i = 1; i <= 15; i++)
     signal (i, catch_sig);
#else
   signal (SIGHUP, catch_sig);
   signal (SIGINT, catch_sig);
   signal (SIGQUIT, catch_sig);
   signal (SIGTERM, catch_sig);
#endif
   signal (SIGCHLD, catch_child);

   comm_pid = fork ();
   if (comm_pid < 0)
     {
	print_error ("can't fork");
	return -1;
     }
   if (comm_pid == 0)			/* command interpreter path */
     {
#ifdef HAVE_UNSETENV
	/*
	 * bash may set these, but they should be clear for the child or
	 * the old settings are passed on and will confuse term size,
	 * unsetenv may not be very portable, surround with HAVE_UNSETENV
	 */
	unsetenv ("LINES");
	unsetenv ("COLUMNS");
#endif	/* HAVE_UNSETENV */

	/*
	 *  Having started a new session, we need to establish
	 *  a controlling teletype for it.  On some systems
	 *  this can be done with an ioctl() but on others
	 *  we need to re-open the slave tty.
	 */
	(void) get_tty ();
	set_ttymodes ();

	/* set window size */
	stty_size (0, RxvtWin.cols, RxvtWin.rows);

	/* reset signals and spin off the command interpreter */
	signal (SIGINT, SIG_DFL);
	signal (SIGQUIT, SIG_DFL);
	signal (SIGCHLD, SIG_DFL);
#ifdef SIGTSTP
	/* mimick login's behavior by disabling the job control   */
	/* signals; a shell that wants them can turn them back on */
	signal (SIGTSTP, SIG_IGN);
	signal (SIGTTIN, SIG_IGN);
	signal (SIGTTOU, SIG_IGN);
#endif /* SIGTSTP */

	cmd = argv [0];
	if (login_shell)
	  {
	     strcpy (&argv0[1], basename (argv[0]));
	     argv0[0] = '-';
	     argv [0] = argv0;
	  }
	execvp (cmd, argv);
	print_error ("couldn't execute %s", cmd);
	exit (EXIT_FAILURE);
     }

   makeutent (&ttynam[5]);	/* stamp /etc/utmp */
   return ptyfd;
}

/*
 * Tell the teletype handler what size the window is.
 * Called after a window size change.
 */
static void
stty_size (int fd, int width, int height)
{
   struct winsize wsize;

   if (fd < 0) return;

   wsize.ws_col = (unsigned short) width;
   wsize.ws_row = (unsigned short) height;
   wsize.ws_xpixel = 0;		/* rgg */
   wsize.ws_ypixel = 0;		/* rgg */
   ioctl (fd, TIOCSWINSZ, (char *)&wsize);
}

/* Tell the teletype handler what size the window is.
 * Called after a window size change.
 */
void
tty_resize (int width, int height)
{
   stty_size (comm_fd, width, height);
}

/*
 * Initialise the command connection.
 * This should be called after the X server connection is established.
 */
void
init_command (char **argv)
{
   greek_init ();
   /* Enable the delete window protocol */
   wm_del_win = XInternAtom (Xdisplay, "WM_DELETE_WINDOW", False);
   XSetWMProtocols (Xdisplay, RxvtWin.parent, &wm_del_win, 1);

   if ((comm_fd = run_command (argv)) < 0)
     {
	print_error ("quitting");
	exit (EXIT_FAILURE);
     }

   x_fd = XConnectionNumber (Xdisplay);
   com_buf_next = com_buf_top = com_buf;
}

#ifdef DEBUG_KEY
static void
print_keystring (char *str, int len)
{
   int i;

   /* Display the keyboard buffer contents -- rgg 06/12/95 */
   fprintf (stderr, "key [%d]: `", len);
   for (i = 0; i < len; i++)
     {
	if (str [i] < ' ' || str [i] >= '\177')
	  fprintf (stderr, "\\%03o", str[i]);
	else
	  fprintf (stderr, "%c", str[i]);
     }
   fprintf (stderr, "'\n");
}
#endif /* DEBUG_KEY */

/*
 * Convert the keypress event into a string
 *
 * use `buffer' as a temporary space for mapping most keys, but
 * use the pointer kbuf to that it is easy to use the KEYSYM_RESOURCE
 * without worrying about length restrictions
 */
static void
lookup_key (XEvent *ev)
{
#ifdef DEBUG_KEY
   static int debug_key = 1;	/* accessible by dbg */
#endif	/* DEBUG_KEY */
#ifdef GREEK_SUPPORT
   static int greek_mode = 0;
#endif
   static int numlock_state = 1;
   static XComposeStatus compose = {NULL, 0};
   static unsigned char buffer [KBUFSZ];
   unsigned char *kbuf;		/* pointer to key-buffer */
   int len, shift, ctrl, meta, appKP_active;
   KeySym keysym;

   /*
    * if numlock has been used -- allow it as a toggle.
    * Otherwise, respect the application keypad set by escape sequence
    * but allow shift to override
    */
   shift= (ev->xkey.state & ShiftMask);
   ctrl = (ev->xkey.state & ControlMask);
   meta = (ev->xkey.state & Mod1Mask);
   len  = (ev->xkey.state & Mod5Mask);	/* tmp value */
   if (numlock_state || len)
     {
	numlock_state = len;	/* numlock toggle */
	appKP = !len;		/* current state */
     }

   /* allow shift to override */
   appKP_active = ((appKP && !shift) || (!appKP && shift));

   len = XLookupString (&ev->xkey, buffer, sizeof(buffer)-1,
			&keysym, &compose);

   kbuf = buffer;		/* it was copied to here */
#ifndef HOTKEY
# define HOTKEY meta
#endif
   if (HOTKEY) {
      if (keysym == ks_pageup)
	{ scr_page (UP, RxvtWin.rows * PAGE_AMOUNT); return; }
      else if (keysym == ks_pagedown)
	{ scr_page (DOWN, RxvtWin.rows * PAGE_AMOUNT); return; }
#ifndef SINGLE_FONT
      else if (keysym == ks_bigfont)	{ new_font (UP); return; }
      else if (keysym == ks_smallfont)	{ new_font (DOWN); return; }
#endif
#ifdef SECURE_KBD
      else if (keysym == ks_secure)	{ scr_secure (); return; }
#endif
#ifdef MAPALERT
      else if (keysym == ks_alert)	{ map_alert = !map_alert; return; }
#endif
   }
#ifdef GREEK_SUPPORT
   if (keysym == ks_greektoggle)
     {
	greek_mode = !greek_mode;
	/* there's no consistent way to set/restore the title since
	 * it could have been set via an XTerm escape sequence
	 * so don't bother trying
	 */
	if (greek_mode) {
	   change_xterm_name (NEW_TITLE_NAME,
			      (GreekMode == GREEK_ELOT928 ?
			       "[GRK elot]" : "[GRK 437]"));
	   greek_reset ();
	} else {
	  change_xterm_name (NEW_TITLE_NAME, rs_title);
	}
	return;
     }
#endif
#ifdef PRINT_PIPE
   else if (keysym == ks_printscreen)
     { scr_printscreen (ctrl|shift); return; }
#else
   else if (keysym == XK_Print)
     { scr_dump_state (); return; }	/* hidden debugging dump */
#endif
   if (keysym >= 0xFF00 && keysym <= 0xFFFF) {
      /* Shift + F1 - F10 generates F11 - F20 */
      if (shift && keysym >= XK_F1 && keysym <= XK_F10)
	{
	   shift = 0;		/* turn off */
	   keysym += (XK_F11 - XK_F1);
	}

#ifdef KEYSYM_RESOURCE
      if (KeySym_map [keysym - 0xFF00] != NULL)
	{
	   /* don't use static_kbuf -- allow arbitrary length strings */
	   shift = meta = ctrl = 0;	/* no post-processing */
	   kbuf = KeySym_map [keysym - 0xFF00];
	   len = strlen (kbuf);
	}
      else
#endif
      switch (keysym) {
       case XK_BackSpace:
	 len = 1;
	 kbuf[0] = ((bs_is_bs && (shift|ctrl))||
		    (!bs_is_bs && !(shift|ctrl))) ? '\177':'\b';
	 break;

       case XK_Tab:	if (shift) { len = 3; strcpy(kbuf,"\033[Z"); }	break;

       case XK_Up:
	 len = 3; strcpy(kbuf, "\033[A");
	 if (appCUR) kbuf [1] = 'O';
#ifdef KC_SHIFT_UP
	 else if (shift) kbuf [2] = KC_SHIFT_UP;
#endif
#ifdef KC_CTRL_UP
	 else if (ctrl) kbuf [2] = KC_CTRL_UP;
#endif
	 break;
       case XK_Down:
	 len = 3; strcpy(kbuf, "\033[B");
	 if (appCUR) kbuf [1] = 'O';
#ifdef KC_SHIFT_DOWN
	 else if (shift) kbuf [2] = KC_SHIFT_DOWN;
#endif
#ifdef KC_CTRL_DOWN
	 else if (ctrl) kbuf [2] = KC_CTRL_DOWN;
#endif
	 break;
       case XK_Right:
	 len = 3; strcpy(kbuf, "\033[C");
	 if (appCUR) kbuf [1] = 'O';
#ifdef KC_SHIFT_RIGHT
	 else if (shift) kbuf [2] = KC_SHIFT_RIGHT;
#endif
#ifdef KC_CTRL_RIGHT
	 else if (ctrl) kbuf [2] = KC_CTRL_RIGHT;
#endif
	 break;
       case XK_Left:
	 len = 3; strcpy(kbuf, "\033[D");
	 if (appCUR) kbuf [1] = 'O';
#ifdef KC_SHIFT_LEFT
	 else if (shift) kbuf [2] = KC_SHIFT_LEFT;
#endif
#ifdef KC_CTRL_LEFT
	 else if (ctrl) kbuf [2] = KC_CTRL_LEFT;
#endif
	 break;

       case XK_Find:	len = strlen(strcpy(kbuf,KS_FIND));	break;
       case XK_Insert:	len = strlen(strcpy(kbuf,KS_INSERT));	break;
       case XK_Execute:	len = strlen(strcpy(kbuf,KS_EXECUTE));	break;
       case XK_Select:	len = strlen(strcpy(kbuf,KS_SELECT));	break;
       case XK_Prior:	len = strlen(strcpy(kbuf,KS_PRIOR));	break;
       case XK_Next:	len = strlen(strcpy(kbuf,KS_NEXT));	break;

       case XK_Delete:	len = strlen(strcpy(kbuf,KS_DELETE));	break;
       case XK_Home:	len = strlen(strcpy(kbuf,KS_HOME));	break;
       case XK_End:	len = strlen(strcpy(kbuf,KS_END));	break;

       case XK_KP_Enter:
	 if (appKP_active)
	   {
	      len = 3; strcpy(kbuf, "\033OM");
	   }
	 else
	   {
	      len = 1; kbuf[0] = '\r';
	   }
	 break;

       case XK_KP_F1:		/* "\033OP" */
       case XK_KP_F2:		/* "\033OQ" */
       case XK_KP_F3:		/* "\033OR" */
       case XK_KP_F4:		/* "\033OS" */
	 len = 3; strcpy(kbuf, "\033OP");
	 kbuf[2] += (keysym - XK_KP_F1);
	 break;

       case XK_KP_Multiply:	/* "\033Oj" : "*" */
       case XK_KP_Add:		/* "\033Ok" : "+" */
       case XK_KP_Separator:	/* "\033Ol" : "," */
       case XK_KP_Subtract:	/* "\033Om" : "-" */
       case XK_KP_Decimal:	/* "\033On" : "." */
       case XK_KP_Divide:	/* "\033Oo" : "/" */
       case XK_KP_0:		/* "\033Op" : "0" */
       case XK_KP_1:		/* "\033Oq" : "1" */
       case XK_KP_2:		/* "\033Or" : "2" */
       case XK_KP_3:		/* "\033Os" : "3" */
       case XK_KP_4:		/* "\033Ot" : "4" */
       case XK_KP_5:		/* "\033Ou" : "5" */
       case XK_KP_6:		/* "\033Ov" : "6" */
       case XK_KP_7:		/* "\033Ow" : "7" */
       case XK_KP_8:		/* "\033Ox" : "8" */
       case XK_KP_9:		/* "\033Oy" : "9" */
	 if (appKP_active)
	   {
	      len = 3; strcpy(kbuf, "\033Oj");
	      kbuf[2] += (keysym - XK_KP_Multiply);
	   }
	 else
	   {
	      len = 1; kbuf[0] = ('*' + (keysym - XK_KP_Multiply));
	   }
	 break;

       case XK_F1:		/* "\033[11~" */
       case XK_F2:		/* "\033[12~" */
       case XK_F3:		/* "\033[13~" */
       case XK_F4:		/* "\033[14~" */
       case XK_F5:		/* "\033[15~" */
	 len = 5; sprintf(kbuf, "\033[%2d~", (11 + (keysym - XK_F1)));	break;

       case XK_F6:		/* "\033[17~" */
       case XK_F7:		/* "\033[18~" */
       case XK_F8:		/* "\033[19~" */
       case XK_F9:		/* "\033[20~" */
       case XK_F10:		/* "\033[21~" */
	 len = 5; sprintf(kbuf, "\033[%2d~", (17 + (keysym - XK_F6)));	break;

       case XK_F11:		/* "\033[23~" */
       case XK_F12:		/* "\033[24~" */
       case XK_F13:		/* "\033[25~" */
       case XK_F14:		/* "\033[26~" */
	 len = 5; sprintf(kbuf, "\033[%2d~", (23 + (keysym - XK_F11)));	break;

       case XK_Help:
       case XK_F15:	len = 5; strcpy(kbuf,"\033[28~");	break;
       case XK_Menu:
       case XK_F16:	len = 5; strcpy(kbuf,"\033[29~");	break;
       case XK_F17:		/* "\033[31~" */
       case XK_F18:		/* "\033[32~" */
       case XK_F19:		/* "\033[33~" */
       case XK_F20:		/* "\033[34~" */
	 len = 5; sprintf(kbuf, "\033[%2d~", (31 + (keysym - XK_F17)));	break;
      }
   }
#ifdef GREEK_SUPPORT
   else if (greek_mode)
     len = greek_xlat (kbuf, len);
#endif

   if (len <= 0) return;	/* not mapped */

   /* only permit these modifications if we are using the static keybuffer */

   /* pass Shift/Control indicators for function keys ending with `~' */
   if (len > 3 && kbuf [len-1] == '~')
     {
	if (shift)
	  kbuf [len-1] = '$';
	else if (ctrl)
	  kbuf [len-1] = '^';
     }
   if (meta && (len < sizeof(buffer)-1))
     {
	unsigned char *c;
#ifndef ALWAYS_META_ESCAPE
	switch (MetaHandling) {
	 case 033:		/* escape prefix */
#endif
	   for (c = kbuf + len; c > kbuf ; c--)
	     *c = *(c-1);

	   kbuf [0] = '\033';
	   len++;
#ifndef ALWAYS_META_ESCAPE
	   break;

	 case 0x80:		/* set 8-bit on */
	   /* can confuses things for some non-english character sets */
	   for (c = kbuf; c < kbuf + len; c++)
	     *c |= 0x80;
	   break;
	}
#endif
     }

   tty_write (kbuf, len);
#ifdef DEBUG_KEY
   if (debug_key)
     print_keystring (kbuf, len);
#endif /* DEBUG_KEY */
}

/*
 * Return the next input character after first passing any keyboard input
 * to the command.
 */
static unsigned char
get_com_char (void)
{
   static int refreshed = 0;
   fd_set in_fdset;
   int count, retval, total;
   struct itimerval value;

   /* If there have been a lot of new lines, then update the screen
    * What the heck I'll cheat and only refresh less than every page-full.
    * the number of pages between refreshes is refresh_nl_limit, which
    * is incremented here because we must be doing flat-out scrolling.
    *
    * refreshing should be correct for small scrolls, because of the
    * time-out */
   if (refresh_nl_count > refresh_nl_limit * RxvtWin.rows)
     {
	if (refresh_nl_limit < refresh_period)
	  refresh_nl_limit++;
	refresh_nl_count = 0;
	refreshed = 1;
	scr_refresh (refresh_type);
     }

   /* If we have characters already read in. return one */
   if (com_buf_top > com_buf_next)
     {
	refreshed = 0;
	return (*com_buf_next++ & hibit_mask);
     }

   while (1)
     {
	/* process any X events that are pending */
	while (XPending (Xdisplay))
	  {
	     refreshed = 0;
	     get_X_event ();
	  }

	/* Nothing to do! */
	FD_ZERO (&in_fdset);
	FD_SET (comm_fd, &in_fdset);
	FD_SET (x_fd, &in_fdset);
	value.it_value.tv_usec = 5000;
	value.it_value.tv_sec = 0;
	if (refreshed)
	  retval = select (fd_width, &in_fdset, NULL, NULL, NULL);
	else
	  retval = select (fd_width, &in_fdset, NULL, NULL, &value.it_value);

	/* See if we can read from the application */
	if (FD_ISSET (comm_fd, &in_fdset))
	  {
	     count = 1;
	     com_buf_next = com_buf_top = com_buf;
	     total = sizeof(com_buf);
	     while ((count > 0) && (total > sizeof(com_buf)/2))
	       {
		  count = read (comm_fd, com_buf_top, total);
		  if (count > 0)
		    {
		       com_buf_top += count;
		       total -= count;
		    }
	       }
	     if (com_buf_top > com_buf_next)
	       {
		  refreshed = 0;
		  return (*com_buf_next++ & hibit_mask);
	       }
	  }
	/* If the select statement timed out, we better update the screen */
	if (retval == 0)
	  {
	     refresh_nl_count = 0;
	     refresh_nl_limit = 1;
	     if (!refreshed)
	       scr_refresh (refresh_type);
	     refreshed = 1;
#ifndef SCROLLBAR_NONE
	     sbar_show (RxvtWin.scrolled + (RxvtWin.rows-1),
			RxvtWin.offset,
			(RxvtWin.rows-1));
#endif
	     /* XFlush (Xdisplay);*/
	  }
     }
   return 0;
}

/*----------------------------------------------------------------------*
 * Receive and process an X event
 *----------------------------------------------------------------------*/
static void
get_X_event (void)
{
   XEvent ev;
   Window root, child;
   int root_x, root_y, x, y;
   unsigned int mods;
   XEvent dummy;
#ifdef MULTIPLE_CLICKS
   static Time buttonpress_time;
   static int clicks = 0;
#endif
   static int size_set = 0;	/* flag that window size has been set */

   XNextEvent (Xdisplay, &ev);

   switch (ev.type) {
    case KeyPress:
      lookup_key (&ev);
      break;

    case ClientMessage:
      if (ev.xclient.format == 32 && ev.xclient.data.l[0] == wm_del_win)
	exit (EXIT_SUCCESS);
      break;

    case MappingNotify:
      XRefreshKeyboardMapping (&ev.xmapping);
      break;

    case GraphicsExpose:
    case Expose:
      if (!size_set)
	{
	   /* Force a window resize if an exposure event
	    * arrives before the first resize event
	    */
	   resize_window (0, 0);
	   size_set = 1;
	}
      if (ev.xany.window == RxvtWin.vt)
	{
	   scr_touch (ev.xexpose.x, ev.xexpose.y,
		      ev.xexpose.width, ev.xexpose.height);
	}
      else
	{
	   while (XCheckTypedWindowEvent (Xdisplay, ev.xany.window,
					  Expose, &dummy));
	   while (XCheckTypedWindowEvent (Xdisplay, ev.xany.window,
					  GraphicsExpose, &dummy));
#ifndef SCROLLBAR_NONE
	   if (ev.xany.window == sbar.sb.win)
	     sbar_show (-1,-1,-1);
#ifndef SCROLLBAR_NOARROWS
	   else if (ev.xany.window == sbar.up.win)
	     sbar_up_reset ();
	   else if (ev.xany.window == sbar.dn.win)
	     sbar_down_reset ();
#endif	/* ! NOARROWS */
	   else
#endif
	     Gr_expose (ev.xany.window);
	}
      break;

    case VisibilityNotify:
      /* Here's my conclusiion:
       * If the window is completely unobscured, use bitblt's
       * to scroll. Even then, they're only used when doing partial
       * screen scrolling. When partially obscured, we have to fill
       * in the GraphicsExpose parts, which means that after each refresh,
       * we need to wait for the graphics expose or Noexpose events,
       * which ought to make things real slow! */
      switch (ev.xvisibility.state) {
       case VisibilityUnobscured:
	 refresh_type = FAST_REFRESH;
/*	 scr_touch (0, 0, RxvtWin.pwidth, RxvtWin.pheight);*/
	 break;
       case VisibilityPartiallyObscured:
	 refresh_type = SLOW_REFRESH;
/*	 scr_touch (0, 0, RxvtWin.pwidth, RxvtWin.pheight);*/
	 break;
       default:
	 refresh_type = NO_REFRESH;
	 break;
      }
      break;

    case FocusIn:	scr_focus (1);	break;
    case FocusOut:	scr_focus (0);	break;
    case ConfigureNotify:
      resize_window (0, 0);
      size_set = 1;
      break;

    case SelectionClear:
      scr_selection_delete ();
      break;

    case SelectionNotify:
      scr_paste_primary (ev.xselection.requestor,
			 ev.xselection.property, True);
      break;

    case SelectionRequest:
      scr_selection_send (&(ev.xselectionrequest));
      break;

    case ButtonPress:
      if (ev.xany.window == RxvtWin.vt)
	{
	   if (ev.xbutton.subwindow != None)
	     Gr_ReportButtonPress (ev.xbutton.x, ev.xbutton.y);
	   else
	     {
#ifndef NO_MOUSE_REPORT
		if (Mouse_Report && !(ev.xbutton.state&(Mod1Mask|ShiftMask)))
		  {
#ifdef MULTIPLE_CLICKS
		     clicks = 0;
#endif
		     if (Mouse_Report & X10_MOUSE)
		       ev.xbutton.state = 0;	/* no state info allowed */
		     mouse_report (&(ev.xbutton));
#ifdef X11_TRACKING
		     if (Mouse_Report & X11_TRACKING)
		       {
			  /* wait for and get reply
			   * "ESC [ Ps ; Ps ; Ps ; Ps ; Ps T"
			   * Parameters =
			   *  func, startx, starty, firstrow, lastrow.
			   * func is non-zero to initiate hilite tracking
			   */
			  if (func)
			    Mouse_Report |= X11_HILITE;
			  else
			    Mouse_Report &= ~X11_HILITE;

			  mouse_tracking (0, x, y, r1, r2);
		       }
#endif
		     return;
		  }
#endif
		switch (ev.xbutton.button) {
		 case Button1:
#ifdef MULTIPLE_CLICKS
		   if (ev.xbutton.time - buttonpress_time < MULTICLICK_TIME)
		     clicks++;
		   else
		     clicks = 1;
		   buttonpress_time = ev.xbutton.time;
		   scr_selection_multiclick (clicks,
					     ev.xbutton.x, ev.xbutton.y);
#else
		   scr_selection_start (ev.xbutton.x, ev.xbutton.y);
#endif
		   break;

		 case Button3:
		   scr_selection_extend (1, ev.xbutton.x, ev.xbutton.y);
		   break;
		}
		return;
	     }
	}
#ifndef SCROLLBAR_NONE
#ifndef NO_MOUSE_REPORT
      /* disabled scrollbar
       * arrow buttons - send up/down
       * click on scrollbar - send pageup/down
       */
      if (Mouse_Report && !(ev.xbutton.state&(Mod1Mask|ShiftMask)))
	{
	   if (ev.xany.window == sbar.sb.win)
	     switch (ev.xbutton.button) {
	      case Button1: tty_write ("\033[6~", 4);	break;
	      case Button3: tty_write ("\033[5~", 4);	break;
	     }
#ifndef SCROLLBAR_NOARROWS
	   else if (ev.xany.window == sbar.up.win) tty_write ("\033[A", 3);
	   else if (ev.xany.window == sbar.dn.win) tty_write ("\033[B", 3);
#endif	/* ! NOARROWS */
	   return;
	}
#endif	/* NO_MOUSE_REPORT */
      if (ev.xany.window == sbar.sb.win)
	{
#ifndef SCROLLBAR_ARROWS
	   if (
#ifdef SCROLLBAR_ANY
	       (sbar.type == SBAR_NOARROWS) &&
#endif	/* ANY */
	       ev.xbutton.button != Button2)
	     /*
	      * Move display proportional to location of the pointer
	      * If pointer is near bottom, scroll full page.
	      * If pointer is near top, scroll one line.
	      */
	     switch (ev.xbutton.button) {
	      case Button1:
		scr_page (DOWN, RxvtWin.rows *
			  (double)(ev.xbutton.y) / (double)(sbar.h-1));
		break;
	      case Button3:
		scr_page (UP, RxvtWin.rows *
			  (double)(ev.xbutton.y) / (double)(sbar.h-1));
		break;
	     }
	   else
#endif	/* ! ARROWS */
	     scr_move_to (ev.xbutton.y);
	}
#ifndef SCROLLBAR_NOARROWS
      else if (ev.xany.window == sbar.dn.win)
	scr_page (DOWN, RxvtWin.rows * PAGE_AMOUNT);
      else if (ev.xany.window == sbar.up.win)
	scr_page (UP, RxvtWin.rows * PAGE_AMOUNT);
#endif	/* ! NOARROWS */
      break;
#endif	/* SCROLLBAR_NONE */

    case ButtonRelease:
      if (ev.xany.window == RxvtWin.vt)
	{
	   if (ev.xbutton.subwindow != None)
	     Gr_ReportButtonRelease (ev.xbutton.x, ev.xbutton.y);
	   else
	     {
#ifndef NO_MOUSE_REPORT
		if (Mouse_Report && !(ev.xbutton.state&(Mod1Mask|ShiftMask)))
		  {
		     switch (Mouse_Report) {
		      case X10_MOUSE:
			break;

		      case X11_MOUSE:
#ifdef MULTIPLE_CLICKS
			clicks = 0;
#endif
			ev.xbutton.button = AnyButton;
			mouse_report (&(ev.xbutton));
			break;

#ifdef X11_TRACKING
		      case X11_TRACKING:
			/* if start/end are  valid text locations:
			 * "ESC [ t CxCy"
			 * If either coordinate is past the end of the line:
			 * "ESC [ T CxCyCxCyCxCy"
			 * The parameters are startx, starty, endx, endy,
			 * mousex, and mousey */
			mouse_tracking (1, ev.xbutton.x, ev.xbutton.y, 0, 0);
			break;
#endif
		       }
		     return;
		  }
#endif

#ifndef NO_MOUSE_REPORT
		/* dumb hack to compensate for the failure of click-and-drag
		 * when overriding mouse reporting
		 */
		if (Mouse_Report && (ev.xbutton.state&(Mod1Mask|ShiftMask))
		    && (ev.xbutton.button == Button1)
#ifdef MULTIPLE_CLICKS
		    && (clicks <= 1)
#endif
		    )
		  scr_selection_extend (1, ev.xbutton.x, ev.xbutton.y);
#endif

		switch (ev.xbutton.button)
		  {
		   case Button1:
		   case Button3:
		     scr_selection_make (ev.xbutton.time);
		     break;

		   case Button2:
		     scr_selection_request (ev.xbutton.time,
					    ev.xbutton.x,
					    ev.xbutton.y);
		     break;
		  }
	     }
	}
      break;

    case MotionNotify:
#ifndef NO_MOUSE_REPORT
      if (Mouse_Report
#ifdef X11_TRACKING
	  && (Mouse_Report ^ X11_TRACKING)
#endif
	  && !(ev.xbutton.state&(Mod1Mask|ShiftMask)))
	return;
#endif
      if (ev.xany.window == RxvtWin.vt)
	{
	   if (((ev.xbutton.state == Button1Mask) ||
		(ev.xbutton.state == Button3Mask))
#ifdef MULTIPLE_CLICKS
	       && (clicks <= 1)
#endif
	       )
	     {
		while (XCheckTypedWindowEvent (Xdisplay, RxvtWin.vt,
					       MotionNotify, &ev));
		XQueryPointer (Xdisplay, RxvtWin.vt, &root, &child,
			      &root_x, &root_y, &x, &y, &mods);
#ifdef MULTIPLE_CLICKS
		/* deal with a `jumpy' mouse */
		if ((ev.xmotion.time - buttonpress_time) >
		    (MULTICLICK_TIME/2))
#endif
		  {
#ifdef X11_TRACKING
		     if (Mouse_Report & X11_TRACKING)
		       {
			  if (Mouse_Report & X11_HILITE)
			    scr_selection_extend (1, x, y);
			  else
			    scr_selection_extend (0, x, y);
		       }
		     else
#endif
		       scr_selection_extend (1, x, y);
		  }

	     }
	}
#ifndef SCROLLBAR_NONE
      else if ((ev.xany.window == sbar.sb.win) &&
	       (ev.xbutton.state&(Button1Mask|Button2Mask|Button3Mask)))
	{
	   while (XCheckTypedWindowEvent (Xdisplay, sbar.sb.win,
					  MotionNotify, &ev));
	   XQueryPointer (Xdisplay, sbar.sb.win, &root, &child,
			  &root_x, &root_y, &x, &y, &mods);
#ifndef SCROLLBAR_ARROWS
	   if (
#ifdef SCROLLBAR_ANY
	       (sbar.type == SBAR_ARROWS) ||
#endif	/* ANY */
	       (ev.xbutton.state & Button2Mask)
	       )
#endif	/* ! ARROWS */
	     {
		scr_move_to (y);
		scr_refresh (refresh_type);
		refresh_nl_count = refresh_nl_limit = 0;
		sbar_show (RxvtWin.scrolled + (RxvtWin.rows-1),
			   RxvtWin.offset,
			   (RxvtWin.rows-1));
	     }
	}
#endif	/* SCROLLBAR_NONE */
      break;
   }
}

/*
 * Send count characters directly to the command
 */
void
tty_write (unsigned char *buf, int count)
{
   while (count > 0)
     {
	int n = write (comm_fd, buf, count);
	if (n > 0)
	  {
	     count -= n;
	     buf += n;
	  }
     }
}

/*
 * Send printf() formatted output to the command.
 * Only used for small ammounts of data.
 */
void
tty_printf (unsigned char *fmt, ...)
{
   static unsigned char buf [256];
   va_list arg_ptr;

   va_start (arg_ptr, fmt);
   vsprintf (buf, fmt, arg_ptr);
   va_end (arg_ptr);
   tty_write (buf, strlen (buf));
}

/*----------------------------------------------------------------------*/

#ifdef PRINT_PIPE
static void
process_print_pipe (void)
{
   const char * escape_seq = "\033[4i";
   const char * rev_escape_seq = "i4[\033";
   int c, index;
   FILE *fd;

   fd = popen (rs_print_pipe, "w");
   if (fd == NULL)
     {
	print_error ("can't open printer pipe!");
	return;
     }

   for (index = 0; index < 4; /* nil*/)
     {
	c = get_com_char ();

	if (c == escape_seq [index]) index++;
	else if (index)
	  for (/*nil*/; index > 0; index--)
	  fputc (rev_escape_seq [index-1], fd);

	if (index == 0) fputc (c, fd);
     }
   fflush (fd);
   pclose (fd);
}
#endif

static void
process_escape_seq (void)
{
   switch (get_com_char ()) {
#if 0
    case 1:	do_tek_mode ();	break;
#endif
    case '#':	if ('8' == get_com_char ()) scr_E ();	break;
    case '(':	scr_set_charset (0, get_com_char ());	break;
    case ')':	scr_set_charset (1, get_com_char ());	break;
    case '*':	scr_set_charset (2, get_com_char ());	break;
    case '+':	scr_set_charset (3, get_com_char ());	break;
    case '7':	scr_save_cursor ();	break;
    case '8':	scr_restore_cursor ();	break;
    case '=':	appKP = 1;		break;
    case '>':	appKP = 0;		break;
    case '@':	(void)get_com_char ();	break;
    case 'D':	scr_index (UP);		break;
    case 'E':	scr_add_lines ("\n\r", 1, 2);	break;
    case 'G':	process_robs_seq ();	break;
    case 'H':	scr_set_tab (1);	break;
    case 'M':	scr_index (DOWN);	break;
    /*case 'N':	scr_single_shift (2);	break;*/
    /*case 'O':	scr_single_shift (3);	break;*/
    case 'Z':	tty_printf (RXVT_ANS);	break;	/* steal obsolete ESC [ c */
    case '[':	process_csi_seq ();	break;
    case ']':	process_xterm_seq ();	break;
    case 'c':	scr_power_on ();	break;
    case 'n':	scr_choose_charset (2);	break;
    case 'o':	scr_choose_charset (3);	break;
   }
}
/*----------------------------------------------------------------------*
 *
 *----------------------------------------------------------------------*/

static void
process_csi_seq (void)
{
   unsigned char c, priv = 0;
   int nargs, arg [ESC_ARGS];

   nargs = 0;
   arg [0] = 0;
   arg [1] = 0;

   c = get_com_char ();
   if (c >= '<' && c <= '?')
     {
	priv = c;
	c = get_com_char ();
     }

   /* read any numerical arguments */
   do
     {
	int n;
	n = 0;
	while (c >= '0' && c <= '9')
	  {
	     n = n * 10 + (c - '0');
	     c = get_com_char ();
	  }
	if (nargs < ESC_ARGS)
	  arg [nargs++] = n;
	if (c == 033)
	  {
	     process_escape_seq ();
	     return;
	  }
	else if (c == '\b')
	  scr_backspace ();
	else if (c < ' ')
	  {
	     scr_add_lines (&c, 0, 1);
	     return;
	  }
	else if (c < '@')
	  c = get_com_char ();
     }
   while (c >= ' ' && c < '@');
   if (c == 033)
     {
	process_escape_seq ();
	return;
     }
   else if (c < ' ')
     return;

   switch (c) {
#ifdef PRINT_PIPE
    case 'i':			/* printing */
      switch (arg[0]) {
       case 0: scr_printscreen (0);		break;
       case 5: process_print_pipe ();		break;
      }
      break;
#endif
    case 'A': case 'e':		/* up <n> */
      scr_move (0, (arg[0] ? -arg[0] : -1), RELATIVE);	break;
    case 'B':			/* down <n> */
      scr_move (0, (arg[0] ? arg[0] : +1), RELATIVE);	break;
    case 'C': case 'a':		/* right <n> */
      scr_move ((arg[0] ? arg[0] : +1), 0, RELATIVE);	break;
    case 'D':			/* left <n> */
      scr_move ((arg[0] ? -arg[0] : -1), 0, RELATIVE);	break;
    case 'E':			/* down <n> & to first column */
      scr_move (0, (arg[0] ? arg[0] : +1), ROW_RELATIVE);	break;
    case 'F':			/* up <n> & to first column */
      scr_move (0, (arg[0] ? -arg[0] : -1), ROW_RELATIVE);	break;
    case 'G': case '`':		/* move to column <n> */
      scr_move ((arg[0] ? arg[0] : +1), 0, ROW_RELATIVE);	break;
    case 'd':			/* move to line <n> */
      scr_move (0, (arg[0] ? arg[0] : +1), COL_RELATIVE);	break;
    case 'H': case 'f':		/* position cursor */
      switch (nargs) {
       case 0: scr_move (0, 0, 0); break;
       case 1: scr_move (0, (arg[0] ? (arg[0]-1) : 0), 0);	break;
       default: scr_move (arg[1] - 1, arg[0] - 1, 0);	break;
      }
      break;
    case 'I':	scr_tab (arg[0] ? arg [0] : +1);	break;
    case 'Z':	scr_tab (arg[0] ? -arg [0] : -1);	break;
    case 'J':	scr_erase_screen (arg[0]);	break;
    case 'K':	scr_erase_line (arg[0]);	break;
    case '@':	scr_insdel_chars ((arg[0] ? arg [0] : 1), INSERT);	break;
    case 'L':	scr_insdel_lines ((arg[0] ? arg [0] : 1), INSERT);	break;
    case 'M':	scr_insdel_lines ((arg[0] ? arg [0] : 1), DELETE);	break;
    case 'X':	scr_insdel_chars ((arg[0] ? arg [0] : 1), ERASE);	break;
    case 'P':	scr_insdel_chars ((arg[0] ? arg [0] : 1), DELETE);	break;

    case 'c':	tty_printf (VT100_ANS);		break;
    case 'm':	process_sgr_mode (nargs, arg);	break;
    case 'n':			/* request for information */
      switch (arg[0]) {
       case 5: tty_write ("\033[0n", 4);		break;	/* ready */
       case 6: scr_report_position ();			break;
       case 7: tty_printf ("%s\n", display_name);	break;
       case 8:
	 change_xterm_name (NEW_TITLE_NAME, "rxvt "RXVT_VERSION);
	 break;
      }
      break;
    case 'r':			/* set top and bottom margins */
      /* what's this about? something to do with vi on ESIX systems */
      if (priv != '?')
	{
	   if (nargs < 2 || arg[0] >= arg[1])
	     scr_scroll_region (0, 10000);
	   else
	     scr_scroll_region (arg[0] - 1,arg[1] - 1);
	   break;
	}
      /* drop */
    case 's':
    case 'h':
    case 'l':
      process_terminal_mode (c, priv, nargs, arg);
      break;
    case 'g':
      switch (arg[0]) {
       case 0: scr_set_tab (0);		break;	/* delete tab */
       case 3: scr_set_tab (-1);	break;	/* clear all tabs */
      }
      break;
    case 'W':
      switch (arg [0]) {
       case 0: scr_set_tab (1);		break;	/* = ESC H */
       case 2: scr_set_tab (0);		break;	/* = ESC [ 0 g */
       case 5: scr_set_tab (-1);	break;	/* = ESC [ 3 g */
      }
      break;
   }
}

static void
process_xterm_seq (void)
{
   unsigned char c, string [STRING_MAX];
   int n, arg;

   c = get_com_char ();
   n = 0;
   while (c >= '0' && c <= '9')
     {
	n = n * 10 + (c - '0');
	c = get_com_char ();
     }
   arg = n;

   c = get_com_char ();
   n = 0;
   while (c != 007 && n < sizeof(string)-1)
     {
	if (c >= ' ')
	  string [n++] = c;
	c = get_com_char ();
     }
   string [n] = 0;
   change_xterm_name (arg, string);
}

/* mode can only have the following values:
 *	'l' = low
 *	'h' = high
 *	's' = save
 *	'r' = restore
 */
static void
process_terminal_mode (unsigned char mode, unsigned char priv,
		       int nargs, int arg[])
{
   /* current and saved mode values */
   static unsigned char saved_appCUR = 0, saved_appKP = 0;
   static unsigned char saved_mode132 = 0, mode132 = 0;
   static unsigned char saved_rvideo = 0, rvideo = 0;
   static unsigned char saved_origin = 0, relative_origin = 0;
   static unsigned char saved_autowrap = 1, autowrap = 1;
   static unsigned char saved_screen = 0, screen = 0;
#ifndef NO_MOUSE_REPORT
   static unsigned char saved_mouse = 0;
#endif
   int i;

   /* make boolean */
   switch (mode) {
    case 'l':	mode = 0;	break;
    case 'h':	mode = 1;	break;
    default:	return;		/* invalid */
   }

   if (nargs == 0) return;

   if (priv == 0)
     {
	for (i = 0; i < nargs; i++)
	  switch (arg[i]) {
	   case 4:	scr_insert_mode (mode);	break;
	   case 36:	bs_is_bs = mode;	break;
	     /* case 38:	TEK mode */
	  }
     }
   else if (priv == '?')
     {
	for (i = 0; i < nargs; i++)
	  switch (arg[i]) {
	   case 1:			/* application cursor keys */
	     switch (mode) {
	      case 's': saved_appCUR = appCUR;	break;
	      case 'r': mode = saved_appCUR;	/*drop*/
	      default:  appCUR = mode;
	     }
	     break;

	   case 3:			/* 80/132 */
	     switch (mode) {
	      case 's': saved_mode132 = mode132;	break;
	      case 'r': mode = saved_mode132;	/*drop*/
	      default:  mode132 = mode; set_width (mode ? 132:80);
	     }
	     break;

	     /* case 4:	- smooth scrolling */
	   case 5:
	     switch (mode) {
	      case 's': saved_rvideo = rvideo;	break;
	      case 'r': mode = saved_rvideo;	/*drop*/
	      default:  rvideo = mode; scr_rvideo_mode (mode);
	     }
	     break;

	   case 6:
	     switch (mode) {
	      case 's': saved_origin = relative_origin;	break;
	      case 'r': mode = saved_origin;	/*drop*/
	      default:  relative_origin = mode; scr_relative_origin (mode);
	     }
	     break;

	   case 7:
	     switch (mode) {
	      case 's': saved_autowrap = autowrap;	break;
	      case 'r': mode = saved_autowrap;	/*drop*/
	      default:  autowrap = mode; scr_autowrap (mode);
	     }
	     break;
	     /* case 8:	- auto repeat, can't do on a per window basis */
#ifndef NO_MOUSE_REPORT
	   case 9:			/* X10 mouse reporting */
	     switch (mode) {
	      case 's': saved_mouse = (Mouse_Report & X10_MOUSE); break;
	      case 'r': mode = (saved_mouse & X10_MOUSE);	/*drop*/
	      default:
		if (mode)
		  Mouse_Report = X10_MOUSE;
		else
		  Mouse_Report &= ~X10_MOUSE;
	     }
	     break;
#endif	/* NO_MOUSE_REPORT */
	   case 47:
	     switch (mode) {
	      case 's': saved_screen = screen;	break;
	      case 'r': mode = saved_screen;	/*drop*/
	      default:  screen = mode; scr_change_screen (mode);
	     }
	   case 66:			/* application key pad */
	     switch (mode) {
	      case 's': saved_appKP = appKP;	break;
	      case 'r': mode = saved_appKP;		/*drop*/
	      default:  appKP = mode;
	     }
	     break;

#ifndef NO_MOUSE_REPORT
	   case 1000:		/* X11 mouse reporting */
	     switch (mode) {
	      case 's': saved_mouse = (Mouse_Report & X11_MOUSE); break;
	      case 'r': mode = (saved_mouse & X11_MOUSE);	/*drop*/
	      default:
		if (mode)
		  Mouse_Report = X11_MOUSE;
		else
		  Mouse_Report &= ~X11_MOUSE;
	     }
	     break;

	   case 1001:		/* X11 mouse highlighting */
#ifdef X11_TRACKING		/* not yet! */
	     switch (mode) {
	      case 's': saved_mouse = (Mouse_Report & X11_TRACKING); break;
	      case 'r': mode = (saved_mouse & X11_TRACKING);	/*drop*/
	      default:
		if (mode)
		  Mouse_Report = X11_TRACKING;
		else
		  Mouse_Report &= ~X11_TRACKING;
	     }
#endif	/* X11_TRACKING */
	     break;
#endif	/* NO_MOUSE_REPORT */
	  }
     }
}

static void
process_sgr_mode (int nargs, int arg[])
{
   int i;

   if (nargs == 0)
     {
	scr_rendition (1, ~RS_NONE);
	return;
     }
   for (i = 0; i < nargs; i++)
     switch (arg [i]) {
      case 0:	scr_rendition (1, ~RS_NONE);	break;
      case 1:	scr_rendition (0, RS_BOLD);	break;
      case 4:	scr_rendition (0, RS_ULINE);	break;
      case 5:	scr_rendition (0, RS_BLINK);	break;
      case 7:	scr_rendition (0, RS_RVID);	break;
      case 22:	scr_rendition (1, RS_BOLD);	break;
      case 24:	scr_rendition (1, RS_ULINE);	break;
      case 25:	scr_rendition (1, RS_BLINK);	break;
      case 27:	scr_rendition (1, RS_RVID);	break;
      case 30: case 31: case 32: case 33:
      case 34: case 35: case 36: case 37:
	scr_fore_color (arg [i] - 30);
	break;
      case 39:			/* set default fg colour */
	scr_fore_color (FG_COLOR - ANSI_COLOR0);
	break;
      case 40: case 41: case 42: case 43:
      case 44: case 45: case 46: case 47:
	scr_back_color (arg [i] - 40);
	break;
      case 49:			/* set default bg colour */
	scr_back_color (BG_COLOR - ANSI_COLOR0);
	break;
     }
}

static void
process_robs_seq (void)
{
   unsigned char c, cmd = get_com_char ();
#ifndef RXVT_GRAPHICS
   if (cmd == 'Q')		/* query graphics */
     {
	tty_printf ("\033G0\n");	/* no graphics */
	return;
     }
   /* swallow other graphics sequences until terminating ':' */
   do c = get_com_char (); while (c != ':');
#else
   int nargs;
   long args [1000];
   unsigned char text[1000];

   if (cmd == 'Q')
     {
#ifdef COLOR
	tty_printf ("\033G2\n");	/* yes, color graphics */
#else
	tty_printf ("\033G1\n");	/* yes, graphics */
#endif
	return;
     }

   for (nargs = 0; nargs < sizeof(args)-1; /*nil*/)
     {
	int neg = 0;
	args [nargs] = 0;

	c = get_com_char ();
	while (1)
	  {
	     if (c >= '0' && c <= '9')
	       args [nargs] = args[nargs] * 10 + (c - '0');
	     else if (c == '-')
	       neg = !neg;
	     else
	       break;
	     c = get_com_char ();
	  }
	if (neg) args [nargs] = -args[nargs];
	nargs++;
	args [nargs] = 0;
	if (c !=  ';')
	  break;
     }

   if ((cmd == 'T') && (nargs >= 5))
     {
	int i;
	for (i = 0; (i < args [nargs-1]) && (i < sizeof(text)-1); i++)
	  text [i] = get_com_char ();
	text [i] = '\0';
     }

   Gr_do_graphics (cmd, nargs, args, text);
#endif
}

/*
 * Read and process output from the application
 */
void
main_loop (void)
{
   int c;
#if 0
   int graphics_mode = 0;	/* graphics pipe (someday?) */
   FILE *graphics_pipe = NULL;
#endif

   do {
      while ((c = get_com_char ()) == 0);	/* wait for something */
      if (c >= ' ' || c == '\t' || c == '\n' || c == '\r') /* (isprint(c))*/
	{
	   /* Read a text string from the input buffer */
	   unsigned char *str;
	   int nl_count = 0;

	   com_buf_next--;	/* decr, already did get_com_char () */
	   str = com_buf_next;	/* point to the start of the string */

	   while (com_buf_next < com_buf_top)
	     {
		c = *com_buf_next;
		if (c >= ' '  || c == '\t' || c == '\n' || c == '\r') /* (isprint(c))*/
		  {
		     com_buf_next++;
		     if (c == '\n')
		       {
			  nl_count++;
			  refresh_nl_count++;

			  if (refresh_nl_count >
			      refresh_nl_limit * RxvtWin.rows)
			    break;
		       }
		  }
		else			/* unprintable */
		  {
		     break;
		  }
	     }
	   scr_add_lines (str, nl_count, com_buf_next - str);
	}
      else
	{
	   switch (c) {
	    case 005:		/* terminal Status */
	      tty_printf (VT100_ANS);
	      break;

	    case 007:		/* bell */
	      scr_bell ();
	      break;
	    case '\b':
	      scr_backspace ();
	      break;
	    case 013:		/* vertical tab */
	    case 014:		/* form feed */
	      scr_index (1);
	      break;
	    case 016:		/* shift out - acs */
	      scr_choose_charset (1);
	      break;
	    case 017:		/* shift in - acs */
	      scr_choose_charset (0);
	      break;
	    case 033:
	      process_escape_seq ();
	      break;
	   }
	}
   } while (c != EOF);
}
/*----------------------- end-of-file (C source) -----------------------*/
