/* X.c - a simple wrapper for X servers that decides whether to let them be
 * run.
 *
 * This program may be distributed under the terms of the GNU public license */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>

#define VT_MAJOR_DEV 4

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif

typedef enum {
  RootOnly,
  Console,
  Anybody
} SecurityLevel;

static SecurityLevel getSecLevel(char *security)
{
  char *c;

  for (c=security; *c; c++) *c=toupper(*c);

  if (strncmp(security,"ROOTONLY",8)==0) return RootOnly;
  if (strncmp(security,"CONSOLE",7)==0) return Console;
  if (strncmp(security,"ANYBODY",7)==0) return Anybody;
  return RootOnly;
}

static int checkSecLevel(SecurityLevel level)
{
  struct stat s;

  switch (level) {
  case RootOnly:
    return FALSE;
    break;
  case Console:
    if (fstat(0,&s)!=0) {
      fprintf(stderr,"X: cannot stat stdin\n");
      return FALSE;
    }
    if (S_ISCHR(s.st_mode) && ((s.st_rdev>>8)&0xff)==VT_MAJOR_DEV &&
	(s.st_rdev&0xff)<128) {
      return TRUE;
    }
    break;
  case Anybody:
    return TRUE;
  }
  return FALSE;
}

int main(int argc, char **argv)
{
  FILE *cf;
  char xserver[1024];
  char security[80];
  char *c;

  if (!(cf=fopen("/etc/X11/Xserver","r"))) {
    fprintf(stderr,"X: cannot find configuration file /etc/X11/Xserver\n"
	    "\nIf you want to run X clients locally you must install an\n"
	    "X server package (eg. xsvga, xs3, xp9000, etc.)\n"
	    "Installation of one of these packages should modify the\n"
	    "configuration file appropriately.\n");
    exit(1);
  }

  if (!fgets(xserver,1024,cf)) {
    fprintf(stderr,"X: cannot read name of X server from /etc/X11/Xserver\n");
    exit(1);
  }

  /* Strip trailing newline from server name */
  for (c=xserver; *c!=0; c++) if (*c=='\n') *c=0;

  if (!fgets(security,80,cf)) {
    fprintf(stderr,"X: cannot read security setting from /etc/X11/Xserver\n");
    exit(1);
  }

  if (getuid()==0 || checkSecLevel(getSecLevel(security))) {
    /* Run the X server */
    execv(xserver,argv);
    fprintf(stderr,"X: exec of %s failed\n",xserver);
    exit(1);
  } else {
    fprintf(stderr,"X: you are not authorised to run the X server\n");
    exit(1);
  }

  exit(1); /* We should never reach this */
}
    
