/* 
 *  File:        math_draw.C
 *  Purpose:     Interaction and drawing for mathed
 *  Author:      Alejandro Aguilar Sierra <asierra@servidor.unam.mx> 
 *  Created:     January 1996
 *  Description: Math drawing and interaction for a WYSIWYG math editor.
 *
 *  Dependencies: Xlib, XForms
 *
 *  Copyright: (c) 1996, Alejandro Aguilar Sierra
 *
 *   Version: 0.5beta, Mathed & Lyx project.
 *
 *   You are free to use and modify this code under the terms of
 *   the GNU General Public Licence version 2 or later.
 */

#include "config.h"
#include <forms.h>
#include "math_draw.h"

extern void mathed_set_font(short type, int style);
extern int mathed_char_width(short type, int style, byte c);
extern int mathed_string_width(short type, int style, byte* s, int ls);
extern int mathed_string_height(short, int, byte*, int, int&, int&);
extern int mathed_char_height(short, int, byte, int&, int&);
   
GC canvasGC=0, mathGC=0, latexGC=0, cursorGC=0;

void LyxMathXIter::SetData(LyxMathInset *pp)
{
   p = pp;
   x = y = 0;
   array = p->GetData();
   if (!array) {
      array = new LyxArrayBase;
      p->SetData(array);
   }
   size = p->GetStyle();
   Reset();
}

byte* LyxMathXIter::GetString(int& ls)
{  
   static byte s[80];
   byte *sx =  LyxMathIter::GetString(ls);
   if (ls>0) {
      if (FCode()==LM_TC_BOP) {
	 byte *ps = &s[0];
	 for (int i=0; i<ls; i++) {
	    *(ps++) = ' ';
	    *(ps++) = sx[i];
	    *(ps++) = ' ';
	 }
//	 *ps = ' ';
	 ls = 3*ls;
      } else
	strncpy((char*)s, (char*)sx, ls);
      x += mathed_string_width(fcode, size, s, ls);
      return &s[0];
   } 	    
   return NULL;
}

Bool LyxMathXIter::Next()
{  
   if (!OK()) return False;
   int w=0;
   if (IsInset()) {
      LyxMathInset* px = GetInset();
     w = px->Width(); 
   } else {  
      byte c = GetChar();
      if (c>=' ') {
	 if (fcode==LM_TC_BOP) {
	    byte s[4] = "   ";
	    s[1] = c;
	    w = mathed_string_width(fcode,size, s, 3);
	 } else	    
	   w = mathed_char_width(fcode, size, c);
      } else
      if (c==LM_TC_TAB && p) {
	 w = p->GetTab(1);
	 //fprintf(stderr, "WW[%d]", w);
      } else 
	fprintf(stderr, "No hubo w[%d]!", (int)c);   
   }
   if (LyxMathIter::Next()) {
      x += w;
      return True;
   } else
     return False;
}


void LyxMathXIter::GoBegin()
{
   Reset();
   x = y = 0;   
   if (p) 
     (void)p->GetTab(-10000);  // This is a dirty trick. I hope you never use
                        //  a matrix with 10,000 columns.  [Alejandro 050596]
}

void LyxMathXIter::GoLast()
{
   while (Next());
}


void LyxMathXIter::Adjust()
{
   int posx = pos;
   GoBegin();
   while (posx>pos) Next();
}

Bool LyxMathXIter::Prev()
{  
   if (LyxMathIter::Prev()) {
      if (IsInset()) {
	 LyxMathInset* px = GetInset();
	 x -= px->Width(); 
      } else {  
	 byte c = GetChar();
	 if (c>=' ') {
	    if (FCode()==LM_TC_BOP) {
	       byte s[4] = "   ";
	       s[1] = c;
	       x -= mathed_string_width(fcode,size, s, 3);
	    } else	    
	      x -= mathed_char_width(fcode, size, c);
	 } else 
	   if (c==LM_TC_TAB && p) { 
	      x -= p->GetTab(0);
	      (void)p->GetTab(-1);
	   }
      }
      return True;
   }
   return False;
}


void LyxMathXIter::Insert(byte c, LyxMathTextCodes t)
{
   if (c>' ') {
      LyxMathIter::Insert(c, t);
      if (t==LM_TC_BOP) {
	 byte s[4] = "   ";
	 s[1] = c;
	 x += mathed_string_width(fcode,size, s, 3);
      } else	    
	x += mathed_char_width(fcode, size, c);
   }
}

void LyxMathXIter::Insert(LyxMathInset* p, int active)
{
   if (p) {
      x += p->Width();
      LyxMathIter::Insert(p, active);      
   }
}


/*
void
MathLatexInset::Draw(long unsigned int pm, int x, int y)
{ 
   mathed_set_font(LM_TC_TEXTRM, size);
   XDrawString(fl_display, pm, latexGC, x, y, name, strlen(name));
}

void MathLatexInset::Metrics() 
{
   if ((unsigned int)ln!=strlen(name)) {
      ln = strlen(name);
      width = mathed_string_width(LM_TC_CONST, size, (byte*)name, ln);
      mathed_string_height(LM_TC_CONST, size, (byte*)name, strlen(name), ascent, descent);
   }
}
*/

void
MathSpaceInset::Draw(long unsigned int pm, int x, int y)
{ 
   XPoint p[4] = {{++x, y-3}, {x, y}, {x+width-1, y}, {x+width-1, y-3}};
   XDrawLines(fl_display, pm, latexGC,p, 4, CoordModeOrigin);
}

void MathSpaceInset::SetSpace(int sp)
{
   space = sp;
   width = space*2+4;
}

void 
MathParInset::Draw(long unsigned pm, int x, int y)
{
   byte cx, cxp=0;
   int xp=0, ls;
   int asc, des;
   // GC gc = mathGC; // unused
           
   xo = x;  yo = y; 
   asc = des = 0;
   if (!array || array->Empty()) {
      mathed_set_font(LM_TC_VAR, 1);
      XDrawRectangle(fl_display,pm,mathGC,x,y-8, 4, 8); 
      return;
   }  
   LyxMathXIter data(this);
   data.Reset();
   while (data.OK()) {
      data.GetPos(x, ls);
      cx = data.GetChar();   
      if (cx >= ' ') {
	 byte *s = data.GetString(ls);
	 mathed_set_font(data.FCode(), size);
	 GC gc = (data.FCode()==LM_TC_TEX) ? latexGC: mathGC;
	 XDrawString(fl_display, pm, gc, x, y, (char*)s, ls);
	 mathed_char_height(data.FCode(), size, s[ls-1], asc, des);
      } else {
	 if (cx==0) break;
	 if (MathIsInset(cx)) {
	    int yy = y;
	    LyxMathInset *p = data.GetInset();
	    if (cx==LM_TC_UP) {
	       xp = (cxp==LM_TC_DOWN) ? xp: x;
	       yy -= asc; 
	    } else
	    if (cx==LM_TC_DOWN) {
	       xp = (cxp==LM_TC_UP) ? xp: x;
	       yy += des + p->Ascent()/2;
	    } else {
	       asc = p->Ascent();
	       des = p->Descent();
	       xp = x;
	    }
	    p->Draw(pm, xp, yy);
	    data.Next();
	 } else 
	   if (cx==LM_TC_TAB) {
	      if ((cxp==cx || data.IsFirst()) && objtype==LM_OT_MATRIX)
		XDrawRectangle(fl_display,pm,mathGC,x,y-8, 4, 8);
//	      fprintf(stderr,"Y[%d] ", y);
	      data.Next();
//	      fprintf(stderr," %d]", y);
	   }
	 else {	 
	    fprintf(stderr, "GLyxMath Error: Unrecognized code[%d]\n", cx);
	    break;
	 }
      }
      cxp = cx;
//      data.GetPos(x, ls);
   }
   if (cxp==LM_TC_TAB && objtype==LM_OT_MATRIX) {
      data.GetPos(x, ls);
      XDrawRectangle(fl_display,pm,mathGC,x,y-8, 4, 8);
   }
}

void 
MathParInset::Metrics()
{
   byte cx, cxp=0, *s;
   int ls;
   int asc, des;
   int tb = 0;
   
   asc = des = ascent = descent = 0;
   if (!array) return;
   if (array->Empty()) {
      ascent = 8;
      width = 4;
      return;
   }
   LyxMathXIter data(this);
   data.Reset();
   while (data.OK()) {
      cx = data.GetChar();      
      if (cx >= ' ') {
	 s = data.GetString(ls);
	 mathed_string_height(data.FCode(), size, s, ls, asc, des);
	 if (asc > ascent) ascent = asc;
	 if (des > descent) descent = des;
      } else
      if (MathIsInset(cx)) {
	 LyxMathInset *p = data.GetInset();
	 int sz = p->GetStyle();
	 if (sz!=size && !(cx==LM_TC_UP || cx==LM_TC_DOWN))
	   p->SetStyle(size);
	 else 
	   if ((cx==LM_TC_UP || cx==LM_TC_DOWN) && size<LM_ST_SCRIPTSCRIPT) {
	      sz = (size<LM_ST_SCRIPT) ? LM_ST_SCRIPT: LM_ST_SCRIPTSCRIPT;
	      p->SetStyle(sz);   
      	   }
	 if (MathIsActive(cx)) 
	   p->Metrics();
	 if (cx==LM_TC_UP) {
	    asc += p->Ascent();
	 } else
	 if (cx==LM_TC_DOWN) {
	    des += p->Ascent();
	 } else {
	    asc = p->Ascent();
	    des = p->Descent();
	 }
	 if (asc > ascent) ascent = asc;
	 if (des > descent) descent = des;
	 data.Next();
      } else 
	if (cx==LM_TC_TAB) {
	   int x, y;
	   data.GetIncPos(x, y);
	   if (data.IsFirst() || cxp==LM_TC_TAB) {
	      if (ascent<8) ascent = 8;
	      SetTab(0);
	      tb = 1;
	   } else {
	      SetTab(x-tb);
	      tb = x;
	   }
	   data.Next();
	}      
      else {
	 fprintf(stderr, "Mathed Error: Unrecognized code[%d]\n", cx);
	 break;
      }       
      cxp = cx;
   }
   data.GetIncPos(width, ls);
   if (tb>0) {
      if (cxp==LM_TC_TAB) {
	 if (ascent<8) ascent = 8;
	 SetTab(0);	 
      } else
	SetTab(width-tb);
   }
}


void
MathSqrtInset::Draw(long unsigned int pm, int x, int y)
{ 
    xo = x;  yo = y; 
   MathParInset::Draw(pm, x+8, yo); 
   int h=ascent, d=descent, h2=Height()/2, w2 = (Height()>25)?6:3;  
   XPoint p[4] = {{x+width-1, yo-h}, {x+6, yo-h}, {x+w2, yo+d}, {x, yo+d-h2}};
   XDrawLines(fl_display, pm, mathGC,p, 4, CoordModeOrigin);
 }

void
MathSqrtInset::Metrics()
{
   MathParInset::Metrics();
   ascent += 2;
   descent += 2;
   width += 10;
}

void
MathDelimInset::Draw(long unsigned int pm, int x, int y)
{ 
   xo = x;  yo = y; 
   MathParInset::Draw(pm, x+8, yo); 
   int h=ascent-1, d=descent-1;  
   XPoint pl[4] = {{x+6, yo-h}, {x+2, yo-h}, {x+2, yo+d}, {x+6, yo+d}};
   x += Width()-8;
   XPoint pr[4] = {{x, yo-h}, {x+4, yo-h}, {x+4, yo+d}, {x, yo+d}};
   XDrawLines(fl_display, pm, mathGC,pl, 4, CoordModeOrigin);
   XDrawLines(fl_display, pm, mathGC,pr, 4, CoordModeOrigin);
}

void
MathDelimInset::Metrics()
{
   MathParInset::Metrics();
   ascent += 2;
   descent += 2;
   width += 20;
}

void
MathFracInset::Draw(long unsigned int pm, int x, int y)
{
   xo = x;  yo = y; 
   num->Draw(pm, x+(width-w0)/2, yo - num->Descent() - 2 - dh);
   den->Draw(pm, x+(width-w1)/2, yo + den->Ascent() + 2 - dh);
   ascent = num->Height() + 2  + dh;
   descent = den->Height()+ 2 - dh;
   XDrawLine(fl_display, pm, mathGC, x+2, yo-dh, x+width-4, yo - dh);
}

void
MathFracInset::Metrics()
{
   if (!dh)
     dh = mathed_char_width(LM_TC_CONST, size, 'I')/2;
   num->Metrics();  den->Metrics();
   w0 = num->Width();  w1 = den->Width();
   width = ((w0 > w1) ? w0: w1) + 12;
   ascent = num->Height() + 2  + dh;
   descent = den->Height()+ 2 - dh;
}


#define MAX_STACK_ITEMS 32

struct MathStackXIter {
   
   int i, imax;
   
   LyxMathXIter *item;
   
   MathStackXIter(int n=MAX_STACK_ITEMS): imax(n) {
      item = new LyxMathXIter[imax];
      i = 0;
   }
   
   ~MathStackXIter() {
      delete[] item;
   }
   
   void push(LyxMathXIter** a) {
      *a = &item[i++];
   }
      
   LyxMathXIter* pop(void) {
      i--;
      return &item[i-1];
   }
      
   LyxMathXIter* Item(int idx) {
      return (idx+1 <= i) ? &item[i-idx-1]: (LyxMathXIter*)NULL;
   }

   void Reset() {
      i = 0;
   }
   
   int Full(void) {
      return (i>=MAX_STACK_ITEMS);
   }
   
   int Empty(void) {
      return (i<=1);
   }
   
   void UpdatePos(int, int);
} mathstk;

void MathStackXIter::UpdatePos(int /*dx*/, int /*dy*/)
{
   fprintf(stderr, "Mathed: Updating pos!\n");
   /*
   int x, y;
   for (int j=0; j<i; j++) {
      item[j].GetOrg(x, y);
      x -= dx; y -= dy;
      item[j].SetOrg(x, y);
   }*/
}

LyxMathCursor::LyxMathCursor(MathParInset *p): par(p)
{
//   flag = 0;
//   win = 0;
//   is_visible = False;
   SetPar(par);
//   mathstk.Reset();
//   mathstk.push(&cursor);
//   cursor->SetData(par);
}


void LyxMathCursor::SetPar(MathParInset */*p*/)
{
   flag = 0;
   win = 0;
   is_visible = False;
   mathstk.Reset();
   mathstk.push(&cursor);
   cursor->SetData(par);
}

void LyxMathCursor::Draw(long unsigned pm, int x, int y)
{
   win = pm;    // win = (mathedCanvas) ? mathedCanvas: pm;
   Hide();
   par->Metrics();
   int w = par->Width(), a = par->Ascent(), h = par->Height();
   if (par->GetType()>LM_OT_PARN) { a += 4;  h += 8; }
   
   if (!canvasGC) mathed_set_font(LM_TC_VAR, 1);
   XFillRectangle(fl_display,pm, canvasGC, x, y-a, w, h);
   par->Draw(pm, x, y);
   if (flag) {
      flag = 0;
      Left();
   }   
   cursor->Adjust();
   Show();
}


void LyxMathCursor::Redraw()
{  
   fprintf(stderr, "Mathed: Redrawing!\n");
   Hide();
   par->Metrics();
   int w = par->Width(), h = par->Height();
   int x, y;
   par->GetXY(x, y);
   mathed_set_font(LM_TC_VAR, 1);
   XFillRectangle(fl_display, win,canvasGC,x, y-par->Ascent(), w, h);
   par->Draw(win, x, y);
   if (flag) {
      flag = 0;
      Left();
   }   
   Show();
}
/* 
void LyxMathCursor::Hide()
{
   if (!cursor) 
      return;
   if (is_visible)
     is_visible = False;
   else
     return;
   XDrawLine(fl_display, win, cursorGC, xc, yc, xc, yc-8);
}

void LyxMathCursor::Show()
{
   if (!cursor) 
      return;  
   if (is_visible)
     return; 
   cursor->GetPos(xc, yc);
   XDrawLine(fl_display, win, cursorGC, xc, yc, xc, yc-8);
   is_visible = True;
}
*/

bool LyxMathCursor::Left()
{
   bool result = cursor->Prev();
   Hide();
   if (!result && !mathstk.Empty()) {
      cursor = mathstk.pop();
      cursor->Adjust();
      result = true;
   } else  
     if (result && cursor->IsActive()) {
	LyxMathInset *p = cursor->GetInset();
	mathstk.push(&cursor);
	cursor->SetData(p);
	cursor->GoLast();
     } 
   Show();
   return result;  
}

bool LyxMathCursor::Right()
{   
   bool result = false;
   Hide();
   if (cursor->IsActive()) {
      LyxMathInset *p = cursor->GetInset();
      mathstk.push(&cursor);
      cursor->SetData(p);
      result = true;
   } else
     if (!cursor->Next() && !mathstk.Empty()) {
	cursor = mathstk.pop();
	cursor->Next();
	result = true;
     }
   Show();
   return result;
}


void LyxMathCursor::SetPos(int x, int y)
{
   Hide();
   mathstk.Reset();
   mathstk.push(&cursor);
   par->SetDataXY(x, y);
   cursor->SetData(par);
   while (cursor->GetX()<x) {
      if (cursor->IsActive()) {
	 LyxMathInset *p = cursor->GetInset();
	 if (p->Inside(x, y)) p->SetDataXY(x, y); 
      }
      Right();
      if (mathstk.Empty() && !cursor->OK())
	break;
   }
   Show();
}
   

void LyxMathCursor::Home()
{
   Hide();
   mathstk.Reset();
   mathstk.push(&cursor);
   cursor->GoBegin();
   Show();
}

void LyxMathCursor::End()
{
   Hide();
   mathstk.Reset();
   mathstk.push(&cursor);
   cursor->GoLast();
   Show();
}

void LyxMathCursor::Insert(byte c, LyxMathTextCodes t)
{  
   if (t==LM_TC_CR) {
      LyxMathInset *p= cursor->p;
      if (p==par && p->GetType()<LM_OT_MPAR && p->GetType()>LM_OT_MIN) {
	 MathMatrixInset* mt = new MathMatrixInset(3);
	 mt->SetAlign(' ', "rcl");
	 mt->SetStyle(LM_ST_DISPLAY);
	 mt->SetType((p->GetType()==LM_OT_PARN) ? LM_OT_MPARN: LM_OT_MPAR);
	 mt->SetData(p->GetData());
	 p->SetData(NULL);
	 delete p;
	 par = mt;
	 cursor->SetData(par);
	 p = mt;
      }
      
      if (p &&  (p->GetType()<=LM_OT_MATRIX && p->GetType()>=LM_OT_MPAR)) {
	 ((MathMatrixInset *)p)->AddRow();
	 cursor->SetData(p);
      }
   } else
   if (t==LM_TC_TAB) {
      LyxMathInset *p = cursor->p;
      if (p &&  (p->GetType()==LM_OT_MATRIX || p->GetType()==LM_OT_MPAR)) {
	 int ct = ((MathMatrixInset *)p)->GetTabPos();
	 while (ct==((MathMatrixInset *)p)->GetTabPos() && cursor->Next());
      }
   } else
     cursor->Insert(c, t);
}

void LyxMathCursor::Insert(LyxMathInset* p, int t)
{
   if (mathstk.i<MAX_STACK_ITEMS-1) {
      cursor->Insert(p, t);
      if (MathIsActive(t)) {
	 flag = 1;
      }
   } else
     fprintf(stderr, "Math error: Full stack.\n");
}

void LyxMathCursor::Delete() 
{
   if (cursor->Empty() && !mathstk.Empty()) {
      cursor = mathstk.pop();
   } 
   if (cursor->GetChar()!=LM_TC_TAB)
     cursor->Delete();
}

void LyxMathCursor::DelLine()
{  
   LyxMathInset *p= cursor->p;
   if (p &&  (p->GetType()<=LM_OT_MATRIX && p->GetType()>=LM_OT_MPAR)) {
      if (!((MathMatrixInset *)p)->DelRow() && !mathstk.Empty()) {
	 cursor = mathstk.pop();
	 cursor->Delete();
      } else  
	cursor->SetData(p);
   }
}

bool LyxMathCursor::Up()
{
   bool result = false;
   Hide();
   LyxMathInset *p= cursor->p;   
   if (p) {
      switch (p->GetType()) {
       case LM_OT_SCRIPT:
	 {
	    LyxMathXIter *cx = mathstk.Item(1);
	    if (cx->GetChar()==LM_TC_DOWN) {
	       cursor = mathstk.pop();
	       cursor->Next();
	       result = true;
	    }
	    break;
	 }
       case LM_OT_FRAC:
	 result = ((MathFracInset *)p)->Up();
	 cursor->SetData(p);
	 break;
       case LM_OT_MPAR:
       case LM_OT_MPARN:
       case LM_OT_MATRIX:	    
	 int ct = p->GetTabPos();
	 if ((result=((MathMatrixInset *)p)->Up())) {
	    cursor->SetData(p);
	    while (ct>p->GetTabPos() && cursor->Next());
	 }
	 break;
      }
      if (!result && !mathstk.Empty()) {
         cursor = mathstk.pop();
         return Up();
      }     
   }
   Show();
   return result;
}
      
bool LyxMathCursor::Down()
{
   bool result = false;
   Hide();
   LyxMathInset *p= cursor->p;   
   if (p) {
      switch (p->GetType()) {
       case LM_OT_SCRIPT:
	 {
	    LyxMathXIter *cx = mathstk.Item(1);
	    if (cx->GetChar()==LM_TC_UP) {
	       cursor = mathstk.pop();
	       cursor->Next();
	       result = true;
	    }
	    break;
	 }
       case LM_OT_FRAC:
	 result = ((MathFracInset *)p)->Down();
	 cursor->SetData(p);
	 break;
       case LM_OT_MPAR:
       case LM_OT_MPARN:
       case LM_OT_MATRIX:
	 int ct = ((MathMatrixInset *)p)->GetTabPos();
//	 fprintf(stderr, "DW[%d]", ct);
	 result = ((MathMatrixInset *)p)->Down();
	 cursor->SetData(p);
	 while (ct>((MathMatrixInset *)p)->GetTabPos()) cursor->Next();
//	 fprintf(stderr, "DW[%d %d]", ct,(MathMatrixInset *)p->GetTabPos());
	 break;
      }
      if (!result && !mathstk.Empty()) {
	 cursor = mathstk.pop();
	 return Down();
      }    
   }
   Show();
   return result;
}

