/*
 * Simple Network Management Protocol (RFC 1067).
 *
 */
/**********************************************************************
	Copyright 1988, 1989, 1991, 1992 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/

#ifdef KINETICS
#include "gw.h"
#include "ab.h"
#include "inet.h"
#include "fp4/cmdmacro.h"
#include "fp4/pbuf.h"
#include "glob.h"
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#ifdef linux
# include <stdlib.h>
# include <ctype.h>
#endif

#if (defined(unix) && !defined(KINETICS))
#include <sys/types.h>
#include <netinet/in.h>
#ifndef NULL
#define NULL 0
#endif
#endif

#ifdef vms
#include <in.h>
#endif

#include "asn1.h"
#include "snmp.h"
#include "snmp_impl.h"

#include "mib.h"

/* tell something... */
int verbose = 0;

void
xdump(cp, length, prefix)
    u_char *cp;
    int length;
    char *prefix;
{
    int col, count;

    count = 0;
    while(count < length){
	printf("%s", prefix);
	for(col = 0;count + col < length && col < 16; col++){
	    if (col != 0 && (col % 4) == 0)
		printf(" ");
	    printf("%02X ", cp[count + col]);
	}
	while(col++ < 16){	/* pad end of buffer with zeros */
	    if ((col % 4) == 0)
		printf(" ");
	    printf("   ");
	}
	printf("  ");
	for(col = 0;count + col < length && col < 16; col++){
	    if (isprint(cp[count + col]))
		printf("%c", cp[count + col]);
	    else
		printf(".");
	}
	printf("\n");
	count += col;
    }

}



u_char *
snmp_parse_var_op(data, var_name, var_name_len, var_val_type, var_val_len, var_val, listlength)
    register u_char *data;  /* IN - pointer to the start of object */
    oid	    *var_name;	    /* OUT - object id of variable */
    int	    *var_name_len;  /* IN/OUT - length of variable name */
    u_char  *var_val_type;  /* OUT - type of variable (int or octet string) (one byte) */
    int	    *var_val_len;   /* OUT - length of variable */
    u_char  **var_val;	    /* OUT - pointer to ASN1 encoded value of variable */
    int	    *listlength;    /* IN/OUT - number of valid bytes left in var_op_list */
{
    u_char	    var_op_type;
    int		    var_op_len = *listlength;
    u_char	    *var_op_start = data;

    data = asn_parse_header(data, &var_op_len, &var_op_type);
    if (data == NULL){
	ERROR("snmp_parse_var_op(): 1 asn_parse_header() == NULL");
	return NULL;
    }
    if (var_op_type != (u_char)(ASN_SEQUENCE | ASN_CONSTRUCTOR))
	return NULL;
    data = asn_parse_objid(data, &var_op_len, &var_op_type, var_name, var_name_len);
    if (data == NULL){
	ERROR("snmp_parse_var_op(): asn_parse_objid() == NULL");
	return NULL;
    }
    if (var_op_type != (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_OBJECT_ID))
	return NULL;
    *var_val = data;	/* save pointer to this object */
    /* find out what type of object this is */
    data = asn_parse_header(data, &var_op_len, var_val_type);
    if (data == NULL){
	ERROR("snmp_parse_var_op(): 2 asn_parse_header() == NULL");
	return NULL;
    }
    *var_val_len = var_op_len;
    data += var_op_len;
    *listlength -= (int)(data - var_op_start);
    return data;
}

u_char *
snmp_build_var_op(data, var_name, var_name_len, var_val_type, var_val_len,
		  var_val, listlength)
    register u_char *data;	/* IN - pointer to the beginning of the output buffer */
    oid		*var_name;	/* IN - object id of variable */
    int		*var_name_len;	/* IN - length of object id */
    u_char	var_val_type;	/* IN - type of variable */
    int		var_val_len;	/* IN - length of variable */
    u_char	*var_val;	/* IN - value of variable */
    register int *listlength;   /* IN/OUT - number of valid bytes left in
				   output buffer */
{
    int		    dummyLen, headerLen, header_shift;
    u_char	    *dataPtr;

    dummyLen = *listlength;
    dataPtr = data;
#if 0
    data = asn_build_sequence(data, &dummyLen,
			      (u_char)(ASN_SEQUENCE | ASN_CONSTRUCTOR), 0);
    if (data == NULL){
	ERROR("");
	return NULL;
    }
#endif
    data += 4;
    dummyLen -=4;
    if (dummyLen < 0)
	return NULL;

    headerLen = data - dataPtr;
    *listlength -= headerLen;
    data = asn_build_objid(data, listlength,
	    (u_char)(ASN_UNIVERSAL | ASN_PRIMITIVE | ASN_OBJECT_ID),
	    var_name, *var_name_len);
    if (data == NULL){
	ERROR("");
	return NULL;
    }
    switch(var_val_type){
	case ASN_INTEGER:
	    data = asn_build_int(data, listlength, var_val_type,
		    (long *)var_val, var_val_len);
	    break;
	case GAUGE:
	case COUNTER:
	case TIMETICKS:
	case UINTEGER:
	    data = asn_build_unsigned_int(data, listlength, var_val_type,
					  (u_long *)var_val, var_val_len);
	    break;
	case COUNTER64:
	    data = asn_build_unsigned_int64(data, listlength, var_val_type,
					   (struct counter64 *)var_val,
					    var_val_len);
	    break;
	case ASN_OCTET_STR:
	case IPADDRESS:
	case OPAQUE:
        case NSAP:
	    data = asn_build_string(data, listlength, var_val_type,
		    var_val, var_val_len);
	    break;
	case ASN_OBJECT_ID:
	    data = asn_build_objid(data, listlength, var_val_type,
		    (oid *)var_val, var_val_len / sizeof(oid));
	    break;
	case ASN_NULL:
	    data = asn_build_null(data, listlength, var_val_type);
	    break;
	case ASN_BIT_STR:
	    data = asn_build_bitstring(data, listlength, var_val_type,
		    var_val, var_val_len);
	    break;
	case SNMP_NOSUCHOBJECT:
	case SNMP_NOSUCHINSTANCE:
	case SNMP_ENDOFMIBVIEW:
	    data = asn_build_null(data, listlength, var_val_type);
	    break;
	default:
	    ERROR("wrong type");
	    return NULL;
    }
    if (data == NULL){
	ERROR("");
	return NULL;
    }
    dummyLen = (data - dataPtr) - headerLen;

    asn_build_sequence(dataPtr, &dummyLen,
		       (u_char)(ASN_SEQUENCE | ASN_CONSTRUCTOR), dummyLen);
    return data;
}


/*
 * store descriptions of interfaces:
 */

conf_if_list *if_list = 0;

static void 
add_interface_description (val)
char *val;
{
  char *name, *type = 0, *speed = 0;
  conf_if_list *nnew;

  name = strtok (val, " \t");
  if (name) type = strtok ((char *) 0, " \t");
  if (type) speed = strtok ((char *) 0, " \t");
  if (! type)
    {
      fprintf (stderr, 
   "warning: reading config: missing type for interface `%s'\n",
	       name ? name : "<unknown>");
      return;
    }

  if (! (nnew = (conf_if_list *) malloc (sizeof (conf_if_list)))
      || ! (nnew->name = strdup (name)))
    {
      fprintf (stderr, "error: reading config: out of memory...aborting.\n");
      exit (1);
    }
  
  nnew->type = atoi (type);
  nnew->speed = speed ? atoi (speed) : 0;

  if (verbose)
    printf ("added from config:  name: %s  type: %d  speed: %d\n", 
	    nnew->name, nnew->type, nnew->speed);

  nnew->next = if_list;
  if_list = nnew;
}


/*
 * read the main-config file (if avail) and initialize the path's 
 * to the {context,party,view,acl}.conf files.
 */

static int main_config_read = 0;		/* still read ? */

/* ``/etc/snmpd.conf'' prefix can be overridden by the env-var
 * ``SNMPCONFIGFILE'' */
char *main_config_fname = "/etc/snmpd.conf";

static char *party_conf_fname = "/etc/party.conf";
static char *view_conf_fname = "/etc/view.conf";
static char *acl_conf_fname = "/etc/acl.conf";
static char *context_conf_fname = "/etc/context.conf";

/* exported to ../agent/snmp_vars.c: */
char sysContact[256] = "Unknown";
char sysLocation[256] = "Unknown";
char sysName[256] = "";
char *public_community = 0;
char *private_community = 0;

/* port read from the configfile: */
int conf_snmp_port = -1;

/* trapsink host and community; setable by configfile: */
char trap_sink [256] = "localhost";
char trap_community [256] = "public";
int conf_authentraps = 0;

void
read_main_config_file ()
{
  FILE *in;
  char line [1024];

  if (main_config_read)
    return;

  /* only do this once: */
  main_config_read = 1;

  /* init path's: */
  { char *pfx = getenv ("SNMPCONFIGFILE");
    if (pfx && (pfx = strdup (pfx)))
      main_config_fname = pfx;
  }

  if (verbose)
    printf ("reading main config file: %s\n", main_config_fname);

  if (! (in = fopen (main_config_fname, "r")))
    {
      fprintf (stderr, "warning: cannot open %s - using default paths.\n",
	       main_config_fname);
      return;
    }

  while (fgets (line, sizeof (line), in))
    {
      char *key, *val;

      if (! *line || *line == '\n' || *line == '#')
	continue;
      if (line [strlen (line) - 1] == '\n')
	line [strlen (line) - 1] = 0;

      if (! (val = strchr (line, ':')))
	{
	  fprintf (stderr, 
   "warning: reading config: don't know what to do with this line:\n\t%s\n",
		   line);
	  continue;
	}

      key = line;

      for (*val++ = 0; *val == ' ' || *val == '\t'; val++)
	continue;

      /* okey dokey; now we have a key and a value: */

/** printf ("got key `%s' and val `%s'\n", key, val); **/
      
      if (! strcmp (key, "party.conf"))
	{
	  party_conf_fname = strdup (val);
	  if (access (party_conf_fname, R_OK))
	    fprintf (stderr, "reading config: warning: cannot access %s\n",
		     party_conf_fname);
	}
      else if (! strcmp (key, "view.conf"))
	{
	  view_conf_fname = strdup (val);
	  if (access (view_conf_fname, R_OK))
	    fprintf (stderr, "reading config: warning: cannot access %s\n",
		     view_conf_fname);
	}
      else if (! strcmp (key, "context.conf"))
	{
	  context_conf_fname = strdup (val);
	  if (access (context_conf_fname, R_OK))
	    fprintf (stderr, "reading config: warning: cannot access %s\n",
		     context_conf_fname);
	}
      else if (! strcmp (key, "acl.conf"))
	{
	  acl_conf_fname = strdup (val);
	  if (access (acl_conf_fname, R_OK))
	    fprintf (stderr, "reading config: warning: cannot access %s\n",
		     acl_conf_fname);
	}
      else if (! strcmp (key, "interface"))
	add_interface_description (val);
      else if (! strcmp (key, "system contact"))
	{
	    int len = sizeof (sysContact);
	    if (strlen (val) < len)
	      strcpy (sysContact, val);
	    else {
		strncpy (sysContact, val, len - 1);
		sysContact [len - 1] = 0;
	    }

	    if (verbose)
	      printf ("added from config: system contact is %s\n", sysContact);
	}
      else if (! strcmp (key, "system location"))
	{
	    int len = sizeof (sysLocation);
	    if (strlen (val) < len)
	      strcpy (sysLocation, val);
	    else {
		strncpy (sysLocation, val, len - 1);
		sysLocation [len - 1] = 0;
	    }
	    
	    if (verbose)
	      printf ("added from config: system location is %s\n", 
		      sysLocation);
	}
      else if (! strcmp (key, "system name"))
	{
	    int len = sizeof (sysName);
	    if (strlen (val) < len)
	      strcpy (sysName, val);
	    else {
		strncpy (sysName, val, len - 1);
		sysName [len - 1] = 0;
	    }
	    
	    if (verbose)
	      printf ("added from config: system name is %s\n", 
		      sysName);
	}
      else if (! strcmp (key, "public") || ! strcmp (key, "private"))
	{
	    static char comm [256];
	    char *s = 0;
	    int err = 0;
	    
	    if (! (s = strdup (val)))
	      {
		  fprintf (stderr, "snmpd: out of mem - over and out.\n");
		  exit (1);
	      }

	    while (isspace (comm [strlen (s) - 1]))
	      s [strlen (s) - 1] = 0;
	    while (*s && isspace (*s))
	      s++;
	    if (! *s)
	      {
		  fprintf (stderr,
			   "snmpd: empty %s community string.\n", key);
		  err = 1;
	      }
	    
	    if (! err) {
		if (! strcmp (key, "private"))
		  private_community = s;
		else
		  public_community = s;
	    }
	    else
	      fprintf (stderr, 
		       "snmpd: default %s community will be used.\n", key);
	}
      else if (! strcmp (key, "port"))
	{
	    unsigned short p = atoi (val);
	    if (! val)
	      fprintf (stderr, "snmpd: no valid port found.\n");
	    else {
		conf_snmp_port = p;
		if (verbose)
		  printf ("port set to %d\n", conf_snmp_port);
	    }
	}
      else if (! strcmp (key, "trap sink"))
	{
	    int len = sizeof (trap_sink);
	    if (strlen (val) < len)
	      strcpy (trap_sink, val);
	    else {
		strncpy (trap_sink, val, len - 1);
		trap_sink [len - 1] = 0;
	    }
	    
	    if (verbose)
	      printf ("added from config: trap sink addess is %s\n", 
		      trap_sink);
	}
      else if (! strcmp (key, "trap community"))
	{
	    int len = sizeof (trap_community);
	    if (strlen (val) < len)
	      strcpy (trap_community, val);
	    else {
		strncpy (trap_community, val, len - 1);
		trap_community [len - 1] = 0;
	    }
	    
	    if (verbose)
	      printf ("added from config: trap community string is %s\n", 
		      trap_community);
	}
      else if (! strcmp (key, "authentraps"))
	{
	    if (! strcmp (val, "yes"))
	      conf_authentraps = 1;
	    else if (! strcmp (val, "no"))
	      conf_authentraps = 2;
	    else
	      fprintf (stderr, 
		       "warning: reading config: unknown val for %s\n", key);
	    
	    if (verbose)
	      printf ("added from config: authentraps set to %s\n", val);
	}
      else
	  fprintf (stderr, 
		   "warning: reading config: unknown key `%s'\n", key);
    }
  fclose (in);
}

char *party_conf ()
{
  if (! main_config_read)
    read_main_config_file ();
  return party_conf_fname;
}

char *view_conf ()
{
  if (! main_config_read)
    read_main_config_file ();
  return view_conf_fname;
}

char *acl_conf ()
{
  if (! main_config_read)
    read_main_config_file ();
  return acl_conf_fname;
}

char *context_conf ()
{
  if (! main_config_read)
    read_main_config_file ();
  return context_conf_fname;
}
