/*** File fitslib/hget.c
 *** August 7, 1995
 *** By Doug Mink, Harvard-Smithsonian Center for Astrophysics

 * Module:	hget.c (Get FITS Header parameter values)
 * Purpose:	Extract values for variables from FITS header string
 * Subroutine:	hgeti2 (hstring,keyword) returns short integer
 * Subroutine:	hgeti4 (hstring,keyword) returns long integer
 * Subroutine:	hgetr4 (hstring,keyword) returns real
 * Subroutine:	hgetr8 (hstring,keyword) returns double
 * Subroutine:	hgetl  (hstring,keyword) returns logical int (0=F, 1=T)
 * Subroutine:	hgetc  (hstring,keyword) returns character string (drops quotes)
 * Subroutine:	ksearch (hstring,keyword) returns pointer to entry in FITS header string

 * Copyright:   1995 Smithsonian Astrophysical Observatory
 *              You may do anything you like with this file except remove
 *              this copyright.  The Smithsonian Astrophysical Observatory
 *              makes no representations about the suitability of this
 *              software for any purpose.  It is provided "as is" without
 *              express or implied warranty.

*/

#include <string.h>		/* NULL, strlen, strstr, strcpy */
#include <stdio.h>
#include "hfiles/fitshead.h"	/* FITS header extraction subroutines */
#include <stdlib.h>

char val[30];

/* hgeti4  extract long value for variable from FITS header string */

int hgeti4 (hstring,keyword,ival)

char *hstring;	/* character string containing FITS header information
		   in the format <keyword>= <value> {/ <comment>} */
char *keyword;	/* character string containing the name of the variable
		   to be returned.  hget searches for a line beginning
		   with this string.  if "[n]" is present, the n'th
		   token in the value is returned.
		   (the first 8 characters must be unique) */
int *ival;
{
char *value;

/* Get value and comment from header string */

	value = hgetc (hstring,keyword);

/* Translate value from ASCII to binary */

	if (value != NULL) {
	    strcpy (val, value);
	    *ival = (int) atol (val);
	    return (1);
	    }
	else {
	    return (0);
	    }
}


/* hgeti2  extract integer*2 value for variable from fits header string */

int hgeti2 (hstring,keyword,ival)

char *hstring;	/* character string containing FITS header information
		   in the format <keyword>= <value> {/ <comment>} */
char *keyword;	/* character string containing the name of the variable
		   to be returned.  hget searches for a line beginning
		   with this string.  if "[n]" is present, the n'th
		   token in the value is returned.
		   (the first 8 characters must be unique) */
short *ival;
{
char *value;

/* Get value and comment from header string */

	value = hgetc (hstring,keyword);

/* Translate value from ASCII to binary */

	if (value != NULL) {
	    strcpy (val, value);
	    *ival = (short) atoi (val);
	    return (1);
	    }
	else {
	    return (0);
	    }
}

/* hgetr4  extract real value for variable from FITS header string */

int hgetr4 (hstring,keyword,rval)

char *hstring;	/* character string containing FITS header information
		   in the format <keyword>= <value> {/ <comment>} */
char *keyword;	/* character string containing the name of the variable
		   to be returned.  hget searches for a line beginning
		   with this string.  if "[n]" is present, the n'th
		   token in the value is returned.
		   (the first 8 characters must be unique) */
float *rval;
{
	char *value;

/* Get value and comment from header string */

	value = hgetc (hstring,keyword);

/* translate value from ASCII to binary */

	if (value != NULL) {
	    strcpy (val, value);
	    *rval = (float) atof (val);
	    return (1);
	    }
	else {
	    return (0);
	    }
}


/* hgetr8  extract real*8 value for variable from FITS header string */

int hgetr8 (hstring,keyword,dval)

char *hstring;	/* character string containing FITS header information
		   in the format <keyword>= <value> {/ <comment>} */
char *keyword;	/* character string containing the name of the variable
		   to be returned.  hget searches for a line beginning
		   with this string.  if "[n]" is present, the n'th
		   token in the value is returned.
		   (the first 8 characters must be unique) */
double *dval;
{
	char *value,val[30];

/* Get value and comment from header string */

	value = hgetc (hstring,keyword);

/* Translate value from ASCII to binary */

	if (value != NULL) {
	    strcpy (val, value);
	    *dval = atof (val);
	    return (1);
	    }
	else {
	    return (0);
	    }
}


/* hgetl  extract logical value for variable from FITS header string */

int hgetl (hstring,keyword,ival)

char *hstring;	/* character string containing FITS header information
		   in the format <keyword>= <value> {/ <comment>} */
char *keyword;	/* character string containing the name of the variable
		   to be returned.  hget searches for a line beginning
		   with this string.  if "[n]" is present, the n'th
		   token in the value is returned.
		   (the first 8 characters must be unique) */
int *ival;
{
	char *value;

/* Get value and comment from header string */

	value = hgetc (hstring,keyword);
	strcpy (val, value);
        value = &val[0];

/* Translate value from ASCII to binary */

	if (value != NULL) {
	    if (value[0] == 't' || value[0] == 'T')
		*ival = 1;
	    else
		*ival = 0;
	    return (1);
	    }
	else {
	    return (0);
	    }
}

/* hgetc  extract character value for variable from FITS header string */

char *hgetc (hstring,keyword0)

char *hstring;
char *keyword0;
{
	static char cval[80];
	char *value;
	char cwhite[2];
	char squot[2],dquot[2],lbracket[2],rbracket[2],slash[2];
	char keyword[16];
	char line[100];
	char *vpos,*cpar;
	char *q1, *q2, *v1, *v2, *c1, *brack1, *brack2;
	char *strsrch();
	int ipar, i;

	squot[0] = 39;
	squot[1] = 0;
	dquot[0] = 34;
	dquot[1] = 0;
	lbracket[0] = 91;
	lbracket[1] = 0;
	rbracket[0] = 93;
	rbracket[1] = 0;
	slash[0] = 47;
	slash[1] = 0;

/* Find length of variable name */
	strcpy (keyword,keyword0);
	brack1 = strsrch (keyword,lbracket);
	if (brack1 != NULL) *brack1 = NULL;

/* Search header string for variable name */
	vpos = ksearch (hstring,keyword);

/* Exit if not found */
	if (vpos == NULL) {
	    return (NULL);
	    }

/* Initialize line to nulls */
	 for (i = 0; i < 100; i++)
	    line[i] = 0;

/* In standard FITS, data lasts until 80th character */

/* Extract entry for this variable from the header */
	strncpy (line,vpos,80);

/* check for quoted value */

	q1 = strsrch (line,squot);
	c1 = strsrch (line,slash);
	if (q1 != NULL) {
	    if (c1 != NULL && q1 < c1)
		q2 = strsrch (q1+1,squot);
	    else if (c1 == NULL)
		q2 = strsrch (q1+1,squot);
	    else
		q1 = NULL;
	    }
	else {
	    q1 = strsrch (line,dquot);
	    if (q1 != NULL) {
		if (c1 != NULL && q1 < c1)
		    q2 = strsrch (q1+1,dquot);
		else if (c1 == NULL)
		    q2 = strsrch (q1+1,dquot);
		else
		    q1 = NULL;
		}
	    else {
		q1 = NULL;
		q2 = line + 10;
		}
	    }

/* Extract value and remove excess spaces */
	if (q1 != NULL) {
	    v1 = q1 + 1;;
	    v2 = q2 - 1;
	    c1 = strsrch (q2+1,"/");
	    }
	else {
	    v1 = strsrch (line,"=") + 1;
	    c1 = strsrch (line,"/");
	    if (c1 != NULL)
		v2 = c1 - 1;
	    else
		v2 = line + 79;
	    }

/* Ignore leading spaces */
	while (*v1 == ' ' && v1 < v2) {
	    v1++;
	    }

/* Drop trailing spaces */
	while (*v2 == ' ' && v2 > v1) {
	    *v2 = 0;
	    v2--;
	    }

	strcpy (cval,v1);
	value = cval;

/* If keyword has brackets, extract appropriate token from value */
	if (brack1 != NULL) {
	    brack2 = strsrch (keyword,rbracket);
	    if (brack2 != NULL) {
		*brack2 = NULL;
		ipar = atoi (brack1);
		if (ipar > 0) {
		    cwhite[0] = 32;
		    cwhite[1] = NULL;
		    for (i = 1; i <= ipar; i++) {
			cpar = strtok (v1,cwhite);
			}
		    if (cpar != NULL) {
			strcpy (cval,cpar);
			}
		    else
			value = NULL;
		    }
		}
	    }

	return (value);
}

char *ksearch (hstring,keyword)

/* Find entry for keyword keyword in FITS header string hstring.
   (the keyword may have a maximum of eight letters)
   NULL is returned if the keyword is not found */

char *hstring;	/* character string containing fits-style header
		information in the format <keyword>= <value> {/ <comment>}
		the default is that each entry is 80 characters long;
		however, lines may be of arbitrary length terminated by
		nulls, carriage returns or linefeeds, if packed is true.  */
char *keyword;	/* character string containing the name of the variable
		to be returned.  ksearch searches for a line beginning
		with this string.  The string may be a character
		literal or a character variable terminated by a null
		or '$'.  it is truncated to 8 characters. */
{
char *loc, *headnext, *headlast, *pval;
int icol, icol0, nline, prevchar, nextchar, lkey;
char *strsrch();

	pval = 0;

/* Search header string for variable name */
	headlast = hstring + strlen (hstring);
	headnext = hstring;
	pval = NULL;
	while (headnext < headlast) {
	    loc = strsrch (headnext,keyword);

	/* Exit if keyword is not found */
	    if (loc == NULL) {
		break;
		}

	/* If parameter name in string entry is longer, keep searching */
	    lkey = strlen (keyword);
	    nextchar = (int) *(loc + lkey);
	    if (nextchar != 61 && nextchar > 32 && nextchar < 127) {
		headnext = loc + 1;
		}

	/* See if this is a legal position for a keyword.  if not, assume
	   it is in the comment or value  field, and keep searching. */
	    else {
		icol = loc - hstring + 1;
		nline = icol / 80;
		icol0 = (80 * nline) + 1;
		icol = icol - icol0;
		if (icol > 8) {
		    headnext = loc + 1;
		    }
		else if (icol > 1) {
		    prevchar = (int) *(loc - 1);
		    if (prevchar > 32 && prevchar < 127) {
			headnext = loc + 1;
			}
		    }
		else {
		    pval = loc;
		    break;
		    }
		}
	    }

/* Return pointer to calling program */
	return (pval);
}


char *strsrch (s1, s2)

char *s1;	/* String to search */
char *s2;	/* String to look for */

{
    char *s,*s1e;
    char cfirst,clast;
    int i,ls1,ls2;

    /* Return null string if either pointer is NULL */
    if (s1 == NULL || s2 == NULL)
	return (NULL);

    /* A zero-length pattern is found in any string */
    ls2 = strlen (s2);
    if (ls2 ==0)
	return (s1);

    /* Only a zero-length string can be found in a zero-length string */
    ls1 = strlen (s1);
    if (ls1 ==0)
	return (NULL);

    cfirst = s2[0];
    clast = s2[ls2-1];
    s1e = s1 + ls1 - ls2;
    s = s1;
    while (s < s1e) { 

	/* Search for first character in pattern string */
	if (*s == cfirst) {

	    /* If single character search, return */
	    if (ls2 == 1)
		return (s);

	    /* Search for last character in pattern string if first found */
	    if (s[ls2-1] == clast) {

		/* If two-character search, return */
		if (ls2 == 2)
		    return (s);

		/* If 3 or more characters, check for rest of search string */
		i = 1;
		while (i++ < ls2 && s[i] == s2[i])
		    ;

		/* If entire string matches, return */
		if (i == ls2)
		    return (s);
		}
	    }
	s++;
	}
    return (NULL);
}
/* Oct 28 1994	New program
 *
 * Mar  1 1995	Search for / after second quote, not first one
 * May  2 1995	Initialize line in HGETC; deal with logicals in HGETL better
 * May  4 1995	Declare STRSRCH in KSEARCH
 * Aug  7 1995  Fix line initialization in HGETC
 */
