/*	Copyright (C) 1995 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */


#include <stdio.h>
#include "debug.h"
#include "gscm.h"
#include "_scm.h"





/* {Mode Flags}
 */

/* This flag tells the interpreter whether it should use the debugging
 * evaluator or the normal one. 
 */

int scm_debug_state;
int scm_check_entry;
int scm_check_exit;

SCM (*scm_ceval_ptr) P ((SCM, SCM));

struct scm_mode_flag scm_debug_flags[] =
{
  { "procnames", 1 },
  { "calls", 0 },
  { "step", 0 },
  { "breakpoints", 0 },
  { "backtrace", 0 }
};

struct scm_mode_flag scm_read_flags[] = 
{
  { "positions", 0 },
  { "copy", 0 }
};

struct scm_mode_flag scm_print_flags[] = 
{
  { "procnames", 0 },
};

#ifdef __STDC__
static SCM
change_mode (SCM new_mode, mode_flag mode_flags[], int n, char *s)
#else
static SCM
change_mode (new_mode, mode_flags, n, s)
     SCM new_mode;
     mode_flag mode_flags[];
     int n;
     char *s;
#endif
{
  int i;
  SCM old = EOL;
  for (i = 0; i < n; ++i)
    if (mode_flags[i].val)
      old = scm_cons (mode_flags[i].sym, old);
  if (!UNBNDP (new_mode))
    {
      int flags[n];
      for (i = 0; i < n; ++i) flags[i] = 0;
      while (NNULLP (new_mode))
	{
	  ASSERT (NIMP (new_mode) && CONSP (new_mode), new_mode, ARG1, s);
	  for (i = 0; i < n; ++i)
	    {
	      if (CAR (new_mode) == mode_flags[i].sym)
		{
		  flags[i] = 1;
		  break;
		}
	      ASSERT (i < n,
		      new_mode,
		      "Unknown debug flag",
		      s);
	    }
	  new_mode = CDR (new_mode);
	}
      for (i = 0; i < n; ++i) mode_flags[i].val = flags[i];
    }
  return old;
}


PROC (s_debug_mode, "debug-mode", 0, 1, 0, scm_debug_mode);
#ifdef __STDC__
SCM
scm_debug_mode (SCM new_mode)
#else
SCM
scm_debug_mode (new_mode)
     SCM new_mode;
#endif
{
  SCM ans;
  DEFER_INTS;
  ans = change_mode (new_mode, scm_debug_flags, N_DEBUG_FLAGS, s_debug_mode);
  if (SINGLE_STEP || BREAKPOINTS) CHECK_ENTRY = 1;
  if (SINGLE_STEP) CHECK_EXIT = 1;
  scm_debug_state = TRACE_CALLS || SINGLE_STEP || BREAKPOINTS;
  scm_ceval_ptr = scm_debug_state ? scm_deval : scm_ceval;
  ALLOW_INTS
  return ans;
}

PROC (s_read_mode, "read-mode", 0, 1, 0, scm_read_mode);
#ifdef __STDC__
SCM
scm_read_mode (SCM new_mode)
#else
SCM
scm_read_mode (new_mode)
     SCM new_mode;
#endif
{
  SCM ans = change_mode (new_mode, scm_read_flags, N_READ_FLAGS, s_read_mode);
  if (COPY_SOURCE)
    RECORD_POSITIONS = 1;
  return ans;
}

PROC (s_print_mode, "print-mode", 0, 1, 0, scm_print_mode);
#ifdef __STDC__
SCM
scm_print_mode (SCM new_mode)
#else
SCM
scm_print_mode (new_mode)
     SCM new_mode;
#endif
{
  SCM ans = change_mode (new_mode, scm_print_flags, N_PRINT_FLAGS, s_print_mode);
  return ans;
}




/* {Accessing Debugging Info}
 */

PROC (s_expr_stack, "expr-stack", 0, 0, 0, scm_expr_stack);
#ifdef __STDC__
SCM
scm_expr_stack (void)
#else
SCM
scm_expr_stack ()
#endif
{
  SCM ls;
  debug_info *frame;

  ls = EOL;
  for (frame = last_debug_info_frame; frame != BIGPTR; frame = frame->prev)
    {
      /* ACK!   Don't export expressions from code graphs. */
      ls = scm_acons (BOOL_F, scm_cons (frame->proc, frame->args), ls);
    }
  return ls;
}

SCM scm_i_procname = BOOL_F;

PROC (s_procedure_name, "procedure-name", 1, 0, 0, scm_procedure_name);
#ifdef __STDC__
SCM
scm_procedure_name (SCM proc)
#else
SCM
scm_procedure_name (proc)
     SCM proc;
#endif
{
  ASSERT(scm_procedurep (proc) == BOOL_T, proc, ARG1, s_procedure_name);
  switch (TYP7 (proc))
    {
    case tcs_closures:
      {
	SCM name = scm_source_property (CDR (CODE (proc)), scm_i_procname);
	return (name != BOOL_F) ? name : gscm_procedure_property (proc, scm_i_name);
      }
    case tcs_subrs:
      return SNAME (proc);
    default:
      return BOOL_F;
    }
}



/* {Object Properties}
 */

PROC (s_object_properties, "object-properties", 1, 0, 0, scm_object_properties);
#ifdef __STDC__
SCM
scm_object_properties (SCM obj)
#else
SCM
scm_object_properties (obj)
     SCM obj;
#endif
{
  SCM h;
  if (WHASHFOUNDP (h = scm_weak_hash_get_handle (scm_object_whash, obj)))
    return WHASHREF (scm_object_whash, h);
  WHASHSET (scm_object_whash, scm_weak_hash_create_handle (scm_object_whash, obj), EOL);
  return EOL;
}


PROC (s_set_object_properties_x, "set-object-properties!", 2, 0, 0, scm_set_object_properties_x);
#ifdef __STDC__
SCM
scm_set_object_properties_x (SCM obj, SCM plist)
#else
SCM
scm_set_object_properties_x (obj, plist)
     SCM obj;
     SCM plist;
#endif
{
  scm_weak_hash_insert (scm_object_whash, obj, plist);
  return UNSPECIFIED;
}

PROC (s_object_property, "object-property", 2, 0, 0, scm_object_property);
#ifdef __STDC__
SCM
scm_object_property (SCM obj, SCM key)
#else
SCM
scm_object_property (obj, key)
     SCM obj;
     SCM key;
#endif
{
  SCM assoc;
  assoc = scm_assoc (key, scm_object_properties (obj));
  return (NIMP (assoc) ? CDR (assoc) : BOOL_F);
}

PROC (s_set_object_property_x, "set-object-property!", 3, 0, 0, scm_set_object_property_x);
#ifdef __STDC__
SCM
scm_set_object_property_x (SCM obj, SCM key, SCM val)
#else
SCM
scm_set_object_property_x (obj, key, val)
     SCM obj;
     SCM key;
     SCM val;
#endif
{
  SCM h;
  h = scm_weak_hash_get_handle (scm_object_whash, obj);
  if (WHASHFOUNDP (h))
    {
      SCM assoc = scm_assoc (key, WHASHREF (object_whash, h));
      if (NIMP (assoc))
	SETCDR (assoc, val);
      else
	WHASHSET (object_whash, h, scm_acons (key, val, WHASHREF (object_whash, h)));
      return UNSPECIFIED;
    }
  h = scm_weak_hash_create_handle (scm_object_whash, obj);
  WHASHSET (object_whash, h, scm_acons (key, val, EOL));
  return UNSPECIFIED;
}

/* {Source Properties}
 *
 * Properties of source list expressions.
 * Five of these have special meaning and optimized storage:
 *
 * filename    string   The name of the source file.
 * line	       integer	The source code line number.
 * column      integer	The source code column number.
 * breakpoint  boolean	Sets a breakpoint on this form.
 * copy	       pair	A copy of this source expression.
 *
 * Most properties above can be set by the reader.
 *
 */

/* Compressed plists */

static SCM scm_i_breakpoint, scm_i_line, scm_i_column;
static SCM scm_i_filename, scm_i_copy;

long tc16_srcprops;

#ifdef __STDC__
static SCM
marksrcprops (SCM obj)
#else
static SCM
marksrcprops (obj)
     SCM obj;
#endif
{
  SETGC8MARK (obj);
  scm_gc_mark (SRCPROPFNAME (obj));
  return SRCPROPCOPY (obj);
}

#ifdef __STDC__
static int
prinsrcprops (SCM obj, SCM port, int writing)
#else
static int
prinsrcprops (obj, port, writing)
     SCM obj;
     SCM port;
     int writing;
#endif
{
  scm_lputs ("#<srcprops ", port);
  scm_iprin1 (scm_srcprops2plist (obj), port, 1);
  scm_lputc ('>', port);
  return 1;
}

static scm_smobfuns srcpropssmob =
{marksrcprops, scm_free0, prinsrcprops, 0};

#ifdef __STDC__
SCM
_scm_make_srcprops (int line, int col, SCM filename, SCM copy)
#else
SCM
_scm_make_srcprops (line, col, filename, copy)
     int line;
     int col;
     SCM filename;
     SCM copy;
#endif
{
  SCM ans;
  DEFER_INTS;
  ans = scm_alloc_obj (SRCPROPS_NCELLS, "srcprops");
  CAR (ans) = tc16_srcprops;
  SETSRCPROPPOS (ans, line, col);
  SRCPROPFNAME (ans) = filename;
  SRCPROPCOPY (ans) = copy;
  ALLOW_INTS;
  return ans;
}

#ifdef __STDC__
SCM
scm_srcprops2plist (SCM obj)
#else
SCM
scm_srcprops2plist (obj)
     SCM obj;
#endif
{
  SCM plist = EOL;
  if (!UNBNDP (SRCPROPCOPY (obj)))
    plist = scm_acons (scm_i_copy, SRCPROPCOPY (obj), plist);
  if (!UNBNDP (SRCPROPFNAME (obj)))
    plist = scm_acons (scm_i_filename, SRCPROPFNAME (obj), plist);
  plist = scm_acons (scm_i_column, MAKINUM (SRCPROPCOL (obj)), plist);
  plist = scm_acons (scm_i_line, MAKINUM (SRCPROPLINE (obj)), plist);
  plist = scm_acons (scm_i_breakpoint, SRCPROPBRK (obj), plist);
  return plist;
}

PROC (s_source_properties, "source-properties", 1, 0, 0, scm_source_properties);
#ifdef __STDC__
SCM
scm_source_properties (SCM obj)
#else
SCM
scm_source_properties (obj)
     SCM obj;
#endif
{
  SCM p;
  p = scm_weak_hash_lookup (scm_object_whash, obj);
  if (IMP (p))
    return p;
  if (SRCPROPSP (p))
    return scm_srcprops2plist (p);
  if (CONSP (p) && NIMP (CAR (p)) && SRCPROPSP (CAR (p)))
    {
      SCM ls = scm_srcprops2plist (CAR (p));
      SCM *lloc = &CDR (CDR (ls)); /* ls contains at least three elements. */
      while (NNULLP (CDR (*lloc)))
	lloc = &CDR (*lloc);
      CDR (*lloc) = CDR (p);
      return ls;
    }
  return p;
}

/*fixme*/
PROC (s_set_source_properties_x, "set-source-properties!", 2, 0, 0, scm_set_source_properties_x);
#ifdef __STDC__
SCM
scm_set_source_properties_x (SCM obj, SCM plist)
#else
SCM
scm_set_source_properties_x (obj, plist)
     SCM obj;
     SCM plist;
#endif
{
  scm_set_object_property_x (scm_object_whash, obj, plist);
  return UNSPECIFIED;
}


PROC (s_source_property, "source-property", 2, 0, 0, scm_source_property);
#ifdef __STDC__
SCM
scm_source_property (SCM obj, SCM key)
#else
SCM
scm_source_property (obj, key)
     SCM obj;
     SCM key;
#endif
{
  SCM plist;
  SCM p = scm_object_properties (obj);
  if (IMP (p))
    return BOOL_F;
  if (CONSP (p) && NIMP (CAR (p)) && SRCPROPSP (CAR (p)))
    {
      plist = CDR (p);
      p = CAR (p);
      goto srcpropsp;
    }
  plist = EOL;
  if (SRCPROPSP (p))
    {
    srcpropsp:
      if      (scm_i_breakpoint == key) p = SRCPROPBRK (p);
      else if (scm_i_line       == key) p = MAKINUM (SRCPROPLINE (p));
      else if (scm_i_column     == key) p = MAKINUM (SRCPROPCOL (p));
      else if (scm_i_filename   == key) p = SRCPROPFNAME (p);
      else if (scm_i_copy       == key) p = SRCPROPCOPY (p);
      else goto plistp;
      return UNBNDP (p) ? BOOL_F : p;
    }
plistp:
  p = scm_assoc (key, plist);
  return (NIMP (p) ? CDR (p) : BOOL_F);
}



#ifdef __STDC__
void
scm_init_debug (void)
#else
void
scm_init_debug ()
#endif
{
  int i;
  
  for (i = 0; i < N_DEBUG_FLAGS; ++i)
    scm_debug_flags[i].sym = CAR (scm_sysintern (scm_debug_flags[i].name, SCM_UNDEFINED));
  scm_debug_mode (SCM_UNDEFINED);
  for (i = 0; i < N_READ_FLAGS; ++i)
    scm_read_flags[i].sym = CAR (scm_sysintern (scm_read_flags[i].name, SCM_UNDEFINED));
  scm_read_mode (SCM_UNDEFINED);
  for (i = 0; i < N_PRINT_FLAGS; ++i)
    scm_print_flags[i].sym = CAR (scm_sysintern (scm_print_flags[i].name, SCM_UNDEFINED));
  scm_print_mode (SCM_UNDEFINED);

  tc16_srcprops = scm_newsmob (&srcpropssmob);
  
  /* scm_alloc_obj which is called from scm_make_whash will induce gc.
     We need scm_stack_base to be set up for gc to be possible, but we
     don't know if this is done before or after this initialization.
     Most probably it's done after. */
  {
    STACKITEM *old = scm_stack_base, s;
    int e = scm_errjmp_bad;
    scm_stack_base = &s;
    scm_errjmp_bad = 0;
    scm_object_whash = scm_make_weak_hash_table (MAKINUM (511));
    scm_errjmp_bad = e;
    scm_stack_base = old;
  }
  scm_sysintern ("object-whash", scm_object_whash);
  scm_add_feature ("debug-extensions");
  scm_i_procname = CAR (scm_sysintern ("procedure-name", SCM_UNDEFINED));
#include "debug.x"
}
