/* texparse.h
   Declarations of classes for parsing TeX.

    Author: John Collins, collins@phys.psu.edu.
    22 Jan 96

    (C) John Collins & Penn State University.
*/

#define TEXPARSE_H
#ifndef BUFFER_H
#include "buffer.h"
#endif
#ifndef SOS_H
#include "sos.h"
#endif


class ListHead;
extern ListHead LostAndFound;


enum catcode {
/* To analyze characters.  Not the TeX numbering.
   The sequence is to allow useful testing by > etc.
*/ Null = NULL,
   Ignore,
   Letter,
   Digit,
   White,
   Other,
   NewLine,
   CommentCh,
   Control,
   EscapeCh,
   BeginGroupCh,
   EndGroupCh,
   MathShiftCh,
   AlignTabCh,
   ParameterCh,
   SuperscriptCh,
   SubscriptCh,
   Invalid
};

extern catcode chcat[256]; // List of cat codes I use.


void set_TeX_cat(void);
   // sets chcat to standard codes


//=============================================================

enum modetype {Text, Math, Display};

class TeXState {
 public:
     modetype mode;
     TeXState(){mode=Text;}
};


class LyxState {
   public:
      int EnvLevel, InPar;
      LyxState() {Reset();}
      void Reset() { EnvLevel = 0; InPar = 0;}
};



//=============================================================

extern Line EmptyLine;


class BaseTeXObject;
class TeXObject;

class TeXPosn: public Posn {
   /* Used for position in a TeX file.  Its methods are used to do
      the parsing of the file.

      I will not allow a TeXPosn to point to nothing.  (I.e., l!=NULL.)
   */


   public:

      inline TeXPosn() {l = &EmptyLine; index = 0;}
      inline TeXPosn(Posn &p) {l = p.l;  index = p.index;};
      inline TeXPosn(TeXPosn &p) {l = p.l;  index = p.index;};
      inline TeXPosn(Line *Pl, unsigned int i = 0) {l = Pl;  index = i;};

      inline catcode cat() {return chcat[c()];}


      virtual int AddOneObject (BaseTeXObject *&Last);
      /* Construct the TeX object that starts at my position,
         add it to a list of TeXObjects, after the element Last,
         update Last to point to the newly added TeXObject,
         and move myself to the next character.

         Return 0 if I succeeded, a positive error code if I failed, and
         -1 if I got to the end of the file.
      */

      virtual int AddTeXToken (BaseTeXObject *&Last);
      /* Assume I am pointing at a TeX special character.
         Construct the TeX token that starts at my position,
         add it to a list of TeXObjects, after the element Last,
         update Last to point to the newly added TeXObject,
         and move myself to the next character.

         Return 0 if I succeeded, a positive error code if I failed, and
         -1 if I got to the end of the file.
      */

      virtual void MakeCS (TeXObject *&t);
         /* Make a Control Sequence object t from the TeXCS I point to.
        Assume I point to '\\'.
         */

      void SkipWhite();
      /* Point me to the first non-white-space character starting from my
         current position.  Don't count newline as white.
      */

      void SkipWhiteAndNL();
      /* Point me to the first non-white-space character starting from my
         current position.  Count newline as white.
      */

      void TeXPosn::SkipCSTrail();
      /* I am pointing at the last character of a CS.  If there is following
         white space point me at the last space, else leave me where I am.
         Here white space is a sequence of white space characters followed by
         zero or one newline.
      */

      void FindTeXSpecial();
      /* Point myself to the first TeX special character forward from
         the current position.  A TeX special character is one that is not
         regular text.
      */
};


//IDs for TeXObjects;
enum TeXObjectID {
   IDDummy, IDListHead, IDText, IDBG, IDEG,
   IDDocClass, IDMathShift, IDComment, IDEnv, IDBegin, IDEnd, IDCS,
   IDPar, IDDocumentBody
};


class BaseTeXObject {
   public:

      BaseTeXObject *prev, *next, *prevlex, *nextlex;
      /* These show the links to the logical structure and the
         lexical structure.
      */

      TeXPosn start, end;
      /* Pointers to the first and last characters in the file that
         correspond to this object.

         No destructor is needed.  Note that start and end are objects that
         are initialized, so the BaseTeXObject's constructor doesn't have
         to initialize them.  They will be initialized to point to nothing,
         i.e., start.l == end.l == NULL.  It is methods derived from
         TeXObject that will point to things.
      */

      TeXObjectID ID;
      int Closes, Opens;
      // This Object closes and opens these number of groups.
   private:
      inline void Init(){prev = next = prevlex = nextlex = NULL;
                 Closes = Opens = 0;
                         ID = IDDummy;}


   public:

      BaseTeXObject();

      virtual ~BaseTeXObject();

      virtual void Show (sos &f = lite_cout) const;
         // For debugging.  Show what kind of an object we've got.

      virtual void ShowStartEnd (sos &f = lite_cout) const;
         // For debugging.  Show start and end.

      virtual void WriteTeX (sos &f = lite_cout) const;
         // Write the TeX representation of this object.
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
         // Write the Lyx representation of this object.

      TeXObjectID IsA(){return ID;}
      BaseTeXObject * head();
      BaseTeXObject * tail();
      virtual char *GetName();
         //Name of environment I control, or documentclass's style, etc.
      virtual int IsTeXText();
      virtual int IsBlankLine();
      virtual int IsComment();

      virtual void Extend(TeXPosn &NewEnd);
         /* Extend the ending position of my text.  Error if I am
            not pointing to anything.
         */

      virtual int Parse (BaseTeXObject *& next_to_parse, TeXState &s);
         /* Parse me, so that I am a complete object.
            Rearrange the logical list to which I am attached as needed.
            Set next_to_parse to the next object to parse.
            Return 0 if no errors, else return non-zero error code.
         */


   protected:
      inline void SetLexFromLogical () {
         // Set logical listing from logical
         prevlex = prev; nextlex = next;
      }

   public:

};


class TeXObject : public BaseTeXObject {
   // Base for TeXObjects that point to something.

   public:

      TeXObject(const TeXPosn &p1, const unsigned int i2);
         // p1 is starting point of the text, i2 is index of end in same line.

      TeXObject(const TeXPosn &p1, const TeXPosn &p2);
         // p1 is starting point of the text, p2 is ending point.


      virtual void Show (sos &f = lite_cout) const;
         // For debugging.  Show what kind of an object we've got.

      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;

      void WriteMyText (sos &f = lite_cout) const;
         // Write out the text I point to
      void WriteMyLyxText (sos &f = lite_cout) const;
         // Write out the text I point to, but appropriately for Lyx.
};



class TeXText : public TeXObject {
   /* This points to a piece of text that does not contain any special
      characters.  It contains the terminating \n if there is one.
      It may span more than one line.

      It should always point to some text. Hence there is no TeXText()
      constructor.

      No destructor is needed.
   */

   public:

      TeXText(const TeXPosn &p1, const unsigned int i2);
         // p1 is starting point of the text, i2 is index of end in same line.

      TeXText(const TeXPosn &p1, const TeXPosn &p2);
         // p1 is starting point of the text, p2 is ending point.

      virtual int IsTeXText();
      virtual int IsBlankLine();
      virtual int IsComment();

      virtual void Show (sos &f = lite_cout) const;

      virtual void WriteTeX (sos &f = lite_cout) const;
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};




class BlankLine : public TeXText {
   /* This points to a piece of text that is one blank line, plus any
      succeeding blank lines and white space.
      It may span more than one line.

      It should always point to some text. Hence there is no BlankLine()
      constructor.  Since this is just a specialization of the TeXText
      class, the destructors are just call those for TeXText.

      No destructor is needed.
   */

   public:


      BlankLine(const TeXPosn &p1);
      /* p1 is starting point of the text; end is necessarily end-of-line.
         But a BlankLine may be extended later beyond the end-of-line,
         to include white space and more blank lines.
      */

      virtual int IsTeXText();
      virtual int IsBlankLine();

      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
      virtual void Show (sos &f = lite_cout) const;

};


class Comment : public TeXText {
   /* This points to a piece of text that is TeX comment (i.e., '%'
      followed by all characters up to and including a new line.
      Add in any following white space.
      It may NOT span more than one line.

      It should always point to some text. Hence there is no Comment()
      constructor.  Since this is just a specialization of the TeXText
      class, the destructors are just call those for TeXText.

      No destructor is needed.
   */

   public:

      Comment(const TeXPosn &p1);
      /* p1 is starting point of the text; end is necessarily end-of-line.
         But a Comment may be extended later beyond the end-of-line,
         to include white space.
      */

      virtual int IsTeXText();
      virtual int IsComment();

      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
      virtual void Show (sos &f = lite_cout) const;

};

class BaseTeXSpecial : public TeXObject {
   /* Base for the macros, and other specials.  This object should
      not be instantiated.
   */
   public:
      BaseTeXSpecial (const TeXPosn &p1): TeXObject (p1,p1){}
      virtual void Show (sos &f = lite_cout) const;
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};


class DummyTeX : public BaseTeXSpecial {
   /* For debugging.  Used for TeX CS's I haven't defined yet.
   */
   public:
      DummyTeX(const TeXPosn &p1): BaseTeXSpecial (p1){}
      virtual void Show (sos &f = lite_cout) const;
      virtual void WriteTeX (sos &f = lite_cout) const;
};

class TeXCS : public BaseTeXSpecial {
   // TeXCS

   public:
      TeXPosn endCS;
      // start..endCS is the CS. endCS++..end is trailing white space
      unsigned int lenCS;
      // an optimization

      TeXCS(const TeXPosn &p1);
      virtual void Show (sos &f = lite_cout) const;
      virtual int Parse (BaseTeXObject *& next_to_parse, TeXState &s);
      virtual void WriteTeX (sos &f = lite_cout) const;
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};

class DocClass : public BaseTeXSpecial {
   // For \documentclass
   public:
      char *style, *options;

      DocClass(const TeXPosn &p1, const TeXPosn &p2);
         // p1 and p2 delimit the \documentclass and the trailing space.
      virtual ~DocClass();
      virtual char *GetName();
         //Name of environment I control, or documentclass's style, etc.
      virtual void Show (sos &f = lite_cout) const;
      virtual void WriteTeX (sos &f = lite_cout) const;
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};


class BeginEnv : public BaseTeXSpecial {
   // For \begin{...}
   public:
      char *name;
      ListHead *Body;

      BeginEnv(const TeXPosn &p1, const TeXPosn &p2);
         // p1 and p2 delimit the \begin and the trailing space.
      virtual ~BeginEnv();
      virtual char *GetName();
         //Name of environment I control, or documentclass's style, etc.
      virtual int Parse (BaseTeXObject *& next_to_parse, TeXState &s);
      virtual void Show (sos &f = lite_cout) const;
      virtual void WriteTeX (sos &f = lite_cout) const;
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};


class EndEnv : public BaseTeXSpecial {
   // For \end{...}
   public:
      char *name;

      EndEnv(const TeXPosn &p1, const TeXPosn &p2);
         // p1 and p2 delimit the \begin and the trailing space.
      virtual ~EndEnv();
      virtual char *GetName();
         //Name of environment I control, or documentclass's style, etc.
      virtual void Show (sos &f = lite_cout) const;
      virtual void WriteTeX (sos &f = lite_cout) const;
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};


class  Parameter: public BaseTeXSpecial {
   public:
      Parameter (const TeXPosn &p1);
      virtual void Show (sos &f = lite_cout) const;
      virtual void WriteTeX (sos &f = lite_cout) const;
};

class  BaseSimple: public BaseTeXSpecial {
   protected:
      char name;
      // Simple object with one character name.
   public:
      BaseSimple(const TeXPosn &p1): BaseTeXSpecial (p1){name=0;}
      virtual void Show (sos &f = lite_cout) const;
      virtual void WriteTeX (sos &f = lite_cout) const;
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};

class  BeginGroup: public BaseSimple {
   public:
      BeginGroup(const TeXPosn &p1): BaseSimple (p1) {
         ID=IDBG; name='{'; Opens=1;
      }
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};

class  EndGroup: public BaseSimple {
   public:
      EndGroup(const TeXPosn &p1): BaseSimple (p1) {
         ID=IDEG; name='}'; Closes=1;
      }
      virtual void WriteLyx (LyxState &, sos &f = lite_cout) const;
};

class  MathShift: public BaseSimple {
   public:
      MathShift (const TeXPosn &p1): BaseSimple (p1){
         ID=IDMathShift; name='$';
      }
};

class  AlignTab: public BaseSimple {
   public:
      AlignTab(const TeXPosn &p1): BaseSimple (p1){name='&';}
};



class ListHead: public BaseTeXObject {
   /* Used to mark the head of lists.
      It is the only BaseTeXObject that should have a pointer to it that
      is not via the prev, etc pointers.
      Thus it is dangerous to delete it except from the controlling routine
      that owns all the pointers to it.
   */

   public:
      ListHead(): BaseTeXObject(){ID=IDListHead;}
      ListHead(BaseTeXObject *head);
      // Extract a list and attach it to me
      ListHead(BaseTeXObject *head, BaseTeXObject *tail);
      // Extract a list and attach it to me
      virtual void Show (sos &f = lite_cout) const;
      virtual void ShowAll (sos &f = lite_cout) const;
      virtual void WriteTeXAll (sos &f = lite_cout) const;
      virtual void WriteLyxAll (LyxState &, sos &f = lite_cout) const;

      virtual int RawTokenize (const Buffer &);
      /* Return 0 if I have success, an error code otherwise.
         This one just puts everthing into one TeXText.
      */

      virtual int Tokenize (const Buffer &);
      /* Construct a list of TeXObjects attached to me, using the data
         in LineList.

         Return 0 if I have success, an error code otherwise.
      */


};



/* ========================================================================
   ========================================================================

   Functions that are not object methods.
*/

void AddToList (BaseTeXObject *&last, TeXObject *newitem);
   /* Add the new item to the list after the item given by last.
      Update last to point to the new object.
      Do not necessarily assume that last is actually the last item
      in the list.
      If the list is empty (last == NULL) just set last = newitem.
   */




int ParseAll (BaseTeXObject * current, TeXState &s);
/* Starting at current, parse all remaining objects.
   Return 0 if no errors, else return non-zero.
*/


void Extract (BaseTeXObject *first, BaseTeXObject *last);
   /* Unlink the list from first to last.
      Set the pointers of the rest of the list correctly.
      first or last ==NULL => do nothing.
   */

void Extract (BaseTeXObject *first);
   /* Unlink the list from first onwards.
      Set the pointers of the rest of the list correctly.
      first==NULL => do nothing.
   */

void MoveTo (BaseTeXObject &dest,
             BaseTeXObject *source, BaseTeXObject *end = NULL);
   /* Move the list source to end to just after dest.
      If end is NULL, the whole of the rest of the list is to be moved.
   */

int FindClose (const BeginGroup &start, BaseTeXObject *&end);
  /*  Assuming that start is a { object.
      Find its matching }.
      Return: 0 if I succeed.
              >=1 if I get to the end-of-file without a match.
                result is the number of unclosed braces.
  */

int FindMatchingEnd (const BeginEnv &start, BaseTeXObject *&end);
  /*  Assuming that start is a \begin object.
      Find its matching \end.
      Return: 0 if I succeed.
              1 if I find a matching \end, but its name is wrong.
                (E.g., \begin{eqnarray} ... \end{equation}
              2 if there was a } without a corresponding { before I found an
                \end. This could indicate start is in the middle of
                a macro definition (i.e., it is legal), or that the document
                is incorrect. end points to the bad brace.
              3 if I find a matching \end, but there are unclosed {s. This
                should indicate that the document is incorrect.
                end points to the \end.
              4 Same as 3, except that the name of the \end is not equal
                to mine. Again, this should indicate that the document
                is incorrect. end points to the bad brace.
              5 if I don't find a matching \end by the end the file, but
                there are matching braces. end is NULL.
              6 if I don't find a matching \end by the end of the file, and
                there are unclosed opening braces. end is NULL.
              7 if there was a mismatched environment before I found the
                matching \end.
  */



void GetSimpleArg (TeXPosn & p, char *& s, int & len);

char *MakeString (char *s, int len);
   // Make null terminated string

int SameName(const BeginEnv &b, const EndEnv &e);
