/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
/*
**  keymap.c
*/

#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "sources.h"
#include "message.h"
#include "output.h"
#include "keymap.h"
#include <ctype.h>
#include "ownkeys.h"
#include "config.h"

#define DEF_ARG  1
#define ARG_LEN  200
#define FUNCMAX  512
#define STACKD   50

typedef struct {
    short funcnr;
    Char  def_arg;
    short mapnr;
    short mapfunc;
} KEY;

#define MAXMAP ((max_keycode)>>5)

typedef KEY PARTMAP[32];

typedef struct {
    PARTMAP *mp[MAXMAP];
    char *desc;
} KEYMAP;

typedef struct {
    PFKI fn;
    char *desc;
} FUNC;

typedef struct {
    int *fnr;
    KEYCODE *kargs;
    Index *args;
    char *desc;
} DEFFUNC;

typedef struct {
    KEYCODE kstack[STACKD];
    int mstack[STACKD];
    int depth;
} KEYSTACK;

static KEYMAP *maps = NULL;
static int nrmaps = 0, maxmaps=0;
static int basicmap = 0;
static int currentmap = 0;

static char **argum = NULL;
static int nrargum=0, maxargum=0;

static FUNC func[FUNCMAX];
static int nrfunc = 1;
static DEFFUNC deffunc[FUNCMAX];
static int nrdeffunc = 1;
static KEYSTACK keystack;
static int inputstack[STACKD];
static int stackdepth = 0;
static Index argument = DEF_ARG;
static char argstr[ARG_LEN];
static int arglen = 0;
static Bool making_argument = False;
static Bool making_string = False;
static char *comment = NULL;
static void (*strfunc)(char *) = NULL;
static KEYCODE redef[max_keycode];
static int call_deffunc_nr = 0;

static char *STRCODE[8] = { "",   "S-",  "M-",  "MS-",
				  "C-", "CS-", "CM-", "CMS-"};
static int K2STR[MAXMAP] = { 4, 0, 0, 0, 6, 2, 2, 6, 4, 4, 2, 6,
			     0, 0, 4, 4, 6, 6, 2, 2, 1, 1, 0, 1 };
static int K2COD[MAXMAP] = { 1, 0, 1, 2, 1, 0, 1, 0, 0, 3, 3, 3,
			     4, 5, 4, 5, 4, 5, 4, 5, 4, 5, 3, 3 };
static char *FUNCKEY5[32] = {
                 "F33", "F34", "F35", "home", "left", "up", "right", "down",
		 "prior", "next", "end", "begin", "PF1", "PF2", "PF3", "PF4",
		 "DEL", "pause", "scrolllock", "multikey", "select",
		 "print", "execute", "insert", "dummy", "undo", "redo",
		 "menu", "find", "cancel", "help", "break" };
static char *FUNCKEY3[32] = {
                 "template", "backspace", "tab", "linefeed", "clear", "ESC",
		 "return", "reset", "system", "user", "clearline",
		 "insertline", "deleteline", "insertchar", "deletechar",
		 "backtab", "kp-backtab", "", "", "", "", "", "", "", "",
		 "", "", "symbol", "", "", "", "" };

#define SHIFT 1
#define META  2
#define CONTROL 4
#define map_part(A,B) (maps[A].mp[(B)>>5])
#define map_func(A,B) ((*map_part(A,B))[(B) & 0x1F])
#define empty_func(A,B) (!map_func(A,B).funcnr && !map_func(A,B).mapnr)

/* SWITCH moves from normal to [-,S,M,MS,C,CS,CM,CMS] key */
static int SWITCH[8][6] =
  { { 0, 0, 0, 0, 0, 0 },
    { 0,                          0,             0,
      K_backspace ^ SK_backspace, K_F1 ^ SK_F1,  K_F33 ^ SK_F33 },
    { K_space^MK_space,           K_at ^ MK_at,  K_quoteleft ^ MK_quoteleft,
      K_backspace ^ MK_backspace, K_F1 ^ MK_F1,  K_F33 ^ MK_F33 },
    { K_space ^ MK_space,         K_at ^ MK_at,  K_quoteleft ^ MK_quoteleft,
      K_backspace ^ MK_backspace, K_F1 ^ MK_F1,  K_F33 ^ MK_F33 },
    { K_space ^ CK_space,         K_at ^ CK_at,  K_quoteleft ^ CK_quoteleft,
      K_backspace ^ CK_backspace, K_F1 ^ CK_F1,  K_F33 ^ CK_F33 },
    { K_space ^ CK_space,         K_at ^ CK_at,  K_quoteleft ^ CK_quoteleft,
      K_backspace ^ CK_backspace, K_F1 ^ CK_F1,  K_F33 ^ CK_F33 },
    { K_space ^ CMK_space,        K_at ^ CMK_at, K_quoteleft ^ CMK_quoteleft,
      K_backspace ^ CMK_backspace,K_F1 ^ CMK_F1, K_F33 ^ CMK_F33 },
    { K_space ^ CMK_space,        K_at ^ CMK_at, K_quoteleft ^ CMK_quoteleft,
      K_backspace ^ CMK_backspace,K_F1 ^ CMK_F1, K_F33 ^ CMK_F33 }
};

static void make_keycodestring(KEYCODE keycode);
static KEYCODE scan_keycodestring(char **str);


static KEYCODE remap_key( unsigned long keysym, unsigned int state)
{
    int i=0;
    KEYCODE temp = NO_KEY;
    int sw = 0;

#define check(A,B,C) if (A<=keysym && keysym<=B) \
    temp = (KEYCODE) (keysym - A + C)

    check(XK_space,       XK_question,   K_space);     else
    check(XK_at,          XK_underscore, K_at);        else
    check(XK_quoteleft,   XK_asciitilde, K_quoteleft); else
    check(XK_Home,        XK_Begin,      K_home);      else
    check(XK_F1,          XK_F35,        K_F1);        else
    check(XK_KP_Home,     XK_KP_Begin,   K_home);      else
    check(XK_KP_F1,       XK_KP_F4,      K_PF1);       else
    check(XK_KP_0,        XK_KP_9,       K_0);         else
    check(XK_Select,      XK_Break,      K_select);    else
    check(XK_Reset,       XK_KP_BackTab, K_reset);     else
    check(XK_BackSpace,   XK_Return,     K_backspace); else
    check(XK_KP_Multiply, XK_KP_Divide,  K_asterisk);  else
    switch ((int) keysym) {
    case XK_Delete:      temp = K_delete;     break;
    case XK_Escape:      temp = K_escape;     break;
    case XK_Pause:       temp = K_pause;      break;
    case XK_Scroll_Lock: temp = K_scrolllock; break;
    case XK_Multi_key:   temp = K_multikey;   break;
    case XK_KP_Space:    temp = K_space;      break;
    case XK_KP_Tab:      temp = K_tab;        break;
    case XK_KP_Enter:    temp = K_return;     break;
    case XK_KP_Equal:    temp = K_equal;      break;
    default:             return NO_KEY;       break;
    }

    i = K2COD[temp>>5];

    if (state & ControlMask) sw |= CONTROL;
    if (state & Mod1Mask)    sw |= META;
    if (state & ShiftMask)   sw |= SHIFT;
    if (sw && sw!=SHIFT && i==2 && K_a<= temp && temp <=K_z) {
	temp ^= (K_a ^ K_A);
	i=1;
    }
    temp ^= SWITCH[sw][i];
    temp = redef[temp];
    return temp;
}

KEYCODE event2code(void *data)
{
    XKeyEvent *event = (XKeyEvent*) data;
    char buf[5];
    int buflen = 5;
    XComposeStatus status;
    int i;
    KeySym keysym;
    unsigned int tempstate;

    tempstate = event->state;
    /* event->state = event->state & (LockMask | ShiftMask); */
    i = XLookupString(event, buf, buflen, &keysym, &status);
    return remap_key(keysym,tempstate);
}

static Bool check_interrupted(Display *d, XEvent *ev, char *a)
{
    KEYCODE k;
    if (d!=display || ev->type!=KeyPress) return False;
    k = event2code((void*) &(ev->xkey));
    return (k==K_cancel || k==CK_G);
}

Bool interrupted(void)
{
    XEvent report;

    if (!XEventsQueued(display, QueuedAfterFlush)) return False;
    return XCheckIfEvent(display, &report, check_interrupted, NULL);
}

static int handle_key(int map, KEYCODE key, int arg)
{
    KEY ft;
    Bool cleared = (keystack.depth == 0);
    if (map<=0 || map>=nrmaps) return 0;
    if (map_part(map,key))
	ft = map_func(map,key);
    else
	ft.mapnr = ft.funcnr = 0;
    if (ft.mapnr) {
	keystack.kstack[keystack.depth] = key;
	keystack.mstack[keystack.depth++] = map;
	if (ft.mapfunc)
	    (*func[ft.mapfunc].fn)(key,arg);
	else {
	    make_keycodestring(key);
	    message(MESSAGECURS, argstr);
	}
	return ft.mapnr;
    }
    if (ft.funcnr) {
	if (keystack.depth) {
	    keystack.depth=0;
	    clear_message(True);
	    (*func[ft.funcnr].fn)(key, (arg>=0 ? arg : ft.def_arg));
	    argstr[0]='\0';
	    arglen = 0;
	    return basicmap;
	} else {
	    (*func[ft.funcnr].fn)(key, (arg>=0 ? arg : ft.def_arg));
	    return map;
	}
    }
    /* clear keystack */
    if (!cleared) keystack.kstack[keystack.depth++] = key;
    while (!cleared) {
	int i=0, pos=-1;
	KEY tc;
	Bool found = (keystack.depth==0);
	clear_message(True);
	tc.funcnr = 0;
	while (!found) {
	    if (map_part(keystack.mstack[i],keystack.kstack[i]))
		ft = map_func(keystack.mstack[i],keystack.kstack[i]);
	    else
		ft.mapnr = ft.funcnr = 0;
	    if (ft.funcnr) {
		pos = i;
		tc = ft;
	    }
	    if (ft.mapnr)
		keystack.mstack[i+1] = ft.mapnr;
	    else if (!tc.funcnr) {
		pos = 0;
		found = True;
	    }
	    i++;
	    found |= (i==keystack.depth);
	}
	if (tc.funcnr)
	    (*func[tc.funcnr].fn)(keystack.kstack[pos], tc.def_arg);
	for (i=0; i+pos<=keystack.depth; i++)
	    keystack.kstack[i] = keystack.kstack[pos+1+i];
	keystack.depth = keystack.depth - pos - 1;
	cleared = (pos<0);
    }
    arglen=0;
    argstr[0]='\0';
    if (keystack.depth) {
	int i;
	for (i=0; i<keystack.depth; i++) {
	    make_keycodestring(key);
	}
	message(MESSAGECURS, argstr);
	return keystack.mstack[keystack.depth];
    } else
	return basicmap;
}

void  call_func( KEYCODE key)
{
    if (key == NO_KEY) return;
    if (making_argument)
	if (key == CK_G) {
	    making_argument = False;
	    argument = DEF_ARG;
	    argstr[0] = '\0';
	    arglen = 0;
	    clear_message(True);
	} else if ((K_0<=key) && (key<=K_9)) {
	    if (argument!=0 || key!=K_0) {
		argstr[arglen] = (char) key;
		argstr[++arglen] = '\0';
		argument = argument *10 + key - K_0;
		message(MESSAGECURS, argstr);
	    }
	} else {
	    making_argument = False;
	    make_keycodestring(key);
	    message(MESSAGE, argstr);
	    currentmap = handle_key(currentmap, key, argument);
	    argument = DEF_ARG;
	    argstr[0] = '\0';
	    arglen = 0;
	}
    else if (making_string) {
	if (key == CK_G) {
	    making_string = False;
	    argument = DEF_ARG;
	    argstr[0] = '\0';
	    arglen = 0;
	    strfunc = NULL;
	    comment = NULL;
	    clear_message(True);
	} else if ((K_space<=key) && (key<=K_tilde)) {
	    if (arglen<ARG_LEN-1) {
		argstr[arglen] = (char) key;
		argstr[++arglen] = '\0';
		message2(MESSAGECURS, comment, argstr);
	    }
	} else if ((key == K_delete) || (key==K_backspace)) {
	    if (arglen) {
		argstr[--arglen] = '\0';
		message2(MESSAGECURS, comment, argstr);
	    }
	} else if (key == K_return) {
	    making_string = False;
	    comment = NULL;
	    clear_message(True);
	    (*strfunc)(argstr);
	    argstr[0] = '\0';
	    arglen = 0;
	}
    } else {
	int i,j;
	i = currentmap;
	j = handle_key(currentmap, key, -1);
	if (i==currentmap) currentmap = j;
    }
}

static void init_arg(int nr, char *str)
{
    int i;
    if (nr>=nrargum) {
	if (nr>maxargum) {
	    char** temp = NULL;
	    int nl = ((nr & 0xFFE0)+1)<<5;
	    temp = (char**) malloc( nl*sizeof(char*));
	    maxargum = nl;
	    for (i=0; i<nrargum; i++)
		temp[i] = argum[i];
	    for (i=nrargum; i<maxargum; i++)
		temp[i] = NULL;
	    myfree(argum);
	    argum = temp;
	}
	argum[nr] = str;
	nrargum = nr+1;
    }
}

static void init_map(int map, char *desc)
{
    int i;
    if (map>=nrmaps) {
	if (map>=maxmaps) {
	    KEYMAP* temp = NULL;
	    int nr = ((map & 0xFFE0)+1)<<5;
	    temp = (KEYMAP*) malloc( nr*sizeof(KEYMAP));
	    maxmaps = nr;
	    for (nr=0; nr<nrmaps; nr++) {
		for (i=0; i<MAXMAP; i++)
		    temp[nr].mp[i] = maps[nr].mp[i];
		temp[nr].desc = maps[nr].desc;
	    }
	    for (nr=nrmaps; nr<maxmaps; nr++) {
		for (i=0; i<MAXMAP; i++)
		    temp[nr].mp[i] = NULL;
		temp[nr].desc = NULL;
	    }
	    myfree(maps);
	    maps = temp;
	}
	maps[map].desc = desc;
	nrmaps = map+1;
    }
}

int new_map(char *desc)
{
    int j=nrmaps;

    if (!j) j++;
    init_map(j,desc);
    return j;
}

static void init_key(int map, KEYCODE key, int funcnr, Index arg)
{
    if (!maps[map].mp[key>>5]) {
	int i;
	maps[map].mp[key>>5] = (PARTMAP*) malloc(sizeof(PARTMAP));
	for (i=0; i<32; i++)
	    maps[map].mp[key>>5][0][i].funcnr =
	    maps[map].mp[key>>5][0][i].mapnr = 0;
    }
    maps[map].mp[key>>5][0][key & 0x1F].funcnr = funcnr;
    maps[map].mp[key>>5][0][key & 0x1F].def_arg = arg;
}

static int define_func(int *funcnrs, KEYCODE *keyargs, Index *args, char *desc)
{
    if (nrdeffunc==FUNCMAX) return 0;
    deffunc[nrdeffunc].fnr = funcnrs;
    deffunc[nrdeffunc].kargs = keyargs;    
    deffunc[nrdeffunc].args = args;
    deffunc[nrdeffunc].desc = desc;
    nrdeffunc++;
    return nrdeffunc-1;
}

static void call_defined_func(KEYCODE key, Index nr)
{
    int i=0, j;
    if (!nr || nr>=nrdeffunc || !deffunc[nr].fnr ||
	!deffunc[nr].args || !deffunc[nr].kargs) return;
    while ((j=deffunc[nr].fnr[i])) {
	(*func[j].fn)(deffunc[nr].kargs[i], deffunc[nr].args[i]);
	i++;
    }
}

char* get_string_arg(int nr)
{
    if (nr>=0 && nr<nrargum)
	return argum[nr];
    else
	return NULL;
}

int add_func(PFKI newfunc, char *desc)
{
    int i;
    for (i=1; i<nrfunc && func[i].fn!=newfunc; i++);
    if (i==nrfunc) {
	func[i].fn = newfunc;
	func[i].desc = desc;
	nrfunc++;
    }
    return i;
}

void add_key(int map_nr, KEYCODE defkey, int funcnr, Index arg)
{
    if (map_nr<=0 || map_nr >= nrmaps)
	message(ERROR, "Keymap does not exist.");
    else if (funcnr<=0 || funcnr>=nrfunc)
	message(ERROR, "Undefined function used.");
    else
	init_key(map_nr, defkey, funcnr, arg);
}

static void  add_keyrange(int map_nr, KEYCODE firstkey, KEYCODE lastkey, int funcnr,
		   Index arg)
{
    if (!map_nr || map_nr >= nrmaps)
	message(ERROR, "Keymap does not exist.");
    else  if (funcnr<=0 || funcnr>=nrfunc)
	message(ERROR, "Undefined function used.");
    else
	for ( ; firstkey<=lastkey; firstkey++)
	    init_key(map_nr, firstkey, funcnr, arg);
}

static void add_keysequence(int map_nr, KEYCODE *keys, int funcnr, Index arg)
{
    if (!map_nr || map_nr >= nrmaps)
	message(ERROR, "Keymap does not exist.");
    else  if (funcnr<=0 || funcnr>=nrfunc)
	message(ERROR, "Undefined function used.");
    else if (keys && keys[0]!=NO_KEY) {
	int i=0, j;

	while (keys[i+1]!=NO_KEY) {
	    if (!map_part(map_nr, keys[i])) init_key(map_nr, keys[i], 0, 0);
	    if ((j=map_func(map_nr, keys[i]).mapnr))
		map_nr = j;
	    else {
		j = new_map(NULL);
		map_func(map_nr, keys[i]).mapnr = j;
		map_func(map_nr, keys[i]).mapfunc = 0;
		map_nr = j;
	    }
	    i++;
	}
	init_key(map_nr, keys[i], funcnr, arg);
    }
}

void  remove_key( int map_nr, KEYCODE undefkey)
{
    if (!map_nr || map_nr >= nrmaps)
	message(ERROR, "Keymap does not exist.");
    else if (map_part(map_nr,undefkey)) {
	int i=0;
	map_func(map_nr, undefkey).funcnr = 0;
	map_func(map_nr, undefkey).def_arg = 0;
	undefkey &= 0xFFE0;
	for (i=0; i<32 && empty_func(map_nr, undefkey+i); i++);
	if (i==32) {
	    myfree(map_part(map_nr,undefkey));
	    map_part(map_nr,undefkey) = NULL;
	}
    }
}

static void add_keymap_sequence(int map_nr, KEYCODE *keys, int nextmap_nr, int funcnr)
{
    if (!map_nr || map_nr >= nrmaps)
	message(ERROR, "Keymap does not exist.");
    else  if (funcnr<0 || funcnr>=nrfunc)
	message(ERROR, "Undefined function used.");
    else if (keys && keys[0]!=NO_KEY) {
	int i=0, j;

	while (keys[i+1]!=NO_KEY) {
	    if (!map_part(map_nr, keys[i])) init_key(map_nr, keys[i], 0, 0);
	    if ((j=map_func(map_nr, keys[i]).mapnr))
		map_nr = j;
	    else {
		j = new_map(NULL);
		map_func(map_nr, keys[i]).mapnr = j;
		map_func(map_nr, keys[i]).mapfunc = 0;
		map_nr = j;
	    }
	    i++;
	}
	if (!map_part(map_nr, keys[i])) init_key(map_nr, keys[i], 0,0);
	if ((j = map_func(map_nr, keys[i]).mapnr)) {
	    if (j!=nextmap_nr)
		message(ERROR, "Keymap already exists.");
	    else
		map_func(map_nr,keys[i]).mapfunc = funcnr;
	} else {
	    map_func(map_nr,keys[i]).mapnr = nextmap_nr;
	    map_func(map_nr,keys[i]).mapfunc = funcnr;
	}
    }
}

static int parse_mapname(char* str);

void use_map_string(KEYCODE mapkey, Index arg)
{
    int c;
    char *s;
    s = get_string_arg(arg);
    if (!s) c=0; else c = parse_mapname(s);
    if (!c) return;
    basicmap = currentmap = c;
}

void use_map(KEYCODE mapkey, Index arg)
{
    if (arg && arg<nrmaps)
	basicmap = currentmap = arg;
}

void reset_map(KEYCODE keycode, Index arg)
{
    currentmap = basicmap = 1;
    stackdepth = 0;
}

static void make_keycodestring(KEYCODE keycode)
{
    KEYCODE temp;
    int inr, modinr;

    if (keycode==NO_KEY) {
	argstr[0] = '\0';
	arglen = 0;
	return;
    }
    inr = (keycode & 0xffe0) >> 5;
    strcpy(argstr+arglen, STRCODE[ K2STR[inr]]);
    while (argstr[arglen]) arglen++;
    modinr = K2COD[inr];
    if (modinr == 3 && K2STR[inr]>1 &&
	((keycode & 0x1F)<1 || (keycode & 0x1F) >26))
	modinr = 2;
    temp = keycode ^ SWITCH[K2STR[inr]][modinr];
    if (modinr == 3) {
	strcat(argstr+arglen, FUNCKEY3[temp & 0x001f]);
	while (argstr[arglen]) arglen++;
    } else if (modinr == 4) {
        int i=temp - K_F1 + 1;
	argstr[arglen++] = 'F';
	if (i>9) argstr[arglen++] = '0'+ i/10;
	argstr[arglen++] = '0' + i%10;
    } else if (modinr == 5) {
        strcat(argstr+arglen, FUNCKEY5[temp & 0x001f]);
	while (argstr[arglen]) arglen++;
    } else {
	argstr[arglen++] = (char) temp;
    }
    argstr[arglen] = ' ';
    argstr[++arglen] = '\0';
}

static KEYCODE scan_keycodestring(char **str)
{
    int i,j;
    int inr = 0;
    KEYCODE temp = NO_KEY;
    char *sstr = *str;

    for (i=7; i>0 && strncmp(STRCODE[i], sstr, strlen(STRCODE[i])); i--);
    sstr += strlen(STRCODE[i]);
    if (!strlen(sstr))
	if (i) {
	    sstr = *str;
	    i = 0;
	} else
	    return NO_KEY;
    if (sstr[0]=='F') {
	sstr++;
	j = sstr[0] - '0';
	if (j>=1 && j<=9) {
	    sstr++;
	    if (sstr[0]) {
		if (j<=3 && sstr[0]>='0' && sstr[0]<='9') {
		    j = j*10 + sstr[0]-'0';
		    if (j>0 && j<36)
			sstr++;
		    else
			j = j/10;
		}
	    }
	    temp = (KEYCODE) K_F1+j-1;
	    if (j>32)
		inr = 5;
	    else
		inr = 4;
	} else
	    sstr--;
    }
    if (temp==NO_KEY) {
	for (j=0; j<32 && (!FUNCKEY3[j][0] ||
			strncmp(FUNCKEY3[j],sstr,strlen(FUNCKEY3[j]))); j++);
	if (j<32) {
	    sstr += strlen(FUNCKEY3[j]);
	    temp = (KEYCODE) K_backspace + j-1;
	    inr = 3;
	}
    }
    if (temp==NO_KEY) {
	for (j=0; j<32 && (!FUNCKEY5[j][0] ||
			strncmp(FUNCKEY5[j],sstr,strlen(FUNCKEY5[j]))); j++);
	if (j<32) {
	    sstr += strlen(FUNCKEY5[j]);
	    temp = (KEYCODE) K_F33+j;
	    inr = 5;
	}
	if ((temp == NOTATION_KEY) || (temp==SYMBOL_KEY)) i=0;
    }
    if (temp==NO_KEY) {
	temp = (KEYCODE) sstr[0];
	inr = ((sstr[0] & 0x60) >> 5) -1;
        if (i==1) i=0;
        if (i && *sstr>='a' && *sstr<='z') {
          temp = temp+ K_A - K_a;
          inr = 1;
        }
	sstr++;
	if (inr<0 || inr>2) return NO_KEY;
    }
    temp = temp ^ SWITCH[i][inr];
    *str = sstr;
    return temp;
}

void print_key(KEYCODE keycode, Index arg)
{
    argstr[0] = '\0';
    arglen = 0;
    make_keycodestring(keycode);
    message(MESSAGE, argstr);
}

void construct_argument(KEYCODE keycode, Index arg)
{
    making_argument = True;
    argstr[0] = '\0';
    arglen = 0;
    make_keycodestring(keycode);
    message(MESSAGECURS, argstr);
    argument = 0;
}

void construct_string(char *cm, char *def, void (*fnc)(char*))
{
    making_string = True;
    strcpy(argstr, def);
    arglen = strlen(argstr);
    comment = cm;
    strfunc = fnc;
    message2(MESSAGECURS, comment, argstr);
    argument = 0;
}

void push_input(void)
{
    if (stackdepth<STACKD)
	inputstack[stackdepth++] = currentmap;
    currentmap = 0;
}

void pop_input(void)
{
    if (stackdepth)
	currentmap = inputstack[--stackdepth];
    else
	currentmap = basicmap;
}

static int parse_funcname(char *str, int *ownfunc)
{   /* func[i].desc == str
    ** ownfunc==0 -> func[i]
    ** ownfunc==1 -> deffunc[i]
    ** ownfunc==2 -> i=0, no function
    */
    int i;

    *ownfunc = 0;
    for (i=0; i<nrfunc; i++) {
	if (func[i].desc && !strcmp(str,func[i].desc))
	    return i;
    }
    *ownfunc = 1;
    for (i=0; i<nrdeffunc; i++) {
	if (deffunc[i].desc && !strcmp(str,deffunc[i].desc))
	    return i;
    }
    *ownfunc = 2;
    return 0;
}

static int parse_mapname(char *str)
{   /* maps[??].desc == str */
    int i;
    for (i=0; i<nrmaps; i++) {
	if (maps[i].desc && !strcmp(str,maps[i].desc))
	    return i;
    }
    return 0;
}

static KEYCODE parse_key(char **str)
{   /* [<key-code>] */
    char *tstr = *str;
    KEYCODE t;

    if (tstr[0]!='[') return NO_KEY;
    tstr++;
    t = scan_keycodestring(&tstr);
    if (t==NO_KEY || tstr[0]!=']') return NO_KEY;
    tstr++;
    *str = tstr;
    return t;
}

static KEYCODE keybuf[STACKD];

#define skip_spaces(A) while (isspace(*(A))) (A)++
#define skip_colon(A)  skip_spaces(A); if (*A!=':') return;A++;skip_spaces(A)
#define skip_colonfunc(A,B) skip_spaces(A);if (*(A)!=':') { B; return;}(A)++;\
skip_spaces(A)

static char *get_name(char** str)
{
    char *h = *str,*g;
    char *ret;
    int st=0;
    skip_spaces(h);
    if (*h!='"') return NULL;
    h++;
    ret = g = h;
    while (*h && *h!='"') {
	if (*h=='%') {
	    *g = '%';
	    h++;
	    st=1;
	}
	if (st) {
	    if (*h!='"') g++;
	    st=0;
	}
	*g = *h;
	h++;
	g++;
    }
    if (ret==h) return NULL;
    *g = '\0';
    h++;
    skip_spaces(h);
    *str = h;
    return ret;
}

static KEYCODE* parse_keysequence(char **str)
{   /* <key><key>...<key> */
    int i=0;
    KEYCODE* t;

    while ((keybuf[i]=parse_key(str))!=NO_KEY) i++;
    t = (KEYCODE *) malloc((i+1)*sizeof(KEYCODE));
    if (t)
	while (i>=0) {
	    t[i]=keybuf[i];
	    i--;
	}
    return t;
}

static char* parse_function(char *str, int *funcnr, KEYCODE *key, Index *arg)
{   /* name(<key>,<argument>)            arg = val(<argument>)
    ** name(<key>,"<argument>")   argum[arg] = <argument>
    ** ownname(<rubbish>)
    */
    int fnr=0, argnr=0;
    int type=0;
    KEYCODE k=NO_KEY;
    char *h = str;
    char *ht;

    while (*h && *h!='(') h++;
    if (!*h || h==str) return NULL;
    *h='\0';
    h++;
    fnr = parse_funcname(str, &type);
    if (!fnr) return NULL;
    skip_spaces(h);
    k = parse_key(&h);
    if (k!=NO_KEY) {
	skip_spaces(h);
	if (*h==',') {
	    h++;
	    skip_spaces(h);
	}
    }
    if (*h=='"') {
	ht = get_name(&h);
	if (ht && *h==')') {
	    char *c;

	    c = concat(ht,"");
	    argnr = nrargum;
	    if (!argnr) argnr++;
	    init_arg(argnr, c);
	} else
	    return NULL;
    } else {
	argnr = mystrtoi(h, &h, 0);
	skip_spaces(h);
	if (*h!=')') return NULL;
    }
    h++;
    if (type==1) {
	*funcnr = call_deffunc_nr;
	*key = NO_KEY;
	*arg = fnr;
    } else {
	*funcnr = fnr;
	*key = k;
	*arg = argnr;
    }
    return h;
}

static char *type[6] = { "", "def-func", "def-key", "def-key-range",
			       "def-map", "redef"};

static int parse_def_type(char **str)
{
    int i=5;

    while (strncmp(*str, type[i], strlen(type[i]))) i--;
    *str = *str+strlen(type[i]);
    return i;
}

static void parse_def_func(char **str)
{ /* <newname>:<function>:<function>:<function> */
    char *h = *str;
    char *name;
    char *t;
    int i,j;
    int fnrs[STACKD];
    Index args[STACKD];
    KEYCODE keys[STACKD];
    int *fnr;
    Index *arg;
    KEYCODE *key;

    name = get_name(&h);
    if (!name) return;
    i=0;

    do {
	skip_colon(h);
	t = parse_function(h, fnrs+i, keys+i, args+i);
	if (!t) return;
	h=t;
	skip_spaces(h);
	i++;
    } while (*h==':' && i<STACKD);
    name = concat(name, "");
    fnr = (int*) malloc((i+1)*sizeof(int));
    arg = (Index*) malloc(i*sizeof(Index));
    key = (KEYCODE*) malloc(i*sizeof(KEYCODE));
    for (j=0; j<i; j++) {
	fnr[j] = fnrs[j];
	arg[j] = args[j];
	key[j] = keys[j];
    }
    fnr[i] = 0;
    (void) define_func(fnr, key, arg, name);
    *str = h;
}

static void parse_def_key(char **str)
{ /* <mapname>:<keysequence>:<function> */
    KEYCODE *keys;
    char *h = *str;
    char *name;
    int mnr;
    int funcnr;
    Index arg;
    KEYCODE key;

    if (!(name = get_name(&h))) return;
    if (!(mnr = parse_mapname(name))) return;
    skip_colon(h);
    keys = parse_keysequence(&h);
    if (!keys) return;
    skip_colonfunc(h, myfree(keys));
    name = parse_function(h, &funcnr, &key, &arg);
    if (name) {
	*str=name;
	add_keysequence(mnr, keys, funcnr, arg);
    }
    myfree(keys);
}

static void parse_def_key_range(char **str)
{ /* <mapname>:<key>:<key>:<function> */
    char *h = *str;
    char *name;
    int mnr, funcnr;
    Index arg;
    KEYCODE key1,key2,key;

    if (!(name=get_name(&h))) return;
    if (!(mnr = parse_mapname(name))) return;
    skip_colon(h);
    if ((key1 = parse_key(&h))==NO_KEY) return;
    skip_colon(h);
    if ((key2 = parse_key(&h))==NO_KEY) return;
    skip_colon(h);
    name = parse_function(h, &funcnr, &key, &arg);
    if (!name) return;
    *str=name;
    if (key1>key2) key1 ^= key2 ^= key1 ^= key2; /* swap */
    add_keyrange(mnr, key1, key2, funcnr, arg);
}

static void parse_def_map(char **str)
{ /* <mapname>:<keysequence>:<newname>[:<function>] */
    KEYCODE *keys;
    char *h = *str;
    char *name, *ns;
    int mnr=0, nmnr=0, funcnr=0;
    Index arg=0;
    KEYCODE key=NO_KEY;

    if (!(name = get_name(&h))) return;
    if (!(mnr = parse_mapname(name))) return;
    skip_colon(h);
    keys = parse_keysequence(&h);
    if (!keys) return;
    skip_colonfunc(h, myfree(keys));
    if (!(name = get_name(&h))) {
	myfree(keys);
	return;
    }
    skip_spaces(h);
    if (*h==':') {
	h++;
	skip_spaces(h);
	ns = parse_function(h, &funcnr, &key, &arg);
	if (ns)
	    h=ns;
	else {
	    myfree(keys);
	    return;
	}
    }
    *str = h;
    nmnr = parse_mapname(name);
    if (!nmnr) {
	name= concat(name,"");
	nmnr = new_map(name);
    }
    add_keymap_sequence(mnr, keys, nmnr, funcnr);
    myfree(keys);
}

static void parse_redef(char **str)
{ /* <key>:<key> */
    char *h = *str;
    KEYCODE key1, key2;

    if ((key1 = parse_key(&h))==NO_KEY) return;
    skip_colon(h);
    key2 = parse_key(&h);
    redef[key1] = key2;
    *str = h;
}

static Bool parse_define_line(char *str)
{ /* (<def_type>:<argument1>:<argument2>:...:<argumentN>) */
    int i;
    char *h=str;

    skip_spaces(h);
    if (*h!='(') return False;
    h++;
    skip_spaces(h);
    i = parse_def_type(&h);
    skip_spaces(h);
    if (!i || *h!=':') return False;
    h++;
    skip_spaces(h);
    switch (i) {
    case 1:  parse_def_func(&h);      break;
    case 2:  parse_def_key(&h);       break;
    case 3:  parse_def_key_range(&h); break;
    case 4:  parse_def_map(&h);       break;
    case 5:  parse_redef(&h);         break;
    default: break;
    }
    skip_spaces(h);
    return (*h==')');
}

static char buffer[5000];

static void read_keymap_file(char *name)
{
    FILE *f;
    char *h;
    int i,l;
    char *dir[4];
    char *fname;

    concat_in(buffer, homedir, "mathspad/");
    dir[1] = buffer;
    h = buffer+strlen(buffer)+1;
    concat_in(h, buffer, "%.mpk");
    dir[0] = h;
    h = h + strlen(h)+1;
    dir[3] = program_dir;
    concat_in(h,program_dir, "%.mpk");
    dir[2] = h;
    fname = search_through_dirs(dir, 4, name);
    if (!fname) return;
    f = fopen(fname, "r");
    if (!f) return;
    h = buffer;
    l=0;
    while (fgets(h, 5000-l,f)) {
	if (*h!='#') {
	    i = strlen(h);
	    i -= 2;
	    if (i>=0) {
		if (h[i]==';') {
		    parse_define_line(buffer);
		    h = buffer;
		    l = 0;
		} else if (h[i]=='\\') {
		    h+=i;
		    l+=i;
		} else {
		    h+=i+1;
		    l+=i+1;
		}
	    }
	}
    }
    fclose(f);
}


Bool keypath_exist(char *path)
{
    char *c, *h, *g;
    char t;
    char *dir[4];
    char *fname;

    if (!strcmp(path,keypath)) return False;
    concat_in(buffer, homedir, "mathspad/");
    dir[1] = buffer;
    h = buffer+strlen(buffer)+1;
    concat_in(h,buffer,"%.mpk");
    dir[0] = h;
    dir[3] = program_dir;
    h = h+ strlen(h)+1;
    concat_in(h, program_dir, "%.mpk");
    dir[2] = h;
    g = buffer+strlen(buffer)+1;
    c = h = path;
    while (*c) {
	c++;
	if (*h==':') h++;
	while (*c && *c!=':') c++;
	t = *c;
	*c = '\0';
	fname = search_through_dirs(dir, 4, h);
	*c = t;
	if (!fname) return False;
	h=c;
    }
    myfree(keypath);
    keypath = path;
    return True;
}    

void load_keypath(void)
{
    char *c, *h;
    char t;

    c = h = program_keypath;
    while (*c) {
	c++;
	if (*h==':') h++;
	while (*c && *c!=':') c++;
	t = *c;
	*c = '\0';
	read_keymap_file(h);
	*c = t;
	h=c;
    }
    if (keypath) {
	c = h = keypath;
	while (*c) {
	    c++;
	    if (*h==':') h++;
	    while (*c && *c!=':') c++;
	    t = *c;
	    *c = '\0';
	    read_keymap_file(h);
	    *c = t;
	    h=c;
	}
    }
}

void keymap_init(void)
{
    int i;

    func[0].fn=NULL;
    func[0].desc=NULL;
    deffunc[0].fnr = NULL;
    deffunc[0].args=NULL;
    deffunc[0].kargs=NULL;
    deffunc[0].desc=NULL;
    keystack.kstack[0]=0;
    keystack.mstack[0]=0;
    keystack.depth=0;
    argstr[0]=0;
    for (i=0; i<max_keycode; i++) {
	redef[i] = (KEYCODE) i;
    }
#ifndef DEL_BACK_SWITCH
#define DEL_BACK_SWITCH 0
#endif
#if DEL_BACK_SWITCH
    redef[  K_delete   ] =   K_backspace;
    redef[  K_backspace] =   K_delete;
    redef[ CK_delete   ] =  CK_backspace;
    redef[ CK_backspace] =  CK_delete;
    redef[ SK_delete   ] =  SK_backspace;
    redef[ SK_backspace] =  SK_delete;
    redef[ MK_delete   ] =  MK_backspace;
    redef[ MK_backspace] =  MK_delete;
    redef[CMK_delete   ] = CMK_backspace;
    redef[CMK_backspace] = CMK_delete;
#endif

    call_deffunc_nr = add_func(call_defined_func, "");
}
