/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/


typedef
struct {
    short height,ascent, descent, width;
} FONTSIZE;

FONTSIZE fontsize[NR_SIZE], fontsizemax;

#define BUFSIZE 1000
static char buffer[BUFSIZE];

typedef
struct {
    short width, ascent, descent, left, right;
} CHARINFO;

typedef
struct {
    char *name;
    char *code;
    short normal[256], math[256];
    short locks;
} LATEXINFO;

typedef
struct {
    char *name;
    Font f_id;
    unsigned min_char, max_char;
    short f_width;
    int f_height, f_ascent;
    CHARINFO *char_info;
    short locks;
} MXFONTINFO;


typedef
struct {
    short *size;
    short *orsize;
    char *loaded;
    char *name;
    short normal;
    short max;
    short label;
    short fnr;
} SIZEINFO;

typedef
struct {
    short max;
    char *loaded;
    SIZEINFO **info;
    int lock;
    char *filename;
} FONTSIZEINFO;

typedef
struct {
    short tnr;
    char *fontname, *openfont, *closefont, *texfilename;
} GENFONTINFO;

typedef
struct {
    short fnr, lnr, fsnr, number, page[2], nrml, max;
    short *size;
    char *loaded;
    char *name;
} FONTINFO;

PAGELIST pagelist;

static INTSTACK *font_stack = NULL;

static LATEXINFO linfo[40];
static MXFONTINFO xfinfo[600];
static FONTSIZEINFO fsinfo[200];

static int nr_latex=0, nr_xfont=0, nr_fsinfo=0;

static FONTINFO finfo[NR_SIZE][MAXFONTS];
static GENFONTINFO gfinfo[256];
static unsigned char fnr2anr[NR_SIZE][256];
static int number_of_fonts[NR_SIZE];
static int alternative[NR_SIZE] = {0};
static int script[NR_SIZE] = {0};

static void remove_return(char *str)
{
    int i;

    if (str && (i=strlen(str)) && (str[i-1] == '\n')) {
	str[i-1] = '\0';
    }
}

static int read_integer_short(char **txt)
{
    int n=0;
    Bool neg=False;

    if (**txt=='-') { neg=True; (*txt)++;}
    while (isdigit(**txt)) {
	n=n*10+ (**txt)-'0';
	(*txt)++;
    }
    return (neg? -n:n);
}

static int read_integer(char **txt)
{
    int n=read_integer_short(txt);

    while (isspace(**txt)) (*txt)++;
    return n;
}

static void make_pagelist(int nr)
{
    int i,j,h,n;

    n=0;

    for (i=0; i<256; i++) {
	h = fnr2anr[nr][i];
	if (finfo[nr][h].number==i && i!=BUTTONFONT)
	    if (finfo[nr][h].page[0]== -1) {
		pagelist.page[n].fnr = i;
		pagelist.page[n++].spos = 0;
		if (xfinfo[finfo[nr][h].fnr].max_char > 127) {
		    pagelist.page[n].fnr = i;
		    pagelist.page[n++].spos = 128;
		}
	    } else
		for (j=0; j<2; j++)
		    if (finfo[nr][h].page[j]!= -2) {
			pagelist.page[n].fnr = i;
			pagelist.page[n++].spos = finfo[nr][h].page[j];
		    }
    }
    pagelist.maxpage = n;
}

static void calc_size(void)
{
    int i,j,h;

    for (j=0; j<NR_SIZE; j++) {
	fontsize[j].descent = 0;
	fontsize[j].ascent = 0;
	fontsize[j].width = 0;
	for (i=0; i<number_of_fonts[j]; i++) {
	    if (finfo[j][i].number!=BUTTONFONT) {
		h = xfinfo[finfo[j][i].fnr].f_height -
		    xfinfo[finfo[j][i].fnr].f_ascent;
		if (fontsize[j].descent < h) fontsize[j].descent = h;
		if (fontsize[j].ascent < xfinfo[finfo[j][i].fnr].f_ascent)
		    fontsize[j].ascent = xfinfo[finfo[j][i].fnr].f_ascent;
		if (fontsize[j].width < xfinfo[finfo[j][i].fnr].f_width)
		    fontsize[j].width = xfinfo[finfo[j][i].fnr].f_width;
	    }
	}
	fontsize[j].height = fontsize[j].ascent + fontsize[j].descent;
    }
    if (number_of_fonts[POPUPFONT]) {
	script[0] = POPUPFONT;
	i=1;
    } else i=0;
    for (; i<NR_SIZE; i++) {
	int k=1;
	h=0;
	for (j=0; j<NR_SIZE; j++)
	    if (fontsize[j].height<fontsize[i].height && fontsize[j].height>h) {
		h= fontsize[j].height;
		k=j;
	    }
	if (h)
	    script[i] = k;
	else
	    script[i] = i;
    }
    fontsizemax = fontsize[grnr];
}

static Bool read_fontsize(char *name, FONTSIZEINFO *fsi)
{
    int i=0,j=0;
    FILE *f;
    char *dirs[2];
    char *fname;
    char b[500];
    short lb[50];
    SIZEINFO *tmp=NULL;

    if (!name) return False;
    dirs[0] = userdir;
    dirs[1] = program_dir;
    concat_in(b, name, ".mpm");
    fname = search_through_dirs(dirs, 2, b);
    if (!fname || !(f=fopen(fname,"r"))) {
	message2(ERROR, "Can't open fontsize file ", name);
	myfree(name);
	return False;
    }
    fsi->filename = name;
    fsi->info = NULL;
    fsi->max = 0;
    fsi->lock=0;
    while (fgets(buffer, BUFSIZE, f)) {
	char *c=buffer;
	while (isspace(*c)) c++;
	i=0;
	while (isdigit(*c)) {
	    i = i*10+ *c-'0';
	    c++;
	}
	if (i) {
	    if (!tmp) tmp = (SIZEINFO*) malloc(sizeof(SIZEINFO));
	    tmp->label=i;
	    c++;
	    while (isspace(*c)) c++;
	    i=0;
	    while (*c && *c!=':') {
		j=0;
		if (*c=='*') {
		    lb[i]=tmp->label;
		    tmp->normal = i;
		    i++;
		    c++;
		} else {
		    while (isdigit(*c)) {
			j = j*10+*c-'0';
			c++;
		    }
		    if (j) lb[i++]=j;
		}
		while (*c && !isdigit(*c) && *c!=':' && *c!='*') c++;
	    }
	    if (*c) {
		c++;
		while (isspace(*c)) c++;
	    }
	    if (*c && i) {
		SIZEINFO **t;
		tmp->size = (short*) malloc(sizeof(short)*i);
		tmp->loaded = (char*) malloc(sizeof(char)*i);
		tmp->orsize = (short*) malloc(sizeof(short)*i);
		for (j=0; j<i; j++) {
		    tmp->orsize[j] = lb[j];
		    tmp->loaded[j] = False;
		    tmp->size[j] = 0;
		}
		tmp->max = i;
		remove_return(c);
		tmp->name = concat(c,"");
		tmp->fnr = 0;
		t = (SIZEINFO**) malloc(sizeof(SIZEINFO*)*(fsi->max+1));
		for (j=0; j<fsi->max; j++) t[j] = fsi->info[j];
		t[j] = tmp;
		myfree(fsi->info);
		fsi->info = t;
		tmp = NULL;
		fsi->max++;
	    }
	}
    }
    fclose(f);
    myfree(tmp);
    if (!fsi->max) {
	message2(ERROR, "Strange fontsize file: ", name);
	myfree(name);
	return False;
    } else {
	fsi->loaded = (char*) malloc(sizeof(char) * fsi->max);
	for (i=0; i<fsi->max; fsi->loaded[i++] = False);
	return True;
    }
}

static void read_general_fontinfo(void)
{
    int i=0;
    FILE *f;
    char *dirs[6];
    char *ind;
    char *fname;
    char b[1000];

    dirs[0]=b;
    concat_in(b+i, userdir, "/%.mpt");
    i+=strlen(b+i)+1;
    dirs[1]=userdir;
    dirs[2]=b+i;
    concat_in(b+i, homedir, "mathspad/%.mpt");
    i+=strlen(b+i)+1;
    dirs[3]=b+i;
    concat_in(b+i, homedir, "mathspad/");
    i+=strlen(b+i)+1;
    dirs[4]=b+i;
    concat_in(b+i, program_dir,"%.mpt");
    dirs[5]=program_dir;
    fname = search_through_dirs(dirs, 6, generalinfofile);
    if (!fname)
	message2(EXIT-1, "Unable to find font information file ",
		 generalinfofile);
    f = fopen(fname, "r");
    if (!f)
	message2(EXIT-1, "Unable to open font information file ",
		 generalinfofile);
    for (i=0; i<256; i++) {
	gfinfo[i].fontname = gfinfo[i].texfilename = gfinfo[i].openfont
	    = gfinfo[i].closefont = NULL;
	gfinfo[i].tnr = 0;
    }
    buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
    while (buffer[0]) {
	if ((ind = begins_with("FONTNR:", buffer))) {
	    int fnr=0;
	    remove_return(ind);
	    if (*ind) fnr = read_integer(&ind);
	    fnr = fnr&0xFF;
	    buffer[0]='\0'; fgets(buffer, BUFSIZE, f);
	    if ((ind = begins_with("FONTNAME:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].fontname);
		gfinfo[fnr].fontname = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }
	    if ((ind = begins_with("LATEX:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].texfilename);
		gfinfo[fnr].texfilename = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }
	    if ((ind = begins_with("OPENFONT:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].openfont);
		gfinfo[fnr].openfont = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }
	    if ((ind = begins_with("CLOSEFONT:", buffer))) {
		remove_return(ind);
		myfree(gfinfo[fnr].closefont);
		gfinfo[fnr].closefont = concat(ind, "");
		buffer[0]='\0'; fgets(buffer,BUFSIZE,f);
	    }
	} else {
	    buffer[0]='\0';
	    fgets(buffer, BUFSIZE, f);
	}	
    }
    fclose(f);
}

static int read_fontpart(FILE *f, FONTINFO finf[MAXFONTS])
{

#define get_new_string (not_finished = ((fgets(buffer, BUFSIZE, f)!=NULL) && \
					!begins_with("STOP", buffer)))

    int nr, i;
    Bool not_finished;
    char *ind;
    unsigned char last_fontnr = 0;

    nr=0;
    get_new_string;
    while (not_finished && nr<MAXFONTS) {
	if ((ind = begins_with("FONT:", buffer)) ||
	    (ind = begins_with("FONTGROUP:", buffer))) {
	    remove_return(ind);
	    finf[nr].name = (char *) malloc(strlen(ind)+1 );
	    strcpy(finf[nr].name,ind);
	    if (buffer[4]!=':') finf[nr].nrml = 1;
	    else finf[nr].nrml = 0;
	    finf[nr].page[0]= -1;       /* alle karakters in symbol window */
	    finf[nr].page[1]= -2;
	    finf[nr].lnr=0;
	    finf[nr].fnr=0;
	    finf[nr].max = 0;
	    finf[nr].size = NULL;
	    finf[nr].number = last_fontnr;
	    if (last_fontnr+1<BUTTONFONT) last_fontnr++;
	    if (get_new_string && (ind = begins_with("NUMBER:", buffer))) {
		remove_return(ind);
		if (*ind) {
		    i = read_integer(&ind);
		    if (0<=i && i<BUTTONFONT)
			finf[nr].number = i;
		}
		get_new_string;
	    }
	    if (not_finished && (ind = begins_with("PAGE:", buffer))) {
		remove_return(ind);
		if (begins_with("NONE", ind))
		    finf[nr].page[0]= -2;  /* geen symbolen in symboolwindow */
		else {
		    for (i=0; i<2 && ind; i++) {
			while (*ind && !isdigit(*ind)) ind++;
			if (*ind)
			    finf[nr].page[i]=read_integer(&ind);
		    }
		}
		get_new_string;
	    }
	    if (gfinfo[finf[nr].number].texfilename) {
		nr++;
	    } else {
		free(finf[nr].name);
	    }
	} else
	    if ((ind = begins_with("BUTTONFONT:", buffer))) {
		remove_return(ind);
		finf[nr].name= (char *) malloc(strlen(ind)+1);
		strcpy(finf[nr].name,ind);
		finf[nr].page[0] = finf[nr].page[1] = -2;
		finf[nr].number = BUTTONFONT;
		nr++;
		get_new_string;
	    } else 
		get_new_string;
    }
    return nr;
}

static short load_latexfile(char *name)
{
#define BIGBUFSIZE 20480
    
    char *tbuf=NULL, *pos;
    char *dirs[3];
    char *fname;
    int count=0, i;
    FILE *tf;

    if (!name) return 0;

    for (i=0; i<nr_latex && strcmp(name, linfo[i].name); i++);
    if (i<nr_latex) {
	linfo[i].locks++;
	myfree(name);
	return i;
    }
    dirs[0] = userdir;
    dirs[1] = program_dir;
    dirs[2] = "";
    fname = search_through_dirs(dirs, 3, name);
    if (!fname || !(tf=fopen(fname, "r"))) {
	message(ERROR,  "Can't open file with LaTeX macros.");
	myfree(name);
	return 0;
    }
    if (!(tbuf = (char *) malloc(BIGBUFSIZE))) {
	message(ERROR, "Out of memory loading LaTeX macros.");
	myfree(name);
	return 0;
    }
    count=1;
    pos=tbuf+1;
    tbuf[0] = '\0';
    for (i=0; i<256; i++)
	linfo[nr_latex].normal[i] = linfo[nr_latex].math[i] = 0;
    while (fgets(tbuf+count, BIGBUFSIZE-count, tf)!=NULL) {
	pos = tbuf+count;
	while (isspace(*pos)) pos++;
	if (isdigit(*pos)) {
	    i = read_integer_short(&pos);
	    tbuf[count] = i%256;
	    if (*pos=='$' || *pos=='&' || isspace(*pos)) {
		count++;
		tbuf[count]= (isspace(*pos) ? ' ' : *pos);
		count++;
		switch (*pos) {
		case '&': linfo[nr_latex].normal[i%256]=count;
		case '$': linfo[nr_latex].math[i%256]=count;
		    break;
		default:  linfo[nr_latex].normal[i%256]=count;
		}
		pos++;
		while (isspace(*pos)) pos++;
		if (!*pos)
		    tbuf[count++] = ' ';
		else
		    while (*pos && *pos!='\n') tbuf[count++] = *pos++;
		tbuf[count++] = '\0';
	    }
	}
    }
    fclose(tf);
    if (count>1) {
	if ((linfo[nr_latex].code = (char *) malloc( count))) {
	    for (i=0; i<count; i++)
		linfo[nr_latex].code[i] = tbuf[i];
	    myfree(tbuf);
	} else
	    linfo[nr_latex].code = tbuf;
    } else
	linfo[nr_latex].code = concat("","");
    linfo[nr_latex].name = name;
    linfo[nr_latex].locks++;
    nr_latex++;
    return nr_latex-1;
}


static int load_xfont(char *name)
{
    CHARINFO *tinf;
    XFontStruct *tempinfo;
    XCharStruct *perchar;
    unsigned i,j;

    for (i=0; i<nr_xfont && strcmp(name, xfinfo[i].name); i++);
    if (i<nr_xfont) {
	xfinfo[i].locks++;
	myfree(name);
	return i;
    }
    if (!(tempinfo = XLoadQueryFont(display, name))) {
	message2(ERROR, "Cannot load font ", name);
	myfree(name);
	return 0;
    }
    for (i=0; i<nr_xfont && xfinfo[i].f_id!=tempinfo->fid; i++);
    if (i<nr_xfont) {
	xfinfo[i].locks++;
	myfree(name);
	XFree((void*)tempinfo);
	return i;
    }
    xfinfo[nr_xfont].min_char = tempinfo->min_char_or_byte2;    
    xfinfo[nr_xfont].max_char = tempinfo->max_char_or_byte2;
    xfinfo[nr_xfont].f_width  = tempinfo->max_bounds.width;
    xfinfo[nr_xfont].f_id     = tempinfo->fid;
    xfinfo[nr_xfont].f_ascent = tempinfo->ascent;
    xfinfo[nr_xfont].f_height = tempinfo->ascent + tempinfo->descent;
    xfinfo[nr_xfont].name     = name;
    j = xfinfo[nr_xfont].max_char - xfinfo[nr_xfont].min_char + 1;
    if ((tinf = xfinfo[nr_xfont].char_info =
	 (CHARINFO *) malloc( 256*sizeof(CHARINFO)))==NULL) {
	message(ERROR, "Out of memory at font loading.");
	myfree(name);
	XFree((void*) tempinfo);
	return 0;
    }
    for (i=0; i<xfinfo[nr_xfont].min_char; i++,tinf++) tinf->width = 0;
    if ((perchar = tempinfo->per_char))
	for (i=0; i<j; i++,perchar++,tinf++) {
	    tinf->width   = perchar->width;
	    tinf->ascent  = perchar->ascent;
	    tinf->descent = perchar->descent;
	    if (perchar->lbearing<0)
		tinf->left = -perchar->lbearing;
	    else
		tinf->left = 0;
	    if (tinf->width<perchar->rbearing)
		tinf->right = perchar->rbearing - tinf->width;
	    else
		tinf->right = 0;
	}
    else
	for (i=0; i<j; i++,tinf++) {
	    tinf->width   = xfinfo[nr_xfont].f_width;
	    tinf->ascent  = xfinfo[nr_xfont].f_ascent;
	    tinf->descent = xfinfo[nr_xfont].f_height-
		xfinfo[nr_xfont].f_ascent;
	    tinf->left = tinf->right = 0;
	}
    for (i=xfinfo[nr_xfont].max_char+1; i<256; i++, tinf++) tinf->width = 0;
    XFree((void*) tempinfo);
    xfinfo[nr_xfont].locks++;
    nr_xfont++;
    return nr_xfont-1;
}

static void free_xfont(int nr)
{
    if (xfinfo[nr].locks) {
	xfinfo[nr].locks--;
	if (!xfinfo[nr].locks) {
	    myfree(xfinfo[nr].name);
	    xfinfo[nr].name = "";
	    myfree(xfinfo[nr].char_info);
	    xfinfo[nr].char_info = NULL;
	    XUnloadFont(display,xfinfo[nr].f_id);
	    xfinfo[nr].f_id = 0;
	}
    }
}

static void free_xfontsize(int nr)
{
    if (fsinfo[nr].lock) {
	fsinfo[nr].lock--;
	if (!fsinfo[nr].lock) {
	    int i;
	    myfree(fsinfo[nr].filename);
	    fsinfo[nr].filename="";
	    for (i=0; i<fsinfo[nr].max; i++) {
		if (fsinfo[nr].loaded[i])
		    free_xfont(fsinfo[nr].info[i]->fnr);
		myfree(fsinfo[nr].info[i]->size);
		myfree(fsinfo[nr].info[i]->orsize);
		myfree(fsinfo[nr].info[i]->loaded);
	    }
	    myfree(fsinfo[nr].loaded);
	}
    }
}

static Bool add_xfontsize(int fsnr, short *size, int nr)
{
    FONTSIZEINFO *fs = fsinfo+fsnr;
    int i, tl,p,j;

    i=0;
    for (i=0; fs->info[i]->size!=size && i<fs->max; i++);
    if (i==fs->max || fs->info[i]->loaded[nr]) return True;
    tl = fs->info[i]->orsize[nr];
    for (i=0; fs->info[i]->label!=tl && i<fs->max; i++);
    if (i==fs->max) return False;
    p = fs->info[i]->fnr = load_xfont(fs->info[i]->name);
    fs->loaded[i] = True;
    if (!p) p = fs->info[i]->fnr = finfo[grnr][fnr2anr[grnr][fsnr]].fnr;
    for (i=0; i<fs->max; i++) {
	for (j=0; j<fs->info[i]->max; j++) {
	    if (fs->info[i]->orsize[j]==tl) {
		fs->info[i]->size[j] = p;
		fs->info[i]->loaded[j] = True;
	    }
	}
    }
    return True;
}

static short load_xfontsize(char *name, short **sizes, char **loaded,
			    short *max, short *nrml, short *fsnr)
{
    int i,j=0,k;
    char *c = name;

    while (isdigit(*c)) {
	j = j*10+*c-'0';
	c++;
    }
    c++;
    while (isspace(*c)) c++;
    i=0;
    while ((name[i++]=*c++));
    for (i=0; i<nr_fsinfo && strcmp(fsinfo[i].filename,name); i++);
    if (i==nr_fsinfo && read_fontsize(name, fsinfo+i)) nr_fsinfo++;
    if (i<nr_fsinfo) {
	fsinfo[i].lock++;
	*fsnr = i;
	for (k=0; k<fsinfo[i].max && fsinfo[i].info[k]->label<j; k++);
	if (k==fsinfo[i].max) k--;
	add_xfontsize(i, fsinfo[i].info[k]->size, fsinfo[i].info[k]->normal);
	*sizes = fsinfo[i].info[k]->size;
	*max = fsinfo[i].info[k]->max;
	*nrml = fsinfo[i].info[k]->normal;
	*loaded = fsinfo[i].info[k]->loaded;
	return (*sizes)[*nrml];
    }
    return 0;
}

static Bool load_font(FONTINFO *finf)
{
    finf->max = 0;
    if (finf->nrml)
	finf->fnr = load_xfontsize(finf->name, &finf->size, &finf->loaded,
				   &finf->max, &finf->nrml, &finf->fsnr);
    else
	finf->fnr = load_xfont(finf->name);
    if (finf->fnr) {
	finf->lnr = gfinfo[finf->number].tnr;
	if (!finf->lnr) {
	    if (finf->max)
		free_xfontsize(finf->fsnr);
	    else
		free_xfont(finf->fnr);
	}
	return (finf->lnr); 
    }
    return 0;
}

static void remove_font(int nr, int pos, FONTINFO *finf)
{
    int i;

    if (fnr2anr[nr][BUTTONFONT] == pos) {
	finfo[nr][pos].number = BUTTONFONT;
	for (i=0; i<BUTTONFONT; i++) {
	    if (fnr2anr[nr][i]==pos) fnr2anr[nr][i] = MAXFONTS+1;
	}
    } else {
	if (finfo[nr][pos].max)
	    free_xfontsize(finfo[nr][pos].fsnr);
	else
	    free_xfont(finfo[nr][pos].fnr);
	i=pos;
	while (i+1<number_of_fonts[nr]) {
	    finfo[nr][i] = finfo[nr][i+1];
	    i++;
	}
	number_of_fonts[nr]--;
	for (i=0; i<256; i++)
	    if (fnr2anr[nr][i]>pos)
		fnr2anr[nr][i]--;
	    else if (fnr2anr[nr][i]==pos)
		fnr2anr[nr][i] = MAXFONTS+1;
    }
}

static Bool can_change_font(int nr, int number)
{
    return ((number != BUTTONFONT) &&
	    ((number_of_fonts[nr]<MAXFONTS) ||
	     (fnr2anr[nr][number]!=fnr2anr[nr][TEXTFONT]) ||
	     (number==TEXTFONT &&
	      fnr2anr[nr][BUTTONFONT]!=fnr2anr[nr][TEXTFONT])));
}

static Bool must_remove_font(int nr, int number)
{
    return ((fnr2anr[nr][number]!=fnr2anr[nr][TEXTFONT])||(number==TEXTFONT));
}

static void add_font(int nr, FONTINFO *finf)
{
    finfo[nr][number_of_fonts[nr]] = *finf;
    if (finf->number == TEXTFONT) {
	int i,j=fnr2anr[nr][TEXTFONT];

	for (i=0; i<256; i++)
	    if (fnr2anr[nr][i]==j)
		fnr2anr[nr][i]=number_of_fonts[nr];
    } else
	fnr2anr[nr][finf->number] = number_of_fonts[nr];
    number_of_fonts[nr]++;
}

int missing_font(FILE *f)
{
    FONTINFO tfi[MAXFONTS], comp;
    int i, n;
    Bool load_tfi[MAXFONTS], lchange=False;
    int wrong1, wrong2, good;
    int oldwidth, oldheight;

#define free_names(A)  for (i=0; i<n; i++) if ((A) || !load_tfi[i]) { \
		       myfree(tfi[i].name); }

    wrong1 = wrong2 = good = 0;
    n = read_fontpart(f, tfi);
    for (i=0; i<n; i++) {
	load_tfi[i]=False;
	comp = finfo[0][fnr2anr[0][tfi[i].number]];
	if (tfi[i].number!=BUTTONFONT) {
	    if (tfi[i].number==TEXTFONT) good++;
	    if (comp.number!= tfi[i].number) {
		wrong1++;
		load_tfi[i]=True;
	    }
	}
    }
    if (!good) {
	free_names(True);
	return EXIT;
    }
    if (wrong1) {
	i=0;
	while (wrong1 && i<n) {
	    if (load_tfi[i])
		if (can_change_font(0,tfi[i].number)) {
		    if (load_font(tfi+i)) {
			add_font(0,tfi+i);
			wrong1--;
			lchange = True;
		    }
		} else
		    load_tfi[i]=False;
	    i++;
	}
	if (lchange) {
	    oldwidth = fontsize[0].width;
	    oldheight = fontsize[0].height;
	    if (!number_of_fonts[SYMBOLFONT] && alternative[SYMBOLFONT]==0)
		make_pagelist(0);
	    calc_size();
	    if ((oldwidth!=fontsize[0].width)||(oldheight!=fontsize[0].height))
		call_layout_change();
	}
	if (wrong1) {
	    message(CLICKREMARK, "Missing fonts for this file.");
	    free_names(False);
	    return EXIT-1;
	}
    }
    free_names(False);
    return 0;
}

static void make_fonts(void)
{
    int i,j;
    int num;
    FILE *f = NULL;
    char *dirs[2];
    char fname[500];
    char *name;

    dirs[0] = userdir;
    dirs[1] = program_dir;

    for (j=0; j<NR_SIZE; j++) {
	number_of_fonts[j]=0;
	for (i=0; i<256; fnr2anr[j][i++]=MAXFONTS+1);
	alternative[j]=j;
	if ((!fontfile[j] || !fontfile[j][0]) && j) {
	    alternative[j]=0;
	} else {
	    concat_in(fname, fontfile[j], ".mpf");
	    name = search_through_dirs(dirs, 2, fname);
	    if (!name) {
		fname[strlen(fname)-4]='\0';
		name = search_through_dirs(dirs, 2, fname);
	    }
	    if (!name && !j) {
		concat_in(fname, program_fontfilename, ".mpf");
		name = search_through_dirs(dirs, 2, fname);
		if (!name) {
		    fname[strlen(fname)-4] = '\0';
		    name = search_through_dirs(dirs, 2, fname);
		}
	    }
	    if (name)
		f = fopen(name, "r");
	    else
		f = NULL;
	    if (!f) {
		if (!j) {
		    message(EXIT-1, "Fontfile not found.");
		    /* ending program */
		} else
		    alternative[j]=0;
	    } else {
		num = read_fontpart(f, finfo[j]);
		fclose(f);
		for (i=0; i<256; i++)
		    fnr2anr[j][i]=255;
		for (i=0; i<num; i++) {
		    if (load_font(&finfo[j][i])) {
			add_font(j, &finfo[j][i]);
		    }
		}
		if (!fnr2anr[j][TEXTFONT]==255)
		    message(EXIT -1, "Fontfile does not contain a textfont.");
	    }
	}
    }
    calc_size();
    make_pagelist(alternative[SYMBOLFONT]);
}

Bool new_fontfile(int nr, char *newname)
{
    if (((!newname||!*newname) && (!fontfile[nr]||!fontfile[nr][0])) ||
	(newname && fontfile[nr] && !strcmp(newname, fontfile[nr])))
	return False;
    else {
	int i;
	Bool fileok;
	int num;
	FILE *f;
	FONTINFO newfonts[MAXFONTS];
	char *dirs[2];
	char fname[500];
	char *name;

	dirs[0] = userdir;
	dirs[1] = program_dir;
	concat_in(fname, newname, ".mpf");
	name = search_through_dirs(dirs, 2, fname);
	if (!name) {
	    fname[strlen(fname)-4] = '\0';
	    name = search_through_dirs(dirs, 2, fname);
	}
	if (!name) {
	    message(CLICKREMARK, "Unable to access new fontfile.");
	    return False;
	}
	f = fopen(name, "r");
	num = read_fontpart(f, newfonts);
	fclose(f);
	fileok = False;
	for (i=0; !fileok && i<num; i++)
	    fileok = (newfonts[i].number == TEXTFONT);
	if (!fileok) {
	    message(CLICKREMARK, "Fontfile does not contain a textfont.");
	    for (i=0; i<num; i++) {
		myfree(newfonts[i].name);
	    }
	    return False;
	}
	for (i=0; i<num; i++) {
	    if (can_change_font(nr, newfonts[i].number)) {
		if (load_font(newfonts+i)) {
		    if (must_remove_font(nr,newfonts[i].number))
			remove_font(nr, fnr2anr[nr][newfonts[i].number],
				    newfonts+i);
		    add_font(nr, newfonts+i);
		}
	    } else {
		myfree(newfonts[i].name);
	    }
	}
	alternative[nr] = nr;
	calc_size();
	make_pagelist(alternative[SYMBOLFONT]);
	myfree(fontfile[nr]);
	fontfile[nr] = newname;
	return True;
    }
}

static void destroy_fonts(int nr)
{
    int i;
    for (i=0;i<number_of_fonts[nr];i++) {
	if (finfo[nr][i].max)
	    free_xfontsize(finfo[nr][i].fsnr);
	else
	    free_xfont(finfo[nr][i].fnr);
    }
}

void push_fontgroup(int nr)
{
    int i;
    i = (nr<0||nr>NR_SIZE ? 0 : alternative[nr]);
    push_int(&font_stack, i);
    if (grnr!=i) {
	grnr = i;
	fontsizemax = fontsize[grnr];
	setsimpletabsize(8*char_width(Font2Char(TEXTFONT,'n'),0));
    }
}

void pop_fontgroup(void)
{
    int i = pop_int(&font_stack);
    i = head_int(font_stack);
    if (i!=grnr) {
	grnr = i;
	fontsizemax = fontsize[grnr];
	setsimpletabsize(8*char_width(Font2Char(TEXTFONT,'n'),0));
    }
}

#define FNR(A,B) fnr2anr[A][B]
#define FINFO(A,B) finfo[A][FNR(A,B)]

static int get_size(int granr, int fontnr, int size)
{
    int j = fnr2anr[granr][fontnr];
    int i = size+finfo[granr][j].nrml;

    if (i<0) i=0; else if (i>=finfo[granr][j].max) i=finfo[granr][j].max-1;
    if (!finfo[granr][j].loaded[i]) {
	message(MESSAGE, "Loading extra font ...");
	add_xfontsize(finfo[granr][j].fsnr, finfo[granr][j].size, i);
	message(MESSAGE, "Loading extra font ... done.");
    }
    return i;
}

#define XINFO(A,B,C) xfinfo[(FINFO(A,B).max?FINFO(A,B).size[get_size(A,B,C)]:FINFO(A,B).fnr)]

Font font_ID(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_id;
}

int font_height(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_height;
}

char *font_opentex(int fontnr, int size)
{
    return gfinfo[fontnr].openfont;
}

char *font_name(int fontnr)
{
    return gfinfo[fontnr].fontname;
}

char *font_closetex(int fontnr, int size)
{
    return gfinfo[fontnr].closefont;
}

int font_ascent(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_ascent;
}

int font_descent(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_height -
	XINFO(grnr,fontnr,size).f_ascent;
}

short font_width(int fontnr, int size)
{
    return XINFO(grnr,fontnr,size).f_width;
}

int char_width(Char data, int size)
{
    int h = FINFO(grnr,Char2Font(data)).lnr;
    int j = Char2ASCII(data);

    if (!linfo[h].math[j] &&
	!linfo[h].normal[j]) return 0;
    else
	return XINFO(grnr,Char2Font(data),size).char_info[j].width;
}

int char_ascent(Char data, int size)
{
    int j = Char2ASCII(data);
    return XINFO(grnr,Char2Font(data), size).char_info[j].ascent;
}

int char_descent(Char data, int size)
{
    int j = Char2ASCII(data);
    return XINFO(grnr,Char2Font(data), size).char_info[j].descent;
}

int char_left(Char data, int size)
{
    return XINFO(grnr,Char2Font(data), size).char_info[Char2ASCII(data)].left;
}

int char_right(Char data, int size)
{
    return XINFO(grnr,Char2Font(data), size).char_info[Char2ASCII(data)].right;
}

char *char_latex(Char data, Bool math)
{
    int h = FINFO(grnr,Char2Font(data)).lnr;
    int j = Char2ASCII(data);
    int idx = (math? linfo[h].math[j] : linfo[h].normal[j]);

    if (idx && XINFO(grnr,Char2Font(data), 0).char_info[j].width)
	return linfo[h].code+idx;
    else
	return NULL;
}

int string_width(int fontnr, char *string, int nr)
{
    int i;
    fontnr = finfo[grnr][fnr2anr[grnr][fontnr]].fnr;
    for (i=0; (nr && *string && *string!='\n') ;nr--,string++)
	i+= xfinfo[fontnr].char_info[*string].width;
    return i;
}
