/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
/*
**   File : edit.c
**   Datum: 11-4-92
**   Doel : Het open en verwerken van gegevens voor een file dat
**          is gekoppeld aan een edit-window. Er kunnen meerdere
**          edit-windows open zijn, met samen 1 primaire en 1
**          secundaire selectie.
*/

#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "sources.h"
#include "keymap.h"
#include "message.h"
#include "button.h"
#include "scrollbar.h"
#include "remark.h"
#include "output.h"
#include "fileread.h"
#include "notatype.h"
#include "edit.h"
#include "editor.h"
#include "menu.h"
#include "fileselc.h"
#include "helpfile.h"
#include "popup.h"

#define EDITNAME  "MathSfile : "
#define EMPTYFILE "noname"
#define LENEMPTYFILE 6
#define CHANGED   " (modified)"
#define VIEWCOM   " (view)"
#define DONECOM   " (done)"
#define RUNCOM    " (running)"
#define EXTENSION ".mpd"

enum button { LOADBUTTON,  SAVEBUTTON,    RENAMEBUTTON,
	      OUTPUTBUTTON, INCLUDEBUTTON, DONEBUTTON, NR_BUTTON  };
#define ASCIIBUTTON NR_BUTTON
static
int perm[NR_BUTTON+5] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10};

static
char *editbutton[NR_BUTTON+1] = { "Load",  "Save", "Rename",
				  "Output", "Include", "Done", "Output" };
static
int edithelp[NR_BUTTON+1] =
{ EDITLOADHELP, EDITSAVEHELP, EDITRENAMEHELP, EDITOUTPUTHELP,
  EDITINCLUDEHELP, EDITDONEHELP, EDITOUTPUTHELP };

static
struct { char *lines;
	 char *extension;
     } textremark[NR_BUTTON+5] = {
	 {"Load document:", "*.mpd"},
	 {"Save document:", "*.mpd"},
	 {"Rename document:", "*.mpd"},
	 {"Make output file:", "*.tex"},
	 {"Include document:", "*.mpd"},
	 {"File not saved!\nSave it?", " Yes \n No \n Cancel "},
	 {"Make output file:", "*"},
	 {"Make LaTeX output file:", "*.tex"},
	 {"Make pe output file:", "*.tex"},
	 {"Make plain LaTeX output file:", "*.tex"},
	 {"Make Ascii output file:", "*"}};

static void handle_output_popup(void*, int);
static Char oname1[6] = { 'L', 'a', 'T','e', 'X', 0 };
static Char oname2[6] = { 'P', 'l', 'a','i', 'n', 0 };
static Char oname3[6] = { 'A', 's', 'c','i', 'i', 0 };
static
MENULINE outputlines[3] =
{ { oname1 ,5,0,0,handle_output_popup,NULL,MPTEX, NULL},
  { oname2 ,5,0,0,handle_output_popup,NULL,PLAINTEX, NULL},
  { oname3 ,5,0,0,handle_output_popup,NULL,ASCII, NULL}};
static
MENU outputmenu =
{
    3,0,-1,0,0,0,-1,-1,-1,-1,0,0,0,"Output",NULL,NULL,outputlines,NULL,NULL,0,0
};

typedef struct { Window win_id, drawwin_id;
                 /* tekstboom, windowtekst, aantal regels, ... */
                 Bool saved, iconized, auto_saved, view_mode, empty,
		 shell,fini,strt;
		 int xpos, ypos, width, height, buflen;
                 void *info;
                 char *headername, *filename, *pathname, *outputname;
		 unsigned char *prcsbuf;
                 void *scrollver, *scrollhor;    /* gegevens van scrollbars */
               } EDITINFO;

/*
** newname wordt gebruikt om de string van remark terug te krijgen.
** newname is alleen niet NULL als functie edit_handle_remark
** wordt aangeroepen
*/

static unsigned long edit_mask;
static XSetWindowAttributes edit_attr;
static char *newname = NULL;
static char *file_name = NULL;
static int number_open=0;
static int number_icon=0;
static int is_opened=False;
static int last_xpos =0, last_ypos = 0, last_width = 0, last_height = 0;
static Bool state_open = False, as_icon = False;
static EDITINFO *state_window = NULL;
static Bool change_check = True;
static Atom inputat, outputat, intestat, outtestat, textat;

#define sub_width(A)   (A) - INTERSPACE*3 -SCROLLBARSIZE
#define sub_height(A)  (A) - INTERSPACE*4 -SCROLLBARSIZE - button_height
#define pos_x_with     INTERSPACE*2 +SCROLLBARSIZE
#define pos_x_without  INTERSPACE
#define pos_y_with     INTERSPACE*3 +SCROLLBARSIZE +button_height
#define pos_y_without  INTERSPACE*2 +button_height

static char buffer[2000];

static void remove_auto_save(EDITINFO *einf)
{
    buffer[0] = '\0';
    strcat(buffer, userdir);
    strcat(buffer, "/#");
    strcat(buffer, strip_name(einf->filename));
    strcat(buffer, "#" EXTENSION);
    remove_file(buffer);
}

static void edit_auto_save(void *data, int dump)
{
    EDITINFO *einf = (EDITINFO *) data;
    FILE *f;
    Bool cont = True;

    if (!einf->auto_saved && !einf->empty) {
        if (dump || !((cont=True)))
	    sprintf(buffer, "dump%s" EXTENSION, strip_name(einf->filename));
	else
	    sprintf(buffer, "#%s#" EXTENSION, strip_name(einf->filename));
	f = open_file(userdir, buffer, "wb");
	if (f) {
	    set_file(f);
	    put_filecode(DOCUMENTFILE);
	    save_editwindow(einf->info);
	    unset_file();
	    cleanup_stencilstack();
	    fclose(f);
	    einf->auto_saved = True;
	} else failure=True;
    }
    if (!cont && !dump) {
	sprintf(buffer,"File %s not consistent.\nFile dumped, no backup made.",
		strip_name(einf->filename));
	message(CLICKREMARK, buffer);
    }
}

static int set_name(void *data, char *pathname)
{
    EDITINFO *einf = (EDITINFO *) data;
    XTextProperty prop_name;
    char *name;
    int namesize;
    char *stripname, *nname;

    if (pathname == NULL) {
	EDITINFO *tinf;
	FLEXARRAY istck;
	int i=0,j;
	int_init(istck);
	while ((tinf=(EDITINFO*)next_data_with_type(MAINEDITWINDOW, &i))) {
	    if (!strncmp(EMPTYFILE,tinf->filename, LENEMPTYFILE)) {
		j=mystrtoi(tinf->filename+LENEMPTYFILE, NULL, 10);
		int_add(istck, j);
	    }
	    i++;
	}
	i=1;
	while (int_contains(istck,i)) i++;
	int_clear(istck);
	nname = (char *) malloc(strlen(userdir) + strlen(EMPTYFILE)
				+ 5 +strlen(EXTENSION));
	concat_in(nname, userdir, "/" EMPTYFILE);
	stripname = nname + strlen(nname);
	sprintf(stripname, "%i", i);
	strcat(stripname, EXTENSION);
    } else
	nname = pathname;
    stripname = concat(strip_name(nname),"");
    if (!strcmp(stripname+strlen(stripname)-strlen(EXTENSION), EXTENSION))
	stripname[strlen(stripname)-strlen(EXTENSION)] = '\0';
    namesize = strlen(EDITNAME) + strlen(stripname) + 1 +
	(einf->saved ? 0 : strlen(CHANGED)) +
	(einf->view_mode && !einf->shell ? strlen(VIEWCOM) : 0) +
	(einf->shell && !einf->fini ? strlen(RUNCOM) : 0) +
	(einf->shell && einf->fini ? strlen(DONECOM) : 0);
    name = (char *) malloc( namesize );
    if (name) {
	name[0]= '\0';
	strcat(name, EDITNAME);
	strcat(name, stripname);
	if (!einf->saved && !einf->shell) strcat(name, CHANGED);
	if (einf->view_mode && !einf->shell) strcat(name, VIEWCOM);
	if (einf->shell && !einf->fini) strcat(name, RUNCOM);
	if (einf->shell && einf->fini) strcat(name, DONECOM);
    }
    if (!name || !XStringListToTextProperty(&name, 1, &prop_name)) {
	message(ERROR, "No location for editname.");
	return 0;
    }
    XSetWMName(display, einf->win_id, &prop_name);
    myfree(einf->headername);
    if (einf->pathname!=nname) myfree(einf->pathname);
    myfree(einf->filename);
    einf->headername = name;
    einf->filename = stripname;
    einf->pathname = nname;

    if (!XStringListToTextProperty(&stripname, 1, &prop_name)) {
	message(ERROR, "No location for editicon.");
	return 0;
    }
    XSetWMIconName(display, einf->win_id, &prop_name);
    return 1;
}

static void set_output_name(EDITINFO *einf)
{
    int i=0;
    if (einf->outputname) free(einf->outputname);
    einf->outputname = (char*) malloc((i=strlen(latexdir))+
				     strlen(einf->filename)+6);
    if (latexdir[i-1]=='/')
	concat_in(einf->outputname,latexdir,einf->filename);
    else {
	concat_in(einf->outputname, latexdir, "/");
	concat_in(einf->outputname, einf->outputname, einf->filename);
    }
    if (output_mode==ASCII)
	strcat(einf->outputname, ".asc");
    else
	strcat(einf->outputname, ".tex");
}

#define BINDOC 1
#define OLDDOC 2
#define NEWDOC 3

static int test_file(FILE *f)
{
    int i;
    i = fgetc(f);
    if (i=='B') {
	if (missing_font(f)==EXIT) {
	    rewind(f);
	    return BINDOC;
	} else
	    return OLDDOC;
    } else {
	rewind(f);
	return (i==27 ? NEWDOC: BINDOC);
    }
}

static int check_name(EDITINFO *einf, char *name)
{
    int i=0,found=0;
    EDITINFO *tinf;
    if (!einf || !name) return 0;
    while (!found &&
	   (tinf=(EDITINFO*) next_data_with_type(MAINEDITWINDOW, &i))) {
	found = (tinf!=einf && !tinf->view_mode &&
		 !strcmp(name, tinf->pathname));
	i++;
    }
    if (found) {
	if (tinf->iconized)
	    XMapWindow(display, tinf->win_id);
	XRaiseWindow(display, tinf->win_id);
    }
    return found;
}

static void handle_filename(void *data, char *name)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (name) {
	FILE *f;
	int i;
	int check_found = check_name(einf, name);

	if (!(f = fopen(name,"rb"))) {
	    message2(CLICKREMARK, "Unable to open file ", name);
	    myfree(name);
	    failure=True;
	    return;
	}
	i = test_file(f);
	set_wait_cursor(einf->win_id);
	switch (i) {
	case BINDOC:
	    message(MESSAGE, "Loading ascii file.");
	    read_file(f,BINARYFILE);
	    unset_file();
	    load_editwindow(einf->info);
	    break;
	case OLDDOC:
	    i = edit_fnr;
	    edit_fnr = 0;
	    load_notation_filenames(f);
	    old_load_editwindow(einf->info,f);
	    edit_fnr = i;
	    if (!state_open) clear_file_ref();
	    break;
	case NEWDOC:
	    i = edit_fnr;
	    edit_fnr = 0;
	    read_file(f,DOCUMENTFILE);
	    unset_file();
	    load_editwindow(einf->info);
	    edit_fnr = i;
	    break;
	default: break;
	}
	fclose(f);
	cleanup_nodestack();
	cleanup_filestack();
	cleanup_stencilstack();
	einf->saved = True;
	einf->auto_saved = True;
	einf->view_mode = check_found;
	set_name(einf, name);
	set_output_name(einf);
	name = NULL;
	remove_wait_cursor();
	if (check_found)
	    message(CLICKREMARK, "The document is already loaded.\n"
		    "This copy has been loaded in view mode\n"
		    "in order to ensure that only one\n"
		    "backup is made.");
	return;
    }
    myfree(name);
}

static void handle_view_filename(void *data, char *name)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (name) {
	FILE *f;
	int i;

	if (!(f = fopen(name,"rb"))) {
	    message2(CLICKREMARK, "Unable to open file ", file_name);
	    myfree(name);
	    failure=True;
	    return;
	}
	i=test_file(f);
	set_wait_cursor(einf->win_id);
	switch (i) {
	case BINDOC:
	    message(MESSAGE, "Viewing ascii file.");
	    read_file(f,BINARYFILE);
	    unset_file();
	    load_editwindow(einf->info);
	    break;
	case OLDDOC:
	    i = edit_fnr;
	    edit_fnr = 0;
	    view_notation_filenames(f);
	    old_load_editwindow(einf->info,f);
	    edit_fnr = i;
	    break;
	case NEWDOC:
	    i = edit_fnr;
	    edit_fnr = 0;
	    read_file(f,DOCUMENTFILE);
	    unset_file();
	    load_editwindow(einf->info);
	    edit_fnr = i;
	    break;
	default: break;
	}
	fclose(f);
	cleanup_filestack();
	cleanup_stencilstack();
	cleanup_nodestack();
	einf->saved = True;
	einf->auto_saved = True;
	einf->view_mode = True;
	set_name(einf, name);
	set_output_name(einf);
	file_name = NULL;
	remove_wait_cursor();
	return;
    }
    myfree(name);
}

static void handle_include_filename(void *data, char *name)
{
    EDITINFO *einf = (EDITINFO *) data;
    FILE *f;
    int i;

    if (!name) return;
    if (!(f = fopen(name,"rb"))) {
	message2(CLICKREMARK, "Unable to open file ", name);
	myfree(name);
	failure=True;
	return;
    }
    i=test_file(f);
    set_wait_cursor(einf->win_id);
    switch (i) {
    case BINDOC:
	message(MESSAGE, "Including ascii file.");
	read_file(f,BINARYFILE);
	unset_file();
	include_editwindow(einf->info);
	break;
    case OLDDOC:
	i = edit_fnr;
	edit_fnr = 0;
	load_notation_filenames(f);
	old_include_editwindow(einf->info,f);
	edit_fnr = i;
	if (!state_open) clear_file_ref();
	break;
    case NEWDOC:
	i = edit_fnr;
	edit_fnr = 0;
	read_file(f, DOCUMENTFILE);
	unset_file();
	include_editwindow(einf->info);
	edit_fnr = i;
	break;
    default: break;
    }
    fclose(f);
    cleanup_filestack();
    cleanup_stencilstack();
    cleanup_nodestack();
    einf->saved = False;
    einf->auto_saved = False;
    remove_wait_cursor();
    myfree(name);
}

static void edit_handle_fileselc_save(void *data, char *name)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (name) {
	FILE *f;
	int check_found = check_name(einf, name);

	f = fopen(name,"wb");
	if (f) {
	    set_wait_cursor(einf->win_id);
	    set_file(f);
	    put_filecode(DOCUMENTFILE);
	    save_editwindow(einf->info);
	    unset_file();
	    cleanup_stencilstack();
	    fclose(f);
	    remove_auto_save(einf);
	    einf->auto_saved = True;
	    einf->saved = True;
	    if (check_found) {
		einf->view_mode = True;
		message(CLICKREMARK, "You have saved the document under a name\n"
			"which is already used by a different window.\n"
			"To make sure that your backups are\n"
			"correct, this copy will be in view mode.");
	    }
	    set_name(einf,name);
	    message(MESSAGE,"File saved.");
	    remove_wait_cursor();
	} else {
	    message(ERROR, "Can't save file! ");
	    myfree(newname);
	    failure=True;
	}
    } else
	myfree(newname);
    kind_of_remark = NO_REMARK;
}

static void edit_handle_fileselc_rename(void *data, char *name)
{
    EDITINFO *einf = (EDITINFO *) data;
    int check_found = check_name(einf, name);
    if (check_found) {
	einf->view_mode=True;
	message(CLICKREMARK, "There is already a document loaded with\n"
		"the same name in a different window.\n"
		"To make sure that your backups are\n"
		"correct, this copy will be in view mode.");
    }
    set_name(einf, name);
}

static int texmode=0;

static void edit_handle_fileselc_output(void *data, char *name)
{
    EDITINFO *einf = (EDITINFO *) data;
    FILE *f;

    if ((f=fopen(name, "w"))) {
	tex_set_file(f);
	tex_mode(texmode);
	tex_placeholders(ON);
	latex_editwindow(einf->info);
	tex_unset();
	fclose(f);
	myfree(einf->outputname);
	einf->outputname = name;
	name = concat("OUTPUTFILE=", einf->outputname);
	putenv(name);
	message(MESSAGE, "Document converted.");
    } else {
	message2(CLICKREMARK, "No output made. Unable to open file ",
		 newname);
	failure=True;
    }
}

static void edit_handle_remark_done(void *data, int bnr)
{
    EDITINFO *einf = (EDITINFO *) data;
    FILE *f;
    int close_next = quit_sequence;

    if (!bnr) {
	f = fopen(newname,"wb");
	if (f) {
	    set_file(f);
	    put_filecode(DOCUMENTFILE);
	    save_editwindow(einf->info);
	    unset_file();
	    cleanup_stencilstack();
	    fclose(f);
	    remove_auto_save(einf);
	    einf->saved = True;
	    edit_close(data);
	} else {
	    message(ERROR, "Can't save file.");
	    failure=True;
	}
    } else if (bnr==1) {
	edit_auto_save(data, 0);
	einf->saved = True;
	edit_close(data);
    } else if (bnr==2) {
	close_next = False;
	quit_sequence = False;
    }
    myfree(newname);
    kind_of_remark = NO_REMARK;
    newname = NULL;
    if (close_next) menu_close();
}

static void (*handle_fileselc_func[NR_BUTTON+5])(void*,char*) = {
    handle_filename,     edit_handle_fileselc_save,
    edit_handle_fileselc_rename,   edit_handle_fileselc_output,
    handle_include_filename, NULL, edit_handle_fileselc_output,
    edit_handle_fileselc_output, edit_handle_fileselc_output,
    edit_handle_fileselc_output, edit_handle_fileselc_output
};


static void set_fileselc(void *data, int nr)
{
    EDITINFO *einf = (EDITINFO *) data;
    char *c=NULL, *h, *s;

    nr = perm[nr];
    if ((nr==OUTPUTBUTTON || nr>=NR_BUTTON) && einf->outputname) {
	h = einf->outputname;
	s = strip_name(einf->outputname);
	if (s!=h) {
	    c=s-1;
	    *c='\0';
	} else h=NULL;
    } else {
	h = einf->pathname;
	s = strip_name(einf->pathname);
	if (s!=h) {
	    c=s-1;
	    *c='\0';
	} else h=NULL;
    }
    fileselc_open(handle_fileselc_func[nr], data, textremark[nr].lines,
		  h, textremark[nr].extension, s, einf->win_id);
    if (c) *c='/';
}

static void edit_draw(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;

    redraw_window(einf->info);
}

static void edit_layout_change(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (!data) return;
    if (!einf->iconized)
	XClearArea(display, einf->drawwin_id, 0, 0, 0, 0, True);
    scrollbar_linesize(einf->scrollver, line_height());
}

static void handle_output_popup(void *data, int nr)
{
    EDITINFO *einf = (EDITINFO *) data;
    if (!einf->outputname) set_output_name(einf);
    texmode = nr;
    set_fileselc(data, NR_BUTTON+nr);
}

static void edit_handle_button(void *data, int b_num)
{
    EDITINFO *einf = (EDITINFO *) data;

    switch (b_num) {
    case OUTPUTBUTTON:
	if (mouse_button==3) {
	    int i;
	    outputmenu.transwin=einf->win_id;
	    for (i=0; i<3; outputmenu.line[i++].fdata=data);
	    popup_make(&outputmenu);
	} else {
	    MENULINE *mp = outputmenu.line+outputmenu.defline;
	    (*(mp->func))(data, mp->fint);
	}
	break;
    case LOADBUTTON:
    case INCLUDEBUTTON:
    case SAVEBUTTON:
    case RENAMEBUTTON:
	set_fileselc( data, b_num);
	break;
    case DONEBUTTON:
	if (can_close_edit) edit_close(data);
	break;
    }
    newname = NULL;
}

static int edit_margin(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;

    einf->empty = window_empty(einf->info);
    if (!einf->view_mode && !einf->shell) {
	if (!einf->auto_saved)
	    einf->auto_saved = einf->empty;
	else if (change_check)
	    einf->auto_saved = !window_changed(einf->info) || einf->empty;
	if ((!einf->auto_saved && einf->saved) || 
	    (!einf->saved && einf->empty)) {
	    einf->saved = !einf->saved;
	    set_name(einf, einf->pathname);
	}
    }
    return -scrollbar_line(einf->scrollhor, 0) * font_width(TEXTFONT,0) + 3;
}

static void edit_press(void *data, XButtonEvent *event)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (event->window == einf->drawwin_id) {
	change_check = False;
	mouse_down(einf->info, event->x-edit_margin(data),
		   event->y, mouse_button);
	get_motion_hints(einf->drawwin_id, -1);
    }
}

static void double_click_func(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;

    change_check = False;
    dbl_click();
    get_motion_hints(einf->drawwin_id, 0);
}

static void edit_release(void *data, XButtonEvent *event)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (event->window == einf->drawwin_id) {
	stop_motion_hints();
	mouse_up(event->x -edit_margin(data), event->y);
	if (mouse_button==1) {
	    call_func(K_escape);
	    use_map(0,1);
	}
	change_check = True;
    }
}

static void edit_motion(void *data, int x, int y)
{
    mouse_move(x-edit_margin(data),y);
}

static void edit_resize(void *data, XConfigureEvent *event)
{
    EDITINFO *einf = (EDITINFO *) data;
    int new_width, new_height;

    einf->width = last_width = event->width;
    einf->height = last_height = event->height;
    einf->xpos = last_xpos = event->x;
    einf->ypos = last_ypos = event->y;
    new_width  = sub_width( event->width );
    new_height = sub_height(event->height );
    XResizeWindow(display, einf->drawwin_id, new_width-2, new_height-2);
    resize_window(einf->info, new_width-2, new_height-2);
    scrollbar_resize(einf->scrollver, new_height);
    scrollbar_resize(einf->scrollhor, new_width);
}

static void edit_scrollto(void *data, int kind)
{
    EDITINFO *einf = (EDITINFO*) data;
    /*
    ** handel scrollbar up/down/left/right af
    ** teken drawwin
    */
    if (!kind) {
	redraw_window(einf->info);
    } else {
	/* vertical scroll */
	int line_nr = scrollbar_line(einf->scrollver,0);
	editwindow_line(einf->info, line_nr);
    }
}

static void edit_iconize(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (!einf->iconized) {
	einf->iconized = True;
	number_icon++;
	edit_iconized =  (number_open==number_icon);
    }
    /*
    **  sluit invoer op einf->info af
    */
}

static void edit_deiconize(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (einf->iconized) {
	einf->iconized = False;
	number_icon--;
	edit_iconized = False;
    }
    /*
    **  maak invoer op einf->info mogelijk
    */
}

static void edit_state(void *data, int *x, int*y, int *w, int *h,
		       int *i, int *s, char **str)
{
    EDITINFO *einf = (EDITINFO *) data;
    int xm,ym;
    window_manager_added(einf->win_id, &xm, &ym);
    *x = einf->xpos-xm;
    *y = einf->ypos-ym;
    *w = einf->width;
    *h = einf->height;
    *i = (einf->iconized ? 0x1 : 0x0);
    *i += (einf->view_mode ? 0x2 : 0x0);
    *s = scrollbar_line(einf->scrollver, 0);
    *str = einf->pathname;
}

static void edit_use_state(int x, int y, int w, int h,
			   int i, int s, char *str)
{
    as_icon = i&0x1;
    state_open = True;
    state_window=NULL;
    if (w>0 && h>0) {
	last_xpos = x;
	last_ypos = y;
	last_width = w;
	last_height = h;
    }
    if (i&0x4) {
	open_helpfile(str, 0);
    } else {
	edit_open();
	if (state_window) {
	    if (i&0x2)
		handle_view_filename( (void *) state_window, str);
	    else
		handle_filename( (void *) state_window, str);
	}
    }
    if (state_window)
	editwindow_line(state_window->info, s);
    state_open = False;
    as_icon = False;
}

static int edit_last_pos(int *x, int *y, int *w, int *h)
{
    *x = last_xpos;
    *y = last_ypos;
    *w = last_width;
    *h = last_height;
    return is_opened;
}

static void edit_set_last_pos(int x, int y, int w, int h)
{
    last_xpos = x;
    last_ypos = y;
    last_width = w;
    last_height = h;
}

void edit_set_number_of_lines(void *window, int numlin)
{
    EDITINFO *einf;
    void *pdata;
    Window pwin;

    (void) get_window_type(*(Window*) window, &pwin, &pdata);
    (void) get_window_type(pwin, &pwin, &pdata);
    einf = (EDITINFO*) pdata;
    scrollbar_set(einf->scrollver, line_number(einf->info), numlin);
}

Bool edit_saved(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;

    return (einf->saved);
}

void edit_bad_end(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;
    /*
    **  save belangrijke informatie in backup-file  #?#  ?~ ?.BAK
    */
    edit_auto_save(data, 0);
    einf->saved = True;
    close_editwindow(einf->info);
    myfree(einf->headername);
    myfree(einf->pathname);
    myfree(einf->filename);
    myfree(einf->outputname);
    myfree(einf->prcsbuf);
    if (einf->iconized) number_icon--;
    edit_is_open = (--number_open >0);
    edit_iconized = (number_icon==number_open);
    popup_remove(einf->win_id);
    destroy_window(einf->win_id);
}

#ifdef DEBUG
static void make_ispell_popup(char *prp, int len)
{
    MENU ispellmenu;
    int i,j,nw;
    char *c;

    /*
    ** layout of Ispell popup:     Action:
    **
    ** Title:   Incorrect word.
    **          -----------------
    ** Actions: Ignore Once          
    **          Ignore Always      Ispell("@Word") (Accept word)
    **          Add to Dictionary  Ispell("*Word") (Personal dictionary)
    **          Stop               Ispell("#")     (Save personal dictionary)
    **          -----------------
    ** Misses:  Alt. 1             Replace("Word", "Alt. 1");   
    **          .... .             Replace("Word", ".... .");   
    **          Alt. n             Replace("Word", "Alt. n");   
    **          -----------------
    ** Guess:   Guess 1            Ispell("@Guess 1");  (Accept Guess)
    **          ..... .            Ispell("*Guess 1");  (Personal dict.)   
    **          Guess n            
    **          -----------------
    */
    for (i=0,nw=0; i<len; i++)
	nw=nw+(prp[i]==',')+(prp[i]==':');
    
}
#endif

static void edit_property_handle(void *data, XPropertyEvent *event)
{
    EDITINFO *einfo = (EDITINFO*) data;
    if (event->window != einfo->win_id) return;
    if (event->atom==inputat && event->state==PropertyNewValue) {
	long n=0;
	long len=4096;
	Atom actt;
	int actf;
	unsigned long nit,baf=1;
	unsigned char *prp;
	while (baf) {
	    XGetWindowProperty(display, einfo->win_id, inputat, n/4, len, True,
			       textat, &actt, &actf, &nit, &baf, &prp);
	    append_editwindow(einfo->info, (char*)prp, nit);
	    XFree(prp);
	    n=n+nit;
	}
	if (!n) {
	    einfo->fini=True;
	    einfo->strt=False;
	    set_name(einfo, einfo->pathname);
	}
    } else if (event->atom==outtestat && event->state==PropertyNewValue) {
	einfo->strt=True;
	if (einfo->buflen) {
	    XChangeProperty(display, einfo->win_id, outputat, textat, 8,
			    PropModeAppend, einfo->prcsbuf, einfo->buflen);
	    XFlush(display);
	    free(einfo->prcsbuf);
	    einfo->prcsbuf=NULL;
	    einfo->buflen=0;
	}
    } else if (event->atom==outtestat && event->state==PropertyDelete) {
	einfo->strt=False;
	einfo->fini=True;
	set_name(einfo, einfo->pathname);
    }
}

static void edit_send_to_proces(void *data, unsigned char *txt, int len)
{
    EDITINFO *einf= (EDITINFO*) data;
    if (!einf->shell || einf->fini) return;
    if (!einf->strt) {
	unsigned char *h;
	h=(unsigned char*) malloc(sizeof(unsigned char)*(einf->buflen+len));
	memcpy(h, einf->prcsbuf, einf->buflen);
	memcpy(h+einf->buflen, txt, len);
	if (einf->prcsbuf) free(einf->prcsbuf);
	einf->prcsbuf=h;
	einf->buflen=einf->buflen+len;
    } else {
	XChangeProperty(display, einf->win_id, outputat, textat, 8,
			PropModeAppend, txt, len);
	XFlush(display);
    }
}

FUNCTIONS maineditfuncs = {
    edit_bad_end, NULL, edit_resize, NULL, NULL, NULL, edit_iconize,
    edit_deiconize, NULL, NULL, edit_layout_change, edit_auto_save,
    edit_use_state, edit_state, NULL, NULL, edit_last_pos,
    edit_set_last_pos, NULL, edit_property_handle };

FUNCTIONS editfuncs = {
    NULL, edit_draw, NULL, edit_press, edit_release, edit_motion,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, edit_margin,
    edit_set_number_of_lines, NULL, NULL, double_click_func };

void edit_string_to_proces(char *txt, char *shname)
{
    EDITINFO *edata;
    char *c,*h;
    char *d;
    char *arg[4];
    Bool only_text=False;
    int len=0,i=0;
    char lpref='\0';
    /*
    ** txt uses the following format sequences:
    **  %t  %1   target selection
    **  %s  %2   source selection
    **  %a  %3   argument selection
    **  %P?      ? is added at the beginning of each line (in a selection)
    **  %T       only text place holders are passed through
    **  %E       all place holders are passed through
    **  %?       ? character
    ** To add: some way to select the output mode.
    */
    while ((edata = (EDITINFO*) next_data_with_type(MAINEDITWINDOW, &i)) &&
	   strcmp(edata->pathname,shname)) i++;
    if (!edata) {
	message2(ERROR, shname," is not running.");
	return;
    }
    if (!edata->shell) {
	message2(ERROR, shname, " can not receive input.");
	return;
    }
    if (edata->fini) {
	message2(ERROR, shname, " is finished.");
	return;
    }
    if (!edata->strt) message2(MESSAGE, shname, " is still busy.");
    /* scan txt for arguments, to determine len */
    d=txt;
    for (i=0; i<4; arg[i++]=NULL);
    while (*d) {
	switch (*d) {
	case '%':
	    d++;
	    i=0;
	    switch (*d) {
	    case 't': case '1': i=1; break;
	    case 's': case '2': i=2; break;
	    case 'a': case '3': i=3; break;
	    case 'P': d++; lpref=*d; if (!*d) d--; break;
	    case 'T': only_text=True; break;
	    case 'E': only_text=False; break;
	    case '\0': c--; break;
	    default: len++; break;
	    }
	    if (i) {
		if (!arg[i]) {
		    tex_set_string(&arg[i]);
		    tex_placeholders(ON);
		    tex_mode(ASCII);
		    latex_text_only(only_text);
		    latex_selection(i);
		    latex_text_only(False);
		    tex_unset();
		}
		if (arg[i]) {
		    len+=strlen(arg[i]);
		    if (lpref>32) {
			h=arg[i];
			while (*h) {
			    if (*h=='\n') len++;
			    h++;
			}
		    }
		} else arg[0]=concat("","");
	    }
	    break;
	default:
	    len++;
	    break;
	}
	d++;
    }
    if (arg[0]) {
	message(ERROR, "Selections not set properly.");
	for (i=0;i<4;i++) if (arg[i]) free(arg[i]);
	return;
    }
    /* make string */
    c=h=(char*)malloc(sizeof(char)*(len+2));
    while (*txt) {
	switch (*txt) {
	case '%':
	    txt++;
	    i=0;
	    switch (*txt) {
	    case 't': case '1': i=1; break;
	    case 's': case '2': i=2; break;
	    case 'a': case '3': i=3; break;
	    case 'P': txt++; lpref=*txt; if (!*txt) txt--; break;
	    case 'T': break;
	    case '\0': txt--; break;
	    default: *h++=*txt; break;
	    }
	    if (i && arg[i]) {
		if (lpref>32) {
		    char *t=arg[i];
		    while ((*h=*t)) {
			if (*h=='\n') {
			    h++;
			    *h=lpref;
			}
			h++;
			t++;
		    }
		} else {
		    strcpy(h, arg[i]);
		    while (*h) h++;
		}
	    }
	    break;
	default:
	    *h++=*txt;
	    break;
	}
	txt++;
    }
    *h='\0';
    /* send string c */
    edit_send_to_proces((void*)edata, (unsigned char*) c, len);
    free(c);
    for (i=0; i<4; i++) if (arg[i]) free(arg[i]);
}

void edit_init(void)
{
    if (output_mode==ASCII)
	perm[OUTPUTBUTTON] = NR_BUTTON;
    edit_mask =
        (CWBackPixel | CWBorderPixel | CWBitGravity |
	 CWEventMask | CWColormap);

    edit_attr.background_pixel = white_pixel;
    edit_attr.border_pixel = black_pixel;
    edit_attr.colormap = colormap;
    edit_attr.bit_gravity = NorthWestGravity;
    edit_attr.event_mask = (ExposureMask | ButtonPressMask | ButtonReleaseMask
			    | ButtonMotionMask | PointerMotionHintMask
			    | KeyPressMask | StructureNotifyMask
			    | VisibilityChangeMask | PropertyChangeMask);
    if (!last_width) {
	last_width = display_width / 2;
	last_height = display_height / 3;
	last_xpos = (display_width-last_width)/2;
	last_ypos = (display_width-last_width)/2;
    }
    inputat  =XInternAtom(display, "MPINPUT", False);
    outputat =XInternAtom(display, "MPOUTPUT", False);
    intestat =XInternAtom(display, "MPINTEST", False);
    outtestat=XInternAtom(display, "MPOUTTEST", False);
    textat   =XInternAtom(display, "TEXT", False);
}

void edit_open(void)
{
    int x = INTERSPACE;
    int y = INTERSPACE;
    int i;
    XSizeHints size_hints;
    EDITINFO *einf;

    state_window = NULL;
    if ( (einf = (EDITINFO *) malloc( sizeof(EDITINFO) )) == NULL)
	message(ERROR, "Out of memory in edit.");
    else {
	if (!state_open)
	    if (!last_xpos && !last_ypos) {
		last_xpos = (display_width - last_width)/2;
		last_ypos = (display_height - last_height)/2;
	    }
	einf->xpos = last_xpos;
	einf->ypos = last_ypos;
	einf->width = last_width;
	einf->height = last_height;
	einf->saved = True;
	einf->auto_saved = True;
	einf->view_mode = False;
	einf->empty = True;
	einf->iconized  = True;
	einf->shell = False;
	einf->fini = False;
	einf->strt = False;
	einf->buflen=0;
	einf->prcsbuf=NULL;
	einf->win_id = XCreateWindow(display, root_window, einf->xpos,
				     einf->ypos, einf->width, einf->height,
				     BORDERWIDTH, CopyFromParent, InputOutput,
				     visual,
				     edit_mask, &edit_attr);
	if (state_open)
	    size_hints.flags = USPosition | USSize | PMinSize;
	else
	    size_hints.flags = PPosition | PSize | PMinSize;
	wm_hints.initial_state =
	    ((iconic || as_icon) ? IconicState : NormalState);
	size_hints.min_width =
	    size_hints.min_height = pos_y_with + SCROLLBARSIZE*3;
	XSetWMProperties(display, einf->win_id, NULL, NULL,
			 NULL, 0, &size_hints, &wm_hints, &class_hints);
	wm_hints.initial_state = NormalState;
	set_protocols(einf->win_id);

	i=0;
	einf->headername = NULL;
	einf->filename = NULL;
	einf->pathname = NULL;
	einf->outputname = NULL;
	if (set_name(einf, NULL) &&
	    add_window(einf->win_id, MAINEDITWINDOW,
		       root_window, (void *) einf, helpname[EDITHELP])) {
	    while (i<NR_BUTTON &&
		   button_make(i, einf->win_id, editbutton[perm[i]], &x, y, 1,
			       (void*) einf, helpname[edithelp[i]],
			       NULL, NULL, edit_handle_button,
			       edit_handle_button, edit_handle_button, NULL))
		i++,x+=BINTERSPACE;
	    x = sub_width(last_width);
	    y = sub_height(last_height);
	    if (i==NR_BUTTON) {
		einf->drawwin_id =
		    XCreateWindow(display, einf->win_id,
				  pos_x_with, pos_y_with, x-2, y-2, 1,
				  CopyFromParent, InputOutput,
				  visual,
				  edit_mask, &edit_attr);
		if (add_window(einf->drawwin_id, EDITWINDOW,
			       einf->win_id, NULL, helpname[EDITSUBHELP]))
		    i++;
	    }
	    if (i==NR_BUTTON +1 &&
		(einf->scrollhor =
		 scrollbar_make(HORIZONTAL, einf->win_id, pos_x_with,
				pos_y_without, x, font_width(TEXTFONT,0),
				edit_scrollto, (void*) einf)))
		i++;
	    if (i==NR_BUTTON+2 &&
		(einf->scrollver =
		 scrollbar_make(VERTICAL, einf->win_id, pos_x_without,
				pos_y_with, y, line_height(),
				edit_scrollto, (void*) einf)))
		i++;
	    
	}
	if (i<NR_BUTTON+3) {
	    myfree(einf->headername);
	    myfree(einf->pathname);
	    myfree(einf->filename);
	    XDestroyWindow(display, einf->win_id);
	    destroy_window(einf->win_id);
	} else {
	    is_opened = True;
	    scrollbar_set(einf->scrollver, 0, 1);
	    scrollbar_set(einf->scrollhor, 0, 80);
	    einf->info = open_editwindow(&einf->drawwin_id, x-2, y-2);
	    (void) window_changed(einf->info);
	    number_icon++;
	    number_open++;
	    edit_is_open = True;
	    edit_iconized = (number_icon==number_open);
	    state_window = einf;
	    XMapSubwindows(display, einf->win_id);
	    XMapWindow(display, einf->win_id);
	}
    }
}

void edit_close(void *data)
{
    EDITINFO *einf = (EDITINFO *) data;

    if (einf->saved) {
	close_editwindow(einf->info);
	myfree(einf->headername);
	myfree(einf->pathname);
	myfree(einf->filename);
	myfree(einf->outputname);
	if (einf->iconized) number_icon--;
	edit_is_open = (--number_open >0);
	edit_iconized = (number_icon==number_open);
	XDestroyWindow(display, einf->win_id);
	popup_remove(einf->win_id);
	destroy_window(einf->win_id);
    } else {
	if (einf->iconized) {
	    XMapWindow(display, einf->win_id);
	    einf->iconized = False;
	    XRaiseWindow(display, einf->win_id);
	}
	XFlush(display);
	newname = einf->pathname;
	remark_make(einf->win_id, data, edit_handle_remark_done,
		    REMARK_BUTTON, char2Char(textremark[DONEBUTTON].lines),
		    textremark[DONEBUTTON].extension, &newname, 300, NULL);
	remark_raise();
	newname = NULL;
    }
}

void open_program(char *c, char *title)
{
    /* c contains the commandline that should be executed.
     * It should contain a %i (or %x) at the location where the window-id
     * should be inserted
     * It has to be a shell script of the form:
     *      xpipeout -window wid | command | xpipein -window wid
     */
    EDITINFO *edata;
    char callbuf[1024];
    move_selection=False;
    edit_open();
    move_selection=True;
    if (state_window) {
	edata=state_window;
	edata->shell=1;
	sprintf(callbuf, c, edata->win_id, edata->win_id);
	XChangeProperty(display, edata->win_id, intestat, textat, 8,
			PropModeReplace, "Yes", 3);
	XDeleteProperty(display, edata->win_id, outputat);
	XFlush(display);
	system(callbuf);
	/* wait till program is ready to read. */
	edata->view_mode=True;
	set_name(edata, concat(title,""));
    }
}

/* header to be able to add this function to a menu */
void open_helpfile(void *data, int nr)
{
    char *c = (char*) data;
    int i=0,hpos=0;
    char *f;
    char *name;
    char *fullname;
    EDITINFO *edata;

    if (!c || !(f = (char*) malloc(strlen(c)+1))) return;
    for (i=0; (f[i]=c[i]) ; i++)
	if (c[i]=='#') if (!i) hpos=-1; else if (hpos) hpos=0; else hpos=i;
    if (hpos>0) {
	name = c+hpos+1;
	f[hpos]='\0';
    } else name = c+i;
    f = standard_dir(f);
    fullname = search_through_dirs(help_dirs, nr_help_dirs, f);
    if (!fullname && f[0]=='/')
	fullname=f;
    else
	free(f);
    if (!fullname) {
	message(ERROR, "Help document not found.");
	return;
    }
    i=0;
    while ((edata = (EDITINFO*) next_data_with_type(MAINEDITWINDOW, &i)) &&
	   strcmp(edata->pathname,fullname))
	i++;
    if (!edata) {
	as_icon=1;
	move_selection=False;
	edit_open();
	move_selection=True;
	as_icon=0;
	if (state_window) {
	    edata = state_window;
	    handle_view_filename((void *) state_window, concat(fullname,""));
	    word_wrap_window(edata->info);
	}
    }
    if (!edata) {
	message2(CLICKREMARK, "Unable to open an edit window for document ",
		 name);
	return;
    }
    if (name[0]) {
	int j=strlen(name)+1;
	Char *cname = (Char*) malloc(j*sizeof(Char));
	if (cname) {
	    for (i=j-1;i>=0; i--) cname[i]=name[i];
	    editwindow_topto(edata->info, cname);
	    free(cname);
	}
    }
    if (edata->iconized) {
	XMapWindow(display, edata->win_id);
    }
    XRaiseWindow(display, edata->win_id);
}

