#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <paths.h>
#include <utmp.h>
#include <signal.h>

struct initline
{
  pid_t pid;
  char tty[7];
  char command[30];
  char argv[10][20];
};

struct initline initline[64];
int num_inittab_lines=0;

void
error(char * errstr)
{
  int error_fd;

  if ((error_fd=open("/dev/console", O_WRONLY))<0)
      exit(1);

  write(error_fd, "tiny-init: ", 11);
  write(error_fd, errstr, strlen(errstr));
  close(error_fd);
}

void
read_inittab(void)
{
  int inittab_fd=0;
  int bytes_read=0;
  int line_num=0;
  char line[80];
  char * ptr, * ptr_2;
  int arg_num=1;

  if ((inittab_fd=open("./inittab", O_RDONLY))<0)
    {
      error("Could not open /etc/inittab!\n");
      exit(1);
    }

  while ((bytes_read=read(inittab_fd, line, sizeof(line)))>0)
    {
      if ((ptr=strchr(line, '\n'))==NULL) 
	{
	  error("Line too long in inittab.\n");
	  exit(1);
	}

      lseek(inittab_fd, (long) (ptr-line)-bytes_read+1, SEEK_CUR);

      if (isalpha(*line))
	{
	  if ((ptr=strchr(line, ':'))==NULL)
	    {
	      error("Malformatted line in inittab.\n");
	      exit(1);
	    }
	  *ptr++='\0';
	  strcpy(initline[num_inittab_lines].tty, line);
	  if ((ptr_2=strchr(ptr, '\n'))==NULL)
	    {
	      error("Malformatted line in inittab.\n");
	      exit(1);
	    }
	  *ptr_2++='\0';
	  ptr_2=strchr(ptr, ' ');
	  *ptr_2++='\0';
	  strcpy(initline[num_inittab_lines].command, ptr);
	  if ((ptr_2=strchr(ptr, '/'))!=NULL)
	    *ptr_2++='\0';
	  strcpy(initline[num_inittab_lines].argv[0], ptr);
	  ptr_2+=strlen(ptr_2)+1;
	  while ((ptr=strchr(ptr_2, ' '))!=NULL)
	    {
	      ptr_2=strchr(ptr_2, ' ');
	      *ptr_2++='\0';
	      strcpy(initline[num_inittab_lines].argv[arg_num++], ptr);
	    }
	  initline[num_inittab_lines].argv[arg_num]=NULL;
	}
      else if (!isspace(*line) && *line!='#')
	{
	  error("Comment line begins with non-space and non-# character.\n");
	  exit(1);
	}
    }
}

/* The wtmp and utmp contain entries which are made everytime the machine
   boots to keep track of uptimes, etc. */
void
make_boot_entries(void)
{
  int fd;
  struct utmp utmp;

  memset(&utmp, 0, sizeof(utmp));
  
  strcpy(utmp.ut_line, "~");
  memset(utmp.ut_name, 0, sizeof(utmp.ut_name));
  time(&utmp.ut_time);
  utmp.ut_type=BOOT_TIME;
  if ((fd=open(_PATH_UTMP, O_CREAT|O_WRONLY|O_TRUNC))>=0)
    {
      write(fd, (char *)&utmp, sizeof(utmp));
      close(fd);
    }
  else error("Couldn't make wtmp boot entry!\n");

  if ((fd=open(_PATH_WTMP, O_WRONLY|O_APPEND))>=0)
    {
      write(fd, (char *)&utmp, sizeof(utmp));
      close(fd);
    }
  else error("Couldn't make utmp boot entry!\n");
}

void
do_reboot(void)
{
  sync();
  reboot(0x1D1E, 0xC0DE, 0x4567);
}

void
int_handler(void)
{
  error("Caught SIGINT - rebooting...\n");
  do_reboot();
}

int
hup_handler(void)
{
  struct initline old_initline[64];
  int old_num_initlines;
  int already_running;
  int i, j;

  signal(SIGHUP, SIG_IGN);

  memcpy(old_initline, initline, 64*sizeof(struct initline));
  old_num_initlines=num_inittab_lines;

  read_inittab();

  for (i=0;i<num_inittab_lines;i++)
    {
      already_running=0;
      for (j=0;j<old_num_initlines;j++)
	if (!strcmp(old_initline[i].tty, initline[i].tty))
	  {
	    already_running=1;
	    if ((initline[i].pid=old_initline[i].pid)<0)
	      spawn(i);
	  }
      if (!already_running) spawn(i);
    }

  signal(SIGHUP, hup_handler);
}

int
boot_single()
{
  pid_t pid;
  int i;
  
  if ((pid=fork())==0)
    {
      execl(_PATH_BSHELL, _PATH_BSHELL, NULL);
      error("Could not execute single user shell!\n");
    }
  else if (pid>0) while (wait(&i)!=pid);
  else if (pid<0)
    {
      error("Couldn't fork() to exec() single-user shell.\n");
      execl(_PATH_BSHELL, _PATH_BSHELL, NULL);
    }

  /* I think the proper behavior after they exit the single user shell
     is to reboot */
  do_reboot();
}

void
run_rc(void)
{
  char * argv[2];
  pid_t pid;
  int i;

  if ((pid=fork())==0)
      execl("/etc/rc", "rc", NULL);
  else if (pid>0) while (wait(&i)!=pid);
  else if (pid<0)
    {
      error("fork() failed when trying to spawn /etc/rc!\n");
      exit(1);
    }
}

void
spawn(int line_num)
{
  pid_t pid;
  int i;

  if ((pid=fork())<0)
    {
      initline[i].pid=-1;
      error("fork failed when trying to spawn something\n");
      return;
    }
  if (pid)
    {
      inittab[i].pid=pid;
      return;
    }
  else
    {
      for (i=0;i<getdtablesize();i++) close(i);
      setsid();
      execv(initline[i].command, initline[i].argv);
      error("exec failed\n");
      /* To prevent rapid respawning... */
      sleep (5);
      exit(1);
    }
    
  
}

void
clear_utmp(int pid)
{
  struct utmp utmp;
  int fd;

  if ((fd=open(_PATH_UTMP, O_RDWR))<0)
    {
      error("Could not open utmp to clear process entry!\n");
      return;
    }

  while (read(fd, (char *) &utmp, sizeof(utmp))>0)
    {
      if (utmp.ut_pid==pid)
	{
	  memset(&utmp.ut_user, 0, sizeof(utmp.ut_user));
	  memset(&utmp.ut_host, 0, sizeof(utmp.ut_host));
	  utmp.ut_type=DEAD_PROCESS;
	  utmp.ut_pid=0;
	  utmp.ut_addr=0;
	  time(&utmp.ut_time);
	  lseek(fd, -sizeof(utmp), SEEK_CUR);
	  write(fd, (char *)&utmp, sizeof(utmp));
	}
    }
}

void
main(int argc, char ** argv)
{
  struct stat stbuf;
  int single_boot=0;
  int i;

  for (i=1;i<argc;i++)
    if (!strcmp(argv[i], "single"))
      single_boot=1;

  if (stat("/etc/singleboot", &stbuf)==0)
    single_boot=1;

  if (single_boot) boot_single();
  make_boot_entries();
  read_inittab();
  signal(SIGHUP, hup_handler);
  signal(SIGINT, int_handler);
  run_rc();

  for (i=0;i<getdtablesize();i++)
    close(i);

  for (i=0;i<num_inittab_lines;i++) spawn(i);

  while(1)
    {
      pid=wait(&vec);
      clear_utmp(pid);
      for (i=0;i<num_inittab_lines;i++)
	{
	  if (pid==initline[i].pid || initline[i].pid<0)
	    {
	      spawn(i);
	      break;
	    }
	}
    } 
}
