/*
  Copyright (C) 1997  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>
#include <pwd.h>
#include <signal.h>
#include "common.h"

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         Global program storage                          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

char *pipefname;							/* named pipe report file */
int pipeFD = 0;								/* report file descriptor */
char *scriptfname;							/* login script filename */
char *envfname;								/* dialing environment filename */
int childPID;								/* PID of running child */

int maxAttempts = MAXNUM_RETRY;				/* max dialing attempts */
int sleepDelay = MAXSEC_DELAY;				/* post-dial sleep delay */
char account[MAXLEN_ACCOUNT+1];				/* account name */
char passwd[MAXLEN_PASSWD+1];				/* password */
char **phone;								/* phone numbers */
int numPhones;								/* how many there are to dial */
char **sline;								/* script lines */
int numSlines;								/* how many there are to include */
char *modemInit;							/* modem init string */				


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                             Utility routines                            |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

void doErr(char *msg)
{
	char emsg[MSGLEN_ERR];

	if (errno < sys_nerr)
		sprintf(emsg, "xispdial: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xispdial: %s: error #%d\n", msg, errno);
	fputs(emsg, stderr);
	if (pipeFD) close(pipeFD);
	exit(1);
}


/* Open a named pipe for writing only */

int namedPipe(char *fname)
{
	struct stat st;
	int fd;

	if (access(fname, F_OK) == -1)				/* check to see if it exists */
		doErr("namedPipe: access");				/* nope, this is not right! */
	else {
		stat(fname, &st);						/* yes, get the node status */
		if (!S_ISFIFO(st.st_mode))				/* is it a FIFO? */
			doErr("namedPipe: stat");			/* nope, still not right! */
	}
	fd = open(fname, O_WRONLY|O_NDELAY);		/* yes, open it for writing */
	if (fd < 0)									/* error means no process has */
		doErr("namedPipe: open");				/* opened it for reading */
	return fd;									/* return the descriptor */
}


/* Write messages printf() style to the named pipe file descriptor. Note
   the implicit total length of MSGLEN_PIPE bytes for the resulting string. */

#define MSGLEN_PIPE 128

int pprintf(va_alist) va_dcl
{
	int iw, bw;
	va_list ap;
	char *fmt, msg[MSGLEN_PIPE];

	va_start(ap);								/* start variable arg list */
	fmt = va_arg(ap, char*);					/* first string is format */
	iw = vsprintf(msg, fmt, ap);				/* pass rest to vsprintf() */
	va_end(ap);									/* end variable arg list */
	bw = write(pipeFD, msg, strlen(msg));		/* write buffer to pipe */
	if (bw < strlen(msg))						/* all bytes written? */
		doErr("xispdial: pprintf");				/* nope, bail out */
	fdatasync(pipeFD);							/* yes, flush the pipe */
	return iw;									/* and return items written */
}


/* Execute a child process, attaching its stderr to a named pipe. The
   pipe output is read by xisp and displayed on its browser window.   */

#define MAXARGS_CHILD 16

int child(va_alist) va_dcl
{
	int i, stat;
	char *args[MAXARGS_CHILD+1];
	va_list ap;

	childPID = fork();							/* fork to create child */
	if (childPID < 0)							/* ret < 0 : fork failed */
		doErr("child: fork");
	if (childPID) {								/* in parrent process */
		i = waitpid(childPID, &stat, 0);		/* get child return status */
		if (i < 0)								/* ret < 0 : wait failed */
			doErr("child: waitpid");
		childPID = 0;							/* OK, indicate child done */
	}
	else {										/* in child process */
		dup2(pipeFD, 2);						/* tack stderr on pipe input */
		va_start(ap);							/* start variable arg list */
		for (									/* parse program arguments */
			i=0; i < MAXARGS_CHILD &&			/* to a max of MAXARGS_CHILD */
			(args[i]=va_arg(ap, char*)) !=
				(char *)0;
			i++
		);
		va_end(ap);								/* end variable arg list */
		execv(args[0], args);					/* exec child */
		doErr("child: execvp");					/* return here means error */
	}
	if (WIFEXITED(stat))						/* parent returns child stat */
		return(WEXITSTATUS(stat));				/* unless something funny */
	else										/* has happened, in which */
		return -1;								/* case -1 is returned */
}

/* Prints out of memory error and exits */

void outOfMem(char *msg)
{
	fprintf(stderr, "xispdial: %s: out of memory!\n", msg);
	if (pipeFD) close(pipeFD);
	exit(1);
}

/* Initializes the dialer file names using the user home directory. */

void initFnames(void)
{
	struct passwd *user;

	user = getpwuid(getuid());
	pipefname = (char *)malloc(6+strlen(user->pw_name)+strlen(PIPEFNAME)+1);
	scriptfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(SCRIPTFNAME)+1);
	envfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(ENVFNAME)+1);
	if (pipefname != NULL && scriptfname != NULL && envfname != NULL) {
		strcpy(pipefname, "/tmp/"); strcat(pipefname, PIPEFNAME);
		strcat(pipefname, "."); strcat(pipefname, user->pw_name);
		strcpy(scriptfname, user->pw_dir); strcat(scriptfname, "/");
		strcat(scriptfname, SCRIPTFNAME);
		strcpy(envfname, user->pw_dir); strcat(envfname, "/");
		strcat(envfname, ENVFNAME);
	}
	else outOfMem("initFnames");
}


/* Reads the environment file created by xISP and then deletes it */

void uEOF(void)
{
	pprintf("%s: %s!\n", "xispdial",
			"environ premature EOF");
	close(pipeFD);
	exit(1);
}

void getISPenv(void)
{
	FILE *envfp;
	int ir = 0, i;

	envfp = fopen(envfname, "r");				/* open temp environment file */
	if (envfp == NULL)							/* if not there bail out */
		doErr("getISPenv: fopen");

	ir += fscanf(envfp,"%s ",account);			/* read all needed dialing */
	ir += fscanf(envfp,"%s ",passwd);			/* environment variables */
	ir += fscanf(envfp,"%d ",&maxAttempts);
	ir += fscanf(envfp,"%d ",&sleepDelay);
	ir += fscanf(envfp,"%d ",&numPhones);		/* read # of phone numbers */
	if (ir < 5)									/* all read so far? */
		uEOF();									/* nope, bail out */
	phone = (char **)							/* allocate space for */
			malloc(numPhones*sizeof(char *));	/* numPhones phone numbers */
	for (i=0; i<numPhones && phone!=NULL;		/* which follow */
		 i++)
	{
		phone[i] = (char *)						/* allocate phone number */
					malloc(MAXLEN_PHONE);
		if (phone[i] != NULL)
			*(phone[i]) = 0;					/* and initialize it */
		else
			phone = NULL;
	}
	if (phone == NULL)							/* allocation completed? */
		outOfMem("getISPenv");					/* nope, bail out */
	for (i=0; i<numPhones; i++) {
		if (fgets(phone[i], MAXLEN_PHONE,		/* read telephone numbers */
				  envfp) == NULL)
			uEOF();								/* bail out on error */
		phone[i][strlen(phone[i])-1] = 0;		/* kill the ending '\n' */
	}
	ir = fscanf(envfp,"%d ",&numSlines);		/* read # of script lines */
	if (! ir)									/* bail out on error */
		uEOF();
	sline = (char **)							/* allocate space for */
			malloc(numSlines*sizeof(char *));	/* numSlines scipt lines */
	for (i=0; i<numSlines && sline!=NULL;		/* which follow */
		 i++)
	{
		sline[i] = (char *)						/* allocate script line */
					malloc(MAXLEN_SLINE);
		if (sline[i] != NULL)
			*(sline[i]) = 0;					/* and initialize it */
		else
			sline = NULL;
	}
	if (sline == NULL)							/* allocation completed? */
		outOfMem("getISPenv");					/* nope, bail out */
	for (i=0; i<numSlines; i++) {
		if (fgets(sline[i], MAXLEN_SLINE,		/* read script lines */
				  envfp) == NULL)
			uEOF();								/* bail out on error */
		sline[i][strlen(sline[i])-1] = 0;		/* kill the ending '\n' */
	}

	fclose(envfp);								/* close and remove the */
	unlink(envfname);							/* temp environment file */
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                             Dialer routines                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

int init_modem(void)							/* initialize modem */
{
	int rcode;

	rcode = child(CHAT,"-V","TIMEOUT","3",
				  "","ATZ","OK-+++\\c-OK",
				  modemInit,(char *)0);
	sleep(1);
	return rcode;
}

int re_init_modem(void)							/* re-initialize modem */
{
	int rcode;

	rcode = child(CHAT,"-V","TIMEOUT","3",
				  "","ATZ","OK-+++\\c-OK",
				  modemInit,(char *)0);
	if (rcode) {
		sleep(1);
		rcode = child(CHAT,"-V","TIMEOUT","3",
					  "","AT","OK-AT-OK",
					  modemInit,(char *)0);
	}
	sleep(1);
	return rcode;
}

int callnumber(char *number)					/* dial the number given */
{
	int i, rcode;
	FILE *scriptF;
	char preamble[] =
		 "TIMEOUT\t\t3\nABORT\t\tBUSY\nABORT\t\t'NO CARRIER'\n"
		 "ABORT\t\tenied\nABORT\t\timeout\n''\t\t\tAT\\sD%c\\s%s\n"
		 "TIMEOUT\t\t50\nCONNECT\t\t''\nTIMEOUT\t\t5\n";
	char script[MAXNUM_SLINES*MAXLEN_SLINE+1];

	umask(077);									/* read/write only by owner */
	scriptF = fopen(scriptfname, "w");			/* open the script file */
	if (scriptF == NULL)						/* failed to do so? */
		doErr("callnumber: fopen");				/* yup, bail out */
	strcpy(script, preamble);					/* OK, copy the preamble */
	for (i=0; i<numSlines; i++) {				/* and now the script lines */
		strcat(script, sline[i]);
		strcat(script, "\n");
	}
	fprintf(scriptF, script, DIAL_TYPE,			/* using the script as fmt */
			number, account, passwd);			/* stick all the rest in */
	fclose(scriptF);							/* close the script file */
	rcode = child(CHAT,"-V","-f",				/* hand it over to chat */
				  scriptfname,(char *)0);
	unlink(scriptfname);						/* done with script, delete */
	if (rcode == 0) {							/* chat terminated OK? */
		pprintf("\n%s: %s %s\n", "xispdial",	/* yes, send "done" message */
				"done dialing", number);
		close(pipeFD);							/* close the report pipe */
		exit(0);								/* and exit with success! */
	}
	if (rcode < 0) {							/* nope, chat problem */
		pprintf("\n%s: %s %d\n", "callnumber",	/* report the error status */
				"chat returned", rcode);
		close(pipeFD);							/* and bail out */
		exit(1);							
	}
	return rcode;								/* else valid report code */
}

void callall(int sn)							/* call the list of numbers */
{
	int i, rcode;

	pprintf("%s: %s #%d\n", "xispdial",			/* report dialing attempt */
			"dialing attempt", sn+1);
	for (i=0; i<numPhones; i++) {				/* try all phones given */
		pprintf("xispdial: dialing: %s\n",		/* report dialed number */
				phone[i]);
		if ((rcode = callnumber(phone[i])))		/* dial it */
			pprintf("\n");						/* not successful, print NL */
		if (rcode == 3) {						/* did chat timeout? */
			pprintf("TIMEOUT\n");				/* yup, report it */
			pprintf("%s: %s %s\n",
				"xispdial", "timeout dialing",
				phone[i]);
			child(CHAT,"-V",""," ",(char *)0);	/* and hang-up */
		}
		sleep(sleepDelay);						/* sleep a while */
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         SIGTERM trap and Main                           |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void Term(int signal)
{
	switch (signal) {
		case SIGTERM: {							/* SIGTERM signal received */
			pprintf("\nxispdial: SIGTERM\n");	/* return status via pipe */
			if (childPID)						/* if a child is in progress */
				kill(childPID, SIGTERM);		/* send it SIGTERM also */
			unlink(envfname);					/* delete environment file */
			unlink(scriptfname);				/* and script file */
			re_init_modem();					/* reset the modem */
			close(pipeFD);						/* close the report pipe */
			exit(1);							/* and exit */
			break;
		}
		default: exit(1);
	}
}

int main()
{
	int rcode, attempt;

	signal(SIGTERM, Term);			/* register SIGTERM handler */
	initFnames();					/* assemble file names used */
	pipeFD = namedPipe(pipefname);	/* initialize named pipe to xISP */
	getISPenv();					/* get info via dialing environment file */

	pprintf("xispdial: PID=%d.\n", getpid());	/* the '.' is important! */
	pprintf("xispdial: dialing ISP: user: %s\n", account);

/* Prepare the modem init string
   ----------------------------- */
	modemInit = malloc(strlen(MODEM_INIT)+5);
	if (modemInit == NULL)
		outOfMem("main");
	strcpy(modemInit, MODEM_INIT);
	strcat(modemInit, " H0");

/* Initialize the modem for dialing
   -------------------------------- */
	if ((rcode=init_modem())) {
		pprintf("xispdial: chat returned %d!\n", rcode);
		pprintf("xispdial: modem init failed!\n");
		return 1;
	}

/* Loop dialing the numbers given maxAttempts times
   ------------------------------------------------ */
	for (attempt=0; attempt<maxAttempts; attempt++) {
		pprintf("\n");
		callall(attempt);
		if ((rcode=re_init_modem())) {
			pprintf("xispdial: chat returned %d!\n", rcode);
			pprintf("xispdial: modem re-init failed!\n");
			return 1;
		}
	}
	close(pipeFD);
	return 1;
}

