
/*
 * Image Split v1.0 -- plug-in for the GIMP.
 * Written by: Danny Sung  dannys@ucla.edu
 *		http://www.seas.ucla.edu/~dannys/
 *
 * This is a simple plug-in that will cut an image into MxN pieces.
 *
 * This program is placed into the public domain by the author, and
 * consequently has no warrenty whatsoever.  Do whatever you wish
 * with it.  My only request is that you give credit where it is due
 * (in other words, please keep my name somewhere in the source) and,
 * if possible, let me know of what changes you've made.
 *
 * $Id: imgsplit.c,v 1.2 1996/05/04 22:26:24 dannys Exp $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "gimp.h"

static int group_ID, dialog_ID, text_ID[2], temp_ID,
	current_ID, split_ID, slider_ID[2];

static void text_callback(int item_ID, void *client_data, void *call_data);
static void slider_callback(int item_ID, void *client_data, void *call_data);
static void cancel_callback(int item_ID, void *client_data, void *call_data);
static void ok_callback(int item_ID, void *client_data, void *call_data);
static void split(Image input);

static int params[2];
#define ROWS 0
#define COLS 1

#define SLIDER_MAX_X		(gimp_image_width(input)/2+1)
#define SLIDER_MAX_Y		(gimp_image_height(input)/2+1)
static Image input;

int main(int argc, char *argv[])
{
	Image output;
	char buf[32];
	void *data;

	if( gimp_init(argc, argv))
	{
		input=0;
		output=0;
		input = gimp_get_input_image(0);
		switch(gimp_image_type(input))
		{
			case GRAY_IMAGE:
			case RGB_IMAGE:
			case INDEXED_IMAGE:
			break;
			default:
				gimp_message("imgsplit: cannot operate on unknown image types");
				gimp_quit();
				break;
		}
		data = gimp_get_params();
		if( data )
		{
			params[ROWS] = ((int*)data)[ROWS];
			params[COLS] = ((int*)data)[COLS];
		}
		else
		{
			params[ROWS] = params[COLS] = 1;
		}

		dialog_ID = gimp_new_dialog("GifSplit");
		group_ID = gimp_new_row_group(dialog_ID, DEFAULT, NORMAL, "");
	
		temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL, "");
		gimp_new_label(dialog_ID, temp_ID, "X direction:");
		sprintf(buf, "%d", params[COLS]);
		text_ID[COLS] = gimp_new_text(dialog_ID, temp_ID, buf);

		slider_ID[COLS] = gimp_new_scale(dialog_ID, group_ID, 
			1, SLIDER_MAX_X, 1, 0);
		gimp_change_item(dialog_ID, slider_ID[COLS], sizeof(params[COLS]), &params[COLS]);

		temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL, "");
		gimp_new_label(dialog_ID, temp_ID, "Y direction:");
		sprintf(buf, "%d", params[ROWS]);
		text_ID[ROWS] = gimp_new_text(dialog_ID, temp_ID, buf);

		slider_ID[ROWS] = gimp_new_scale(dialog_ID, group_ID,
			1, SLIDER_MAX_Y, 1, 0);
		gimp_change_item(dialog_ID, slider_ID[ROWS], sizeof(params[ROWS]), &params[ROWS]);

		temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL, "");
		gimp_new_label(dialog_ID, temp_ID, "Current resolution: ");
		sprintf(buf, "%ldx%ld", gimp_image_width(input), gimp_image_height(input));
		current_ID = gimp_new_text(dialog_ID, temp_ID, buf);

		temp_ID = gimp_new_column_group(dialog_ID, group_ID, NORMAL, "");
		gimp_new_label(dialog_ID, temp_ID, "Split resolution: ");
		sprintf(buf, "%ldx%ld", gimp_image_width(input), gimp_image_height(input));
		split_ID = gimp_new_text(dialog_ID, temp_ID, buf);
	
		gimp_add_callback(dialog_ID, text_ID[COLS], text_callback, (void *)COLS);
		gimp_add_callback(dialog_ID, slider_ID[COLS], slider_callback, (void *)COLS);
		gimp_add_callback(dialog_ID, text_ID[ROWS], text_callback, (void *)ROWS);
		gimp_add_callback(dialog_ID, slider_ID[ROWS], slider_callback, (void *)ROWS);
		gimp_add_callback(dialog_ID, gimp_ok_item_id(dialog_ID), ok_callback,0);
		gimp_add_callback(dialog_ID, gimp_cancel_item_id(dialog_ID), cancel_callback, 0);
	
		if( gimp_show_dialog(dialog_ID))
		{
			gimp_set_params(sizeof(int)*2, params);
			if( input )
			{
				split(input);
			}
		}
		if( input ) gimp_free_image(input);
		gimp_quit();
	}

	return 0;
}

static void slider_callback(int item_ID, void *client_data, void *call_data)
{
	char buf[64];
	int n;
	n = (int)client_data;
	switch(n)
	{
		case COLS:
		case ROWS:
			break;
		default:
			return;
	}
	params[n] = *((int*)call_data);

	sprintf(buf, "%d", params[n]);
	gimp_change_item(dialog_ID, text_ID[n], strlen(buf)+1, buf);
}

static void text_callback(int item_ID, void *client_data, void *call_data)
{
	char str[256];
	int rx, ry, n;

	n = (int)client_data;
	switch( n ){
		case COLS:
		case ROWS:
			break;
		default:
			return;
	}
	/* Make the change to global variable */
	params[n] = atoi(call_data);

	/* Update the split label */
	if( params[COLS] < 1 ) params[COLS] = 1;
	if( params[ROWS] < 1 ) params[ROWS] = 1;
	rx = params[COLS];
	ry = params[ROWS];
	sprintf(str, "%ldx%ld",
		rx==0 ? 0 : gimp_image_width(input)/rx,
		ry==0 ? 0 : gimp_image_height(input)/ry );
	gimp_change_item(dialog_ID, split_ID, strlen(str)+1, str);

	/* Update the slider */
	if( params[n] <=
		(n==COLS ?  SLIDER_MAX_X : SLIDER_MAX_Y ) )
		gimp_change_item(dialog_ID, slider_ID[n], sizeof(params[n]),
			&params[n]);
}

static void ok_callback(int item_ID, void *client_data, void *call_data)
{
	gimp_close_dialog(dialog_ID, 1);
}

static void cancel_callback(int item_ID, void *client_data, void *call_data)
{
	gimp_close_dialog(dialog_ID, 0);
}

static void split(Image input)
{
	Image output;
	int owidth, oheight;
	int x, y, nwidth, nheight;
	int bytes;
	int c;
	unsigned char *src, *dest;

	if( params[COLS] <= 0 || params[ROWS] <= 0 )
		return;

	bytes = gimp_image_channels(input);
	src = gimp_image_data(input);
	owidth = gimp_image_width(input);
	oheight = gimp_image_height(input);

	nwidth = owidth / params[COLS];
	nheight = oheight / params[ROWS];

	for(y=0; y<params[ROWS]; y++)
		for(x=0; x<params[COLS]; x++)
		{
			char *ptr;
			char str[256];
			int n,m;
			sprintf(str, "%s%02dx%02d", gimp_image_name(input), x, y);
			output = gimp_new_image(str, nwidth, nheight,
				gimp_image_type(input));
			if( !output )
				continue;
			ptr = gimp_image_name(output);
			sprintf(ptr, "%02d%02d\n", x,  y);
			dest = gimp_image_data(output);
			gimp_display_image(output);
			for(n=0; n<nwidth; n++)
				for(m=0; m<nheight; m++)
					for(c=0; c<bytes; c++)
					{
						*(dest+n*bytes+c+m*nwidth*bytes) =
							*(src+(x*nwidth*bytes+n*bytes+c)+(y*nheight+m)*owidth*bytes);
					}
			gimp_update_image(output);
			gimp_free_image(output);
		}
}
