
/* blending.c V1.0 - Copyright (C) 1993/94/95/96 Jim Geuther */

/*
 *
 * Function:	Blend to rgb images.
 * Author: 	Jim GEUTHER
 * Date:    	24-May-1996
 * Environment: Personal Power System 850 + AIX V4.1.3.0
 *
 * This Gimp plugin blends two rgb-images. This
 * filter can be extremely usefull to replace the background of an
 * image.
 *
 * Processing options:
 *
 * Operation - specified which type of operation should be performed.
 *	Following operations are supported:
 *	add - Blends primary and secondary image by adding their rgb-values.
 *	sub - Blends primary and secondary image by subtracting the 
 *		rgb-values of the secondary image from the primary image. The
 *		resultating rgb values are corrected to be within the range
 *		0-255.
 *	mul - Blends primary and secondary image by multiplying the rgb-values,
 *		of both images. This operator ignores the weighting factors.
 *	div - Blends the primary and secondary image by dividing primary image
 *		rgb values, by the secondary image rgb values.
 *	not - Copies the secondary image into the primary image, where each
 *		primaries pixel matches the specified background color.
 *	diff - Blends the primary and the secondary image by calculating
 *		the differences between the rgb-values.
 *
 *
 * Weight 1 - Weight factor to be applied to the first image.
 * Weight 2 - Weight factor to be applied to the second image.
 *
 * Background color - This is used only when both weighting factors are
 * 	100. Wherever a pixel in the primary image matches the specified
 *	rgb value, it will be completely replaced with a pixel from the
 *	secondary image.
 *
 * Example:
 *
 * Load an image.
 * Convert to colormap indexed.
 * Remap colormap from dark to light.
 * Perform colormapped sobel edge detect.
 * Convert image to rgb-image.
 * Now perform some of the blend operations.
 *
 * History:
 * V1.00	Jim GEUTHER, ported from ImageKnife (Amiga) to Gimp (AIX)
 *		Heavily based on blend.c
 *
 */
 
/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * Blends two images together.
 */

#include "gimp.h"

/* #define	_DEBUG */

static char *prog_name;
static int dialog_ID;

typedef enum {
	bl_add = 1,
	bl_sub = 2,
	bl_mul = 3,
	bl_div = 4,
	bl_not = 5,
	bl_diff = 6
} opers;

typedef struct {
	int	oper;
	int	weight1;
	int	weight2;
	int	red;
	int	green;
	int	blue;
} Data;

static Data *data=NULL;


/* Declare local functions.
 */
 
static void radio_callback (int, void *, void *);
static void image_menu_callback (int, void *, void *);
static void scale_callback (int, void *, void *);
static void ok_callback (int, void *, void *);
static void cancel_callback (int, void *, void *);
static void blending(Image, Image,opers,int,int,int,int,int);

int
main (argc, argv)
     int argc;
     char **argv;
{
  Image src1, src2;
  int group_ID;
  int frame1_ID, frame2_ID, frame3_ID, color_ID;
  int weight1_scale_ID, weight2_scale_ID,
  	red_ID, green_ID, blue_ID;
  int image_menu1_ID;
  int image_menu2_ID;
  int src1_ID, src2_ID;
int 	radio_frame_ID, radio_group_ID, radio_add_ID, radio_sub_ID,
  	radio_mul_ID, radio_div_ID, radio_not_ID, radio_diff_ID;
static int	one=1;  
  /* Save the program name so we can use it later in reporting errors
   */
  prog_name = argv[0];

  /* Call 'gimp_init' to initialize this filter.
   * 'gimp_init' makes sure that the filter was properly called and
   *  it opens pipes for reading and writing.
   */
  if (gimp_init (argc, argv))
    {
      src1 = src2 = 0;
      src1_ID = src2_ID = 0;

	data = gimp_get_params();
	if( !data ) {
		if( data = calloc( sizeof( Data ), 1 ) ) {
			data->oper = bl_add;	/* Add */
			data->weight1 = 50;
			data->weight2 = 50;
			data->red = data->green = data->blue = 0;
		} 
	}
      dialog_ID = gimp_new_dialog ("Blending");
      group_ID = gimp_new_row_group (dialog_ID, DEFAULT, NORMAL, "");
      image_menu1_ID = gimp_new_image_menu (dialog_ID, group_ID, 
					    IMAGE_CONSTRAIN_RGB ,
					    "First Image");
      image_menu2_ID = gimp_new_image_menu (dialog_ID, group_ID, 
					    IMAGE_CONSTRAIN_RGB,
					    "Second Image");
	radio_frame_ID = gimp_new_frame( dialog_ID, group_ID, "Operation" );
	radio_group_ID = gimp_new_row_group( dialog_ID, radio_frame_ID, RADIO, "" );
	radio_add_ID = gimp_new_radio_button( dialog_ID, radio_group_ID, "Add" );
	radio_sub_ID = gimp_new_radio_button( dialog_ID, radio_group_ID, "Sub" );
	radio_mul_ID = gimp_new_radio_button( dialog_ID, radio_group_ID, "Mul" );
	radio_div_ID = gimp_new_radio_button( dialog_ID, radio_group_ID, "Div" );
	radio_not_ID = gimp_new_radio_button( dialog_ID, radio_group_ID, "Not" );
	radio_diff_ID = gimp_new_radio_button( dialog_ID, radio_group_ID, "Diff" );
	
	gimp_change_item( dialog_ID, radio_add_ID, sizeof(one), &one );
	
      frame1_ID = gimp_new_frame (dialog_ID, group_ID, "Weight 1"); 
      weight1_scale_ID = gimp_new_scale (dialog_ID, frame1_ID, 0, 100, data->weight1, 0);
      frame2_ID = gimp_new_frame( dialog_ID, group_ID, "Weight 2");
      weight2_scale_ID = gimp_new_scale( dialog_ID, frame2_ID, 0, 100, data->weight2, 0);
      
      frame3_ID = gimp_new_frame( dialog_ID, group_ID, "Background color");
      color_ID = gimp_new_row_group( dialog_ID, frame3_ID, NORMAL, "" );
      red_ID = gimp_new_scale( dialog_ID, color_ID, 0, 255, data->red, 0);
      gimp_new_label(dialog_ID, red_ID, "Red");
      green_ID = gimp_new_scale( dialog_ID, color_ID, 0, 255, data->green, 0);
      gimp_new_label(dialog_ID, green_ID, "Green");
      blue_ID = gimp_new_scale( dialog_ID, color_ID, 0, 255, data->blue, 0 );
      gimp_new_label(dialog_ID, blue_ID, "Blue");
        
      gimp_add_callback( dialog_ID, radio_add_ID, radio_callback, (void *)bl_add );
      gimp_add_callback( dialog_ID, radio_sub_ID, radio_callback, (void *)bl_sub );
      gimp_add_callback( dialog_ID, radio_mul_ID, radio_callback, (void *)bl_mul );
      gimp_add_callback( dialog_ID, radio_div_ID, radio_callback, (void *)bl_div );
      gimp_add_callback( dialog_ID, radio_not_ID, radio_callback, (void *)bl_not );
	gimp_add_callback( dialog_ID, radio_diff_ID, radio_callback, (void *)bl_diff );
      gimp_add_callback (dialog_ID, image_menu1_ID, image_menu_callback, &src1_ID);
      gimp_add_callback (dialog_ID, image_menu2_ID, image_menu_callback, &src2_ID);
      gimp_add_callback (dialog_ID, weight1_scale_ID, scale_callback, &data->weight1);
      gimp_add_callback( dialog_ID, weight2_scale_ID, scale_callback, &data->weight2);
      gimp_add_callback( dialog_ID, red_ID, scale_callback, &data->red );
      gimp_add_callback( dialog_ID, green_ID, scale_callback, &data->green );
      gimp_add_callback( dialog_ID, blue_ID, scale_callback, &data->blue );
      gimp_add_callback (dialog_ID, gimp_ok_item_id (dialog_ID), ok_callback, 0);
      gimp_add_callback (dialog_ID, gimp_cancel_item_id (dialog_ID), cancel_callback, 0);

      if (gimp_show_dialog (dialog_ID))
	{
	  src1 = gimp_get_input_image (src1_ID);
	  src2 = (src2_ID != src1_ID) ? gimp_get_input_image (src2_ID) : src1;
	  
	  if (src1 && src2)
	    {
	      gimp_init_progress ("Blending");
	      blending (src1, src2, data->oper, data->weight1, data->weight2, 
	      		data->red, data->green, data->blue);
	    }
	}

    gimp_set_params(sizeof(Data), data);
      /* Free the images.
       */
      if (src1)
	gimp_free_image (src1);
      if (src2)
	gimp_free_image (src2);

      /* Quit
       */
      gimp_quit ();
    }

  return 0;
}


static void
radio_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{

 data->oper=(int)client_data; 

}

static void
image_menu_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  *((long*) client_data) = *((long*) call_data);
}

static void
scale_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  *((long*) client_data) = *((long*) call_data);
}

static void
ok_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  gimp_close_dialog (dialog_ID, 1);
}

static void
cancel_callback (item_ID, client_data, call_data)
     int item_ID;
     void *client_data;
     void *call_data;
{
  gimp_close_dialog (dialog_ID, 0);
}

static void
blending(src1, src2, oper, weight1, weight2, br, bg, bb)
     Image src1, src2;
opers	oper;	/* 1 = Add
		 * 2 = Sub
		 * 3 = Mul     
		 * 4 = Div
		 * 5 = Not
		 */
int	weight1, weight2, 
	br, 	/* background red 	*/
	bg,	/* background green 	*/
	bb;	/* background blue	*/
{
  Image dest;
  ImageType dest_type;
  unsigned char *src1p;
  unsigned char *src2p;
  unsigned char *destp;
  long width, height;
  long src1_channels;
  long src2_channels;
  long dest_channels;
  int src1r, src1g, src1b;
  int src2r, src2g, src2b;
  int destr, destg, destb;

  int i, j, same;

double tempr, tempg, tempb;

  src1p = gimp_image_data (src1);
  src2p = gimp_image_data (src2);

  width = gimp_image_width (src1);
  height = gimp_image_height (src1);
  
  src1_channels = gimp_image_channels (src1);
  src2_channels = gimp_image_channels (src2);

dest_channels = 3;

dest_type = RGB_IMAGE;

  dest = gimp_new_image (0, width, height, dest_type);
  destp = gimp_image_data (dest);


#ifdef	_DEBUG
printf("%s.%ld: w1=%ld, w2=%ld, r=%ld, g=%ld, b=%ld\n",__FILE__,__LINE__,weight1,weight2,br,bg,bb);
#endif

same = 0;
for (i = 0; i < height; i++) {
	for (j = 0; j < width; j++) {
		src1r = *src1p++;
		src1g = *src1p++;
		src1b = *src1p++;
              	src2r = *src2p++;
       		src2g = *src2p++;
       		src2b = *src2p++;			
       		if( weight1 == 100 && weight2 == 100 ) {
       			same = 0;
			if( (src1r == br)
			&&  (src1g == bg)
			&&  (src1b == bb) ) {
				destr = src2r;
				destg = src2g;
				destb = src2b;
				same = 1;
			}
		}
		if( !same ) {
			switch( oper ) {
			case bl_add :	/* add */
				destr = ((double)(src1r*weight1) + 
					 (double)(src2r*weight2)) / 
					 (double)(weight1 + weight2);	
				destg = ((double)(src1g*weight1) + 
					 (double)(src2g*weight2)) / 
					 (double)(weight1 + weight2);	
				destb = ((double)(src1b*weight1) + 
					 (double)(src2b*weight2)) / 
					 (double)(weight1 + weight2);	
				break;
			case bl_sub :	/* sub */
				destr = ((double)(src1r*weight1) - 
					 (double)(src2r*weight2)) / 
					 (double)(weight1 + weight2);	
				destg = ((double)(src1g*weight1) - 
					 (double)(src2g*weight2)) / 
					 (double)(weight1 + weight2);	
				destb = ((double)(src1b*weight1) - 
					 (double)(src2b*weight2)) / 
					 (double)(weight1 + weight2);	

				if( destr < 0 ) destr = 0;
				if( destg < 0 ) destg = 0;
				if( destb < 0 ) destb = 0;
				if( destr > 255 ) destr = 255;
				if( destg > 255 ) destg = 255;
				if( destb > 255 ) destb = 255;
				
				break;
			case bl_mul :	/* mul */
#ifdef	OBSOLETE			
				destr = ((double)(src1r*weight1) * 
					 (double)(src2r*weight2)) / 
					 (double)(weight1 + weight2);	
				destg = ((double)(src1g*weight1) * 
					 (double)(src2g*weight2)) / 
					 (double)(weight1 + weight2);	
				destb = ((double)(src1b*weight1) * 
					 (double)(src2b*weight2)) / 
					 (double)(weight1 + weight2);	
#else
				/* Weighting factor for multiplication
				 * doesnot make sense because result
				 * will be alwas > 255 
				 */
				destr = src1r * src2r / 255;
				if( destr < 0 ) destr = 0;
				else if( destr > 255 ) destr = 255;
				
				destg = src1g * src2g / 255;
				if( destg < 0 ) destg = 0;
				else if( destg > 255 ) destg = 255;
				
				destb = src1b * src2b / 255;				
				if( destb < 0 ) destb = 0;
				else if( destb > 255 ) destb = 255;
						
				
#endif			
				break;
			case bl_div :	/* div */

				tempr = (double)src2r /* * (double)weight2 */;
				tempg = (double)src2g /* * (double)weight2 */;
				tempb = (double)src2b /* * (double)weight2 */;
				if( (tempr > 0.0) && (tempg > 0.0) && (tempb > 0.0 ) ) {
#ifdef	OBSOLETE
					destr = ((double)(src1r*weight1) /
						 tempr ) / 
						 (double)(weight1 + weight2);	
					destg = ((double)(src1g*weight1) / 
						 tempg ) / 
						 (double)(weight1 + weight2);	
					destb = ((double)(src1b*weight1) / 
						 tempb) / 
						 (double)(weight1 + weight2);	
#else
	
				destr = src1r * weight1 * 255 / (src2r * weight2);
				destg = src1g * weight1 * 255 / (src2g * weight2);
				destb = src1b * weight1 * 255 / (src2b * weight2);
#endif					
				if( destr < 0 ) destr = 0;
				if( destg < 0 ) destg = 0;
				if( destb < 0 ) destb = 0;
				if( destr > 255 ) destr = 255;
				if( destg > 255 ) destg = 255;
				if( destb > 255 ) destb = 255;
						 
				} else {
					destr = src1r;
					destg = src1g;
					destb = src1b;
				
				}
				break;
			case bl_not :	/* not */
				if( (src1r == br)
				&&  (src1g == bg)
				&&  (src1b == bb) ) {
					destr = src2r;
					destg = src2g;
					destb = src2b;
				} else {
					destr = src1r;
					destg = src1g;
					destb = src1b;
				}		
				break;
			case bl_diff :
				destr = (src1r > src2r) ? (src1r - src2r) : (src2r - src1r);
	  			destg = (src1g > src2g) ? (src1g - src2g) : (src2g - src1g);
	  			destb = (src1b > src2b) ? (src1b - src2b) : (src2b - src1b);

	  			if (destr < 0) destr = 0;
	  			if (destg < 0) destg = 0;
	  			if (destb < 0) destb = 0;
	  			if (destr > 255) destr = 255;
	  			if (destg > 255) destg = 255;
	 	 		if (destb > 255) destb = 255;
				break;
			default :
				break;
			}
		}
		*destp++ = destr;
       		*destp++ = destg;
       		*destp++ = destb;
       	}
	if ((i % 5) == 0) gimp_do_progress (i, height);		
}
	
  gimp_display_image (dest);
  gimp_update_image (dest);
  gimp_free_image (dest);
}

