
/******************************************************************************
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
******************************************************************************/

/*****************************************************************************
**
**  File:    logging.c
**  Date:    February 17, 2002
**
**  Author:  Brian Wotring
**  Purpose: log handling routines
**
******************************************************************************/

#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"

#include "md_hosts.h"
#include "md_compare.h"

#ifdef WIN32
#include "windows_service.h"
#else
#include <syslog.h>
#endif

#include "logging.h"

extern osi_bool run_as_daemon;

#ifdef WIN32
extern HANDLE event_source;
#else
extern int syslog_facility;
#endif


void log_error( int id, const char *host, const char *message, ... )
{
    va_list ap;
    char *host_name = "*";

#ifdef WIN32
    LPCSTR  lpszStrings[1];
#endif

    char buffer[MAX_LOG_MESSAGE_LENGTH] = "";
    char header[MAX_LOG_MESSAGE_LENGTH] = "";

    /* setup defaults. */

    if( host != NULL )
    {
        host_name = (char *)host;
    }

    /* write the header. */

    osi_snprintf( header, sizeof(header), "[%d][%s][err] ", id, host_name );

    /* write buffer if we have one. */

    if( message != NULL )
    {
        va_start( ap, message );
        osi_vsnprintf( buffer, sizeof(buffer), message, ap );
        va_end( ap );
    }

    /* now piece the entire thing together and write to the system log. */

    osi_strlcat( header, buffer, sizeof( header ) );

#ifdef WIN32
        lpszStrings[0] = header;

        if( event_source != NULL)
        {
            ReportEvent( event_source,     /* handle of event source */
                     EVENTLOG_ERROR_TYPE,  /* event type */
                     0,                    /* event category */
                     1001,                 /* event ID */
                     NULL,                 /* current user's SID */
                     1,                    /* strings in lpszStrings */
                     0,                    /* no bytes of raw data */
                     lpszStrings,          /* array of error strings */
                     NULL);                /* no raw data */
        }

        if( run_as_daemon )
        {
            fprintf( stdout, header );
            fprintf( stdout, "\n" );
        }
#else
    syslog( ( syslog_facility | LOG_ERR ), header );
#endif
}

void log_info( int id, const char *host, const char *message, ... )
{
    va_list ap;
    char *host_name = "*";

#ifdef WIN32
    LPCSTR  lpszStrings[2];
#endif

    char buffer[MAX_LOG_MESSAGE_LENGTH] = "";
    char header[MAX_LOG_MESSAGE_LENGTH] = "";

    /* setup defaults. */

    if( host != NULL )
    {
        host_name = (char *)host;
    }

    /* write the header. */

    osi_snprintf( header, sizeof(header), "[%d][%s][info] ", id,host_name );

    /* write buffer if we have one. */

    if( message != NULL )
    {
        va_start( ap, message );
        osi_vsnprintf( buffer, sizeof(buffer), message, ap );
        va_end( ap );
    }

    /* now piece the entire thing together and write to the system log. */

    osi_strlcat( header, buffer, sizeof( header ) );

#ifdef WIN32
        lpszStrings[0] = header;

        if( event_source != NULL)
        {
            ReportEvent( event_source,     /* handle of event source */
                     EVENTLOG_INFORMATION_TYPE,  /* event type */
                     0,                    /* event category */
                     1001,                 /* event ID */
                     NULL,                 /* current user's SID */
                     1,                    /* strings in lpszStrings */
                     0,                    /* no bytes of raw data */
                     lpszStrings,          /* array of error strings */
                     NULL);                /* no raw data */
        }

        if( run_as_daemon )
        {
            fprintf( stdout, header );
            fprintf( stdout, "\n" );
        }
#else
    syslog( ( syslog_facility | LOG_INFO ), header );
#endif
}

void log_warning( int id, const char *host, const char *message, ... )
{
    va_list ap;
    char *host_name = "*";

#ifdef WIN32
    LPCSTR  lpszStrings[2];
#endif

    char buffer[MAX_LOG_MESSAGE_LENGTH] = "";
    char header[MAX_LOG_MESSAGE_LENGTH] = "";

    /* setup defaults. */

    if( host != NULL )
    {
        host_name = (char *)host;
    }

    /* write the header. */

    osi_snprintf( header, sizeof(header), "[%d][%s][warning] ", id, host_name );

    /* write buffer if we have one. */

    if( message != NULL )
    {
        va_start( ap, message );
        osi_vsnprintf( buffer, sizeof(buffer), message, ap );
        va_end( ap );
    }

    /* now piece the entire thing together and write to the system log. */

    osi_strlcat( header, buffer, sizeof( header ) );

#ifdef WIN32
        lpszStrings[0] = header;

        if( event_source != NULL)
        {
            ReportEvent( event_source,     /* handle of event source */
                     EVENTLOG_WARNING_TYPE,  /* event type */
                     0,                    /* event category */
                     1001,                 /* event ID */
                     NULL,                 /* current user's SID */
                     1,                    /* strings in lpszStrings */
                     0,                    /* no bytes of raw data */
                     lpszStrings,          /* array of error strings */
                     NULL);                /* no raw data */
        }

        if( run_as_daemon )
        {
            fprintf( stdout, header );
            fprintf( stdout, "\n" );
        }
#else
    syslog( ( syslog_facility | LOG_WARNING ), header );
#endif
}


void log_cmp_message( int id, OSI_DB_CTX *context, const char *message, ... )
{
    va_list ap;

    char buffer[MAX_LOG_MESSAGE_LENGTH] = "";
    
#ifdef WIN32
    LPTSTR  lpszStrings[2];
#endif
    
    if( message == NULL )
    {
        return;    
    }

    if( ( context == NULL ) || ( context->use_system_log ) )
    {
        char *next = buffer;
        int header_size = 0;

        osi_snprintf( buffer, sizeof(buffer), "[%d]", id );
       
        /* skip past the header. */ 
        header_size = strlen( buffer );
        next += header_size;

        va_start( ap, message );
        osi_vsnprintf( next, ( sizeof(buffer) - header_size ), message, ap );
        va_end( ap );

#ifdef WIN32
        lpszStrings[0] = buffer;

        if( event_source != NULL)
        {
            ReportEvent( event_source, 	   /* handle of event source */
                        EVENTLOG_INFORMATION_TYPE,  /* event type */
                    0,                    /* event category */
                    0,                    /* event ID */
                    NULL,                 /* current user's SID */
                    1,                    /* strings in lpszStrings */
                    0,                    /* no bytes of raw data */
                    lpszStrings,          /* array of error strings */
                    NULL);                /* no raw data */
        }
#else
        syslog( ( syslog_facility | LOG_INFO ), buffer );
#endif
    }

    /* now, see if we have to log to a file. */

    if( ( context == NULL ) || ( context->filepath[0] == '\0' ) )
    {
        return;
    }

    /* we are logging to a file. first, make sure it is open. */

    if( context->file_handle == NULL )
    {
        return;
    }

    /* we have a valid file handle, so we write our message to it. */

    fprintf( context->file_handle, buffer );
    
#ifdef WIN32
    fprintf( context->file_handle, "\r\n" );
#else
    fprintf( context->file_handle, "\n" );
#endif
}



