/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/

#include <fstream>
#include <iostream>
#include "key_consts.h"
#include "main.h"
#include "input.h"
#include "font.h"
#include "console.h"
#include "basic_editor.h"
#include "solver.h"

using namespace std;

extern console cons;

basic_editor::basic_editor (const std::string& settingsf, const std::string& helpf) : helptext (helpf) {
  snap_what = SNAP_NONE;
  load (settingsf);
  update_snaps = 1;
  edit_sustain = 0;
  lmb_clicked = 0;
  pan = zoom = 1;
}

basic_editor::~basic_editor () {}

void basic_editor::obj2win (const point<float>& v, float& wx, float& wy) {
  wx = win_per_obj.x * v.x;
  wy = win_per_obj.y * v.y;
}

void basic_editor::obj2win (const float& ox, const float& oy, float& wx, float& wy) {
  wx = win_per_obj.x * ox;
  wy = win_per_obj.y * oy;
}

void basic_editor::win2obj (const float& wx, const float& wy, float& ox, float& oy) {
  ox = obj_per_win.x * wx;
  oy = obj_per_win.y * wy;
}

void basic_editor::load (const std::string& fname) {

  extern std::string dotdin;
  ifstream file ((dotdin + fname).c_str (), ios::in);
  if (!file) return;
  load (file);

}

void basic_editor::load  (ifstream& file) {

  std::string ignore;
	
	file >> ignore >> id;
	
  file >> ignore >> name;

  float l, b, r,  t;
  file >> ignore >> l >> b >> r >> t;
  win.set (l, b, r, t);

  file >> ignore >> win_chunk.x >> win_chunk.y;
  file >> ignore >> obj_chunk.x >> obj_chunk.y;

  win_per_obj (win_chunk.x / obj_chunk.x, win_chunk.y / obj_chunk.y);
  obj_per_win (obj_chunk.x / win_chunk.x , obj_chunk.y / win_chunk.y);

  file >> ignore >> snap_what;
  update_snaps = 1;

  file >> ignore >> win_resolution;
  int dummy1 = 0; float dummy2 = 0;
  win2obj (win_resolution, dummy1, obj_resolution, dummy2);

}

void basic_editor::save (ofstream& file) {
  file << "id " << id << endl;
  file << "editor " << name << endl;
  file << "window " << win.left << ' ' << win.bottom << ' ' << win.right << ' ' << win.top << endl;
  file << "win_chunk " << win_chunk.x << ' ' << win_chunk.y << endl;
  file << "obj_chunk " << obj_chunk.x << ' ' << obj_chunk.y << endl;
  file << "snap " << snap_what << endl;
  file << "win_resolution " << win_resolution << endl;
}

void basic_editor::calc_win_mouse () {
  win.update_mouse ();
}

bool basic_editor::handle_input () {
	
	if (id == KB_KB_ATTACK) {
		
		extern int lmb;
		if (lmb) {
			if (lmb_clicked == 0) {
				if (edit_sustain) 
					edit_sustain = 0;
				else if (inbox (susbox, win.mousex, win.mousey)) 
					edit_sustain = 1;
				lmb_clicked = 1;
			}
		} else {
			if (edit_sustain) {
				float sx, sy; snap (sx, sy);
				float cx, cy; win2obj (sx, sy, cx, cy);
				_gotog.set (cx);
			}
			lmb_clicked = 0;
		}
		
	}
	
  // mouse capture

  if (mocap0.state == mocap::capturing) mocap0.add (win.mousex, win.mousey);

  // movement

  double pan_rept = window::PAN_REPEAT, zoom_rept = window::ZOOM_REPEAT;
  if  (keypressedd (k_a, pan_rept, pan_rept)) {win.panx (-pan); calc_visual_params ();}
  else if (keypressedd (k_d, pan_rept, pan_rept)) {win.panx (+pan); calc_visual_params ();}
  else if (keypressedd (k_w, pan_rept, pan_rept)) {win.pany (+pan); calc_visual_params ();}
  else if (keypressedd (k_s, pan_rept, pan_rept)) {win.pany (-pan); calc_visual_params ();}
  else if (keypressedd (k_q, zoom_rept, zoom_rept)) {win.zoom (zoom); calc_visual_params ();}
  else if (keypressedd (k_e, zoom_rept, zoom_rept)) {win.zoom (-zoom); calc_visual_params ();}

  // snap
  else if (keypressed (k_x)) snap_what = SNAP_X;
  else if (keypressed (k_y)) snap_what =  SNAP_Y;
  else if (keypressed (k_b)) snap_what = SNAP_BOTH;
  else if (keypressed (k_n)) snap_what = SNAP_NONE;

  else if (keypressedd (k_f5)) {
    if (keydown (k_lshift)) set_win_chunk (win_chunk.x, --win_chunk.y); else set_win_chunk (--win_chunk.x, win_chunk.y);
  } else if (keypressedd (k_f6)) {
    if (keydown (k_lshift)) set_win_chunk (win_chunk.x, ++win_chunk.y); else set_win_chunk (++win_chunk.x, win_chunk.y);
  }
  else if (keypressed (k_f7)) {
    if (mocap0.state != mocap::capturing) {
      mocap0.clear ();
      mocap0.state = mocap::capturing;
      cons << console::red << "capturing mouse" << eol;
    } else {
      mocap0.finish (this);
      cons << console::green << "mouse captured" << eol;
    }
  }

  return true;
}

void basic_editor::set_win_chunk (int x, int y) {

  if (x < 1) x = 1;
  if (y < 1) y = 1;

  win_chunk.x = x;
  win_chunk.y = y;

  float wx = win_chunk.x / obj_chunk.x, wy = win_chunk.y / obj_chunk.y;
  win_per_obj (wx, wy);
  obj_per_win (1./wx, 1./wy);

}

int basic_editor::is_snapx () {
  return ((snap_what == SNAP_X) || (snap_what == SNAP_BOTH));
}

int basic_editor::is_snapy () {
  return ((snap_what == SNAP_Y) || (snap_what == SNAP_BOTH));
}

void basic_editor::snap (float& x, float& y) {
  x = win.mousex;
  y = win.mousey;
  if (is_snapx()) {
    float s = win.mousex / win_chunk.x;
    if (s < 0) s -= 0.5; else s += 0.5;
    x = (int) s * win_chunk.x;
  }
  if (is_snapy()) {
    float s = win.mousey / win_chunk.y;
    if (s < 0) s -= 0.5; else s += 0.5;
    y = (int)s * win_chunk.y;
  }
}

void basic_editor::project () {
  glMatrixMode (GL_MODELVIEW);
    glPushMatrix ();
      glLoadIdentity ();
      glMatrixMode (GL_PROJECTION);
      glPushMatrix ();
        glLoadIdentity ();
        glOrtho (win.left, win.right, win.bottom, win.top, -1, 1);
}

void basic_editor::unproject () {
    glMatrixMode (GL_PROJECTION);
    glPopMatrix ();
    glMatrixMode (GL_MODELVIEW);
    glPopMatrix ();
}

void basic_editor::draw  () {
  // must always be wrapped by project, unproject
  //
  draw_snaps ();
}

void basic_editor::draw_cursor () {

  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
    glLoadIdentity ();
    extern viewport view;
    
    glOrtho (0, view.xmax, 0, view.ymax, 0, 1);
    
    float wx, wy; snap (wx, wy);
    float cx, cy; win2obj (wx, wy, cx, cy);
    int vx, vy; win2view (wx, wy, vx, vy, win, view);
    
    glColor3f (1, 1, 1);
    sprintf (buf, " %.3f, %.3f", cx, cy);
    draw_string (buf, vx, vy);
    
    if (edit_sustain) {
			vy -= get_line_height ();
			extern gotog _gotog;
			sprintf (buf, " sustain @ %.3f", _gotog.g);
			draw_string (buf, vx, vy);
		}

  glPopMatrix ();

  glColor3f (0.4, 0.4, 0.4);
  glBegin (GL_LINE_STRIP);
    glVertex2f (win.left, wy);
    glVertex2f (wx, wy);
    glVertex2f (wx, win.bottom);
  glEnd ();

}

void basic_editor::draw_snaps () {

  //
  // draw snap lines
  //

  static const float sr = 0.1, sg = sr, sb = sg; // snap color

  glColor3f (sr, sg, sb);
  
	int l, r, b, t;
  
		if (update_snaps) {
			
			if (is_snapx()) {

				l = (int)(win.left / win_chunk.x);
				startx = l * win_chunk.x;
				r = (int)(win.right / win_chunk.x);
				endx = r * win_chunk.x;
				
				xlines.clear ();
				nxpts = 0;
				while (startx <= endx) {
					xlines.push_back (startx);
					xlines.push_back (win.bottom);
					xlines.push_back (startx);
					xlines.push_back (win.top);
					startx += win_chunk.x;
					nxpts += 2;
				}
				
				update_snaps = 0;
				
			}
			
			if (is_snapy ()) {

				b = (int)(win.bottom / win_chunk.y);
				starty = b * win_chunk.y;
				t = (int)(win.top / win_chunk.y);
				endy = t * win_chunk.y;
			
				ylines.clear ();
				nypts = 0;
				while (starty <= endy) {
					ylines.push_back (win.left);
					ylines.push_back (starty);
					ylines.push_back (win.right);
					ylines.push_back (starty);
					starty += win_chunk.y;
					nypts += 2;
				}

			}  
	 
		}
		
		glEnableClientState (GL_VERTEX_ARRAY);
		
		if (is_snapx()) { // lines along y
			glVertexPointer (2, GL_INT, 0, xlines.data ());
			glDrawArrays (GL_LINES, 0, nxpts);
		}

		if (is_snapy()) { // lines along x
			glVertexPointer (2, GL_INT, 0, ylines.data ());
			glDrawArrays (GL_LINES, 0, nypts);
		}
		
		glDisableClientState (GL_VERTEX_ARRAY);	
	
}

void basic_editor::calc_visual_params () {
	if (id == KB_KB_ATTACK) susbox (susx - win.handle_radius, win.top - win.sus_handle_radius, susx + win.handle_radius, win.top); 
  tb.refresh (this);
  update_snaps = 1;
}

void basic_editor::update_sustain (float f) {
	float ox = f, oy = 0; 
	obj2win (ox, oy, susx, susy);
	calc_visual_params ();
}
