#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <pthread.h>
#include <signal.h>

#include <simclist.h>

#include "attack_parser.h"

#include "sshguard_log.h"
#include "sshguard_addresskind.h"
#include "sshguard_whitelist.h"
#include "sshguard_fw.h"
#include "sshguard.h"

#define MAX_LOGLINE_LEN     1000

list_t limbo;
list_t hell;

typedef struct {
    char addr[ADDRLEN];
    int addrkind;
    int service;
    time_t whenfirst;
    time_t whenlast;
    unsigned int numhits;
} ipentry_t;

struct {
    time_t pardon_threshold, stale_threshold;
    unsigned int abuse_threshold;
} opts;

static inline void ipentryinit(ipentry_t *ipe, char *ipaddr, int addrkind, int service);
static void usage(void);
void sigfin_handler(int signo);
void report_address(char *addr, int addrkind, int service);
void purge_limbo_stale(void);
void *pardonBlocked(void *par);
void sighand(int sig);
void finishup(void);



int main(int argc, char *argv[]) {
    pthread_t tid;
    char optch;
    int retv;
    char buf[MAX_LOGLINE_LEN];
    

    /* initializations */

    /* pending and blocked address lists */
    list_init(&limbo);
    list_init(&hell);

    /* logging system */
    sshguard_log_init();

    /* address blocking system */
    if (fw_init() != FWALL_OK) {
        fprintf(stderr, "Could not init firewall.\n");
        exit(1);
    }

    /* whitelisting system */
    if (whitelist_init() != 0 || whitelist_conf_init() != 0) {
        sshguard_log(LOG_ERR, "Could not initialize the whitelist engine.");
        fprintf(stderr, "Could not nitialize the whitelist engine.\n");
        exit(1);
    }

    /* parsing the command line */
    opts.pardon_threshold = DEFAULT_PARDON_THRESHOLD;
    opts.stale_threshold = DEFAULT_STALE_THRESHOLD;
    opts.abuse_threshold = DEFAULT_ABUSE_THRESHOLD;
    while ((optch = getopt(argc, argv, "p:s:a:w:h")) != -1) {
        switch (optch) {
            case 'p':   /* pardon threshold interval */
                opts.pardon_threshold = strtol(optarg, (char **)NULL, 10);
                break;
            case 's':   /* stale threshold interval */
                opts.stale_threshold = strtol(optarg, (char **)NULL, 10);
                break;
            case 'a':   /* abuse threshold count */
                opts.abuse_threshold = strtol(optarg, (char **)NULL, 10);
                break;
            case 'w':   /* whitelist entries */
                if (optarg[0] == '/' || optarg[0] == '.') {
                    /* add from file */
                    if (whitelist_file(optarg) != 0) {
                        fprintf(stderr, "Could not handle whitelisting for %s.\n", optarg);
                        sshguard_log(LOG_ERR, "Could not handle whitelisting for %s.", optarg);
                        exit(1);
                    }
                } else {
                    /* add raw content */
                    if (whitelist_add(optarg) != 0) {
                        fprintf(stderr, "Could not handle whitelisting for %s.\n", optarg);
                        sshguard_log(LOG_ERR, "Could not handle whitelisting for %s.", optarg);
                        exit(1);
                    }
                }
                break;
            case 'h':
            default:
                usage();
                exit(1);
        }
    }

    whitelist_conf_fin();

    /* set finalization routine */
    signal(SIGINT, sigfin_handler);
    signal(SIGTERM, sigfin_handler);
    signal(SIGHUP, sigfin_handler);
    atexit(finishup);

    /* start thread for purging stale blocked addresses */
    if (pthread_create(&tid, NULL, pardonBlocked, NULL) != 0) {
        perror("pthread_create()");
        exit(2);
    }


    /* initialization successful */
    
    sshguard_log(LOG_INFO, "Started successfully [(a,p,s)=(%u, %u, %u)], now ready to scan.", \
            opts.abuse_threshold, (unsigned int)opts.pardon_threshold, (unsigned int)opts.stale_threshold);


    while (fgets(buf, MAX_LOGLINE_LEN, stdin) != NULL) {
        retv = parse_line(buf);
        /* retv = yyparse(); */
        if (retv != 0) {
            /* sshguard_log(LOG_DEBUG, "Parsing line '%s': skip.", buf); */
            continue;
        }

        /* extract the IP address */
        sshguard_log(LOG_DEBUG, "Matched IP address %s", attackparser_targetaddr);
        
        /* report IP */
        report_address(attackparser_targetaddr, attackparser_targetaddrkind, attackparser_service);
    }

    if (fw_fin() != FWALL_OK) sshguard_log(LOG_ERR, "Cound not finalize firewall.");

    sshguard_log_fin();

    return 0;
}


void report_address(char *addr, int addrkind, int service) {
    ipentry_t *tmpent = NULL;
    unsigned int pos;
    int ret;


    /* clean list from stale entries */
    purge_limbo_stale();

    /* protected address? */
    if (whitelist_match(addr, addrkind)) {
        sshguard_log(LOG_INFO, "Pass over address %s because it's been whitelisted.", addr);
        return;
    }
    
    /* search entry in list */
    pos = 0;
    list_iterator_start(&limbo);
    while (list_iterator_hasnext(&limbo)) {
        tmpent = list_iterator_next(&limbo);
        if (strcmp(tmpent->addr, addr) == 0) /* found! */
            break;
        pos++;
    }
    list_iterator_stop(&limbo);
    
    if (pos < list_size(&limbo)) { /* then found */
        /* update last hit */
        tmpent->whenlast = time(NULL);
        tmpent->numhits++;
        if (tmpent->numhits >= opts.abuse_threshold) { /* block IP */
            sshguard_log(LOG_NOTICE, "Blocking %s: %u failures over %u seconds.\n", tmpent->addr, tmpent->numhits, tmpent->whenlast - tmpent->whenfirst);
            ret = fw_block(addr, addrkind, service);
            if (ret != FWALL_OK) sshguard_log(LOG_ERR, "Blocking command failed. Exited: %d", ret);
            /* re-get elem for deleting it from limbo list */
            tmpent = list_extract_at(& limbo, pos);
            list_append(&hell, tmpent);
        }
        return;
    }
    
    /* otherwise: insert the new item */
    tmpent = malloc(sizeof(ipentry_t));
    ipentryinit(tmpent, addr, addrkind, service);
    list_append(&limbo, tmpent);
}

static inline void ipentryinit(ipentry_t *ipe, char *ipaddr, int addrkind, int service) {
    strcpy(ipe->addr, ipaddr);
    ipe->addrkind = addrkind;
    ipe->service = service;
    ipe->whenfirst = ipe->whenlast = time(NULL);
    ipe->numhits = 1;
}

void purge_limbo_stale(void) {
    ipentry_t *tmpent;
    time_t now;
    unsigned int pos = 0;


    now = time(NULL);
    for (pos = 0; pos < list_size(&limbo); pos++) {
        tmpent = list_get_at(&limbo, pos);
        if (now - tmpent->whenfirst > opts.stale_threshold)
            list_delete_at(&limbo, pos);
    }
}

void *pardonBlocked(void *par) {
    time_t now;
    ipentry_t *tmpel;
    int ret;


    srandom(time(NULL));

    while (1) {
        /* wait some time, at most opts.pardon_threshold/3 + 1 sec */
        sleep(1 + random() % (opts.pardon_threshold/2));
        now = time(NULL);
        tmpel = list_get_at(&hell, 0);
        while (tmpel != NULL && now - tmpel->whenlast > opts.pardon_threshold) {
            sshguard_log(LOG_INFO, "Releasing %s after %u seconds.\n", tmpel->addr, now - tmpel->whenlast);
            /* fprintf(stderr, "Releasing %s after %u seconds.\n", tmpel->ip, now - tmpel->whenlast); */
            ret = fw_release(tmpel->addr, tmpel->addrkind, tmpel->service);
            if (ret != FWALL_OK) sshguard_log(LOG_ERR, "Release command failed. Exited: %d", ret);
            list_delete_at(&hell, 0);
            free(tmpel);
            tmpel = list_get_at(&hell, 0);
        }
    }

    pthread_exit(NULL);
    return NULL;
}

/* finalization routine */
void finishup(void) {
    /* flush blocking rules */
    sshguard_log(LOG_INFO, "Got exit signal, flushing blocked addresses and exiting...");
    fw_flush();
    if (fw_fin() != FWALL_OK) sshguard_log(LOG_ERR, "Cound not finalize firewall.");
    if (whitelist_fin() != 0) sshguard_log(LOG_ERR, "Could not finalize the whitelisting system.");
    sshguard_log_fin();
}

void sigfin_handler(int signo) {
    /* let exit() call finishup() */
    exit(0);
}

static void usage(void) {
    fprintf(stderr, "Usage:\nsshguard [-a num] [-p sec] [-w <addr>|<host>|<block>|<file>]{0,n} [-s sec] \n");
    fprintf(stderr, "\t-a\tNumber of hits after which blocking an address (%d)\n", DEFAULT_ABUSE_THRESHOLD);
    fprintf(stderr, "\t-p\tSeconds after which unblocking a blocked address (%d)\n", DEFAULT_PARDON_THRESHOLD);
    fprintf(stderr, "\t-w\tWhitelisting of addr/host/block, or take from file if starts with \"/\" or \".\" (repeatable)\n");
    fprintf(stderr, "\t-s\tSeconds after which forgetting about a cracker candidate (%d)\n", DEFAULT_STALE_THRESHOLD);
}


