/*
 * Copyright (c) 1987, 1988, 1989, 1990, 1991 Stanford University
 * Copyright (c) 1991 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Stanford and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Stanford and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */

/*
 * Raster - rasterized image
 */

/*
 * $Id: raster.h,v 1.5 1996/07/01 13:35:11 bmarsch Exp $
 *
 * $Log: raster.h,v $
 * Revision 1.5  1996/07/01 13:35:11  bmarsch
 * Bugfix in scaling
 *
 * Revision 1.4  1996/07/01 08:17:50  bmarsch
 * Implemented scaling of images
 *
 * Revision 1.3  1996/05/07 16:14:48  bmarsch
 * Replaced RGBAarray's with RGBA_BT_array's
 *
 * Revision 1.2  1996/03/01 14:12:41  bmarsch
 * Extented alpha channel to [0;255]
 *
 */

/* bmarsch: 10 Nov 94: poke and peek are now fast and the raster is
                       dithered on pseudocolor visuals              */

#ifndef iv_raster_h
#define iv_raster_h

#include <InterViews/color.h>
#include <InterViews/coord.h>
#include <InterViews/resource.h>

class Canvas;
class Color;
class RasterRep;


class Raster : public Resource {
public:
  Raster(unsigned long width, unsigned long height, boolean dither = true,
         long scaledWidth = -1, long scaledHeight = -1);
  Raster(const Raster&);
  virtual ~Raster();

  // size of original image in Coords
  virtual Coord width() const;
  virtual Coord height() const;

  // size of original image in pixel
  virtual unsigned long pwidth() const;
  virtual unsigned long pheight() const;

  // size of scaled image in pixel
  unsigned long unscaledWidth() const;
  unsigned long unscaledHeight() const;

  void setSize(long scaledWidth, long scaledHeight);

  virtual Coord left_bearing() const;
  virtual Coord right_bearing() const;
  virtual Coord ascent() const;
  virtual Coord descent() const;

  virtual void peek(
    unsigned long x, unsigned long y,
    ColorIntensity& red, ColorIntensity& green, ColorIntensity& blue,
    float& alpha) const;

  virtual void poke(
    unsigned long x, unsigned long y,
    ColorIntensity red, ColorIntensity green, ColorIntensity blue,
    float alpha);

  virtual void flush() const;

  const unsigned char* data() const;
  const unsigned char* alpha() const;
  RasterRep* rep() const;
  void modified();

  // bmarsch/mpichler, 19941123: peek and poke also with unsigned char
  virtual void peekChr (
    unsigned long x, unsigned long y,
    unsigned char& red, unsigned char& green, unsigned char& blue,
    unsigned char& alpha) const;

  virtual void pokeChr (
    unsigned long x, unsigned long y,
    unsigned char red, unsigned char green, unsigned char blue,
    unsigned char alpha = 255);

  // bmarsch: 20 Dec 94: support transparency
  void useAlphaTransparency();

  // bmarsch: 09 Jan 95: return data array
  const unsigned char* getRGBarray();
  void freeRGBarray();
  const unsigned char* getRGBA_BT_array();
  void freeRGBA_BT_array();

private:
  boolean modified_;

  // this is the original (untransformed image size)
  Coord width_;
  Coord height_;
  int pwidth_;
  int pheight_;

  // scaled size
  Coord scwidth_;
  Coord scheight_;
  int swidth_;
  int sheight_;
  Coord left_;
  Coord right_;
  Coord bottom_;
  Coord top_;

  boolean doDither_;
  unsigned char* data_;       // data array (rgb)
  unsigned char* alpha_;      // alpha channel
  unsigned char* data_rgba_;  // data array (rgba): mixed data_ and alpha_
  RasterRep* rep_;            // all displaying functions and resources

  friend class RasterRep;

  /* anachronisms */
public:
  unsigned int Width() const;
  unsigned int Height() const;
};

inline const unsigned char* Raster::data() const {
  return data_;
}

inline const unsigned char* Raster::alpha() const {
  return alpha_;
}

inline RasterRep* Raster::rep() const {
  return rep_;
}

inline void Raster::modified() {
  modified_ = true;
}

inline unsigned int Raster::Width() const { return (unsigned int)pwidth(); }
inline unsigned int Raster::Height() const { return (unsigned int)pheight(); }


#endif
