/* -*- mode: c; c-file-style: "gnu" -*-
 * stats.c -- server statistics
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file stats.c
 * Server statistics.
 *
 * This module contains statistic collector and reporter functions,
 * used by the thy_stats() in main.c.
 */

#include <time.h>

#include "stats.h"

/** Accepted requests. */
static long int _stats_requests = 0;
/** Transferred bytes. */
static size_t _stats_transfers = 0;
/** Time of server startup. */
static time_t _stats_server_start = 0;

/** Increase the number of accepted requests.
 * @param v is the number to increase with.
 */
void
thy_stats_requests_inc (int v)
{
  _stats_requests += v;
}

/** Increase the number of served bytes.
 * @param size is the number to increase with.
 */
void
thy_stats_transfer_inc (size_t size)
{
  _stats_transfers += size;
}

/** Return the number of transferred bytes.
 * @returns the number of transferred bytes.
 *
 * @note Since this is a size_t, it could easily wrap around on a
 * heavily loaded site.
 */
size_t
thy_stats_transfers (void)
{
  return _stats_transfers;
}

/** Return the number of accepted requests.
 * @returns teh number of accepted requests.
 *
 * @note This is the number of accepted connections, not requests.
 */
long int
thy_stats_requests (void)
{
  return _stats_requests;
}

/** Get the server uptime.
 * @returns the uptime of the server.
 */
int
thy_stats_uptime (void)
{
  return (int)(time (NULL) - _stats_server_start);
}

/** Set server start time.
 */
void
thy_stats_start (void)
{
  _stats_server_start = time (NULL);
}
