/*
 * The Sleuth Kit
 *
 * $Date: 2005/01/17 22:40:17 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003-2005 Brian Carrier.  All rights reserved
 *
 * mm_open - wrapper function for specific partition type
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "mm_tools.h"


/* 
 * THe main wrapper function for opening an image
 */
MM_INFO *
mm_open(IMG_INFO * img_info, const char *type, DADDR_T offset)
{
    /* Autodetect mode 
     * We need to try all of them in case there are multiple 
     * installations
     */
    if (type == NULL) {
	MM_INFO *mm_info, *mm_set = NULL;
	char *set = NULL;

	if ((mm_info = dos_open(img_info, offset, 1)) != NULL) {
	    set = "DOS";
	    mm_set = mm_info;
	}
	if ((mm_info = bsd_open(img_info, offset, 1)) != NULL) {
	    // if (set == NULL) {
	    // In this case, BSD takes priority because BSD partitions start off with
	    // the DOS magic value in the first sector with the boot code.
	    set = "BSD";
	    mm_set = mm_info;
	    /*
	       }
	       else {
	       mm_set->close(mm_set);
	       mm_info->close(mm_info);
	       error("Cannot determine if BSD or %s at %"PRIuDADDR"", set, offset);
	       }
	     */
	}
	if ((mm_info = gpt_open(img_info, offset, 1)) != NULL) {
	    if (set == NULL) {
		set = "GPT";
		mm_set = mm_info;
	    }
	    else {
		mm_set->close(mm_set);
		mm_info->close(mm_info);
		error("Cannot determine if GPT or %s at %" PRIuDADDR "",
		      set, offset);
	    }
	}
	if ((mm_info = sun_open(img_info, offset, 1)) != NULL) {
	    if (set == NULL) {
		set = "Sun";
		mm_set = mm_info;
	    }
	    else {
		mm_set->close(mm_set);
		mm_info->close(mm_info);
		error("Cannot determine if Sun or %s at %" PRIuDADDR "",
		      set, offset);
	    }
	}
	if ((mm_info = mac_open(img_info, offset, 1)) != NULL) {
	    if (set == NULL) {
		set = "Mac";
		mm_set = mm_info;
	    }
	    else {
		mm_set->close(mm_set);
		mm_info->close(mm_info);
		error("Cannot determine if Mac or %s at %" PRIuDADDR "",
		      set, offset);
	    }
	}

	return mm_set;
    }
    else {
	uint8_t mmtype;

	/* Transate the string into the number */
	mmtype = mm_parse_type(type);

	switch (mmtype) {
	case MM_DOS:
	    return dos_open(img_info, offset, 0);
	case MM_MAC:
	    return mac_open(img_info, offset, 0);
	case MM_BSD:
	    return bsd_open(img_info, offset, 0);
	case MM_SUN:
	    return sun_open(img_info, offset, 0);
	case MM_GPT:
	    return gpt_open(img_info, offset, 0);
	case MM_UNSUPP:
	default:
	    fprintf(stderr, "unknown partition type: %s\n", type);
	    fprintf(stderr, "Supported types:\n");
	    mm_print_types(stderr);
	    return NULL;
	}
    }
}
