//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General)mm Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------
#include "event.h"
#include "perfroll.h"

const int c_perfroll_background_x = (c_ppqn * 16) / c_perf_scale_x;


perfroll::perfroll( perform *a_perf,
		    Adjustment * a_hadjust,
		    Adjustment * a_vadjust  ) : DrawingArea() 
{    
  
    Glib::RefPtr<Gdk::Colormap> colormap = get_default_colormap();

    m_black = Gdk::Color( "black" );
    m_white = Gdk::Color( "white" );
    m_grey = Gdk::Color( "grey" );
    
    //m_text_font_6_12 = Gdk_Font( c_font_6_12 );
  
    colormap->alloc_color( m_black );
    colormap->alloc_color( m_white );
    colormap->alloc_color( m_grey );
    
    m_mainperf = a_perf;
    m_vadjust = a_vadjust;
    m_hadjust = a_hadjust;

    m_adding = false;
    m_adding_pressed = false;

    m_old_progress_ticks = 0;

    add_events( Gdk::BUTTON_PRESS_MASK | 
		Gdk::BUTTON_RELEASE_MASK |
		Gdk::POINTER_MOTION_MASK |
		Gdk::KEY_PRESS_MASK |
		Gdk::KEY_RELEASE_MASK |
		Gdk::FOCUS_CHANGE_MASK	);

    set_size_request( 10, 10 );

    set_snap( c_ppqn / 4 );

    m_4bar_offset = 0;
    m_sequence_offset = 0;
    m_roll_length_ticks = 0;

    m_roll_length_ticks = m_mainperf->get_max_trigger();
    m_roll_length_ticks = m_roll_length_ticks - 
	( m_roll_length_ticks % ( c_ppqn * 16 ));
    m_roll_length_ticks +=  c_ppqn * 128;

    set_double_buffered( false );

} 

perfroll::~perfroll( )
{

}


void
perfroll::change_horz( )
{
    if ( m_4bar_offset != (int) m_hadjust->get_value() ){
	
	m_4bar_offset = (int) m_hadjust->get_value();
	queue_draw();
    }
}

void
perfroll::change_vert( )
{
    if ( m_sequence_offset != (int) m_vadjust->get_value() ){
	
	m_sequence_offset = (int) m_vadjust->get_value();
	queue_draw();
    }
}

/* popup menu calls this */
void 
perfroll::set_adding( bool a_adding )
{
    if ( a_adding ){

	get_window()->set_cursor(  Gdk::Cursor( Gdk::PENCIL ));
	m_adding = true;
    
    } else {

	get_window()->set_cursor( Gdk::Cursor( Gdk::LEFT_PTR ));
	m_adding = false;
    }
}


void 
perfroll::on_realize()
{
    // we need to do the default realize
    Gtk::DrawingArea::on_realize();

    set_flags( Gtk::CAN_FOCUS );

    // Now we can allocate any additional resources we need
    m_window = get_window();
    m_gc = Gdk::GC::create( m_window );
    m_window->clear();

    m_background = Gdk::Pixmap::create( m_window,
                                        c_perfroll_background_x,
                                        c_names_y, -1 );

    /* and fill the background ( dotted lines n' such ) */
    fill_background_pixmap();
    update_sizes();

    m_hadjust->signal_value_changed().connect( slot( *this, &perfroll::change_horz ));
    m_vadjust->signal_value_changed().connect( slot( *this, &perfroll::change_vert ));

}

void 
perfroll::update_sizes()
{
    m_hadjust->set_lower( 0 );
    m_hadjust->set_upper( m_roll_length_ticks / (c_ppqn * 16) );

    int seqs_showing = (m_window_y / c_names_y) - 1;

    m_vadjust->set_lower( 0 );
    m_vadjust->set_upper( c_total_seqs - seqs_showing );

    if ( is_realized() ) 
	m_pixmap = Gdk::Pixmap::create( m_window,
                                        m_window_x,
                                        m_window_y, -1 );
}

void
perfroll::increment_size()
{
    m_roll_length_ticks += (c_ppqn * 512);
    update_sizes( );
}

/* updates background */
void 
perfroll::fill_background_pixmap()
{
    /* clear background */
    m_gc->set_foreground(m_white);
    m_background->draw_rectangle(m_gc,true,
 				0,
 				0, 
 				c_perfroll_background_x, 
 				c_names_y );

    /* draw horz grey lines */
    m_gc->set_foreground(m_grey);

    gint8 dash = 1;
    m_gc->set_dashes( 0, &dash, 1 );  	
    
    m_gc->set_line_attributes( 1,
                               Gdk::LINE_ON_OFF_DASH,
                               Gdk::CAP_NOT_LAST,
                               Gdk::JOIN_MITER );

    m_background->draw_line(m_gc,
			   0,
			   0,
			   c_perfroll_background_x,
			   0 );
    
    int distance = c_ppqn / c_perf_scale_x; 
    
    /* draw vert lines */
    for ( int i=0; i< 16 ; ++i ){

        m_gc->set_line_attributes( 1,
                                   Gdk::LINE_SOLID,
                                   Gdk::CAP_NOT_LAST,
                                   Gdk::JOIN_MITER );

 	if ( i == 0 ){

 	    m_gc->set_foreground(m_black);

	}
	else if ( i % 4 == 0 ){
	    
 	    m_gc->set_foreground(m_grey);

	}
 	else  {
	    
            m_gc->set_line_attributes( 1,
                               Gdk::LINE_ON_OFF_DASH,
                               Gdk::CAP_NOT_LAST,
                               Gdk::JOIN_MITER );
	    m_gc->set_foreground(m_grey);
	}
	    

 	/* solid line on every beat */
 	m_background->draw_line(m_gc,
 			       i * distance ,
 			       0,
 			       i * distance,
 			       c_names_y );
	

    }

    /* reset line style */

    m_gc->set_line_attributes( 1,
                               Gdk::LINE_SOLID,
                               Gdk::CAP_NOT_LAST,
                               Gdk::JOIN_MITER );
}


/* simply sets the snap member */
void 
perfroll::set_snap( int a_snap )
{
    m_snap = a_snap;
}

void 
perfroll::draw_progress()
{
//     m_old_progress_x = m_progress_x;

    long tick = m_mainperf->get_tick(); 
    long tick_offset = m_4bar_offset * c_ppqn * 16;
    
    int progress_x =     ( tick - tick_offset ) / c_perf_scale_x ; 
    int old_progress_x = ( m_old_progress_ticks - tick_offset ) / c_perf_scale_x ; 

    /* draw old */
    m_window->draw_drawable(m_gc, 
			 m_pixmap, 
			 old_progress_x, 0,
			 old_progress_x, 0,
			 1, m_window_y );

    m_gc->set_foreground(m_black);
    m_window->draw_line(m_gc,
		       progress_x, 0,
		       progress_x, m_window_y);
    
    m_old_progress_ticks = tick;

}



void perfroll::draw_sequence_on( Glib::RefPtr<Gdk::Drawable> a_draw, int a_sequence )
{
    
    long tick_on;
    long tick_off;

    long tick_offset = m_4bar_offset * c_ppqn * 16;

    if ( a_sequence < c_total_seqs ){

	if ( m_mainperf->is_active( a_sequence )){
	    
	    sequence *seq =  m_mainperf->get_sequence( a_sequence );
	    
	    seq->reset_draw_trigger_marker();
	    
	    a_sequence -= m_sequence_offset;

	    long sequence_length = seq->get_length();
	    int length_w = sequence_length / c_perf_scale_x;
	    
	    while ( seq->get_next_trigger( &tick_on, &tick_off )){
	
		tick_on  -= tick_offset;
		tick_off -= tick_offset;

		if ( tick_off > 0 ){

		    long x_on  = tick_on  / c_perf_scale_x;
		    long x_off = tick_off / c_perf_scale_x;
		    int  w     = x_off - x_on;
		    
		    int x = x_on;
		    int y = c_names_y * a_sequence + 2;
		    int h = c_names_y - 4;

		    
		    
		    m_gc->set_foreground(m_grey);
		    a_draw->draw_rectangle(m_gc,true,
					   x,
					   y,
					   w,
					   h );
		    
		    m_gc->set_foreground(m_black);
		    a_draw->draw_rectangle(m_gc,false,
					   x,
					   y,
					   w,
					   h );

		    m_gc->set_foreground(m_black);
		    long length_marker_first_tick = ((tick_on + tick_offset)  -
						     ((tick_on + tick_offset) % sequence_length) + sequence_length -
						     tick_offset);
		    int lm_x = length_marker_first_tick / c_perf_scale_x;
		    int lm_w = sequence_length / c_perf_scale_x;
		    
		    while ( lm_x < x_off ){

		      a_draw->draw_line(m_gc,
					lm_x,
					y + 3,
					lm_x,
					y + h - 3 );
		      
		      lm_x += lm_w;
		      
		    }   
		}
	    }
	}
    }
}

bool 
perfroll::on_expose_event(GdkEventExpose* e)
{

    int x_s = e->area.x / c_perfroll_background_x;
    int x_f = (e->area.x + e->area.width) / c_perfroll_background_x;

    int y_s = e->area.y / c_names_y;
    int y_f = (e->area.y  + e->area.height) / c_names_y;
  
    for ( int y=y_s; y<=y_f; y++ ){

	for ( int x=x_s; x<=x_f; x++ ){
	    
	    m_pixmap->draw_drawable(m_gc, m_background,
				 0,
				 0,
				 x * c_perfroll_background_x,
				 c_names_y * y,
				 c_perfroll_background_x,
				 c_names_y );
	}
	draw_sequence_on(m_pixmap, y + m_sequence_offset );
    }

    m_window->draw_drawable( m_gc, m_pixmap,
			  e->area.x,
			  e->area.y,
			  e->area.x,
			  e->area.y,
			  e->area.width,
			  e->area.height );
    return true;
}

void 
perfroll::draw_background_at(  Glib::RefPtr<Gdk::Drawable> a_draw, int a_x, long a_y )
{
    int t = a_x / c_perfroll_background_x;
    int s = a_y / c_names_y;

    a_draw->draw_drawable(m_gc, m_background,
			0,
			0,
			t * c_perfroll_background_x,
			c_names_y * s,
			c_perfroll_background_x,
			c_names_y );
}


bool
perfroll::on_button_press_event(GdkEventButton* a_ev)
{
    long tick;
    int sequence;

    grab_focus( ); 

    int x = (int) a_ev->x;
    int y = (int) a_ev->y;
    
    /*      left mouse button     */
    if ( a_ev->button == 1 ){ 
	
 	/* add a new note if we didnt select anything */
	if (  m_adding ){
	    
	    m_adding_pressed = true;

	    snap_x( &x );
 	    convert_xy( x, y, &tick, &sequence );

	    if ( m_mainperf->is_active( sequence )){

		bool state = m_mainperf->get_sequence( sequence )->get_trigger_state( tick );

		m_adding_pressed_state = ! state;

		m_mainperf->get_sequence( sequence )->add_trigger( tick, m_snap, ! state );
		draw_background_at( m_window, x, y );
		draw_sequence_on( m_window, sequence );
		draw_background_at( m_pixmap, x, y );
		draw_sequence_on( m_pixmap, sequence );


	    }
	}
    }
    

    /*     right mouse button      */
    if ( a_ev->button == 3 ){
 	set_adding( true );
    }

    return true;
    
}


bool
perfroll::on_button_release_event(GdkEventButton* a_ev)
{
    if ( a_ev->button == 1 ){
	m_adding_pressed = false;
    }

    if ( a_ev->button == 3 ){	
 	set_adding( false );
    }

    return true;
}

 

bool
perfroll::on_motion_notify_event(GdkEventMotion* a_ev)
{

    long tick;
    int sequence;
    
    int x = (int) a_ev->x;
    int y = (int) a_ev->y;
    
    if (  m_adding && m_adding_pressed ){
	    
	snap_x( &x );
	convert_xy( x, y, &tick, &sequence );

	if ( m_mainperf->is_active( sequence )){

	    m_mainperf->get_sequence( sequence )->add_trigger( tick, m_snap, m_adding_pressed_state );
	    draw_background_at( m_window, x, y );
	    draw_sequence_on( m_window, sequence );
	    draw_background_at( m_pixmap, x, y );
	    draw_sequence_on( m_pixmap, sequence );
	}
    }

    return true;
}




/* performs a 'snap' on x */
void 
perfroll::snap_x( int *a_x )
{
    // snap = number pulses to snap to
    // m_scale = number of pulses per pixel
    //	so snap / m_scale  = number pixels to snap to

    int mod = (m_snap / c_perf_scale_x );

    if ( mod <= 0 )
 	mod = 1;
    
    *a_x = *a_x - (*a_x % mod );
}



void 
perfroll::convert_xy( int a_x, int a_y, long *a_tick, int *a_seq)
{

    long tick_offset = m_4bar_offset * c_ppqn * 16;

    *a_tick = a_x * c_perf_scale_x;  
    *a_seq = a_y / c_names_y;  

    *a_tick += tick_offset;
    *a_seq  += m_sequence_offset;

    if ( *a_seq >= c_total_seqs )
	*a_seq = c_total_seqs - 1;

    if ( *a_seq < 0 )
	*a_seq = 0;
}


bool
perfroll::on_focus_in_event(GdkEventFocus*)
{
    set_flags(Gtk::HAS_FOCUS);
    return false;
}


bool
perfroll::on_focus_out_event(GdkEventFocus*)
{
    unset_flags(Gtk::HAS_FOCUS);
    return false;
}


void
perfroll::on_size_allocate(GtkAllocation* a_r )
{
    Gtk::DrawingArea::on_size_allocate( a_r );

    m_window_x = a_r->width;
    m_window_y = a_r->height;

    update_sizes();
}

void 
perfroll::on_size_request(GtkRequisition* a_r )
{
}

