#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <sys/types.h>
#include <malloc.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include "nntp.h"
#include "tcp.h"
#include "config.h"
#include "postit.h"

#define TIMEOUT 300

char verbosity=2;		/* verbosity :), the higher the more messages */

char queuefile[STRING];
char fqdn[STRING];
char queuefilename[LONGSTRING];
char workfilename[LONGSTRING];
char failfilename[LONGSTRING];

int sd;

int articles_processed;
int articles_posted;
int articles_failed;


void strscpy(char *t, char *s)
{
  while (*s&&!isspace(*s)) *t++=*s++;
  *t=0;
}

void spacestrip(char *s)
{
  while (*s&&!isspace(*s)) s++;
  *s=0;
}

void args(int argc, char **argv)
{
  int i,j,k;
  char *s,*t;

  k=0;
  for (i=1; i<argc; i++) {
    if (k||argv[i][0]!='-') {
      if (!k) strcpy(queuefile,argv[i]);
      else if (k==1) strcpy(fqdn,argv[i]);
      else goto display_help;
      k++;
      continue;
    }
    for (j=1; argv[i][j]; j++) {
      switch (argv[i][j]) {
      case 'v':
	if (isdigit(argv[i][j+1])) verbosity=argv[i][(j++)+1]-'0';
	break;
      case '?':
      display_help:
	printf("%s [-v] queuefile fqdn\n"
	       "Command line options:\n"
	       "  -v[#]   verbose mode (1=default)\n");
	exit(1);
      default:
	printf("ignoring unknown option %c\n",argv[i][j]);
      }
    }
  }
  return;
}

void prepare_queuefile()
{
  struct stat s;
  FILE *w,*f;
  char buf[VERYLONGSTRING];

  /* generate the pathnames */
  sprintf(queuefilename,"%s/%s/%s",SPOOLPATH,OUTGOING,queuefile);
  sprintf(workfilename,"%s%s",queuefilename,WORK);
  sprintf(failfilename,"%s%s",queuefilename,FAIL);
  /* see if there's a workfile; if yes, there's probably a postit job running already for this file */
  if (!stat(workfilename,&s)) {
    printf("%s exists!\nRemove this file if there's no other postit running for %s.\n",workfilename,queuefile);
    exit(1);
  }
  /* now do an atomic move */
  if (rename(queuefilename,workfilename)) {
    if (verbosity>=3) printf("Can't rename queuefile %s, probably nothing to do\n",queuefilename);
  }
  if (verbosity>=3) printf("Creating .work file\n");
  /* see if there's a failfile left over; if yes, add its contents to the workfile */
  f=fopen(failfilename,"r");
  if (f) {
    w=fopen(workfilename,"a");
    if (!w) {
      printf("can't append to %s, exiting\n",workfilename);
      exit(1);
    }
    if (verbosity>=2) printf("Found a .fail file, re-queueing articles\n");
    while (fgets(buf,sizeof(buf),f)) fputs(buf,w);
    fclose(f);
    fclose(w);
    unlink(failfilename);
  }
}


void connect_server()
{
  if (verbosity>=2) printf("Opening connection to %s\n",fqdn);
  sd=nntp_open_connection(fqdn,119);
  if (sd<0) {
    printf("can't connect to %s, exiting\n",fqdn);
    rename(workfilename,failfilename);
    exit(1);
  }
  /* OK, we're connected.  Now switch to reader mode */
  if (verbosity>=3) printf("Switching to reader mode, checking if we are allowed to post\n");
  if (nntp_send_command(sd,"MODE READER")!=200) {
    printf("we're not allowed to post on %s, exiting\n",fqdn);
    nntp_close_connection(sd);
    rename(workfilename,failfilename);
    exit(1);
  }
  if (verbosity>=3) printf("Server welcomed us, posting OK\n");
}


int send_article(char *artname, char *messageid)
{
  char buf[VERYLONGSTRING];
  FILE *a; /* article file */
  FILE *t; /* temporary file */

  /* check if the server will accept this message ID as a new article */
  sprintf(buf,"STAT %s",messageid);
  if (nntp_send_command(sd,buf)!=430) {
    if (verbosity>=1) printf("Already have %s\n",messageid);
    return 1;
  }
  if (verbosity>=3) printf("Server will likely accept our article, let's prepare it\n");
  t=tmpfile();
  if (!t) {
    printf("fatal error: can't create tempfile\n");
    exit(1);
  }
  sprintf(buf,"%s/%s",SPOOLPATH,artname);
  a=fopen(buf,"r");
  if (!a) {
    printf("can't read article file %s\n",buf);
    return -1;
  }
  /* copy header, weed out several header lines that we are not
     allowed to send */
  while (fgets(buf,sizeof(buf),a)) {
    if (!strncasecmp(buf,"NNTP-Posting",12)) continue;
    if (!strncasecmp(buf,"Sender:",7)) continue;
    if (!strncasecmp(buf,"Xref:",5)) continue;
    /* add more here if necessary */
    fputs(buf,t);
    if (*buf=='\n') break;
  }
  /* now copy the body, paying special attention on lines starting with '.' (compare RfC977) */
  while (fgets(buf,sizeof(buf),a)) {
    if (*buf=='.') fprintf(t,".%s",buf);
    else fputs(buf,t);
  }
  /* close article file */
  fclose(a);
  /* rewind tempfile */
  rewind(t);
  if (verbosity>=3) printf("Offering article\n");
  if (nntp_send_command(sd,"POST")!=340) {
    printf("can't post articles\n");
    fclose(t);
    return -1;
  }
  /* now send the article */
  nntp_send_stream(sd,t);
  fclose(t);
  /* send a dot on a line by itself */
  if (nntp_end_of_article(sd)!=240) {
    if (verbosity>=1) printf("article %s could not be posted\n",messageid);
    return -1;
  }
  if (verbosity>=1) printf("article %s posted\n",messageid);
  return 0;
}


void process_queuefile()
{
  char line[VERYLONGSTRING];
  FILE *w, *f;
  char artfilename[STRING];
  char messageid[LONGSTRING];
  char *c;
  struct stat s;
  int x;

  w=fopen(workfilename,"r");
  if (!w) {
    printf("can't open %s, exiting\n",workfilename);
  outtahere:
    nntp_close_connection(sd);
    rename(workfilename,failfilename);
    exit(1);
  }
  f=fopen(failfilename,"a");
  if (!f) {
    printf("can't open %s, exiting\n",failfilename);
    goto outtahere;
  }
  while (fgets(line,sizeof(line),w)) {
    strscpy(artfilename,line);
    c=line;
    while (!isspace(*c)&&*c) c++;
    while (*c&&isspace(*c)) c++;
    strscpy(messageid,c);
    articles_processed++;
    if ((x=send_article(artfilename,messageid))<0) {
      fputs(line,f);
      articles_failed++;
    }
    else if (!x) articles_posted++;
  }
  fclose(w);
  unlink(workfilename);
  fclose(f);
  /* if the failfile is empty, we unlink it */
  if (!stat(failfilename,&s)) {
    if (!s.st_size) unlink(failfilename);
  }
}


void disconnect_server()
{
  nntp_close_connection(sd);
}


void show_statistics()
{
  if (verbosity<1) return;
  printf("Articles processed: %5d\n",articles_processed);
  printf("Articles posted:    %5d\n",articles_posted);
  printf("Articles failed:    %5d\n",articles_failed);
}


int main(int argc, char **argv)
{
  args(argc,argv);
  articles_processed=articles_posted=articles_failed=0;
  prepare_queuefile();
  connect_server();
  process_queuefile();
  disconnect_server();
  show_statistics();
  return 0;
}
