// -*- C++ -*- 
/*
** Copyright (C) 2000 Alan McIvor <alan@mcivor.gen.nz>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifndef PROJECT_H
#define PROJECT_H

#include <cstdio>
#include <string>
#include <time.h>

#ifdef __EMX__
#define strcasecmp(s1,s2) stricmp((s1),(s2))
#define strncasecmp(s1,s2,l) strnicmp((s1),(s2),(l))
#endif

#include "task.h"
#include "resource.h"
#include "milestone.h"
#include "projectExceptions.h"

/* Constants */

/* these are used to load different start and finish times from the file */
#define TNORM 0
#define TACTUAL 1
#define TBASE 2

#ifdef VARIABLE
#  undef VARIABLE
#endif

typedef struct dayname
{
    time_t t;
    char *s;
    int  num;
}
DAYNAME;

class Project
{

public:

    typedef TASK::PTRLIST::const_iterator TPLCI;
    typedef RESOURCE::PTRLIST::const_iterator RPLCI;
    typedef MILESTONE::PTRLIST::const_iterator MPLCI;
    typedef ITEM::PTRLIST::const_iterator IPLCI;

    Project();
    ~Project();

    void checkComplete();
    void DoScheduling();
    void PrintFile();

    int ResListPosition(const char *id);	// for qsort
    RESOURCE *FindResource(const char *id);
    void AddResource(char *id, char *name);
    void SetEfficiency(char *id, double d);
    TASK *FindTask(const char *id);
    void AddTask(TASK *task);
    void AddTask(char *id, char *name, int duration);
    int FindDay(char *name);
    void Vacation(char *resid, int d1, int d2);
    void AddCandidate(const char *taskid, const char *resid);
    void BookResource( TASK *t, RESOURCE *parent, RESOURCE *r,
		       int tstart, int tfinish, TimeBlock::Type type );
    void WorkBlock(const char *taskid, const char *resid,
		   int d1, int d2, TimeBlock::Type type);
    void WorkBlock( TASK *t, RESOURCE * r, 
		    int d1, int d2, TimeBlock::Type type);
    void AddMilestone(char *id, char *name);
    MILESTONE * FindMilestone(char *id);
    void TaskNote(char *taskid, char *text);
    void ResourceNote(char *resid, char *text);
    void Blockify(char *id);
    void AddDescription(char *id, char *desc);
    void AddGroup(char *id, char **members, int Nmembers);
    void AddDependencies(char *taskid, char **tasks, int Ntasks);
    void AddAfter(char *msid, char **tasks, int Ntasks);
    void SetCompletion(char *task, char *complete);
    void StartTask(char *taskid, char *dayname, int type);
    void FinishTask(char *taskid, char *dayname, int type);
    void StartTask(TASK *t, int  d, int type);
    void FinishTask(TASK *t, int d, int type);
    void DaysDone(char *taskid, int days);
    void SetResourceRate(char *resid, char *camount);
    void AddTaskItem(char *taskid, char *camount, char *desc);
    void SetProjectRate(char *camount);
    void AddProjectItem(char *camount, char *desc);
    void SetStartTime(const char *year, const char *month, const char *mday);
    void SetStartTime(time_t when);
    void SetFinishDate(char *dayname);
    void AddNetworkX(char *taskid, char *xstr);
    void AddNetworkY(char *taskid, char *ystr);
    void AddStartNetworkY(char *ystr);
    void AddFinishNetworkY(char *ystr);
    void SetDateFormat( char * format );
    void SortTasks(bool tg_sortbyresource);

    char *sStartDay() { return days[0].s; };
    char *sFinishDay() { return days[finishDay].s; };
    int  nTasks() { return mTaskList.size(); };
    TASK *sortedTask(int i) { return mSortedTaskList[i]; };
    char *sDays(int i) { return days[i].s; };
    void SortMilestones();
    int nMilestones() { return mMilestoneList.size(); };
    MILESTONE *sortedMilestone(int i);
    TPLCI beginTaskList() { return mTaskList.begin(); };
    TPLCI endTaskList() { return mTaskList.end(); };
    int nDays(int i) { return days[i].num; };
    int tDays(int i) { return days[i].t; };
    int networkStartY() { return pc_start_ny; };
    int networkFinishY() { return pc_finish_ny; };

    MPLCI beginMilestoneList() { return mMilestoneList.begin(); };
    MPLCI endMilestoneList() { return mMilestoneList.end(); };

    RPLCI beginResourceList() { return mResourceList.begin(); };
    RPLCI endResourceList() { return mResourceList.end(); };

    double itemCost();
    int numItems() { return mItems.size(); };
    IPLCI begin_items() { return mItems.begin(); };
    IPLCI end_items() { return mItems.end(); };


    // utils
    int ResourceIsUsedForTask( RESOURCE *r, TASK *t, int dayNo );
    char *dayNames[8];
    void Error(char *fmt, ...);

# define VARIABLE(AA,BB,CC) AA BB;
#include "globals.i"

private:

    DAYNAME  days[MAX_TIME];

    TASK::PTRLIST mTaskList;
    TASK::PTRLIST mSortedTaskList;
    MILESTONE::PTRLIST mMilestoneList;
    MILESTONE::PTRLIST mSortedMilestoneList;
    RESOURCE::PTRLIST mResourceList;

    void LoadDays();

    void ReverseBookResource( TASK *t, RESOURCE *parent, RESOURCE *r,
			      int tstart, int tfinish, TimeBlock::Type type );
    void AssignResource(TASK *t, RESOURCE *r, int tstart);
    int do_match(const char *str, const char *regexp, int case_sig);

    int FirstFreeTime(TASK *task, RESOURCE *r, int earliest, int duration);
    void ReverseAssignResource(TASK *t, RESOURCE *r, int tfinish);
    RESOURCE *FindEarliestResource(TASK *t, int t_earliest);
    void AssignTask(TASK *t);
    void ReverseAssignTask(TASK *t);
    void PredictOverruns();
    void ReverseScheduleTasks();
    void ScheduleMilestones();
    void ScheduleTasks();
    void printTaskList();

    char *monthNames[13];

    void Debug(char *fmt, ...);
    void Warning(char *fmt, ...);

    vector<ITEM *> mItems;

};


#endif


