// smartmem.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#include "application.h"
#include "localdefs.h"
#include "smartmem.h"
#ifdef NeXT
#include <m68k/math.h>
#elif defined (__ppc__)
#include <limits.h>
#else
#include <values.h>
#endif

#include <sys/mman.h>

#undef USE_MMAP

size_t SmartMemory::totalBytes_Allocated = 0;
size_t SmartMemory::max_TotalAllocation = size_t(5.0e+08);	// 500 Mb.
size_t SmartMemory::max_SingleAllocation = size_t(1.0e+08);	// 100 Mb.

#ifdef USE_MMAP
struct MapTable {
	MapTable(int fd, void *addr) : fdesc(fd), address(addr), 
								   prev(NULL), next(NULL) {}
	void remove() { prev->next = next;  next->prev = prev; }
	int fdesc;
	void *address;
	struct MapTable *prev;
	struct MapTable *next;
};

MapTable *s_MapTable = NULL;

MapTable *
GetTableForAddress(void *addr) {
	MapTable *table = s_MapTable;
	while (table != NULL) {
		if (table->address == addr)
			return table;
		table = table->next;
	}
	return NULL;
}

void *
GetNewMappedAddress(int length) {
	void *memory = NULL;
}

#endif

void
SmartMemory::setMaxTotalAllocation(size_t maxTot) {
	max_TotalAllocation = maxTot;
}

void
SmartMemory::setMaxSingleAllocation(size_t maxSingle) {
	max_SingleAllocation = maxSingle;
}

void *
SmartMemory::allocate(size_t* size) {
	void* mem = 0;
	if(isLegalAllocation(*size)) {
#ifdef USE_MMAP
		
#else
#ifdef USE_MALLOC
		// This is extremely slow for large allocations due to cache hits
		if((mem = ::malloc(*size)) != 0) {
			zero(mem, *size);
			totalBytes_Allocated += *size;
		}
#else
		if((mem = ::calloc(*size, 1)) != 0) {
			totalBytes_Allocated += *size;
		}
#endif
		else {
			Application::error("Memory allocation failure:");
			*size = 0;
		}
#endif
	}
	else *size = 0;
	return mem;
}

void
SmartMemory::free(void* ptr, size_t size) {
	if(ptr) {
		::free(ptr);
		totalBytes_Allocated -= size;
	}
}

void*
SmartMemory::changeAllocation(void* ptr, size_t oldSize, size_t* size) {
	void* oldMem = ptr;
	size_t newSize = *size;
	int delta = 0;
	if (newSize == 0) {
		Application::alert("Illegal request for zero bytes");
		*size = oldSize;
		return ptr;
	}
	else if(newSize < oldSize)					// deallocating
		delta = int(newSize) - int(oldSize);
	else {								// allocating
		delta = int(newSize - oldSize);
		if(delta < 0) delta = INT_MAX;	// check for int overflow
	}
	if(delta < 0 || isLegalAllocation(delta)) {
		if((ptr = ::realloc(ptr, newSize)) != 0) {
			totalBytes_Allocated += delta;
			if(delta > 0)
				zero(addr(ptr) + oldSize, delta);	// zero out extended space
		}
		else {
			Application::error("Memory reallocation failure:");
			ptr = oldMem;
			*size = oldSize;
		}
	}
	else *size = oldSize;
	return ptr;
}

void*
SmartMemory::zero(void* ptr, size_t size, int zero) {
	if (ptr) {
#ifdef USE_MEMSET
		::memset(ptr, zero, size);
#else
		char *cptr = (char *) ptr;
		const char czero = (char) zero;
		size_t bsize = size / 8;
		size_t remainder = size - (bsize * 8);
		for (int n = 0; n < bsize; ++n, cptr += 8) {
			cptr[0] = czero;
			cptr[1] = czero;
			cptr[2] = czero;
			cptr[3] = czero;
			cptr[4] = czero;
			cptr[5] = czero;
			cptr[6] = czero;
			cptr[7] = czero;
		}
		for (int nn = 0; nn < remainder; ++nn)
			*cptr++ = zero;
#endif
	}
	return ptr;
}

void*
SmartMemory::copy(void* from, void* to, size_t size) {
	if(from != nil && to != nil)
		::memmove(to, from, size);
	return to;
}

int
SmartMemory::isLegalAllocation(size_t size) {
	int status = false;
	char msg[120];
	const float million = 1.0e+06;
	if (size > maxSingleAllocation()) {
		sprintf(msg, "Request for %.2fmB exceeds single allocation limit (%.2fmB).",
			size/million, maxSingleAllocation()/million);
		status = Application::confirm("Memory limit alert:",
			msg, "Attempt anyway (and possibly extend total limit)?");
		if (status == true && size >  maxTotalAllocation())
			setMaxTotalAllocation(size);
	}
	else if (size == 0) {
		Application::alert("Illegal request for zero bytes or",
						   "amount exceeding system limits");
		status = false;
	}
	else if (totalBytes_Allocated + size > maxTotalAllocation()) {
		sprintf(msg, "Request for additional %.1fkB puts system beyond total allocation limit (%.2fmB).",
			size/1000.0, maxTotalAllocation()/million);
		status = Application::confirm("Memory limit alert:",
			msg, "Attempt anyway (and extend total limit)?");
		if (status == true)
			setMaxTotalAllocation(totalBytes_Allocated + size);
	}
	else status = true;	
	return status;
}

