/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */
package mx4j.tools.remote.soap;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Map;

import javax.management.MBeanServerConnection;
import javax.management.remote.JMXServiceURL;
import javax.security.auth.Subject;

import mx4j.remote.ConnectionResolver;
import mx4j.remote.HeartBeat;
import mx4j.remote.RemoteNotificationClientHandler;
import mx4j.tools.remote.AbstractJMXConnector;

/**
 * @author <a href="mailto:a_taherkordi@users.sourceforge.net">Alireza Taherkordi</a>
 * @version $Revision: 1.16 $
 */
public class SOAPConnector extends AbstractJMXConnector
{
   public static final String SOAP_TRANSPORT_PROTOCOL = "jmx.remote.x.soap.transport.protocol";

   private transient ClientSOAPConnection connection;
   private transient RemoteNotificationClientHandler notificationHandler;
   private transient HeartBeat heartbeat;

   public SOAPConnector(JMXServiceURL url, Map environment) throws IOException
   {
      super(url);
   }

   protected void doConnect(Map environment) throws IOException, SecurityException
   {
      JMXServiceURL address = getAddress();
      String protocol = address.getProtocol();
      ConnectionResolver resolver = ConnectionResolver.newConnectionResolver(protocol, environment);
      if (resolver == null) throw new MalformedURLException("Unsupported protocol: " + protocol);

      connection = (ClientSOAPConnection)resolver.lookupClient(address, environment);
      connection = (ClientSOAPConnection)resolver.bindClient(connection, environment);

      Object credentials = environment == null ? null : environment.get(CREDENTIALS);
      connection.connect(credentials);

      this.heartbeat = new SOAPHeartBeat(connection, getConnectionNotificationEmitter(), environment);
      this.notificationHandler = new SOAPRemoteNotificationClientHandler(connection, getConnectionNotificationEmitter(), heartbeat, environment);

      this.heartbeat.start();
      this.notificationHandler.start();
   }

   protected void doClose() throws IOException
   {
      if (notificationHandler != null) notificationHandler.stop();
      if (heartbeat != null) heartbeat.stop();
      if (connection != null) connection.close();
   }

   protected MBeanServerConnection doGetMBeanServerConnection(Subject delegate) throws IOException
   {
      SOAPConnection catcher = ClientExceptionCatcher.newInstance(connection);
      return new ClientInvoker(catcher, notificationHandler, delegate);
   }

   public String getConnectionId() throws IOException
   {
      return connection.getConnectionId();
   }
}
